/*
 *  Copyright (C) 1999 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <stdlib.h>
#include "gtm.h"
#include "file-data.h"
#include "file-list.h"
#include "dialogs.h"
#include "download-info.h"
#include "main-window.h"
#include "finish.xpm"
#include "run.xpm"
#include "stop.xpm"

/* Timeout for auto-scroll on drag */
#define SCROLL_TIMEOUT 100

/* Name of the columns of file list */
#define FILE_LIST_TOTAL_COLUMNS 4
#define FILE_LIST_COL_STATE 0
#define FILE_LIST_COL_RETRIEVED 1
#define FILE_LIST_COL_REMAIN 2
#define FILE_LIST_COL_FILENAME 3

/* The types of stuff we accept on dnd */
enum {
    TARGET_CLIST = 0,
    TARGET_URL,
    TARGET_NETSCAPE_URL
};

/* Structure used to hold the auto-scroll on drag data */
typedef struct 
{
    GtkWidget *list;
    GtkWidget *sw;
    gint timer_id;
    gint drag_motion_x;
    gint drag_motion_y;
} Panel;

/* File list popup menu */
static GnomeUIInfo menu_popup[] = {
#define MENU_POPUP_REMOVE 0
    GNOMEUIINFO_ITEM_STOCK (N_ ("Remove"), NULL, remove_cmd,
			    GNOME_STOCK_PIXMAP_CLOSE),
    GNOMEUIINFO_SEPARATOR,
#define MENU_POPUP_START 2
    GNOMEUIINFO_ITEM_STOCK (N_ ("Start"), NULL, start_cmd,
			    GNOME_STOCK_PIXMAP_EXEC),
#define MENU_POPUP_STOP 3
    GNOMEUIINFO_ITEM_STOCK (N_ ("Stop"), NULL, stop_cmd, 
			    GNOME_STOCK_PIXMAP_STOP),
#define MENU_POPUP_RESTART 4
    GNOMEUIINFO_ITEM_STOCK (N_ ("Restart"), NULL, restart_cmd, 
			    GNOME_STOCK_PIXMAP_REFRESH),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_PROPERTIES_ITEM (properties_cmd, NULL),
    GNOMEUIINFO_END
};

/* Function to update the selected items of the popup menu of the file list */
void
file_list_menu_popup_set_state (GtkCList *file_list)
{
    FileData *file_data;

    if (file_list->selection == NULL) {
        /* No file selected */
        gtk_widget_set_sensitive (
            menu_popup[MENU_POPUP_REMOVE].widget, FALSE);
        gtk_widget_set_sensitive (
            menu_popup[MENU_POPUP_START].widget, FALSE);
        gtk_widget_set_sensitive (
            menu_popup[MENU_POPUP_STOP].widget, FALSE);
        gtk_widget_set_sensitive (
            menu_popup[MENU_POPUP_RESTART].widget, FALSE);
    } else if (file_list->selection->next == NULL) {
        /* One file selected */
        gtk_widget_set_sensitive (
            menu_popup[MENU_POPUP_REMOVE].widget, TRUE);
        file_data = gtk_clist_get_row_data (file_list, 
                                            (gint) file_list->selection->data);
        if (file_data->state == DL_NOT_STARTED
            || file_data->state == DL_NOT_RUNNING)
            gtk_widget_set_sensitive (
                menu_popup[MENU_POPUP_START].widget, TRUE);
        else
            gtk_widget_set_sensitive (
                menu_popup[MENU_POPUP_START].widget, FALSE);
        if (file_data->state != DL_NOT_STARTED
            && file_data->state != DL_NOT_RUNNING
            && file_data->state != DL_COMPLETED)
            gtk_widget_set_sensitive (
                menu_popup[MENU_POPUP_STOP].widget, TRUE);
        else
            gtk_widget_set_sensitive (
                menu_popup[MENU_POPUP_STOP].widget, FALSE);
        if (file_data->state == DL_NOT_RUNNING
            || file_data->state == DL_COMPLETED)
            gtk_widget_set_sensitive (
                menu_popup[MENU_POPUP_RESTART].widget, TRUE);
        else
            gtk_widget_set_sensitive (
                menu_popup[MENU_POPUP_RESTART].widget, FALSE);
    } else {
        /* More than one file selected */
        gtk_widget_set_sensitive (
            menu_popup[MENU_POPUP_REMOVE].widget, TRUE);
        gtk_widget_set_sensitive (
            menu_popup[MENU_POPUP_START].widget, TRUE);
        gtk_widget_set_sensitive (
            menu_popup[MENU_POPUP_STOP].widget, TRUE);
        gtk_widget_set_sensitive (
            menu_popup[MENU_POPUP_RESTART].widget, TRUE);
    }
}

/* Callbacks functions */
static int
file_list_event (GtkCList *file_list, GdkEvent *event, GtkMenu *popup)
{
    gint row, column;
    static gint row_pressed = -1;
    GdkEventButton *button_event = (GdkEventButton *) event;
	
    if (event->type == GDK_BUTTON_PRESS && button_event->button == 3) {
        /* Right mouse button pressed */
        file_list_menu_popup_set_state (file_list);
        gtk_menu_popup (popup, NULL, NULL, NULL, NULL, 3, 
                        button_event->time);
        return TRUE;
    } else if (event->type == GDK_BUTTON_PRESS
               && button_event->button == 1) {
        /* Left mouse button pressed */
        if (gtk_clist_get_selection_info (file_list, button_event->x,
                                          button_event->y, &row, &column) == 0)
            row_pressed = -1;
        else
            row_pressed = row;
    } else if (event->type == GDK_BUTTON_RELEASE
               && button_event->button == 1) {
        /* Left mouse button released */
        FileData *file_data;

        if (gtk_clist_get_selection_info (file_list, button_event->x,
                                          button_event->y, &row, &column) == 0
            && row != row_pressed)
            return FALSE;

        file_data =
            (FileData *) gtk_clist_get_row_data (file_list,
                                                 file_list->focus_row);
        download_info_update (file_data);
        dialog_prop (FALSE, file_data);
    }
    return FALSE;
}

static void 
file_list_select_row (GtkCList *file_list, gint row, gint column, 
		      GdkEventButton *event, gpointer data)
{
    FileData *file_data;
    
    file_data = (FileData *) gtk_clist_get_row_data (file_list,
                                                     file_list->focus_row);
    menu_toolbar_set_state (file_list);
    file_list_menu_popup_set_state (file_list);
    download_info_update (file_data);
    dialog_prop (FALSE, file_data);
}

static void 
file_list_unselect_row (GtkCList *file_list, gint row, gint column, 
                        GdkEventButton *event, gpointer data)
{
    menu_toolbar_set_state (file_list);
    file_list_menu_popup_set_state (file_list);
    if (g_list_find (file_list->selection,
                     (gpointer) file_list->focus_row) == NULL) {
        download_info_update (NULL);
        dialog_prop (FALSE, NULL);
    }
}


/* Functions used to construct the auto-scroll on drag */
static GtkAdjustment *
file_list_get_vadjustment (Panel *panel)
{
    GtkRange *vsb = GTK_RANGE (GTK_SCROLLED_WINDOW (panel->sw)->vscrollbar);
    GtkAdjustment *va = vsb->adjustment;
    
    return va;
}

static gboolean
file_list_scrolling_is_desirable (Panel *panel, int x, int y)
{
    GtkAdjustment *va;
    
    va = file_list_get_vadjustment (panel);
    if (y < 10) {
        if (va->value > va->lower)
            return TRUE;
    } else {
        if (y > GTK_CLIST (panel->list)->clist_window_height - 10) {
            if (va->value < va->upper - va->page_size)
                return TRUE;
        }
    }
    
    return FALSE;
}

static gboolean
file_list_scroll (gpointer data)
{
    Panel *panel = data;
    GtkAdjustment *va;
    gdouble v;

    va = file_list_get_vadjustment (panel);

    if (panel->drag_motion_y < 10) {
        v = va->value - va->step_increment;
        if (v < va->lower)
            v = va->lower;
    } else {
        v = va->value + va->step_increment;
        if (v > va->upper - va->page_size)
            v = va->upper - va->page_size;
    }
    gtk_adjustment_set_value (va, v);

    return TRUE;
}

static void
file_list_cancel_drag_scroll (Panel *panel)
{
    if (panel->timer_id != -1) {
        gtk_timeout_remove (panel->timer_id);
        panel->timer_id = -1;
    }
}

static void
file_list_setup_drag_scroll (Panel *panel, int x, int y)
{
    file_list_cancel_drag_scroll (panel);
    
    panel->drag_motion_x = x;
    panel->drag_motion_y = y;

    if (file_list_scrolling_is_desirable (panel, x, y))
        panel->timer_id = gtk_timeout_add (SCROLL_TIMEOUT, file_list_scroll,
                                           panel);
}

static gboolean
file_list_drag_motion (GtkCList *file_list, GdkDragContext *context, gint x, 
                       gint y, guint time, gpointer data)
{
    Panel *panel = data;

    /* Normalize the y coordinate to the file list */
    y -= (GTK_CONTAINER (file_list)->border_width
          + file_list->column_title_area.y
          + file_list->column_title_area.height);
    if (y < 0){
        return TRUE;
    }

    /* Set up auto-scrolling */
    file_list_setup_drag_scroll (panel, x, y);
    return TRUE;
}

static void
file_list_drag_leave (GtkCList *file_list, GdkDragContext *context,
                      guint time, gpointer data)
{
    file_list_cancel_drag_scroll ((Panel *) data);
}

static void
file_list_drag_begin (GtkCList *file_list, GdkDragContext *context,
                      gpointer data)
{
}

static void
file_list_drag_end (GtkCList *file_list, GdkDragContext *context,
                    gpointer data)
{
    file_list_cancel_drag_scroll ((Panel *) data);
}

static void
file_list_drag_data_received (GtkWidget *widget, GdkDragContext *context,
                              gint x, gint y, GtkSelectionData *selection_data,
                              guint info, guint time)
{
    g_return_if_fail (widget != NULL);

    switch (info) {
        case TARGET_URL:
	case TARGET_NETSCAPE_URL:
            dialog_add (selection_data->data);
	    break;
    }
}

/* Function to create a file list */
GtkWidget *
file_list_create (void)
{
    GtkWidget *file_list, *file_list_sw;
    Panel *panel;
    int i;
    static GtkWidget *popup = NULL;
    gchar *titles[] = { 
        N_ ("S"), 
        N_ ("%"),
        N_ ("Remain"),
        N_ ("Filename")
    };
    gchar *t_titles[FILE_LIST_TOTAL_COLUMNS];
    /* FIXME: To avoid building a new widget I add here the target for the
     * reorder stuff using dnd on the list. */
    static GtkTargetEntry target_table[] = {
        { "gtk-clist-drag-reorder", 0, TARGET_CLIST },
        { "text/uri-list", 0, TARGET_URL },
        { "text/plain", 0, TARGET_URL },
        { "x-url/http", 0, TARGET_NETSCAPE_URL },
        { "x-url/ftp", 0, TARGET_NETSCAPE_URL },
        { "_NETSCAPE_URL", 0, TARGET_NETSCAPE_URL }
    };

    /* This panel data is used for drag auto-scroll */
    panel = g_new (Panel, 1);

    /* Create the clist and set it's attributes */
    for (i = 0; i < FILE_LIST_TOTAL_COLUMNS; i++)
        t_titles[i] = _ (titles[i]);
    file_list = gtk_clist_new_with_titles (4, t_titles);
    gtk_clist_set_reorderable (GTK_CLIST (file_list), TRUE);
    gtk_clist_set_use_drag_icons (GTK_CLIST (file_list), FALSE);
    gtk_clist_column_titles_passive (GTK_CLIST (file_list));
    gtk_clist_set_column_auto_resize (GTK_CLIST (file_list),
                                      FILE_LIST_COL_STATE, TRUE);
    gtk_clist_set_column_auto_resize (GTK_CLIST (file_list), 
                                      FILE_LIST_COL_RETRIEVED, TRUE);
    gtk_clist_set_column_auto_resize (GTK_CLIST (file_list), 
                                      FILE_LIST_COL_REMAIN, TRUE);
    gtk_clist_set_column_auto_resize (GTK_CLIST (file_list), 
                                      FILE_LIST_COL_FILENAME, TRUE);
    gtk_clist_set_shadow_type (GTK_CLIST (file_list), GTK_SHADOW_IN);
    gtk_clist_set_selection_mode (GTK_CLIST (file_list),
                                  GTK_SELECTION_EXTENDED);
    gtk_widget_show (file_list);
    panel->list = file_list;

    /* Create the scrolled window and add the file list */
    file_list_sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_container_add (GTK_CONTAINER (file_list_sw), GTK_WIDGET (file_list));
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (file_list_sw),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
    panel->sw = file_list_sw;
    panel->timer_id = -1;

    /* Create the popup menu, if not created already, for the file list */
    if (popup == NULL)
	popup = gnome_popup_menu_new (menu_popup);

    /* Handle the important signals of the file list */
    gtk_signal_connect (GTK_OBJECT (file_list), "event",
                        GTK_SIGNAL_FUNC (file_list_event),
                        popup);
    gtk_signal_connect (GTK_OBJECT (file_list), "select_row", 
                        GTK_SIGNAL_FUNC (file_list_select_row),
                        NULL);
    gtk_signal_connect (GTK_OBJECT (file_list), "unselect_row", 
                        GTK_SIGNAL_FUNC (file_list_unselect_row),
                        NULL);

    /* These are for auto-scroll on drag */
    gtk_signal_connect (GTK_OBJECT (file_list), "drag_motion",
                        GTK_SIGNAL_FUNC (file_list_drag_motion), panel);
    gtk_signal_connect (GTK_OBJECT (file_list), "drag_leave",
                        GTK_SIGNAL_FUNC (file_list_drag_leave), panel);
    gtk_signal_connect (GTK_OBJECT (file_list), "drag_begin",
                        GTK_SIGNAL_FUNC (file_list_drag_begin), panel);
    gtk_signal_connect (GTK_OBJECT (file_list), "drag_end",
                        GTK_SIGNAL_FUNC (file_list_drag_end), panel);

    /* Set the drag 'n drop stuff */
    /* FIXME: To avoid building a new widget I unset the drag_dest (was set
     * when I used gtk_clist_set_reorderable) and then set it again with the
     * gtk-clist reorder stuff */
    gtk_drag_dest_unset (file_list);
    gtk_drag_dest_set (file_list, GTK_DEST_DEFAULT_MOTION 
		       | GTK_DEST_DEFAULT_DROP,
                       target_table, 
		       sizeof (target_table) / sizeof (target_table[0]),
                       GDK_ACTION_COPY | GDK_ACTION_MOVE);
    gtk_signal_connect (GTK_OBJECT (file_list), "drag_data_received",
                        GTK_SIGNAL_FUNC(file_list_drag_data_received), NULL);

    return file_list_sw;
}

void
file_list_set_row_state (GtkCList *file_list, gint row, DlState state)
{
    static GdkPixmap *pm_stop = NULL, *pm_run = NULL, *pm_finish = NULL;
    static GdkBitmap *pm_stop_mask = NULL, *pm_run_mask = NULL,
        *pm_finish_mask = NULL;

    if (pm_stop == NULL) {
        gdk_imlib_data_to_pixmap (finish_xpm, &pm_finish, &pm_finish_mask);
        gdk_imlib_data_to_pixmap (run_xpm, &pm_run, &pm_run_mask);
        gdk_imlib_data_to_pixmap (stop_xpm, &pm_stop, &pm_stop_mask);
    }
    switch (state) {
        case DL_NOT_STARTED:
        case DL_NOT_RUNNING:
            gtk_clist_set_pixmap (GTK_CLIST (file_list), row,
                                  FILE_LIST_COL_STATE, pm_stop, pm_stop_mask);
            break;
        case DL_COMPLETED:
            gtk_clist_set_pixmap (GTK_CLIST (file_list), row, 
                                  FILE_LIST_COL_STATE, pm_finish,
                                  pm_finish_mask);
            break;
        default:
            gtk_clist_set_pixmap (GTK_CLIST (file_list), row,
                                  FILE_LIST_COL_STATE, pm_run,
                                  pm_run_mask);
            break;
    }
}

void
file_list_set_row_statistics (GtkCList *file_list, gint row, 
                              FileData *file_data)
{
    gchar buffer[20];
    guint32 retr_size;
    time_t estimated, remain_time;

    /* Update retrieved */
    if (file_data->state == DL_COMPLETED)
        strcpy (buffer, "100%");
    else if (file_data->total_size != 0)
        sprintf (buffer, "%d%%",
                 (gint) (((gfloat) file_data->cur_size / file_data->total_size)
                         * 100));
    else
        strcpy (buffer, "");
    gtk_clist_set_text (GTK_CLIST (file_list), row,
                        FILE_LIST_COL_RETRIEVED, buffer);

    /* The rest of the statistics is calculated so we show them only
     * if requested */
    if (!gtm_pref.show_list_stat) {
        strcpy (buffer, "");
        gtk_clist_set_text (GTK_CLIST (file_list), row, 
                            FILE_LIST_COL_REMAIN, buffer);
        return;
    }
    
    /* Update remain time */
    retr_size = file_data->cur_size - file_data->session_start_size;
    if (file_data->state != DL_NOT_STARTED
        && file_data->state != DL_NOT_RUNNING
        && file_data->state != DL_COMPLETED) {
        if (retr_size != 0)
            estimated =
                (((gfloat) (file_data->total_size
                            - file_data->session_start_size)
                  * file_data->session_elapsed) / retr_size)
                + ((gfloat) (file_data->total_time
                             - file_data->session_elapsed));
        else
            estimated = 0;
    } else {
        if (file_data->cur_size != 0 && file_data->state != DL_COMPLETED)
            estimated =
                ((gfloat) file_data->total_size * file_data->total_time)
                / file_data->cur_size;
        else
            estimated = 0;
    }
    if (estimated == 0)
        strcpy (buffer, "");
    else {
        remain_time = estimated - file_data->total_time;
        sprintf (buffer, "%02d:%02d:%02d", (gint)(remain_time / 3600), 
                 (gint)((remain_time % 3600) / 60), (gint)(remain_time % 60));
    }
    gtk_clist_set_text (GTK_CLIST (file_list), row,
                        FILE_LIST_COL_REMAIN, buffer);
}

gint
file_list_find_url (GtkCList *file_list, gchar *url)
{
    gint row;
    FileData *file_data;

    row = 0;
    while ((file_data = gtk_clist_get_row_data (GTK_CLIST (file_list),
                                                row)) != NULL) {
        if (strcmp (file_data->url, url) == 0)
            return row;
        row++;
    }
    return -1;
}

gboolean
file_list_add (GtkCList *file_list, FileData *file_data)
{
    gint row;
    gchar *data[FILE_LIST_TOTAL_COLUMNS];

    /* Check if the URL is not on the file list */
    if (file_list_find_url (file_list, file_data->url) != -1)
        return FALSE;
    else {
        file_data->file_list = GTK_WIDGET (file_list);
        data[FILE_LIST_COL_STATE] = NULL;
        data[FILE_LIST_COL_RETRIEVED] = NULL;
        data[FILE_LIST_COL_REMAIN] = NULL;
        data[FILE_LIST_COL_FILENAME] = file_data->filename;
        row = gtk_clist_append (GTK_CLIST (file_list), data);
        gtk_clist_set_row_data_full (GTK_CLIST (file_list), row, file_data,
                                     (GtkDestroyNotify) file_data_free);
        file_data_set_state (file_data, file_data->state);
        file_data_set_total_size (file_data, file_data->total_size);
        file_data_update_statistics (file_data);
        report_event ("add");
    }
    return TRUE;
}

void 
file_list_remove_selection (GtkCList *file_list)
{
    FileData *file_data;
    
    gtk_clist_freeze (GTK_CLIST (dl_file_list));
    while (GTK_CLIST (dl_file_list)->selection != NULL) {
        file_data = 
            (FileData *) gtk_clist_get_row_data (
                GTK_CLIST (dl_file_list), 
                (gint) GTK_CLIST (dl_file_list)->selection->data);
        file_list_remove_file (file_list, file_data);
    }
    gtk_clist_thaw (GTK_CLIST (dl_file_list));
}

void
file_list_remove_file (GtkCList *file_list, FileData *file_data)
{
    file_data_stop_download (file_data);
    gtk_clist_remove (file_list, 
                      gtk_clist_find_row_from_data (file_list, file_data));
}

static void
file_list_load_old_format_data (GtkCList *file_list, gint total_files)
{
    FileData *file_data;
    gchar *prefix, *filename, **files;
    gint i;

    /* Load in old format */
    for (i = 0; i < total_files; i++) {
        file_data = g_new (FileData, 1);
        prefix = g_strdup_printf ("/gtm/File%d/", i);
        gnome_config_push_prefix (prefix);
        file_data->id = next_file_id++;
        file_data->log_tag = -1;
        file_data->url = gnome_config_get_string ("url");
        file_data->dir = gnome_config_get_string ("dir");
        filename = &file_data->url[strlen (file_data->url)];
        while (*filename != '/' && filename != file_data->url)
            filename--;
        filename++;
        file_data->filename = g_strdup (filename);
        file_data->local_filename = g_strconcat (file_data->dir,
                                                 file_data->filename,
                                                 NULL);
        file_data->disable_proxy = 
            gnome_config_get_bool ("disable_proxy=FALSE");
        file_data->disable_auto_dl = 
            gnome_config_get_bool ("disable_auto_dl=FALSE");
        file_data->line = NULL;
        file_data->session_start_time = 0;
        file_data->session_start_size = 0;
        file_data->session_elapsed = 0;
        file_data->cur_size = 0;
        file_data->state = gnome_config_get_int ("state=0");
        file_data->total_size = gnome_config_get_int ("size=0");
        file_data->total_time = gnome_config_get_int ("time=0");
        file_data->file_list = NULL;
        gnome_config_pop_prefix ();
        g_free (prefix);
        file_list_add (file_list, file_data);
    }

    /* Delete the records in old format */
    gnome_config_clean_key ("/gtm/FileList/total_files");
    for (i = 0; i < total_files; i++) {
        prefix = g_strdup_printf ("/gtm/File%d/", i);
	gnome_config_clean_section (prefix);
        g_free (prefix);
    }

    /* Save data in new format */
    files = g_new0 (gchar *, file_list->rows);
    for (i = 0; i < file_list->rows; i++) {
	file_data = gtk_clist_get_row_data (file_list, i);
        files[i] = g_strdup_printf ("%d", file_data->id);
        prefix = g_strdup_printf ("/gtm/File%d/", file_data->id);
	gnome_config_push_prefix (prefix);
	gnome_config_set_string ("url", file_data->url);
	gnome_config_set_string ("dir", file_data->dir);
	gnome_config_set_bool ("disable_proxy", file_data->disable_proxy);
	gnome_config_set_bool ("disable_auto_dl",
                               file_data->disable_auto_dl);
	gnome_config_set_int ("state", file_data->state);
	gnome_config_set_int ("size", file_data->total_size);
	gnome_config_set_int ("time", file_data->total_time);
	gnome_config_pop_prefix ();
        g_free (prefix);
    }
    gnome_config_set_int ("/gtm/FileList/next_file_id", next_file_id);
    gnome_config_set_vector ("/gtm/FileList/file_ids", file_list->rows,
                             (const gchar **) files);
    for (i = 0; i < file_list->rows; i++)
        g_free (files[i]);
    g_free (files);
    gnome_config_sync ();
}

void
file_list_load_data (GtkCList *file_list)
{
    FileData *file_data;
    gint i, total_files;
    gchar *prefix, *filename;
    gchar **files;

    gtk_clist_freeze (file_list);
    if (gnome_config_has_section ("/gtm/FileList/")) {
	gnome_config_push_prefix ("/gtm/FileList/");
	if ((total_files = gnome_config_get_int ("total_files=0")) != 0) {
            /* Old list format */
            gnome_config_pop_prefix ();
            file_list_load_old_format_data (file_list, total_files);
        } else {
            /* New list format */
            next_file_id = gnome_config_get_int ("next_file_id=0");
            gnome_config_get_vector ("file_ids", &total_files, &files);
            gnome_config_pop_prefix ();
            for (i = 0; i < total_files; i++) {
                prefix = g_strdup_printf ("/gtm/File%s/", files[i]);
                if (gnome_config_has_section (prefix)) {
                    file_data = g_new (FileData, 1);
                    gnome_config_push_prefix (prefix);
                    file_data->id = atoi (files[i]);
                    file_data->log_tag = -1;
                    file_data->url = gnome_config_get_string ("url");
                    file_data->dir = gnome_config_get_string ("dir");
                    filename = &file_data->url[strlen (file_data->url)];
                    while (*filename != '/' && filename != file_data->url)
                        filename--;
                    filename++;
                    file_data->filename = g_strdup (filename);
                    file_data->local_filename =
                        g_strconcat (file_data->dir,
                                     file_data->filename,
                                     NULL);
                    file_data->disable_proxy = 
                        gnome_config_get_bool ("disable_proxy=FALSE");
                    file_data->disable_auto_dl = 
                        gnome_config_get_bool ("disable_auto_dl=FALSE");
                    file_data->line = NULL;
                    file_data->session_start_time = 0;
                    file_data->session_start_size = 0;
                    file_data->session_elapsed = 0;
                    file_data->cur_size = 0;
                    file_data->state = gnome_config_get_int ("state=0");
                    file_data->total_size = gnome_config_get_int ("size=0");
                    file_data->total_time = gnome_config_get_int ("time=0");
                    file_data->file_list = NULL;
                    gnome_config_pop_prefix ();
                    file_list_add (file_list, file_data);
                }
                g_free (prefix);
            }
            for (i = 0; i < total_files; i++)
                g_free (files[i]);
            g_free (files);
        }
    }
    gtk_clist_thaw (file_list);
}

void
file_list_func_on_selection (GtkCList *file_list,
                             void (*func) (FileData *file_data))
{
    GList *p;
    FileData *file_data;
    
    gtk_clist_freeze (file_list);
    p = file_list->selection;
    while (p != NULL) {
        file_data = (FileData *) gtk_clist_get_row_data (file_list, 
                                                         (gint) p->data);
        (*func) (file_data);
        p = p->next;
    }
    gtk_clist_thaw (file_list);
}

void
file_list_start_auto_downloads (GtkCList *file_list)
{
    gint row;
    FileData *file_data;

    /* check before to start to download */
    row = 0;
    while (num_of_download < gtm_pref.num_of_auto_download
           && (file_data = 
               gtk_clist_get_row_data (GTK_CLIST (file_list), row)) != NULL) {
        if (file_data->disable_auto_dl == FALSE)
            file_data_start_download (file_data);
        row++;
    }
}

void
file_list_select_all (GtkCList *file_list)
{
    /* This test is needed because I think there is a bug no GtkCList */
    if (file_list->rows > 0)
        gtk_clist_select_all (file_list);
}

void
file_list_unselect_all (GtkCList *file_list)
{
    /* This test is needed because I think there is a bug no GtkCList */
    if (file_list->rows > 0)
        gtk_clist_unselect_all (file_list);
}
