/* sync.h - Main header file for synchronizer.

   Copyright (C) 1998 Tom Tromey

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

#ifndef SYNC_H
#define SYNC_H

#include <libgnome/libgnome.h>
#include <libgnomeui/libgnomeui.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <limits.h>

#ifdef HAVE_DB_185_H
# include <db_185.h>
#else
# include <db.h>
#endif

#include "md5.h"

/* The name of the config option holding cached file size.  */
#define KEY_FILE_SIZE "/Synchronize/Transfers/FileSize"

/* The name of the config option holding the number of transfers.  */
#define KEY_TRANSFERS "/Synchronize/Transfers/Number"


/* The magic checksum value indicating that the checksum has never
   been computed.  */
#define CHECKSUM_NEVER  ""

/* An entry describes an entry in the database.  The local file name
   is used as the key.  */
struct entry
{
  /* Name of remote file.  */
  char *name;
  /* Checksum for file.  */
  char *checksum;
};


/* State values for a file.  */
#define STATE_NEW      0
#define STATE_READING  1
#define STATE_WRITING  2
#define STATE_DONE     3
#define STATE_ERROR    4
#define STATE_MISSING  5

/* Flag values for a file.  */
#define FLAG_MAPPED    (1 << 0)
#define FLAG_FULL      (1 << 1)

/* A file describes a single file that we are manipulating.  */
struct file
{
  /* File name.  */
  char *name;
  /* State of file.  */
  int state;
  /* Checksum for file.  */
  char checksum[17];
  /* Flags.  */
  int flags;

  /* File descriptor if the file is open, -1 otherwise.  */
  int fd;
  /* Tag returned by gdk_input_add.  */
  int tag;

  /* Information for checksum processing.  */
  struct md5_ctx context;

  /* Mode of file.  */
  umode_t mode;

  /* When checksumming, the total size of buffer.  When copying, this
     is overloaded to mean the total number of data bytes in the
     buffer.  */
  size_t size;
  /* Location of next free slot in buffer.  This is only used when
     reading and caching the data.  When copying, this is overloaded
     to mean the current location in the buffer.  */
  size_t slot;
  /* If not NULL, holds pointer to the data in the file.  If mapped,
     then this points to the mapped data; otherwise it points to data
     which must be free()d.  */
  char *buffer;
};


/* A reader holds information about two different files.  This is used
   to synchronize reads between the files, so that the appropriate
   scheduling can be done when one or the other finishes (or fails).  */
struct reader
{
  /* Local file.  */
  struct file *local;
  /* Remote file.  */
  struct file *remote;
  /* The local file's entry in the database.  */
  struct entry *entry;
  /* The database and window.  */
  struct database *database;
};

/* Flags for database structure.  */
#define FLAG_DB_SYNC   (1 << 0)
#define FLAG_DB_ERROR  (1 << 1)

/* This describes a single edit window and the associated database.  */
struct database
{
  /* Top level window.  */
  GtkWidget *top;
  /* List widget displaying contents.  NULL if not yet created.  */
  GtkCList *list;
  /* File name of database.  */
  char *filename;
  /* Database itself.  */
  DB *database;
  /* Some flags; values are FLAG_DB_*.  */
  int flags;
  /* List of all current i/o operations.  Each element is a
     struct reader *.  */
  GList *ios;
};


/* Functions from main.c.  */
void new_sync_app (struct database *database);
void force_quit (struct database *database);

/* Database access functions from db.c.  */
void database_fill_list (struct database *database, GtkCList *list);
struct entry *lookup (const DB *database, const char *local_name);
DB *database_open (const char *filename);
int update_entry (const DB *database, const char *local_name,
		  const char *remote_name, const char *checksum);
void free_entry (struct entry *entry);
int remove_entry (const DB *database, const char *name);

/* Functions from sync.c.  */
void synchronize (struct database *database);
void free_reader (struct reader *reader);

/* Functions from new.c.  */
void new_entry (struct database *database);
void edit_entry (struct database *database, const char *local_file);

/* Functions from copy.c.  */
/* Values for DIRECTION argument to copy().  */
#define COPY_FROM_LOCAL  0
#define COPY_FROM_REMOTE 1
/* Note that copy() frees READER.  */
void copy (struct reader *reader, int direction);

/* Functions from errors.c.  */
void display_error (struct database *database, const char *local_file,
		    const char *remote_file, const char *error_msg);
void display_posix_error (struct database *database, const char *local_file,
			  const char *remote_file, const char *leader);
void clear_errors (struct database *database);
void db_posix_error (const char *message, const char *filename);
void add_display_entry (struct database *database, const char *local_file,
			const char *remote_file);
void display_message (struct database *database, const char *local_file,
		      const char *remote_file, const char *message);

/* Functions from property.c.  */
void show_properties (void);

/* Some globals from main.c.  */
extern int concurrent_transfers;
extern int max_file_size;

#endif /* SYNC_H */
