/* gnome-napster, a GNOME Napster client.
 * Copyright (C) 1999 Evan Martin <eeyem@u.washington.edu>
 */
#include <gtk/gtk.h>
#include <gnome.h>
#include <pthread.h>
#include <glib.h>
#include <zvt/zvtterm.h>
#include "ui.h"
#include "util.h"
#include "lnap.h"
#include "login.h"
#include "search.h"
#include "host.h"
#include "download.h"
#include "gconfig.h"

/* from configure.in */
#include "config.h"

#include "../pixmaps/napsmall.xpm"
#include "../pixmaps/online.xpm"
#include "../pixmaps/offline.xpm"

void about_cb(GtkWidget *w, gpointer d);
static void init_login(GtkWidget *w, gpointer d);
void log_complete(int type);

GtkWidget *app;
static GtkWidget *term, *appbar, *connpix, *statuslabel;
static gint gtag_server = 0;

/* colors "borrowed" from crescendo/src/color.c :) */
static gushort blu[] =
{
	0x0000,0x0000,0x0000,0x0000,0xaaaa,0xaaaa,0xaaaa,0xaaaa,
	0x5555,0x5555,0x5555,0x5555,0xffff,0xffff,0xffff,0xffff,
	0xaaaa, 0x0000
	/* 0xd7d7, 0x0000 */
};
static gushort red[] =
{
	0x0000,0xaaaa,0x0000,0xaaaa,0x0000,0xaaaa,0x0000,0xaaaa,
	0x5555,0xffff,0x5555,0xffff,0x5555,0xffff,0x5555,0xffff,
	0xaaaa, 0x0000
	/* 0xfafa, 0x0000 */
};
static gushort grn[] =
{ 
	0x0000,0x0000,0xaaaa,0x5555,0x0000,0x0000,0xaaaa,0xaaaa,
	0x5555,0x5555,0xffff,0xffff,0x5555,0x5555,0xffff,0xffff,
	0xaaaa, 0x0000
	/* 0xebeb, 0x0000 */
}; 

GnomeUIInfo file_menu[] = {
	GNOMEUIINFO_MENU_EXIT_ITEM(gtk_main_quit, NULL),
	GNOMEUIINFO_END
};

GnomeUIInfo napster_menu[] = {
	GNOMEUIINFO_ITEM(N_("_Login/Logout"), N_("Login/logout to Napster"), 
			init_login, napsmall_xpm),
	GNOMEUIINFO_ITEM_NONE(N_("_Relog"), N_("Logout, then log back in to Napster"), 
			login_relog_cb),
	GNOMEUIINFO_END
};

GnomeUIInfo settings_menu[] = {
	GNOMEUIINFO_MENU_PREFERENCES_ITEM(gconfig_dlg_run, NULL),
	GNOMEUIINFO_END
};

GnomeUIInfo help_menu[] = {
	GNOMEUIINFO_MENU_ABOUT_ITEM(about_cb, NULL),
	GNOMEUIINFO_END
};

GnomeUIInfo menubar[] = {
	GNOMEUIINFO_MENU_FILE_TREE(file_menu),
	GNOMEUIINFO_SUBTREE(N_("_Napster"), napster_menu),
	GNOMEUIINFO_MENU_SETTINGS_TREE(settings_menu),
	GNOMEUIINFO_MENU_HELP_TREE(help_menu),
	GNOMEUIINFO_END
};

static void delete_event(GtkWidget *w, GdkEvent *e, gpointer d) {
	gtk_main_quit();
}

inline void printc(char *text) {
	zvt_term_feed(ZVT_TERM(term), text, strlen(text));
}

inline void printcn(char *text) {
	printc(text); printc("\r\n");
	qprintf("%s\n", text);
}

void printcf(char *fmt, ...) {
	char *text;
	va_list ap;
	
	va_start(ap, fmt);
	text = g_strdup_vprintf(fmt, ap);
	va_end(ap);
	
	printc(text); qprintf(text);
	g_free(text);
}


static void status_cb(char *text, void *d) {
	gdk_threads_enter();
	printcn(text);
	gdk_threads_leave();
}

static void enter_cb(GtkWidget *w, gpointer d) {
	GtkEntry *e = GTK_ENTRY(d);
	gchar *text;
	
	text = gtk_entry_get_text(e);
	printcn(text); 
	gtk_entry_set_text(e, "");
}

static GtkWidget* create_console() {
	GtkWidget *vbox, *hbox, *e, *sb;
	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
		hbox = gtk_hbox_new(FALSE, 0);
			term = zvt_term_new_with_size(80, 24);
			zvt_term_set_blink(ZVT_TERM(term), FALSE);
			zvt_term_set_scrollback(ZVT_TERM(term), 1000);
			GTK_WIDGET_UNSET_FLAGS(term, GTK_CAN_FOCUS);
		gtk_box_pack_start(GTK_BOX(hbox), term, TRUE, TRUE, 0);
			sb = gtk_vscrollbar_new(GTK_ADJUSTMENT(ZVT_TERM(term)->adjustment));
			GTK_WIDGET_UNSET_FLAGS(sb, GTK_CAN_FOCUS);
		gtk_box_pack_start(GTK_BOX(hbox), sb, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
		e = gtk_entry_new();
		gtk_signal_connect(GTK_OBJECT(e), "activate",
				GTK_SIGNAL_FUNC(enter_cb), e);
	gtk_box_pack_start(GTK_BOX(vbox), e, FALSE, FALSE, 0);
	nap_setstatusfunc(status_cb, NULL);
	return vbox;
}

#if 0
static GtkWidget* create_queue() {
	GtkWidget *box;
	GtkWidget *p;
	GtkWidget *list;
	GtkWidget *itembox, *li;
	GtkWidget *scroll;
	GtkWidget *frame;

	GList *items = NULL;
	int i;
	static gchar* samples[] = {
		"01_-_squarepusher_-_tequila_fish.mp3", "02_-_autechre_-_vose_in.mp3",
		"03_-_meat_beat_manifesto_-_acid_again_(dub_again_remix).mp3",
		"04_-_plaid_-_undoneson.mp3", "05_-_plug_-_cheesy_(amen_mix).mp3",
		"06_-_the_bowling_green_-_think_what_you.re_doing.mp3",
		"07_-_meat_beat_manifesto_-_book_of_shadows.mp3"
	};

	box = gtk_vbox_new(FALSE, 10);

	gtk_container_set_border_width(GTK_CONTAINER(box), 10);
	list = gtk_list_new();
	for (i = 0; i < 7; i++) {
		itembox = gtk_vbox_new(FALSE, 0);
		frame = gtk_frame_new(samples[i]);
		/*gtk_frame_set_shadow_type(GTK_FRAME(frame), i % GTK_SHADOW_ETCHED_IN);*/
		gtk_container_set_border_width(GTK_CONTAINER(itembox), 5);
		/*	text = gtk_label_new(samples[i]);
		gtk_box_pack_start(GTK_BOX(itembox), text, FALSE, FALSE, 0);*/
			p = gtk_progress_bar_new();
			gtk_progress_configure(GTK_PROGRESS(p), (float)i/7.0, 0, 1);
			gtk_progress_set_show_text(GTK_PROGRESS(p), TRUE);
		gtk_box_pack_start(GTK_BOX(itembox), p, FALSE, FALSE, 0);
		gtk_container_add(GTK_CONTAINER(frame), itembox);

		li = gtk_list_item_new();
		gtk_container_add(GTK_CONTAINER(li), frame);

		items = g_list_append(items, li);
	}

	gtk_list_append_items(GTK_LIST(list), items);

	scroll = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), list);
	
	gtk_box_pack_start(GTK_BOX(box), scroll, TRUE, TRUE, 0);

	return box;
}
#endif 

static void init_login(GtkWidget *w, gpointer d) {
	/*gtk_widget_set_sensitive(menubar[1].widget, FALSE);*/
	login_loginout_cb(NULL, NULL); /* popup login box */
}

void 
ui_run(void) {
	GtkWidget *book;

	app = gnome_app_new(PACKAGE, _("GNOME Napster Client " VERSION));
	gnome_app_create_menus_with_data(GNOME_APP(app), menubar, app);
	gtk_window_set_policy(GTK_WINDOW(app), TRUE, TRUE, FALSE);
	gtk_signal_connect(GTK_OBJECT(app), "delete_event", 
		GTK_SIGNAL_FUNC(delete_event), NULL);

	appbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
	gnome_appbar_set_default(GNOME_APPBAR(appbar), _("Ready"));
	gnome_app_set_statusbar(GNOME_APP(app), appbar);

	statuslabel = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(appbar), statuslabel, FALSE, FALSE, 5);

	connpix = gnome_pixmap_new_from_xpm_d(offline_xpm);

	gtk_box_pack_start(GTK_BOX(appbar), connpix, FALSE, FALSE, 0);

	gnome_app_install_menu_hints(GNOME_APP(app), menubar);
	
	gtk_widget_realize(app);

	book = gtk_notebook_new();
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(book), TRUE);
	gtk_notebook_append_page(GTK_NOTEBOOK(book), 
			create_console(), gtk_label_new("Console"));
	gtk_notebook_append_page(GTK_NOTEBOOK(book), 
			create_search(), gtk_label_new("Search"));
	search_login_status(FALSE);
	gtk_notebook_append_page(GTK_NOTEBOOK(book), 
			create_host(), gtk_label_new("Host"));
	host_login_status(FALSE);
	/* gtk_notebook_append_page(GTK_NOTEBOOK(book), create_queue(), gtk_label_new("Queue")); */
	gtk_notebook_append_page(GTK_NOTEBOOK(book), 
			downloadtab_create(), gtk_label_new("Download"));

	gnome_app_set_contents(GNOME_APP(app), book);

	gtk_widget_show_all(app);
	zvt_term_set_color_scheme(ZVT_TERM(term), red, grn, blu);

	gdk_threads_enter();
	/* gtk_idle_add(init_login, NULL);
	 * can't add it as an idle process, because it pops up a modal dialog. */
	init_login(NULL, NULL);
	gtk_main();
	gdk_threads_leave();
}

static inline GtkWidget *about_url_new(char *link, char *text, char *caption) {
	GtkWidget *hb;
	GtkWidget *url, *l;
	hb = gtk_hbox_new(FALSE, 0);
		url = gnome_href_new(link, text); 
	gtk_box_pack_start(GTK_BOX(hb), url, FALSE, FALSE, 0);
		l = gtk_label_new(caption);
		gtk_label_set_justify(GTK_LABEL(l), GTK_JUSTIFY_RIGHT);
		gtk_label_set_line_wrap(GTK_LABEL(l), TRUE);
	gtk_box_pack_end(GTK_BOX(hb), l, FALSE, FALSE, 0);
	return hb;
}

void about_cb(GtkWidget *w, gpointer d) {
	static GtkWidget *about = NULL;
	const gchar *authors[] = {
		"Evan Martin",
		NULL
	};
	GtkWidget *vbox;
	GtkWidget *url;
	gchar *logofile;
	
	if ((logofile = gnome_pixmap_file("gnome-napster-largelogo.png")) == NULL) {
		g_warning("gnome-napster-animate.png not in gnome datadir; trying"
				" a relative path.");
		logofile = "../pixmaps/gnome-napster-largelogo.png";
	} else {
		g_free(logofile);
		logofile = "gnome-napster-largelogo.png";
	}
	about = gnome_about_new(_("GNOME Napster Client"), VERSION, 
			"Copyright (C) 1999 Evan Martin",
			authors, _("Client for Napster, a music sharing system."), 
			logofile);

	gnome_dialog_set_parent(GNOME_DIALOG(about), GTK_WINDOW(app));

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
	gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new("Relevant web links:"), FALSE, FALSE, 0);
	url = about_url_new("http://students.washington.edu/eeyem", 
				"Gnome Napster", "Home of this program.");
	gtk_box_pack_start(GTK_BOX(vbox), url, TRUE, TRUE, 0);
	url = about_url_new("http://www.napster.com", 
			"www.napster.com", "Home of Napster.");
	gtk_box_pack_start(GTK_BOX(vbox), url, TRUE, TRUE, 0);
	url = about_url_new("http://www.napster.com/faq", 
			"Napster FAQ", "Many of these FAQs also\napply to this client.");
	gtk_box_pack_start(GTK_BOX(vbox), url, TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(about)->vbox), vbox, FALSE, FALSE, 0);
	gtk_widget_show_all(GNOME_DIALOG(about)->vbox);

	/*gtk_window_set_modal(GTK_WINDOW(about), TRUE);*/
	gnome_dialog_run_and_close(GNOME_DIALOG(about));
}

static void handlemsg(snap_packet *p) {
	int a, b, c;
	char buf[128];

	switch (p->msg) {
		case NM_ERROR:
			gnome_error_dialog_parented(p->txt, GTK_WINDOW(app));
			break;
		case NM_SEARCH_RESULT: 
			search_result(p->txt);
			break;
		case NM_SEARCH_COMPLETE:
			search_complete();
			break;
		case NM_BROWSE_NOTFOUND: 
			host_notfound(p->txt);
			break;
		case NM_BROWSE_RESULT: 
			host_result(p->txt);
			break;
		case NM_BROWSE_COMPLETE:
			host_complete();
			break;
		case NM_FILEINFO_RESULT:
			download_getinfo_cb(p->txt);
			break;
		case NM_REMOTEQUEUEFULL:
			nap_dumppacket(p);
			download_remotequeuefull_cb(p->txt);
			break;
		case NM_FILECOUNT:
			sscanf(p->txt, "%d %d %d", &a, &b, &c);
			sprintf(buf, "%d songs in %d libraries (%dgb) on %s", 
					b, a, c, g_logininfo.serv);
			printcn(buf);
			gtk_label_set_text(GTK_LABEL(statuslabel), buf);
			break;
		case NM_SYSMSG: /* system message */
			printcn(p->txt);
			break;
		default:
			nap_dumppacket(p);
			break;
	}
	free(p);
}

static void
server_input(gpointer d, gint source, GdkInputCondition condition) {
	snap_packet *p;
	p = nap_readpacket_nonblock(nap_socknap);
	if (p == NULL) return;
	handlemsg(p);
}

/*static void* dataproc(void *d) {
	char buf[1024];
	int len, i, sock;
	int sock;
	struct sockaddr_in addr;
	int addr_size = sizeof(struct sockaddr_in);
	if (nap_socklocal == -1) return NULL;
	while (1) {
		if ((sock = accept(nap_socklocal, (struct sockaddr*)&addr, &addr_size)) < 0) {
			perror("accept");
			continue;
		}
		qprintf("accepted connection from %s...\n", inet_ntoa(addr.sin_addr));
	}
	return NULL;
}*/

extern void ui_loginstatus(void) {
	if (g_logininfo.connected) {
		/* login complete */
		gnome_pixmap_load_xpm_d(GNOME_PIXMAP(connpix), online_xpm);
		//printf("spawing server/data threads...\n");
		gtag_server = gdk_input_add(nap_socknap, GDK_INPUT_READ, server_input, NULL);
		//pthread_create(&datathread, NULL, &dataproc, NULL);
		/* printcn("Logged in.");  HACK: lnap.c's nap_login already prints this... */
		search_login_status(TRUE);
		host_login_status(TRUE);
		/*gtk_widget_set_sensitive(menubar[1].widget, TRUE);*/
	} else {
		gnome_pixmap_load_xpm_d(GNOME_PIXMAP(connpix), offline_xpm);
		qprintf("killing server/data threads...\n");
		gdk_input_remove(gtag_server);
		gtag_server = 0;
		//pthread_cancel(datathread);
		printcn("Logged out.");
		search_login_status(FALSE);
		host_login_status(FALSE);
	}
}
