/*
 protocol.c : GMasqDialer

    Copyright (C) 1998-1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gmasqdialer.h"

#include <ctype.h>

gboolean ready;

static CALLBACK_DATA data;
static GList *callbacks;

static gint mserver_handle, inputtag;
static GString *mserver_buffer;

inline void callback_run(CALLBACK_DATA *data)
{
    CALLBACK_FUNC func;

    if (callbacks != NULL)
    {
        callbacks = g_list_first(callbacks);
        func = (CALLBACK_FUNC) callbacks->data;
        callbacks = g_list_remove_link(callbacks, callbacks);
        if (func != NULL) func(data);
    }
}

inline void callback_run_keepfunc(CALLBACK_DATA *data)
{
    CALLBACK_FUNC func;

    if (callbacks != NULL)
    {
        func = (CALLBACK_FUNC) g_list_first(callbacks)->data;
        func(data);
    }
}

inline void mserver_cmd(gchar *cmd, CALLBACK_FUNC func)
{
    if (mserver_handle != -1)
    {
	callbacks = g_list_append(callbacks, func);
        net_transmit(mserver_handle, cmd, strlen(cmd));
        ready = FALSE;
    }
}

void protocol_init(void)
{
    mserver_handle = -1;
    ready = FALSE;
}

int mserver_connected(void)
{
    return mserver_handle != -1;
}

void mserver_hangup(void)
{
    mserver_cmd("KILL\n", NULL);
}

void mserver_dial(char *name, CALLBACK_FUNC func)
{
    GString *str;

    str = g_string_new(NULL);
    g_string_sprintf(str, "DIAL:%s\n", name);
    mserver_cmd(str->str, func);
    g_string_free(str, TRUE);
}

void mserver_setuser(gchar *user, CALLBACK_FUNC func)
{
    GString *str;

    str = g_string_new(NULL);
    g_string_sprintf(str, "USER:%s\n", user);
    mserver_cmd(str->str, func);
    g_string_free(str, TRUE);
}

void mserver_setpass(gchar *pass, CALLBACK_FUNC func)
{
    GString *str;

    str = g_string_new(NULL);
    g_string_sprintf(str, "PASS:%s\n", pass);
    mserver_cmd(str->str, func);
    g_string_free(str, TRUE);
}

void mserver_connectinfo(gchar *name, CALLBACK_FUNC func)
{
    GString *str;

    str = g_string_new(NULL);
    g_string_sprintf(str, "CINFO:%s\n", name);
    mserver_cmd(str->str, func);
    g_string_free(str, TRUE);
}

void mserver_netload(CALLBACK_FUNC func)
{
    mserver_cmd("NETLOAD\n", func);
}

void mserver_status(CALLBACK_FUNC func)
{
    mserver_cmd("STAT\n", func);
}

void mserver_time(CALLBACK_FUNC func)
{
    mserver_cmd("CTIME\n", func);
}

void mserver_connectiontime(CALLBACK_FUNC func)
{
    mserver_cmd("TIME\n", func);
}

void mserver_list_connections(CALLBACK_FUNC func)
{
    mserver_cmd("LIST\n", func);
}

void mserver_version(CALLBACK_FUNC func)
{
    mserver_cmd("VERSION\n", func);
}

void mserver_author(CALLBACK_FUNC func)
{
    mserver_cmd("AUTHOR\n", func);
}

void mserver_history(CALLBACK_FUNC func)
{
    mserver_cmd("HISTORY\n", func);
}

void mserver_license(CALLBACK_FUNC func)
{
    mserver_cmd("LICENSE\n", func);
}

void mserver_clientcount(CALLBACK_FUNC func)
{
    mserver_cmd("CCOUNT\n", func);
}

void mserver_listclients(CALLBACK_FUNC func)
{
    mserver_cmd("WHO\n", func);
}

static void process_line(gchar *str)
{
    static gboolean list_running = FALSE;

    /* BEGIN ... END list running */
    if (list_running)
    {
        if (strcmp(str, "END") == 0)
        {
            list_running = FALSE;
            data.u.list.type = LIST_END;
        }
        else
        {
            data.u.list.data = str;
            data.u.list.type = LIST_ITEM;
            callback_run_keepfunc(&data);
        }
        return;
    }

    /* Number ? */
    if (isdigit(str[0]))
    {
        glong digit;

        sscanf(str, "%ld", &digit);
        data.u.digit = (time_t) digit;
    }
    else if (strcmp(str, "READY") == 0)
    {
        ready = TRUE;
        callback_run(&data);
        data.status = 1;
    }
    else if (strcmp(str, "BUSY") == 0)
    {
        /* Old server versions let only one client to connect at time.. */
        masqerror(_("The masqserver is busy"));
        menu_disconnect();
    }
    else if (strcmp(str, "BEGIN") == 0)
    {
        /* BEGIN ... END list started. */
        list_running = TRUE;
        data.u.list.type = LIST_START;
        callback_run_keepfunc(&data);
    }
    else if (strncmp(str, "ERROR:", 6) == 0)
    {
        /* Error? naah, couldn't be.. */
        masqerror(str+6);
        data.status = 0;
    }
    else if (strcmp(str, "UNKNOWN") == 0)
    {
        /* Buggy old server didn't understand our command. */
        data.status = 0;
    }
    else if (strcmp(str, "DOWN") == 0/* || strcmp(str, "SUCCESS") == 0*/)
    {
        /* Connection is down */
        data.u.connection.connection = NULL;
        data.u.connection.bps = 0;
    }
    else if (strncmp(str, "UP:", 3) == 0)
    {
        /* Connection is up */
        gchar *name, *bps;
        gint bpsrate;

        bps = name = str+3;
        while (*bps != '\0' && *bps != ':') bps++;
        if (*bps == ':') *bps++ = '\0';
        sscanf(bps, "%d", &bpsrate);

        data.u.connection.connection = g_strdup(name);
        data.u.connection.bps = bpsrate;
    }
    else
    {
        if (callbacks != NULL) data.u.text = g_strdup(str);
    }
}

static void masqinput(gint handle)
{
    GString *str;
    gint ret;

    str = g_string_new(NULL);
    do
    {
        ret = read_line(handle, str, mserver_buffer);

        if (ret == -1)
        {
            /* disconnected */
            menu_disconnect();
            return;
        }
        if (ret != 0) process_line(str->str);
    }
    while (ret > 0);

    g_string_free(str, TRUE);
}

int mserver_connect(gchar *host, gint port)
{
    if (mserver_handle != -1) mserver_disconnect();

    data.status = 1; callbacks = NULL; user_sent = FALSE; ready = FALSE;

    mserver_handle = net_connect(host, port);
    if (mserver_handle == -1)
    {
        masqerror(_("Could not connect to masqserver"));
        return 0;
    }

    mserver_buffer = g_string_new(NULL);
    inputtag = gdk_input_add(mserver_handle, GDK_INPUT_READ, (GdkInputFunction) masqinput, (gpointer) mserver_handle);

    return 1;
}

int mserver_disconnect(void)
{
    if (mserver_handle == -1) return 0;

    net_disconnect(mserver_handle);
    gdk_input_remove(inputtag);
    g_string_free(mserver_buffer, TRUE);
    mserver_handle = -1;
    return 1;
}

