/*****************************************************************************/
/*  file_transfer.c - contains the ui and network routines for file transfer */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

struct recursive_transfer {
   struct ftp_transfer_data *tdata;
   GList *current_file;
   char *path;
   char *dest;
   char *root_path;
   char *start_file;
};

static GtkWidget *dialog, *status;
static int skip_same_state;
static void get_local_files_and_dirs(struct recursive_transfer *transdata);
static void asktrans(struct ftp_transfer_data *tdata);
static void trans_selectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, struct ftp_transfer_data *tdata);
static void trans_unselectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, struct ftp_transfer_data *tdata);
static void trans_selectall (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void trans_unselectall (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void overwrite (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void resume (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void skip (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void ok(GtkWidget *widget, struct ftp_transfer_data *tdata);
static void cancel(GtkWidget *widget, struct ftp_transfer_data *tdata);
static void calc_kbs (struct ftp_transfer_data *tdata, ssize_t num_read);
static void set_file_transfer_mode (struct ftp_transfer_data *tdata);
static void goto_next_file (struct ftp_transfer_data *tdata);
static int get_status (struct ftp_transfer_data *tdata, ssize_t num_read);
static mode_t parse_local_attribs (char *attribs);
static mode_t parse_remote_attribs (char *attribs);
 
void retrCB (GtkWidget *widget, gpointer data) {
   struct ftp_transfer_data *tdata;
   
   if (!check_status (_("Retrieve Files"), &window2, 0, 1, window2.hdata->ftpdata->get_file != NULL)) return;
   if ((tdata = inittrans (1, 1, window2.hdata->files)) != NULL) dotrans (tdata);
}
/*****************************************************************************/
void putCB (GtkWidget *widget, gpointer data) {
   struct ftp_transfer_data *tdata;
   
   if (!check_status (_("Put Files"), &window1, 0, 1, window1.hdata->ftpdata->put_file != NULL)) return;
   if (window2.local == -1) {
      ftp_log (gftp_logging_misc, NULL, _("Put Files: Not connected to a remote site\n"));
      return;
   }
   if ((tdata = inittrans (0, 1, window1.hdata->files)) != NULL) dotrans (tdata);
}
/*****************************************************************************/
struct ftp_transfer_data *inittrans (int direction, int add_to_queue, GList *files) {
   struct ftp_file_data *newfle, *tempfle;
   struct recursive_transfer transdata;
   struct ftp_transfer_data *tdata;
   struct ftp_window_data *fromwdata;
   GtkWidget *tempwid, *vbox;
   char *tempstr, *temp1str;
   GList *templist;
   FILE *oldsockfd;
   int oldfd;
   
   skip_same_state = 0;
   dialog = NULL;
   tdata = new_tdata ();
   window2.hdata->ftpdata->init (tdata->hdata->ftpdata);
   if (direction) {
      tdata->direction = 1;
      fromwdata = &window2;
   }
   else fromwdata = &window1;
   copy_hdata_struct (window2.hdata, tdata->hdata);
   tdata->hdata->totalfiles = tdata->hdata->wdata->numselected;
   if ((templist = get_next_selected_filename (files)) == NULL) {
      ftp_log (gftp_logging_misc, NULL, "Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n");
      return (NULL);
   }
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (strcmp (tempfle->file, "..") == 0) {
         ftp_log (gftp_logging_misc, NULL, _("Transfer Files: Skipping the transfer of the .. directory\n"));
         templist = get_next_selected_filename (templist->next);
         continue;
      }
      else if (tempfle->isdir) {
         if (dialog == NULL) {
            dialog = gtk_window_new (GTK_WINDOW_DIALOG);
            gtk_grab_add (dialog);
            gtk_window_set_title (GTK_WINDOW (dialog), _("Getting directory listings"));
            gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
            gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
    
            vbox = gtk_vbox_new (FALSE, 5);
            gtk_container_border_width (GTK_CONTAINER (vbox), 10);
            gtk_container_add (GTK_CONTAINER (dialog), vbox);
            gtk_widget_show (vbox);

            tempwid = gtk_label_new (_("Please wait while getting directory listings"));
            gtk_box_pack_start (GTK_BOX (vbox), tempwid, TRUE, TRUE, 0);
            gtk_widget_show (tempwid);

            status = gtk_label_new ("");
            gtk_box_pack_start (GTK_BOX (vbox), status, TRUE, TRUE, 0);
            gtk_widget_show (status);
            gtk_widget_show (dialog);
            fix_display ();
         }

         if (direction) {
            tempstr = g_strconcat (GFTP_GET_DIRECTORY (window2.hdata->ftpdata), "/", tempfle->file, NULL);
            temp1str = g_strconcat (GFTP_GET_DIRECTORY (window1.hdata->ftpdata), "/", tempfle->file, NULL);
         }
         else {
            temp1str = g_strconcat (GFTP_GET_DIRECTORY (window2.hdata->ftpdata), "/", tempfle->file, NULL);
            tempstr = g_strconcat (GFTP_GET_DIRECTORY (window1.hdata->ftpdata), "/", tempfle->file, NULL);
         }
         memset (&transdata, 0, sizeof (struct recursive_transfer));
         transdata.tdata = tdata;
         transdata.path = tempstr;
         transdata.dest = temp1str;
         transdata.root_path = g_malloc (strlen (tempstr) + 1);
         strcpy (transdata.root_path, tempstr);
         transdata.start_file = NULL;
         oldsockfd = GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata);
         oldfd = tdata->hdata->ftpdata->sockfd_unbuffered;
         GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) = GFTP_GET_CONTROL_FD (window2.hdata->ftpdata);
         tdata->hdata->ftpdata->sockfd_unbuffered = window2.hdata->ftpdata->sockfd_unbuffered;
         get_local_files_and_dirs (&transdata);
         GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) = oldsockfd;
         tdata->hdata->ftpdata->sockfd_unbuffered = oldfd;
      }
      else {
         newfle = g_malloc0 (sizeof (struct ftp_file_data));
         copy_fdata_struct (tempfle, newfle);

         if (newfle->file) g_free (newfle->file);
         if (newfle->remote_file) g_free (newfle->remote_file);
         newfle->file = g_strconcat (GFTP_GET_DIRECTORY (window1.hdata->ftpdata), "/", tempfle->file, NULL);
         remove_double_slashes (newfle->file);
         newfle->remote_file = g_strconcat (GFTP_GET_DIRECTORY (window2.hdata->ftpdata), "/", tempfle->file, NULL);
         remove_double_slashes (newfle->remote_file);

         newfle->ascii = get_file_transfer_mode (newfle->file);
         tdata->hdata->files = g_list_append (tdata->hdata->files, newfle);
      }
      templist = get_next_selected_filename (templist->next);
   }
   
   if (dialog != NULL) {
      if (fromwdata->local) {
         chdir (GFTP_GET_DIRECTORY (fromwdata->hdata->ftpdata));
      }
      else {
         gftp_set_directory (fromwdata->hdata->ftpdata, GFTP_GET_DIRECTORY (fromwdata->hdata->ftpdata));
      }
      gtk_widget_destroy (dialog);
   }

   if (tdata->hdata->files == NULL) {
      free_tdata (tdata);
      return (NULL);
   }
   tdata->curfle = tdata->hdata->files;
   if (add_to_queue) add_file_transfer (tdata);
   return (tdata);
}
/*****************************************************************************/
static void get_local_files_and_dirs (struct recursive_transfer *transdata) {
   struct ftp_file_data *tempfle;
   char *newpath, *newdest;
   GList *templist;
   int i;
   
   for (i=0; i<transdata->tdata->num_dirs_to_be_made; i++) {
      if (strcmp (transdata->tdata->dirs_to_be_made[i], transdata->dest) == 0) {
         break;
      }
   }
   if (i == transdata->tdata->num_dirs_to_be_made) {
      transdata->tdata->num_dirs_to_be_made++;
      transdata->tdata->dirs_to_be_made = g_realloc (transdata->tdata->dirs_to_be_made, transdata->tdata->num_dirs_to_be_made * sizeof(char *));
      transdata->tdata->dirs_to_be_made[transdata->tdata->num_dirs_to_be_made-1] = g_malloc (strlen (transdata->dest) + 1);
      strcpy (transdata->tdata->dirs_to_be_made[transdata->tdata->num_dirs_to_be_made-1], transdata->dest);
   }
   if (transdata->tdata->direction) templist = get_remote_files (transdata->tdata->hdata, transdata->path, &i, 1, GTK_LABEL (status));
   else templist = get_local_files (transdata->path, &i);

   if (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      free_fdata (tempfle);
      templist = g_list_remove_link (templist, templist);
   }
   if (transdata->tdata->hdata->files == NULL) {
      transdata->tdata->hdata->files = templist;
   }
   else {
      transdata->tdata->hdata->files = g_list_concat (transdata->tdata->hdata->files, templist);
   }
   if (transdata->current_file == NULL) {
      transdata->current_file = templist;
   }
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      newpath = g_strconcat (transdata->path, "/", tempfle->file, NULL);
      newdest = g_strconcat (transdata->dest, "/", tempfle->file, NULL);
      g_free (tempfle->file);
      if (transdata->tdata->direction) {
         tempfle->file = newdest;
         tempfle->remote_file = newpath;
      }
      else {
         tempfle->file = newpath;
         tempfle->remote_file = newdest;
      }
      tempfle->remote_size = tempfle->size;
      templist = templist->next;
   }
   while (transdata->current_file != NULL) {
      tempfle = (struct ftp_file_data *) transdata->current_file->data;
      if (tempfle->isdir) {
         if (transdata->tdata->direction) {
            newpath = tempfle->remote_file;
            newdest = tempfle->file;
         }
         else {
            newpath = tempfle->file;
            newdest = tempfle->remote_file;
         }
         transdata->path = newpath;
         transdata->dest = newdest;
         templist = transdata->current_file;
         transdata->current_file = transdata->current_file->next;
         transdata->tdata->hdata->files = g_list_remove_link (transdata->tdata->hdata->files, templist);
         get_local_files_and_dirs (transdata);
         return;
      }
      else {
         transdata->tdata->hdata->totalfiles++;
         tempfle->ascii = get_file_transfer_mode (tempfle->file);
      }
      transdata->current_file = transdata->current_file->next;
   }
}
/*****************************************************************************/
void dotrans (struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle, *remotefle;
   char *pos, origch, *filelist_dir;
   struct stat flestat;
   GList *templist;
   int same;
   
   same = 0;
   filelist_dir = NULL;
   pthread_mutex_lock (&tdata->mutex);
   while (tdata->curfle != NULL) {
      tempfle = (struct ftp_file_data *) tdata->curfle->data;
      if (tdata->direction) {
         /* We are downloading these files */
         if (stat (tempfle->file, &flestat) != -1) {
            tempfle->remote_size = tempfle->size;
            tempfle->size = flestat.st_size;
            tempfle->exists = 1;
            same = 1;
         }
      }
      else {
         /* We are uploading these files */
         pos = strrchr (tempfle->remote_file, '/');
         if (pos == NULL) pos = tempfle->remote_file;
         origch = *pos;
         *pos = '\0';

         if (strcmp (tempfle->remote_file, GFTP_GET_DIRECTORY (window2.hdata->ftpdata)) == 0) {
            templist = window2.hdata->files;
         }
         else templist = NULL; /* FIXME */

         *pos++ = origch;
         while (templist != NULL) {
            remotefle = (struct ftp_file_data *) templist->data;
            if (strcmp (remotefle->file, pos) == 0) {
               tempfle->remote_size = remotefle->size;
               tempfle->exists = 1;
               same = 1;
            }
            templist = templist->next;
         }
      }
      tdata->curfle = tdata->curfle->next;
   }
   if (same == 0) {
      tdata->show = 1;
      pthread_mutex_unlock (&tdata->mutex);
   }
   else {
      pthread_mutex_unlock (&tdata->mutex);
      asktrans (tdata);
   }
}
/*****************************************************************************/
static void asktrans (struct ftp_transfer_data *tdata) {
   char *dltitles[4], *add_data[4] = {NULL, NULL, NULL, NULL};
   GtkWidget *tempwid, *scroll, *hbox;
   struct ftp_file_data *tempfle;
   char *tempstr, *pos;
   GList *templist;
   size_t len;
   int i;

   dltitles[0] = _("Filename");
   dltitles[1] = _("Local Size");
   dltitles[2] = _("Remote Size");
   dltitles[3] = _("Action");
   dialog = gtk_dialog_new ();
   gtk_grab_add (dialog);
   gtk_window_set_title (GTK_WINDOW (dialog), tdata->direction ? _("Download Files") : _("Upload Files"));
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 10);
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->action_area), 5);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 5);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->action_area), 35);
   gtk_box_set_homogeneous (GTK_BOX (GTK_DIALOG (dialog)->action_area), TRUE);
   gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
               
   tempwid = gtk_label_new (_("The following file(s) exist on both the local and remote computer\nPlease select what you would like to do"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->vbox), tempwid, FALSE, FALSE, 0);
   gtk_widget_show (tempwid);

   scroll = gtk_scrolled_window_new (NULL, NULL);
   gtk_widget_set_usize (scroll, 450, 200);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
   	GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   tdata->clist = gtk_clist_new_with_titles (4, dltitles);
   gtk_container_add (GTK_CONTAINER (scroll), tdata->clist);
   gtk_clist_set_selection_mode (GTK_CLIST (tdata->clist), GTK_SELECTION_EXTENDED);
   gtk_clist_set_column_width (GTK_CLIST (tdata->clist), 0, 100);
   gtk_clist_set_column_justification (GTK_CLIST (tdata->clist), 1, GTK_JUSTIFY_RIGHT);
   gtk_clist_set_column_width (GTK_CLIST (tdata->clist), 1, 85);
   gtk_clist_set_column_justification (GTK_CLIST (tdata->clist), 2, GTK_JUSTIFY_RIGHT);
   gtk_clist_set_column_width (GTK_CLIST (tdata->clist), 2, 85);
   gtk_clist_set_column_width (GTK_CLIST (tdata->clist), 3, 85);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), scroll, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tdata->clist), "select_row", GTK_SIGNAL_FUNC (trans_selectrow), (gpointer) tdata);
   gtk_signal_connect (GTK_OBJECT (tdata->clist), "unselect_row", GTK_SIGNAL_FUNC (trans_unselectrow), (gpointer) tdata);
   gtk_widget_show (tdata->clist);
   gtk_widget_show (scroll);

   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      tempfle->selected = 0;
      if (tempfle->exists) {
         pos = tempfle->remote_file;
         len = strlen (GFTP_GET_DIRECTORY (window2.hdata->ftpdata));
         if (strncmp (pos, GFTP_GET_DIRECTORY (window2.hdata->ftpdata), len) == 0) {
            pos = tempfle->remote_file + len + 1;
         }
         add_data[0] = pos;
         add_data[3] = _("Overwrite");
         i = gtk_clist_append (GTK_CLIST (tdata->clist), add_data);
         
         tempstr = insert_commas (tempfle->size);
         gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 1, tempstr);
         g_free (tempstr);

         tempstr = insert_commas (tempfle->remote_size);
         gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 2, tempstr);
         g_free (tempstr);
      }
      templist = templist->next;
   }

   gtk_clist_select_all (GTK_CLIST (tdata->clist));

   hbox = gtk_hbox_new (TRUE, 20);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE, TRUE, 0);
   gtk_widget_show (hbox);

   tempwid = gtk_button_new_with_label (_("Overwrite"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (overwrite), (gpointer) tdata);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Resume"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (resume), (gpointer) tdata);
   gtk_widget_set_sensitive (tempwid, GFTP_SUPPORTS_RESUME (tdata->hdata->ftpdata));
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Skip File"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (skip), (gpointer) tdata);
   gtk_widget_show (tempwid);

   hbox = gtk_hbox_new (TRUE, 20);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE, TRUE, 0);
   gtk_widget_show (hbox);

   tempwid = gtk_button_new_with_label (_("Select All"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (trans_selectall), (gpointer) tdata);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Deselect All"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (trans_unselectall), (gpointer) tdata);
   gtk_widget_show (tempwid);
   
   tempwid = gtk_button_new_with_label (_("OK"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (ok), (gpointer) tdata);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_grab_default (tempwid);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("  Cancel  "));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (cancel), (gpointer) tdata);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_show (tempwid);

   gtk_widget_show(dialog);
}
/*****************************************************************************/
static void trans_selectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i;
   
   i = 0;
   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->exists) {
         if (i == row) {
            tempfle->selected = 1;
            break;
         }
         i++;
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void trans_unselectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i;
   
   i = 0;
   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->exists) {
         if (i == row) {
            tempfle->selected = 0;
            break;
         }
         i++;
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void trans_selectall (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i;
   
   i = 0;
   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->exists) {
         gtk_clist_select_row (GTK_CLIST (tdata->clist), i++, 0);
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void trans_unselectall (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i;
   
   i = 0;
   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->exists) {
         gtk_clist_unselect_row (GTK_CLIST (tdata->clist), i++, 0);
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void overwrite (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i;
   
   i = 0;
   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->exists) {
         if (tempfle->selected) {
            tempfle->restart = 0;
            tempfle->skip = 0;
            gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 3, _("Overwrite"));
         }
         i++;
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void resume (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i;
   
   i = 0;
   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->exists) {
         if (tempfle->selected) {
            tempfle->skip = 0;
            tempfle->restart = 1;
            gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 3, _("Resume"));
         }
         i++;
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void skip (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i;
   
   i = 0;
   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->exists) {
         if (tempfle->selected) {
            tempfle->restart = 0;
            tempfle->skip = 1;
            gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 3, _("Skip"));
         }
         i++;
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void ok (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   GList *templist, *dellist;
   
   pthread_mutex_lock (&tdata->mutex);
   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->skip) {
         free_fdata (tempfle);
         dellist = templist;
         templist = templist->next;
         tdata->hdata->files = g_list_remove_link (tdata->hdata->files, dellist);
         tdata->hdata->totalfiles--;
      }
      else templist = templist->next;
   }
   if (tdata->hdata->files == NULL) tdata->done = 1;
   else tdata->show = 1;
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
static void cancel (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   pthread_mutex_lock (&tdata->mutex);
   tdata->done = 1;
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
void *ftp_get_files(void *ptr) {
   struct ftp_transfer_data *tdata;
   struct ftp_file_data *tempfle;
   int i, j, success, writefd;
   struct utimbuf time_buf;
   long startsize, size;
   ssize_t num_read;
   char buf[8192];

   tdata = (struct ftp_transfer_data *) ptr;
   pthread_detach (pthread_self ());
   pthread_mutex_lock (&tdata->mutex);
   gettimeofday (&tdata->inittime, NULL);
   pthread_mutex_unlock (&tdata->mutex);
   gftp_logging (tdata->hdata->ftpdata, 1, queue_log, NULL);
   if (ftp_connect (tdata->hdata, 1)) {
      for (i=0; i<tdata->num_dirs_to_be_made; i++) {
         mkdir (tdata->dirs_to_be_made[i], 488);
         g_free (tdata->dirs_to_be_made[i]);
      }
      g_free (tdata->dirs_to_be_made);

      while (!tdata->cancel && tdata->curfle != NULL) {
         tempfle = (struct ftp_file_data *) tdata->curfle->data;
         success = 1;
         set_file_transfer_mode (tdata);

         if ((writefd = open (tempfle->file, tempfle->restart ? O_APPEND | O_CREAT | O_WRONLY : O_CREAT | O_TRUNC | O_WRONLY, S_IRUSR | S_IWUSR)) == -1) {
            queue_log (gftp_logging_error, NULL, _("Cannot create %s: %s\n"), 
            	tempfle->file , g_strerror (errno));
            goto_next_file (tdata);
            continue;
         }
         if (tempfle->restart) {
            startsize = tempfle->size;
            if (GFTP_GET_DATA_TYPE (tdata->hdata->ftpdata)) {
               startsize += file_countlf (writefd, startsize);
            }
         }
         else startsize = 0;

         if ((size = gftp_get_file (tdata->hdata->ftpdata, tempfle->remote_file, startsize)) < 0) {
            queue_log (gftp_logging_error, NULL, _("Cannot download file %s from %s\n"), 
            	tempfle->file , GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            goto_next_file (tdata);
            close (writefd);
            continue;
         }
         
         pthread_mutex_lock (&tdata->mutex);
         tdata->current_file_retries++;
         tdata->current_file_number++;
         tempfle->size = size;
         gettimeofday (&tdata->starttime, NULL);
         memcpy (&tdata->lasttime, &tdata->starttime, sizeof (tdata->lasttime));
         tdata->curtrans = startsize;
         tdata->curmax = tempfle->remote_size;
         tdata->trans_bytes += startsize;
         tdata->sttrans = startsize;
         pthread_mutex_unlock (&tdata->mutex);

         num_read = 0;
         while (!tdata->cancel && 
         	(num_read = fread (buf, 1, sizeof (buf), GFTP_GET_DATA_FD (tdata->hdata->ftpdata))) > 0) {
            calc_kbs (tdata, num_read);
            
            if (GFTP_GET_DATA_TYPE (tdata->hdata->ftpdata) == GFTP_TYPE_ASCII) { 
               for (i=0, j=0; i<num_read; i++) {
                  if (buf[i] != '\r') buf[j++] = buf[i];
               }
            }
            else j = 0;
            
            if (write (writefd, buf, GFTP_GET_DATA_TYPE (tdata->hdata->ftpdata) == GFTP_TYPE_ASCII ? j : num_read) == -1) {
               queue_log (gftp_logging_error, NULL, _("Error writing %s to disk: %s...Aborting entire transfer...\n"), 
               		tempfle->file, g_strerror (errno));
               success = 0;
               break;
            }
         }
         gftp_end_transfer (tdata->hdata->ftpdata);

         if (success && !tdata->cancel) {
            switch (get_status (tdata, num_read)) {
               case -1: success = 0;
               case  1: continue;
               default: success = 1;
            }
         }

         if (success && !tdata->cancel) {
            queue_log (gftp_logging_misc, NULL, _("Successfully downloaded %s\n"), 
            	tempfle->remote_file);
         }
         else {
            queue_log (gftp_logging_error, NULL, _("Could not download %s from %s\n"), 
            	tempfle->remote_file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            gftp_disconnect (tdata->hdata->ftpdata);
            break;
         }

         if (preserve_permissions) {
            if (tempfle->attribs) {
               chmod (tempfle->file, parse_local_attribs (tempfle->attribs));
            }
            if (tempfle->datetime) {
               time_buf.modtime = time_buf.actime = tempfle->datetime;
               utime (tempfle->file, &time_buf);
            }
         }
         close (writefd);
         goto_next_file (tdata);
      }
   }

   pthread_mutex_lock (&tdata->mutex);
   tdata->done = 1;
   pthread_mutex_unlock (&tdata->mutex);
   pthread_exit (NULL);
}
/*****************************************************************************/
void *ftp_put_files(void *ptr) {
   struct ftp_transfer_data *tdata;
   struct ftp_file_data *tempfle;
   int i, j, success, readfd;
   long startsize, newsize;
   char buf[8192], *newbuf;
   ssize_t num_read;

   tdata = (struct ftp_transfer_data *) ptr;
   pthread_detach (pthread_self ());
   pthread_mutex_lock (&tdata->mutex);
   gettimeofday (&tdata->inittime, NULL);
   pthread_mutex_unlock (&tdata->mutex);
   gftp_logging (tdata->hdata->ftpdata, 1, queue_log, NULL);
   if (ftp_connect (tdata->hdata, 1)) {
      for (i=0; i<tdata->num_dirs_to_be_made; i++) {
         gftp_make_directory (tdata->hdata->ftpdata, tdata->dirs_to_be_made[i]);
         g_free (tdata->dirs_to_be_made[i]);
      }
      g_free (tdata->dirs_to_be_made);

      while (!tdata->cancel && tdata->curfle != NULL) {
         tempfle = (struct ftp_file_data *) tdata->curfle->data;
         success = 1;
         set_file_transfer_mode (tdata);

         if (!(readfd = open (tempfle->file, O_RDONLY))) {
            queue_log (gftp_logging_error, NULL, _("Error: Cannot open local file %s: %s\n"), 
            	tempfle->file, g_strerror (errno));
            goto_next_file (tdata);
            continue;
         }

         if (tempfle->restart) {
            startsize = gftp_get_file_size (tdata->hdata->ftpdata, tempfle->remote_file);
            if (GFTP_GET_DATA_TYPE (tdata->hdata->ftpdata) == GFTP_TYPE_ASCII) {
               startsize += file_countlf(readfd, startsize);
            }
            lseek (readfd, startsize, SEEK_SET);
         }
         else startsize = 0;
         
         if (gftp_put_file (tdata->hdata->ftpdata, tempfle->remote_file, startsize) != 0) {
            queue_log (gftp_logging_error, NULL, _("Cannot put file %s to %s\n"), 
            	tempfle->remote_file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            goto_next_file (tdata);
            close (readfd);
            continue;
         }

         pthread_mutex_lock (&tdata->mutex);
         tdata->current_file_retries++;
         tdata->current_file_number++;
         gettimeofday (&tdata->starttime, NULL);
         memcpy (&tdata->lasttime, &tdata->starttime, sizeof (tdata->lasttime));
         tdata->curtrans = startsize;
         tdata->curmax = tempfle->size;
         tdata->trans_bytes += startsize;
         tdata->sttrans = startsize;
         pthread_mutex_unlock (&tdata->mutex);

         num_read = 0;
         while (!tdata->cancel && 
         	(num_read = read (readfd, buf, sizeof (buf))) > 0) {

            calc_kbs (tdata, num_read);

            if (GFTP_GET_DATA_TYPE (tdata->hdata->ftpdata) == GFTP_TYPE_ASCII) { 
               newsize = 0;
               for (i=0; i<num_read; i++) {
                  newsize++;
                  if(i > 0 && buf[i] == '\n' && buf[i-1] != '\r') newsize++;
               }
               newbuf = g_malloc (newsize);
               for (i=0,j=0; i<num_read; i++) {
                  if (i > 0 && buf[i] == '\n' && buf[i-1] != '\r') newbuf[j++] = '\r';
                  newbuf[j++] = buf[i];
               }
            }
            else {
               newbuf = buf;
               newsize = num_read;
            }

            fwrite (newbuf, 1, newsize, GFTP_GET_DATA_FD (tdata->hdata->ftpdata));
            if (ferror (GFTP_GET_DATA_FD (tdata->hdata->ftpdata)) != 0) {
               queue_log (gftp_logging_error, NULL, _("Cannot write %s to %s. Aborting entire transfer...\n"), 
               		tempfle->file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
               if (GFTP_GET_DATA_TYPE (tdata->hdata->ftpdata) == GFTP_TYPE_ASCII) g_free (newbuf);
               success = 0;
               continue;
            }
            if (GFTP_GET_DATA_TYPE (tdata->hdata->ftpdata) == GFTP_TYPE_ASCII) g_free (newbuf);
         }   
         close (readfd);
         
         gftp_end_transfer (tdata->hdata->ftpdata);
         if (success && !tdata->cancel) {
            switch (get_status (tdata, num_read)) {
               case -1: success = 0;
               case  1: continue;
               default: success = 1;
            }
            
         }

         if (success && !tdata->cancel) {
            queue_log (gftp_logging_misc, NULL, _("Successfully uploaded %s to %s\n"), 
            	tempfle->remote_file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            if (preserve_permissions) {
               if (tempfle->attribs) {
                  gftp_chmod (tdata->hdata->ftpdata, tempfle->remote_file, parse_remote_attribs (tempfle->attribs));
               }
            }
         }
         else {
            queue_log (gftp_logging_error, NULL, _("Could not upload %s to %s\n"), 
            	tempfle->remote_file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            gftp_disconnect (tdata->hdata->ftpdata);
            break;
         }
         
         goto_next_file (tdata);
         close (readfd);
      }
   }

   pthread_mutex_lock (&tdata->mutex);
   tdata->done = 1;
   pthread_mutex_unlock (&tdata->mutex);
   pthread_exit (NULL);
}
/*****************************************************************************/
static void calc_kbs (struct ftp_transfer_data *tdata, ssize_t num_read) {
   unsigned long waitusecs, difftime;
   struct timeval tv;
   float curkbs;
   
   gettimeofday (&tv, NULL);
   pthread_mutex_lock (&tdata->mutex);
   difftime = (tv.tv_sec - tdata->starttime.tv_sec) * 1000000 + tv.tv_usec - tdata->starttime.tv_usec;
   if (difftime == 0) tdata->kbs = 0;
   else tdata->kbs = ((float) (tdata->curtrans - tdata->sttrans) / 1024) / ((float) difftime / 1000000);
   tdata->curtrans += num_read;
   tdata->trans_bytes += num_read;
   tdata->stalled = 0;
   pthread_mutex_unlock (&tdata->mutex);

   difftime = (tv.tv_sec - tdata->lasttime.tv_sec) * 1000000 + tv.tv_usec - tdata->lasttime.tv_usec;
   if (difftime == 0) curkbs = (float) num_read / 1024;
   else curkbs = ((float) num_read / 1024) / ((float) difftime / 1000000);
   if (tdata->maxkbs > 0 && curkbs > tdata->maxkbs) {
      waitusecs = ((float) num_read / 1024) / tdata->maxkbs * 1000000 - difftime;
      usleep (waitusecs);
   }
   gettimeofday (&tdata->lasttime, NULL);
}
/*****************************************************************************/
static void set_file_transfer_mode (struct ftp_transfer_data *tdata) {
   if (((struct ftp_file_data *) tdata->curfle->data)->ascii) {
      gftp_set_data_type (tdata->hdata->ftpdata, GFTP_TYPE_ASCII);
   }
   else {
      gftp_set_data_type (tdata->hdata->ftpdata, GFTP_TYPE_BINARY);
   }
}
/*****************************************************************************/
static void goto_next_file (struct ftp_transfer_data *tdata) {
   pthread_mutex_lock (&tdata->mutex);
   tdata->next_file = 1;
   tdata->curfle = tdata->curfle->next;
   tdata->current_file_retries = 0;
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
static int get_status (struct ftp_transfer_data *tdata, ssize_t num_read) {
   struct ftp_file_data *tempfle;
   struct timeval tv;
   
   tempfle = (struct ftp_file_data *) tdata->curfle->data;
   if (tdata->hdata->ftpdata->add_onek) tdata->curtrans += 1024;
   if (num_read < 0 || tdata->curtrans < tdata->curmax) {
      if (tdata->current_file_retries >= retries) {
         queue_log (gftp_logging_error, NULL, _("Error: Remote site %s disconnected. Max retries reached...giving up\n"), 
         	GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
         return (-1);
      }
      else {
         queue_log (gftp_logging_error, NULL, _("Error: Remote site %s disconnected. Will reconnect in %d seconds\n"), 
         	GFTP_GET_HOSTNAME (tdata->hdata->ftpdata), sleep_time);
      }
      gftp_disconnect (tdata->hdata->ftpdata);
      tv.tv_sec = sleep_time;
      tv.tv_usec = 0;              
      select (0, NULL, NULL, NULL, &tv);
      if (ftp_connect (tdata->hdata, 1)) {
         fclose (GFTP_GET_DATA_FD (tdata->hdata->ftpdata));
         pthread_mutex_lock (&tdata->mutex);
         tempfle->restart = 1;
         tempfle->size = tdata->curtrans;
         if (tdata->hdata->ftpdata->add_onek) tempfle->size -= 1024;
         pthread_mutex_unlock (&tdata->mutex);
         tdata->current_file_number--; /* Decrement this because it will 
               			be incremented when we continue in the loop */
         return (1);
      }
   }
   
   return (0);
}
/*****************************************************************************/
static mode_t parse_local_attribs (char *attribs) {
   mode_t mode;
   
   mode = 0;
   if (tolower (attribs[3]) == 's') mode |= S_ISUID;
   if (tolower (attribs[6]) == 's') mode |= S_ISGID;
   if (attribs[1] == 'r') mode |= S_IRUSR;
   if (attribs[2] == 'w') mode |= S_IWUSR;
   if (attribs[3] == 'x') mode |= S_IXUSR;
   if (attribs[4] == 'r') mode |= S_IRGRP;
   if (attribs[5] == 'w') mode |= S_IWGRP;
   if (attribs[6] == 'x') mode |= S_IXGRP;
   if (attribs[7] == 'r') mode |= S_IROTH;
   if (attribs[8] == 'w') mode |= S_IWOTH;
   if (attribs[9] == 'x') mode |= S_IXOTH;
   return (mode);
}
/*****************************************************************************/
static mode_t parse_remote_attribs (char *attribs) {
   mode_t mode;
   int cur;
   
   mode = 0;
   cur = 0;
   if (tolower (attribs[3]) == 's') cur += 4;
   if (tolower (attribs[6]) == 's') cur += 2;
   mode = (mode * 10) + cur;

   cur = 0;
   if (attribs[1] == 'r') cur += 4;
   if (attribs[2] == 'w') cur += 2;
   if (attribs[3] == 'x') cur += 1;
   mode = (mode * 10) + cur;

   cur = 0;
   if (attribs[4] == 'r') cur += 4;
   if (attribs[5] == 'w') cur += 2;
   if (attribs[6] == 'x') cur += 1;
   mode = (mode * 10) + cur;

   cur = 0;
   if (attribs[7] == 'r') cur += 4;
   if (attribs[8] == 'w') cur += 2;
   if (attribs[9] == 'x') cur += 1;
   mode = (mode * 10) + cur;
   return (mode);
}
/*****************************************************************************/
