#include <includes.h>
#include <darxite.h>
#include "child.h"
#include "global.h"
#include "res.h"

static struct lookup_t {
	int r_fd; /* fileno of cstdout */

	FILE *cstdin, *cstdout;

	/* if true, return an IP (in_addr)--the hostent->h_addr;
	 * otherwise the hostent->h_name */
	int byname;
	struct lookup_t *next, *prev;
} *lookups = NULL;
MUTEX(ResMutex);
static ONCE_ID(ResMutexInit);

void Init_ResMutex()
{
    INIT_MUTEX(ResMutex);
}

static struct lookup_t *res_addlookup()
{
    struct lookup_t *new;

    new = dxmalloc(sizeof(struct lookup_t));
    new->next = lookups;
    if(new->next) new->next->prev = new;
    lookups = new;

    return new;
}

static void res_dellookup(struct lookup_t *l)
{
    if(l->next) l->next->prev = l->prev;
    if(l->prev) l->prev->next = l->next;

    if(l == lookups) lookups = l->next;

    free(l);
}

/* request a host lookup.  return a tag for this
 * lookup (in particular, the pipe with the child doing
 * it), or -1 if there was an error.  you may use the
 * pipe fd to select on, or poll periodically.
 * 
 * if byname is true, host should be a char * and a gethostbyname
 * will be done.
 * 
 * if byname is false, host should be a struct in_addr *, and a
 * gethostbyaddr will be done. */
int res_gethost(const char *host, int byname)
{
    struct lookup_t *lookup;
    int pid, rfds[2], wfds[2];
    int oldcanceltype;

    if(!host) {
	errno = EINVAL;
	return -1;
    }

    SET_CANCEL_TYPE(PTHREAD_CANCEL_DEFERRED, &oldcanceltype);
    ONCE(ResMutexInit, Init_ResMutex);

    LOCK_MUTEX(ResMutex);
    lookup = res_addlookup();

    /* rfds becomes the child's cstdout, and our readable fd;
     * wfds becomes the child's cstdin, and our writable fd */
    pipe(rfds);
    pipe(wfds);

    lookup->byname = byname;
    if(access(LIBEXEC "/darxite_resolve", R_OK|X_OK) == -1) {
	/* bad thing */
	error(E_FATAL, "Couldn't exec " LIBEXEC "/darxite_resolve!");
	exit(0);
    }

    pid = make_child(NULL, "dns of %s", host);
    if(!pid) {
	close(rfds[0]);
	close(wfds[1]);
	dup2(wfds[0], 0);
	dup2(rfds[1], 1);
	execl(LIBEXEC "/darxite_resolve", NULL);
	/* notreached */
	exit(1);
    }
    close(rfds[1]);
    close(wfds[0]);

    lookup->cstdin = fdopen(wfds[1], "w");

    if(!lookup->cstdin) fprintf(stderr, "? %s\n", strerror(errno));
    lookup->r_fd = rfds[0];
    lookup->cstdout = fdopen(lookup->r_fd, "r");

    fprintf(lookup->cstdin, "%i\n", byname);
    fprintf(lookup->cstdin, "%s\n", host);
    fflush(lookup->cstdin);

    error(E_TRACE, "Started lookup: byname=%i; host=%s; fd=%i (%i)",
	    byname, host, lookup->r_fd, wfds[1]);
    UNLOCK_MUTEX(ResMutex);
    return lookup->r_fd;
}

/* check pid for completion.  return 0 if complete,
 * in which case host will be filled in (up to host_s)
 * and the resolve finished; 1 if the host lookup failed
 * because no host was available; 2 if the lookup timed
 * out; 3 if we're not done yet.  -1 is returned on error */
int res_checkhost(int rfd, char *host, size_t host_s)
{
    /* find the child */
    struct lookup_t *lookup = lookups;
    char buf[512];
    int ret = 0;
    
    LOCK_MUTEX(ResMutex);

    while(lookup && lookup->r_fd != rfd) lookup=lookup->next;

    if(!lookup) {
	errno = EINVAL;
	return -1;
    }

    /* is there anything on the pipe? */
    {
	struct timeval tv = { 0, 0 };
	fd_set fds;
	FD_ZERO(&fds);
	FD_SET(lookup->r_fd, &fds);

	if(select(lookup->r_fd+1, &fds, NULL, NULL, &tv) <= 0) {
	    /* Nothing in the pipe. */
	    UNLOCK_MUTEX(ResMutex);
	    return 3;
	}
    }

    /* Yes */
    fgets(buf, sizeof(buf), lookup->cstdout);

    switch(buf[0]) {
	case 'F': /* failed */
	    ret = 1;
	    break;
	case 'T': /* timed out */
	    ret = 2;
	    break;
	case 'S':
	    fgets(host, host_s, lookup->cstdout);
	    strip(host);

	    ret = 0;
	    break;
    }

    fclose(lookup->cstdin);
    fclose(lookup->cstdout);
    error(E_TRACE, "Closed lookup: byname=%i; fd=%i; ret=%i; host=%s",
	    lookup->byname, lookup->r_fd, ret, host);
    res_dellookup(lookup);
    UNLOCK_MUTEX(ResMutex);

    return ret;
}

void res_cancel(int rfd)
{
    struct lookup_t *lookup = lookups;
    
    LOCK_MUTEX(ResMutex);

    while(lookup && lookup->r_fd != rfd) lookup=lookup->next;

    if(!lookup) return;

    fclose(lookup->cstdin);
    fclose(lookup->cstdout);

    res_dellookup(lookup);
    UNLOCK_MUTEX(ResMutex);
}

