// File containing FTP-related functions

#include "ftp.h"
#include "sockets.h"
#include "network.h"
#include "files.h"

#include <dirent.h>

BOOL FtpStartGettingFile(FileInfo *file)
{
    char buffer[256], buffer2[256], *end;

    file->ControlSocket->CmdResponse = (char *)dxmalloc(RESPONSE_MAX);
    sprintf(file->Activity, "Connected; waiting for reply...");
    if (GetResponse(file->ControlSocket, file->ControlSocket->CmdResponse,
                    RESPONSE_MAX) == DX_ERROR)
    {
        CancelStartGettingFile(file);
        return FALSE;
    }
    error(E_TRACE, "Got response for host %s OK", file->ActualServer->Name);
    
    if (!strcmp(file->LogIn, ""))
    {
        // if no login info's provided, use the server's defaults
        if (!LogIn(file->ControlSocket, file->Server->UserName,
                   file->Server->Password, file->Activity))
        {
            CancelStartGettingFile(file);
            FileComplete(file, "Couldn't log in");
            return FALSE;
        }
    }
    else
    {
        if (!LogIn(file->ControlSocket, file->LogIn, file->Password,
                   file->Activity))
        {
            CancelStartGettingFile(file);
            FileComplete(file, "Couldn't log in");
            return FALSE;
        }
    }
    
    if (strchr(file->Flags, 'd'))
    {
        file->TotalSize = 0;
        file->CurrentSize = 0;
        // turn off spooling to avoid weirdnesses
        if (!strchr(file->Flags, 'S'))
            strcat(file->Flags, "S");
        memset(buffer, 0, sizeof(buffer));
        memset(buffer2, 0, sizeof(buffer2));
        strcpy(buffer, file->Path);
        if (lastchr(buffer) == '/')
            lastchr(buffer) = '\0';
        // if we want to transfer the root
        if (!strcmp(buffer, ""))
        {
            FtpRecursiveGet(file, "/", "");
        }
        else
        {
            end = strrchr(buffer, '/');
            strncpy(buffer2, buffer, end - buffer + 1);
            error(E_TRACE, "root path: %s extra: %s", buffer2, end + 1);
            FtpRecursiveGet(file, buffer2, end + 1);
        }
        DisconnectFile(file, FALSE);
        file->Starting = FALSE;
        FileComplete(file, "Recursive transfer started");
        // even if we normally leave complete files in the batch,
        // get rid of it (confusing otherwise)
        if (DX_LeaveCompleteFiles)
            RemoveFromBatch(file);
        return TRUE;
    }
    else
    {
        return FtpSingleGet(file);
    }
}

BOOL FtpStartPuttingFile(FileInfo *file)
{
    char buffer[256], buffer2[256], file_buf[256], *end;

    file->ControlSocket->CmdResponse = (char *)dxmalloc(RESPONSE_MAX);
    sprintf(file->Activity, "Connected; waiting for reply...");
    if (GetResponse(file->ControlSocket, file->ControlSocket->CmdResponse,
                    RESPONSE_MAX) == DX_ERROR)
    {
        CancelStartGettingFile(file);
        return FALSE;
    }
    error(E_TRACE, "Got response for host %s OK", file->ActualServer->Name);
    
    if (!strcmp(file->LogIn, ""))
    {
        // if no login info's provided, use the server's defaults
        if (!LogIn(file->ControlSocket, file->Server->UserName,
                   file->Server->Password, file->Activity))
        {
            CancelStartGettingFile(file);
            FileComplete(file, "Couldn't log in");
            return FALSE;
        }
    }
    else
    {
        if (!LogIn(file->ControlSocket, file->LogIn, file->Password,
                   file->Activity))
        {
            CancelStartGettingFile(file);
            FileComplete(file, "Couldn't log in");
            return FALSE;
        }
    }
    if (strchr(file->Flags, 'd'))
    {
        // do recursive put
        file->TotalSize = 0;
        file->CurrentSize = 0;
        memset(buffer, 0, sizeof(buffer));
        memset(buffer2, 0, sizeof(buffer2));
        strcpy(buffer, file->LocalPath);
        if (lastchr(buffer) == '/')
            lastchr(buffer) = '\0';
        // strip the last part of the remote path off and ensure it
        // ends with a /
        if (strrchr(file->Path, '/'))
        {
            strcpy(file_buf, strrchr(file->Path, '/'));
            *(strrchr(file->Path, '/')) = '\0';
        }
        else
        {
            strcpy(file_buf, "");
        }
        if (lastchr(file->Path) != '/')
            strcat(file->Path, "/");
        error(E_TRACE, "File's path is now %s", file->Path);
        // if we want to transfer the root
        if (!strcmp(buffer, ""))
        {
            FtpRecursivePut(file, "/", "");
        }
        else
        {
            end = strrchr(buffer, '/');
            strncpy(buffer2, buffer, end - buffer + 1);
            error(E_TRACE, "root path: %s extra: %s", buffer2, end + 1);
            FtpRecursivePut(file, buffer2, end + 1);
        }
        DisconnectFile(file, FALSE);
        file->Starting = FALSE;
        // we need to restore the file's original path before we remove it
        // because otherwise the batch-removal function will not find it
        lastchr(file->Path) = '\0';
        strcat(file->Path, file_buf);
        FileComplete(file, "Recursive upload started");
        // even if we normally leave complete files in the batch,
        // get rid of it (confusing otherwise)
        if (DX_LeaveCompleteFiles)
            RemoveFromBatch(file);
        return TRUE;
    }
    else
    {
        return FtpSinglePut(file);
    }
}

// Arguments: the original file to transfer (needed to get flags etc.), the
// original path, and the extra path of the directory we want to get. Root
// path should begin and end with a /, extra path should not start or end
// with one.
BOOL FtpRecursiveGet(FileInfo *file, const char *root_path,
                     const char *extra_path)
{
    char local_path[256], line_buffer[257];
    char path_buffer[256], get_buffer[256], *line_ptr;
    char *dirs_to_get[100], new_flags[10], size[20], *size_ptr;
    int i, dircount = 0;
    SockInfo *data_sock;
    
    error(E_TRACE, "Recursively getting ftp://%s%s...", root_path, extra_path);
    sprintf(local_path, "%s/%s", DX_OutputDir, extra_path);
    if (lastchr(local_path) == '/')
        lastchr(local_path) = '\0';
    error(E_TRACE, "Creating %s...", local_path);
    mkdir(local_path, 0755);
    sprintf(path_buffer, "%s%s", root_path, extra_path);
    if (lastchr(path_buffer) == '/')
        lastchr(path_buffer) = '\0';
    
    data_sock = DataConnect(file->ControlSocket);
    if (data_sock == NULL)
        return FALSE;
    
    if (strcmp(path_buffer, ""))
    {
        if (!List(file->ControlSocket, path_buffer, TRUE))
            return FALSE;
    }
    else
    {
        if (!List(file->ControlSocket, "/", TRUE))
            return FALSE;
    }
    
    memset(line_buffer, 0, sizeof(line_buffer));
    data_sock->Eof = FALSE;
    // Read the listing from the server
    while ((TransferData(&data_sock, 1, NULL, 0, 10) > 0) && !data_sock->Eof)
    {
        while (buf_hasline(data_sock->DataBuf, DX_BufferSize))
        {
            buf_getline(data_sock->DataBuf, DX_BufferSize, line_buffer, 256);
            error(E_TRACE, "line buf: \"%s\"", line_buffer);
            // now we've got a line, let's download the file
            if (strncasecmp(line_buffer, "total", 5))
            {
                size_ptr = line_buffer;
                memset(size, 0, sizeof(size));
                // try to find the size first
                for (i = 0; i < 4; i++)
                {
                    size_ptr = strchr(size_ptr, ' ');
                    if (size_ptr == NULL)
                        break;
                    while (*size_ptr == ' ')
                        size_ptr++;
                }
                line_ptr = size_ptr;
                while (isdigit(*line_ptr))
                    line_ptr++;
                strncpy(size, size_ptr, line_ptr - size_ptr);
                
                // try to find the file name next
                while (isspace(*line_ptr))
                    line_ptr++;
                for (i = 0; i < 3; i++)
                {
                    line_ptr = strchr(line_ptr, ' ');
                    if (line_ptr == NULL)
                        break;
                    while (*line_ptr == ' ')
                        line_ptr++;
                }
                
                if (line_ptr && *line_ptr && strcmp(line_ptr, "..") &&
                    strcmp(line_ptr, "."))
                {
                    if (*line_buffer == 'd')
                    {
                        if (strcmp(extra_path, ""))
                            sprintf(get_buffer, "%s/%s", extra_path,
                                    line_ptr);
                        else
                            strcpy(get_buffer, line_ptr);
                        dirs_to_get[dircount] = strdup(get_buffer);
                        dircount++;
                    }
                    else if (*line_buffer != 'l')
                    {
                        // get the file if it isn't a symlink
                        memset(new_flags, 0, sizeof(new_flags));
                        for (i = 0; i < strlen(file->Flags); i++)
                        {
                            // remove the "recursive" flag
                            if (file->Flags[i] != 'd')
                            {
                                new_flags[strlen(new_flags)] =
                                    file->Flags[i];
                            }
                        }
                        sprintf(get_buffer, "%s://%s%s/%s | %s/%s | %s | "
                                "%s | %s | %s", file->Protocol,
                                file->ActualServer->Name, path_buffer,
                                line_ptr, local_path, line_ptr,
                                file->LogIn, file->Password, new_flags,
                                size);
                        AddFileToBatch(TRUE, get_buffer, NULL);
                    }
                    else
                    {
                        error(E_TRACE, "link: \"%s\"", line_ptr);
                    }
                }
            }
        }
    }
    GetResponse(file->ControlSocket, NULL, 0);
    DataDisconnect(data_sock);
    FreeSocket(&data_sock);
    for (i = 0; i < dircount; i++)
    {
        FtpRecursiveGet(file, root_path, dirs_to_get[i]);
        dxfree(dirs_to_get[i]);
    }
    return FALSE;
}

BOOL FtpSingleGet(FileInfo *file)
{
    THREAD_ID(thread_id)
    FILE *local_file;
    char buffer[256];
    
    // if the file's total size hasn't been specified in the batch
    if (file->TotalSize == SIZE_UNKNOWN)
    {
        file->TotalSize = GetFileSize(file);
        sprintf(buffer, "\"%s://%s%s\" | \"%s\" | %s | %s | %s | %d",
                file->Protocol, file->Server->Name, file->Path,
                file->LocalPath, file->LogIn, file->Password, file->Flags,
                file->TotalSize);
        RewriteFileInfo(file, buffer);
    }
    
    switch (file->TotalSize)
    {
    case SIZE_ERROR:
        CancelStartGettingFile(file);
        FileComplete(file, "Error getting file size");
        return FALSE;
        
    case SIZE_NO_FILE:
        CancelStartGettingFile(file);
        error(E_WARN, "File ftp://%s%s does not exist",
              file->ActualServer->Name, file->Path);
        FileComplete(file, "File does not exist");
        return FALSE;
        
    case 0:
        error(E_TRACE, "Not bothering to transfer zero-length file");
        CancelStartGettingFile(file);
        FileComplete(file, "File is zero-length");
        return FALSE;
    }
    
    if ((file->CurrentSize >= file->TotalSize) &&
        (file->TotalSize != SIZE_UNKNOWN))
    {
        file->Complete = TRUE;
        file->Starting = FALSE;
        DisconnectFile(file, FALSE);
        error(E_TRACE, "This file is already complete");
        if ((DX_EnableSpooling && !strchr(file->Flags, 'S')) ||
            strchr(file->Flags, 's'))
        {
            error(E_TRACE, "Trying to despool file");
            strcpy(file->Activity, "Despooling file...");
            SPAWN_THREAD(ThreadedMoveFile, thread_id, file);
            DETACH_THREAD(thread_id);
            Sleep(50);
        }
        else
        {
            // if we need to rename it now that the transfer's complete,
            // then do so - note that SpooledPath should be the same as
            // LocalPath
            if ((DX_EnableRenaming && !strchr(file->Flags, 'R')) ||
                strchr(file->Flags, 'r'))
            {
                snprintf(buffer, sizeof(buffer), "%s.darxite",
                         file->SpooledPath);
                rename(buffer, file->SpooledPath);
            }
            FileComplete(file, "File complete");
        }
        return TRUE;
    }
    
    // massive download stuff
    error(E_TRACE, "FTP total size: %d limit: %d maxconns: %d, ratio: %d "
          "current: %d connectagain: %d",
          file->TotalSize, DX_MassiveLimit, DX_MassiveMaxConnections,
          (DX_MassiveMaxConnections > 0) ?
          (file->TotalSize / DX_MassiveMaxConnections) :
          -1, file->CurrentSize,
          DX_MassiveConnectAgainLimit);
    if ((DX_EnableMassiveDownload && !strchr(file->Flags, 'v') &&
         !strchr(file->Flags, 'E') &&
        (file->TotalSize > (DX_MassiveLimit * 1024))) ||
        strchr(file->Flags, 'e'))
    {
        StartMassiveDownload(file);
    }
    
    strcpy(file->Activity, "Preparing for data transfer...");
    if (!SetType(file->ControlSocket, 'I'))
    {
        error(E_TRACE, "Couldn't set type to I");
        CancelStartGettingFile(file);
        return FALSE;
    }
    file->DataSocket = DataConnect(file->ControlSocket);
    if (!file->DataSocket)
    {
        error(E_TRACE, "Couldn't DataConnect");
        CancelStartGettingFile(file);
        return FALSE;
    }
    
    file->DataSocket->Eof = FALSE;
    // append to file if possible
    if (file->CurrentSize > 0)
    {
        if (!Restore(file->ControlSocket, file->CurrentSize))
        {
            // clear the local file if no luck
            error(E_TRACE, "Server doesn't support file resume");
            file->CurrentSize = 0;
            file->Server->SupportsFileResume = FALSE;
            if ((DX_EnableRenaming && !strchr(file->Flags, 'R')) ||
                strchr(file->Flags, 'r'))
            {
                sprintf(buffer, "%s.darxite", file->SpooledPath);
            }
            else
            {
                strcpy(buffer, file->SpooledPath);
            }
            local_file = fopen(buffer, "w");
            fclose(local_file);
        }
        else
        {
            file->Server->SupportsFileResume = TRUE;
        }
    }
    
    if (Get(file->ControlSocket, file->Path))
    {
        strcpy(file->Activity, "Transferring file");
        file->TimeOfLastRxTx = time(NULL);
        file->StartTime = time(NULL);
        file->RxTxOverall = 0;
        file->Started = TRUE;
        file->Starting = FALSE;
        //error(E_TRACE, "File %s is no longer starting", file->Path);
        return TRUE;
    }
    else
    {
        error(E_WARN, "Couldn't get the file: %s",
              file->ControlSocket->CmdResponse);
        DataDisconnect(file->DataSocket);
        if (file->DataSocket)
            FreeSocket(&file->DataSocket);
        CancelStartGettingFile(file);
        FileComplete(file, "Couldn't get file");
    }
    return FALSE;
}

// used by FtpRecursivePut for scandir
static int one(const struct dirent *unused)
{
    return 1;
}

// root path and extra path refer to the LOCAL path
BOOL FtpRecursivePut(FileInfo *file, const char *root_path,
                     const char *extra_path)
{
    char buffer[256], remote_path[256];
    char path_buffer[256], put_buffer[256], new_flags[10];
    struct dirent **eps;
    struct stat stat_buf;
    int i, n, rc;
    
    error(E_TRACE, "Recursively putting %s%s...", root_path, extra_path);
    sprintf(remote_path, "%s%s", file->Path, extra_path);
    if (lastchr(remote_path) == '/')
        lastchr(remote_path) = '\0';
    error(E_TRACE, "Creating %s...", remote_path);
    // send an MKDIR command
    NetWrite(file->ControlSocket, "MKD %s", remote_path);
    sprintf(path_buffer, "%s%s", root_path, extra_path);
    if (lastchr(path_buffer) == '/')
        lastchr(path_buffer) = '\0';
    
    strcpy(new_flags, file->Flags);
    RemoveFlag(new_flags, 'd');
    
    n = scandir(path_buffer, &eps, one, alphasort);
    if (n >= 0)
    {
        for (i = 0; i < n; i++)
        {
            // get information about the file
            sprintf(buffer, "%s%s/%s", root_path, extra_path, eps[i]->d_name);
            rc = stat(buffer, &stat_buf);
            if (!strcmp(eps[i]->d_name, ".") ||
                !strcmp(eps[i]->d_name, "..") || (rc == -1))
            {
                free(eps[i]);
                continue;
            }
            // if it's a directory, then recursively put it
            else if (S_ISDIR(stat_buf.st_mode))
            {
                sprintf(buffer, "%s/%s", extra_path, eps[i]->d_name);
                FtpRecursivePut(file, root_path, buffer);
            }
            else
            {
                sprintf(put_buffer, "%s://%s%s/%s | %s%s/%s | %s | "
                        "%s | %s | %d", file->Protocol,
                        file->ActualServer->Name, remote_path,
                        eps[i]->d_name, root_path, extra_path, eps[i]->d_name,
                        file->LogIn, file->Password, new_flags,
                        (int)stat_buf.st_size);
                AddFileToBatch(TRUE, put_buffer, NULL);
            }
            free(eps[i]);
        }
    }
    return FALSE;
}

BOOL FtpSinglePut(FileInfo *file)
{
    int rc;

    file->CurrentSize = GetFileSize(file);
    switch (file->CurrentSize)
    {
    case SIZE_ERROR:
        CancelStartGettingFile(file);
        FileComplete(file, "Error getting file size");
        return FALSE;
        break;
        
    case SIZE_NO_FILE:
        file->CurrentSize = 0;
        break;
        
    case 0:
        file->CurrentSize = 0;
        break;
    }
    
    strcpy(file->Activity, "Preparing for data transfer...");
    if (!SetType(file->ControlSocket, 'I'))
    {
        error(E_TRACE, "Couldn't set type to I");
        CancelStartGettingFile(file);
        return FALSE;
    }
    file->DataSocket = DataConnect(file->ControlSocket);
    if (!file->DataSocket)
    {
        error(E_TRACE, "Couldn't DataConnect");
        CancelStartGettingFile(file);
        return FALSE;
    }
    file->DataSocket->Eof = FALSE;
    
    // clear the remote file if necessary
    if (!strchr(file->Flags, 'c'))
    {
        file->CurrentSize = 0;
        strcat(file->Flags, "c"); // next time, append properly
        rc = Put(file->ControlSocket, file->Path, FALSE);
    }
    else
    {
        rc = Put(file->ControlSocket, file->Path, TRUE);
    }
    if (rc == TRUE)
    {
        strcpy(file->Activity, "Transferring file");
        file->TimeOfLastRxTx= time(NULL);
        file->StartTime = time(NULL);
        file->RxTxOverall = 0;
        file->Started = TRUE;
        file->Starting = FALSE;
        //error(E_TRACE, "File %s is no longer starting", file->Path);
        return TRUE;
    }
    else
    {
        error(E_WARN, "Couldn't put the file: %s",
              file->ControlSocket->CmdResponse);
        DataDisconnect(file->DataSocket);
        if (file->DataSocket)
            FreeSocket(&file->DataSocket);
        CancelStartGettingFile(file);
        FileComplete(file, "Couldn't put file");
    }
    return FALSE;
}

int GetFileSize(FileInfo *file)
{
    int size = 0, i;
    char buffer[256], path[256], link_buf[10], *buf_ptr;
    BOOL is_link = FALSE;

    strcpy(file->Activity, "Getting file size...");
    // Get the file's size on the server so we know when we've finished it
    // Note that this function now traverses symbolic links
    
    if (!SetType(file->ControlSocket, 'A'))
        return SIZE_ERROR;
    
    // For some reason, some FTP servers (eg. wu-ftpd 2.5.0) don't allow you
    // to do a LIST (or NLST) of files with spaces in their names! This is
    // extremely lame, but I have to work around it. Note that ncftp also has
    // this problem; try doing "ls "test dir"" for instance.
    if (strchr(file->Path, ' '))
        return SIZE_UNKNOWN;
    
    strcpy(path, file->Path);
    do {
        file->DataSocket = DataConnect(file->ControlSocket);
        if (!file->DataSocket)
            return SIZE_ERROR;
        
        memset(buffer, 0, sizeof(buffer));
        if (!ListFile(file, path, buffer, sizeof(buffer) - 1))
            return SIZE_ERROR;
        
        DataDisconnect(file->DataSocket);
        FreeSocket(&file->DataSocket);
        
        if (strlen(buffer) == 0)
            return SIZE_NO_FILE;
        
        error(E_TRACE, "List buffer is \"%s\"", buffer);
        // if there's an extraneous "total" line (ProFTPD)
        if (strchr(buffer, '\n') &&
            (strchr(buffer, '\n') < buffer + strlen(buffer) - 1))
        {
            buf_ptr = strchr(buffer, '\n') + 1;
        }
        else
        {
            buf_ptr = buffer;
        }
        
        if (*buf_ptr == 'l') // if it's a symlink
        {
            error(E_TRACE, "It's a link - recursing...");
            memclr(link_buf);
            do {
                strncpy(link_buf, buf_ptr, 4);
                buf_ptr++;
            } while ((*buf_ptr != '\0') && (strcmp(link_buf, " -> ")));
            if (!strcmp(link_buf, " -> "))
            {
                strcpy(path, buf_ptr + 3);
                if (lastchr(path) == '\n')
                    lastchr(path) = '\0';
                if (lastchr(path) == '\r')
                    lastchr(path) = '\0';
                error(E_TRACE, "Found link to %s", path);
                is_link = TRUE;
            }
            else
            {
                error(E_TRACE, "There appears to be a dodgy symlink");
                return SIZE_NO_FILE;
            }
        }
        else
        {
            is_link = FALSE;
            buf_ptr = strtok(buffer, " ");
            for (i = 0; buf_ptr && (i < 4); i++)
            {
                buf_ptr = strtok(NULL, " ");
            }
            if (buf_ptr)
                size = atoi(buf_ptr);
            else
                size = SIZE_UNKNOWN;
        }
    } while (is_link);
    error(E_TRACE, "File size is %d", size);
    return size;
}

BOOL ListFile(FileInfo *file, const char *path, char *buffer, int buf_size)
{
    int total_bytes = 0;

    if (!List(file->ControlSocket, path, TRUE))
        return FALSE;
    
    memset(buffer, 0, buf_size);
    file->DataSocket->Eof = FALSE;
    // Read the listing from the server
    /* FIXME: should use a buffer and a buf-to-buf transfer */
    while ((TransferData(&file->DataSocket, 1, NULL, 0, 10) > 0) &&
           !file->DataSocket->Eof)
    {
        // if the data to transfer fits in the buffer
        if (total_bytes + file->DataSocket->DataLength <= buf_size)
        {
            memcpy(buffer + total_bytes, file->DataSocket->DataBuf,
                   file->DataSocket->DataLength);
            total_bytes += file->DataSocket->DataLength;
        }
        // if there is some overrun
        else if (total_bytes < buf_size)
        {
            memcpy(buffer + total_bytes, file->DataSocket->DataBuf,
                   buf_size - total_bytes);
            total_bytes = buf_size;
        }
    }
    GetResponse(file->ControlSocket, NULL, 0);
    return TRUE;
}

// === FTP commands below this point ===

BOOL LogIn(SockInfo *sock, const char *user, const char *pass, char *activity)
{
    int rc;

    if (sock == NULL)
    {
        error(E_TRACE, "LogIn() passed null socket");
        return FALSE;
    }
    else if (!sock->Connected)
    {
        error(E_TRACE, "LogIn() passed unconnected socket");
        return FALSE;
    }
    error(E_TRACE, "  LogIn(%s, %s, %s)", sock->HostName, user, pass);

    if (activity)
        sprintf(activity, "Log in as \"%s\"...", user);
    rc = NetWrite(sock, "USER %s", user);
    if (rc >= FTP_ERROR_START)
    {
        error(E_WARN, "Couldn't log in: server response was \"%s\"",
              sock->CmdResponse);
        return FALSE;
    }
    else if (rc == -1)
    {
        return FALSE;
    }
    
    if (activity)
        strcpy(activity, "Sending password...");
    rc = NetWrite(sock, "PASS %s", pass);
    if (rc >= FTP_ERROR_START)
    {
        error(E_WARN, "Couldn't log in: server response was \"%s\"",
              sock->CmdResponse);
        return FALSE;
    }
    else if (rc == -1)
    {
        return FALSE;
    }

    return TRUE;
}

BOOL SetType(SockInfo *sock, char type)
{
    int rc;

    if (sock == NULL)
    {
        error(E_TRACE, "SetType() passed null socket");
        return FALSE;
    }
    else if (!sock->Connected)
    {
        error(E_TRACE, "SetType() passed unconnected socket");
        return FALSE;
    }
    error(E_TRACE, "  SetType(%s, %c)", sock->HostName, type);

    rc = NetWrite(sock, "TYPE %c", type);
    if (rc >= FTP_ERROR_START)
    {
        error(E_WARN, "Couldn't set type to '%c': server response was \"%s\"",
              type, sock->CmdResponse);
        return FALSE;
    }
    else if (rc == -1)
    {
        return FALSE;
    }
    
    return TRUE;
}

BOOL List(SockInfo *sock, const char *path, BOOL verbose)
{
    int err_code;

    if (sock == NULL)
    {
        error(E_TRACE, "List() passed null socket");
        return FALSE;
    }
    else if (!sock->Connected)
    {
        error(E_TRACE, "List() passed unconnected socket");
        return FALSE;
    }
    error(E_TRACE, "  List(%s, %s, %d)", sock->HostName, path, verbose);
    
    if (verbose)
        err_code = NetWrite(sock, "LIST %s", path);
    else
        err_code = NetWrite(sock, "NLST %s", path);
    
    if (err_code >= FTP_ERROR_START)
    {
        error(E_WARN, "Couldn't list \"%s\": server response was \"%s\"",
              path, sock->CmdResponse);
        return FALSE;
    }
    else if (err_code == -1)
    {
        return FALSE;
    }
    return TRUE;
}

BOOL Get(SockInfo *sock, const char *path)
{
    int rc;

    if (sock == NULL)
    {
        error(E_TRACE, "Get() passed null socket");
        return FALSE;
    }
    else if (!sock->Connected)
    {
        error(E_TRACE, "Get() passed unconnected socket");
        return FALSE;
    }
    error(E_TRACE, "  Get(%s, %s)", sock->HostName, path);
    
    rc = NetWrite(sock, "RETR %s", path);
    if (rc >= FTP_ERROR_START)
    {
        //error(E_WARN, "Couldn't get file \"%s\": server response was \"%s\"",
        //      path, sock->CmdResponse);
        return FALSE;
    }
    else if (rc == -1)
    {
        return FALSE;
    }
    
    sock->Transferring = TRUE;
    return TRUE;
}

BOOL Put(SockInfo *sock, const char *path, BOOL append)
{
    int rc;

    if (sock == NULL)
    {
        error(E_TRACE, "Put() was passed null socket");
        return FALSE;
    }
    else if (!sock->Connected)
    {
        error(E_TRACE, "Put() was passed unconnected socket");
        return FALSE;
    }
    error(E_TRACE, "  Put(%s, %s, %d)", sock->HostName, path, append);
    
    if (append)
        rc = NetWrite(sock, "APPE %s", path);
    else
        rc = NetWrite(sock, "STOR %s", path);
    if (rc >= FTP_ERROR_START)
    {
        //error(E_WARN, "Couldn't put file \"%s\": server response was \"%s\"",
        //      path, sock->CmdResponse);
        return FALSE;
    }
    else if (rc == -1)
    {
        return FALSE;
    }
    
    sock->Transferring = TRUE;
    return TRUE;
}

BOOL Restore(SockInfo *sock, int position)
{
    int rc;

    if (sock == NULL)
    {
        error(E_TRACE, "Restore() passed null socket");
        return FALSE;
    }
    else if (!sock->Connected)
    {
        error(E_TRACE, "Restore() passed unconnected socket");
        return FALSE;
    }
    error(E_TRACE, "  Restore(%s, %d)", sock->HostName, position);

    rc = NetWrite(sock, "REST %d", position);
    if (rc >= FTP_ERROR_START)
    {
        //error(E_WARN,"Couldn't restore file pos to %d bytes:"
        //      "server response was \"%s\"",
        //      position, sock->CmdResponse);
        return FALSE;
    }
    else if (rc == -1)
    {
        return FALSE;
    }
    
    return TRUE;
}
