// Error logging facilities

#include "error.h"
#include <darxite.h>
#include "global.h"
#include "clients.h"
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>

#define ERROR_FILE "log-errors.txt"

static char *bold_on = "\x1B[1m", *bold_off = "\x1B[0m";

char *DX_ErrorText[] = {
    "Success",
    "An error occurred",
    "Bad command",
    "Network error",
    "Not connected to an FTP server",
    "FTP error",
    "Socket error",
    "Host not found",
    "Couldn't connect to host",
    "There's more data to read"
};

void init_logging(void)
{
    if (DX_EnableErrorLogging)
    {
        FILE *error_file;
        
        error_file = fopen(ERROR_FILE, "w");
        if (error_file)
        {
            fputs("===Start of error log===\n", error_file);
            fclose(error_file);
        }
        else
        {
            fprintf(stderr, "Couldn't open error log file!");
        }
    }
}

void error(int level, const char *error_str, ...)
{
    va_list ap;
    char error_text[2048], print_text[2048];
    FILE *error_file;
    int olderrno;

    if (error_str == NULL)
        return;
   
    /* Since we're used for debugging, we really don't want to hose errno. */
    olderrno = errno;
    va_start(ap, error_str);
    memset(error_text, 0, sizeof(error_text));
    memset(print_text, 0, sizeof(print_text));
    vsnprintf(error_text, sizeof(error_text) - 1, error_str, ap);
    strcat(error_text, "\n");
    if (DX_EnableErrorLogging)
    {
        error_file = fopen(ERROR_FILE, "a");
        if (error_file)
        {
            fputs(error_text, error_file);
            fclose(error_file);
        }
    }
    
    snprintf(print_text, sizeof(print_text) - 1, "DX: %s", error_text);
    if (level <= E_INFO)
    {
        if (strcmp(DX_OutputToProgram, ""))
        {
            error_file = popen(DX_OutputToProgram, "w");
            if (error_file)
            {
                fprintf(error_file, print_text);
                pclose(error_file);
            }
        }
        // only print the output when a file to send it to hasn't
        // been specified
        if (strcmp(DX_OutputToFile, ""))
        {
            error_file = fopen(DX_OutputToFile, "a");
            if (error_file)
            {
                fprintf(error_file, print_text);
                fclose(error_file);
            }
        }
        else
        {
            snprintf(print_text, sizeof(print_text) - 1, "%sDX%s: %s", bold_on,
                     bold_off, error_text);
            printf(print_text);
        }
        // notify any clients that have asked for it
        SendEvent(ERROR_EVENT, print_text);
        snprintf(print_text, sizeof(print_text) - 1, "%d %s", DX_ERROR_EVENT,
                 error_text);
    }
    // if it's really fatal (eg. segfault) we don't want to run exit handlers
    if (level == E_FATAL)
        abort();
    if (level == E_FAIL)
        exit(1);
    va_end(ap);
    errno = olderrno;
}

