// File containing implementation for Ashley's version of buffering code

#include <darxite.h>
#include "buffer.h"
#include "global.h"

// a version of read() that ignores signals and checks for s NULL buffer
ssize_t buf_read(int fd, char *buf, size_t count)
{
    ssize_t rc;
	
    if (buf == NULL)
    {
        error(E_TRACE, "buf_read() was passed a null buffer");
        errno = EINVAL;
        return -1;
    }
    
    do {
        rc = read(fd, buf, count);
    } while ((rc == -1) && (errno == EINTR));
    
    return rc;
}

// uses buf_read() to read repeatedly until <count> bytes have been read
ssize_t buf_repeat_read(int fd, char *buf, size_t count)
{
    ssize_t rc, bytes_read = 0;
	
    do {
        rc = buf_read(fd, buf + bytes_read, count - bytes_read);
        bytes_read += rc;
    } while ((bytes_read < count) && (rc != -1));
    
    if (rc == -1)
        return rc;
    else
        return bytes_read;
}

size_t buf_fread(char *ptr, size_t size, size_t nmemb, FILE *stream)
{
    size_t rc;
	
    if (ptr == NULL)
    {
        error(E_TRACE, "buf_fread() was passed a null buffer");
        errno = EINVAL;
        return -1;
    }
    
    do {
        rc = fread(ptr, size, nmemb, stream);
    } while ((rc == -1) && (errno == EINTR));
    
    return rc;
}

ssize_t buf_write(int fd, char *buf, size_t count)
{
    ssize_t rc;
	
    if (buf == NULL)
    {
        error(E_TRACE, "buf_write() was passed a null buffer");
        errno = EINVAL;
        return -1;
    }
    
    do {
        rc = write(fd, buf, count);
    } while ((rc == -1) && (errno == EINTR));
    
    return rc;
}

// uses buf_write() to write repeatedly until <count> bytes have been written
ssize_t buf_repeat_write(int fd, char *buf, size_t count)
{
    ssize_t rc, bytes_written = 0;
	
    do {
        rc = buf_write(fd, buf + bytes_written, count - bytes_written);
        bytes_written += rc;
    } while ((bytes_written < count) && (rc != -1));
    
    if (rc == -1)
        return rc;
    else
        return bytes_written;
}

size_t buf_fwrite(char *ptr, size_t size, size_t nmemb, FILE *stream)
{
    size_t rc;
	
    if (ptr == NULL)
    {
        error(E_TRACE, "buf_fwrite() was passed a null buffer");
        errno = EINVAL;
        return -1;
    }
    
    do {
        rc = fwrite(ptr, size, nmemb, stream);
    } while ((rc == -1) && (errno == EINTR));
    
    error(E_TRACE, "    buf_fwrite: wrote %d items", rc);
    
    return rc;
}

size_t buf_repeat_fwrite(char *ptr, size_t nmemb, FILE *stream)
{
    ssize_t rc, bytes_written = 0;
	
    do {
        rc = buf_fwrite(ptr + bytes_written, 1, nmemb - bytes_written, stream);
        bytes_written += rc;
    } while ((bytes_written < nmemb) && (rc != -1));
    
    if (rc == -1)
        return rc;
    else
        return bytes_written;
}

/* see if the buffer has a full line; if so, return its length, including
 * the \n; otherwise return 0 */
int buf_hasline(char *buf, int buf_size)
{
    int i;
    
    /* scan to see if the buffer has a full line; if it does,
     * set i to the newline */
    for (i = 0; i < buf_size; i++)
    {
        // note that we do NOT want to return if we've got a \r because
        // it is not actually the end of the line until we get the \n
        if (buf[i] == '\n')
            break;
    }
    // if i = the size then we didn't find anything
    if ((i == (buf_size - 1)) && (i != '\n'))
        return 0;
    
    //error(E_TRACE, "Line has %d chars", i + 1);
    return i + 1;
}

// Reads a line from the buffer, moving its contents appropriately.
// Returns 0 for failure, the length of the line for success.
int buf_getline(char *buf, size_t buf_size, char *line, size_t line_size)
{
    int length;

    length = buf_hasline(buf, buf_size);
    
    // if there is no line to get, return FALSE
    if (length == 0)
        return FALSE;
    
    if (line_size > length)
        line_size = length;
    
    // copy the line data across
    strncpy(line, buf, line_size);
    
    /* now, replace the newline with a \0; this will go to the end of
     * the buffer if the buffer wasn't big enough */
    line[line_size - 1] = '\0';
    
    /* strip whitespace etc */
    strip(line);
    
    // move the data along in the buffer
    memmove(buf, buf + line_size, buf_size - line_size);
    
    //error(E_TRACE, "    getline: Got \"%s\"", line);
    
    return line_size;
}
