#include <darxite.h>
#include "buf.h"
#include "global.h"

struct buf_t {
	/* FIXME: dyn */
	char data[32000]; /* real men don't need powers of 2 */
	int used;
	/* Never read more then this in one call (-1 = unlimited) */
	int trans_input_size;
	/* Never write more then this in one call (-1 = unlimited) */
	int trans_output_size;
};

struct buf_t *buf_new(void)
{
	struct buf_t *new;

	new = (struct buf_t *) dxmalloc(sizeof(struct buf_t));

	new->used = 0;
	new->trans_input_size = new->trans_output_size = -1;
	return new;
}

void buf_free(struct buf_t *buf)
{
	free(buf);
}

void buf_set_input_size(struct buf_t *buf, size_t size)
{
	buf->trans_input_size = size;
}

int buf_get_input_size(const struct buf_t *buf)
{
	return buf->trans_input_size;
}

void buf_set_output_size(struct buf_t *buf, size_t size)
{
	buf->trans_output_size = size;
}

int buf_get_output_size(const struct buf_t *buf)
{
	return buf->trans_output_size;
}

int buf_used(const struct buf_t *buf)
{
	return buf->used;
}

/* Return the max current read size */
static int buf_readsize(const struct buf_t *buf)
{
	if(buf->trans_input_size == -1) return(sizeof(buf->data)-(buf->used));
	return (min(sizeof(buf->data)-(buf->used), buf->trans_input_size));
}

/* Return the max current write size */
static int buf_writesize(const struct buf_t *buf)
{
	if(buf->trans_output_size == -1) return buf->used;
	return (min(buf->used, buf->trans_output_size));
}

void buf_flush(struct buf_t *buf)
{
	buf->used = 0;
}

int buf_read(struct buf_t *buf, int fd)
{
	int ret;
	
	if(fd == -1 || buf == NULL) {
		/* bad fd passed, or bad buf_t; invalid argument */
		errno = EINVAL;
		return -1;
	}

	while(1) {
		if(buf->used >= sizeof(buf->data)) {
			/* buffer is full. return error EAGAIN, to pretend
			 * there was nothing else in the buffer. this probably
			 * should never happen */
			errno = EAGAIN;
			return -1;
		}
		
		/* do the read */
		ret = read(fd, buf->data+buf->used,
				buf_readsize(buf));

		/* EINTR means 'try again now' */
		if(ret == -1 && errno == EINTR) continue;
		
		if(ret > 0) {
			/* got data; adjust used */
			buf->used += ret;
		}
		return ret;
	}
	/* not reached */
}

int buf_fread(struct buf_t *buf, FILE *f)
{
	int ret;
	
	if(f == NULL || buf == NULL) {
		/* bad f passed, or bad buf_t; invalid argument */
		errno = EINVAL;
		return -1;
	}

	while(1) {
		if(buf->used >= sizeof(buf->data)) {
			/* buffer is full. return error EAGAIN, to pretend
			 * there was nothing else in the buffer. this probably
			 * should never happen */
			errno = EAGAIN;
			return -1;
		}
		
		/* do the read */
		ret = fread(buf->data+buf->used, 1, 
				buf_readsize(buf), f);

		/* EINTR means 'try again now' */
		if(ret == -1 && errno == EINTR) continue;
		
		if(ret > 0) {
			/* got data; adjust used */
			buf->used += ret;
		}
		
		return ret;
	}
	/* not reached */
}

int buf_write(struct buf_t *buf, int fd)
{
	int ret;
	
	if(fd == -1 || buf == NULL) {
		/* bad fd passed, or bad buf_t; invalid argument */
		errno = EINVAL;
		return -1;
	}

	while(1) {
		/* do the write */
		ret = write(fd, buf->data, buf_writesize(buf));

		/* EINTR means 'try again now' */
		if(ret == -1 && errno == EINTR) continue;
		
		if(ret > 0) {
			/* sent data; adjust used and delete it */
			memmove(buf->data, buf->data+ret, buf->used-ret);
			buf->used -= ret;
		}
		
		return ret;
	}
	/* not reached */
}

int buf_fwrite(struct buf_t *buf, FILE *f)
{
	int ret;
	
	if(f == NULL || buf == NULL) {
		/* bad f passed, or bad buf_t; invalid argument */
		errno = EINVAL;
		return -1;
	}

	while(1) {
		/* do the write */
		ret = fwrite(buf->data, 1, buf_writesize(buf), f);
        //error(E_TRACE, "Wrote \"%*s\"", buf_writesize(buf), buf->data);

		/* EINTR means 'try again now' */
		if(ret == -1 && errno == EINTR) continue;

		if(ret > 0) {
			/* sent data; adjust used and delete it */
			memmove(buf->data, buf->data+ret, buf->used-ret);
			buf->used -= ret;
		}
		
		return ret;
	}
	/* not reached */
}

/* see if the buffer has a full line; if so, return its length, including
 * the \n; otherwise return -1 */
int buf_hasline(const struct buf_t *buf)
{
	int i;

	/* scan to see if the buffer has a full line; if it does,
	 * set i to the newline */
	for(i = 0; i < buf->used; i++) {
		if(buf->data[i] == '\n' || buf->data[i] == '\r') break;
	}
	/* if i == buf->used, we found nothing, so return -1 */
	if(i == buf->used)
		return -1;

	return i+1;
}

/* quick hack to remove leading and trailing nulls from the buf */
void buf_stripnulls(struct buf_t *buf)
{
	while(buf->data[0] == '\0') {
		memmove(buf->data, buf->data+1, buf->used-1);
		buf->used--;
	}
	while(*(buf->data+buf->used-1) == '\0') {
		buf->used--;
	}
}

int buf_getline(struct buf_t *buf, char *str, size_t bufsize)
{
	int i, num;

	/* sanity: */
	buf_stripnulls(buf);
	
	i = buf_hasline(buf);
	
	/* if i == -1, there is no new line, so return 0 */
	if(i == -1)
		return 0;

	/* i is now the position of the newline for the upcoming
	 * line.  strncpy at most, the max of bufsize and i characters,
	 * to str.  */
	num = bufsize > i? i:bufsize;
	strncpy(str, buf->data, num);

	/* now, replace the newline with a \0; this will go to the end of
	 * the buffer if the buffer wasn't big enough */
	str[num-1] = '\0';

	/* move the line out of the buffer, including the \n/\r.  also, if the
	 * next char is \r and this is \n, or vice versa, chop that too, so we
	 * deal with \r\n and \n\r without mangling \n\n */
	{
		int move = i;
		char last;
		last = buf->data[move-1];
		if(buf->used > move) {
			if(buf->data[move-1] == '\n' && buf->data[move] == '\r') move++;
			else if(buf->data[move-1] == '\r' && buf->data[move] == '\n') move++;
		}
		memmove(buf->data, buf->data+move, buf->used-move);
		buf->used -= move;
	}

	/* strip whitespace etc */
	strip(str);

	/* if i > bufsize, return 2, indicating we didn't have quite
	 * enough space */
	if(i > bufsize) return 2;

	/* success */
	return 1;
}

int buf_printf(struct buf_t *buf, const char *fmt, ...)
{
	va_list msg;
	int ret;

	va_start(msg, fmt);
	ret = vsnprintf(buf->data + buf->used, sizeof(buf->data) - buf->used, fmt, msg);
	va_end(msg);

	buf->used += ret;

	return ret;
}

int buf_grab(struct buf_t *buf, char *s, size_t size)
{
	if(size > buf->used) size = buf->used;
	memmove(s, buf->data, size);
	return size;
}

