// Simple utility to send a command to the Darxite daemon.

#include <stdio.h>  // for printf() etc.
#include <getopt.h> // for getopt()
#include <stdlib.h> // for atoi()
#include <string.h>	// for strerror()
#include <unistd.h> // for write()
#include <errno.h>  // for errno

#include <darxite.h>

#define OPTION_STRING	"vhc:ef:qrt:"
#define VERSION			"0.2"
#define AUTHORS			"Ashley Montanaro"

BOOL Echo = FALSE, Quiet = FALSE, NoResponse = FALSE;
char RemoteHost[256], FileToRead[256];
int RemotePort = 0, Timeout = 10;

char *ReadCmdLine(int argc, char **argv);
void PrintVersion(void);
void Usage(char *prog_name);

void PrintVersion(void)
{
    printf("DarxCmd utility v%s, release %s (%s) by %s\n", VERSION,
           RELEASE_VER, RELEASE_NAME, AUTHORS);
}

void Usage(char *prog_name)
{
    printf("Usage: %s [OPTIONS] command\n"
           "where options are:\n"
           "-v, --version\t\t\tShow version and exit\n"
           "-h, --help\t\t\tShow some usage information\n"
           "-c<host>:<port> --connect<host>\tConnect to <host> on <port>\n"
           "-e, --echo\t\t\tEcho each command as it's sent\n"
           "-q, --quiet\t\t\tDon't echo the daemon's responses\n"
           "-r, --noresponse\t\tDon't get the daemon's responses\n"
           "-f<file>, --file <file>\t\tRead commands from file\n"
           "-t<secs>, --timeout <secs>\tTime out response after secs\n"
           "If command is blank or -, will read from standard input\n",
           prog_name);
    exit(0);
}

char *ReadCmdLine(int argc, char **argv)
{
    char opt;
    int option_index;

    static struct option long_options[] = {
        { "version", 	0, 0, 'v' },
        { "help",		0, 0, 'h' },
        { "connect",	1, 0, 'c' },
        { "echo",		0, 0, 'e' },
        { "quiet",		0, 0, 'q' },
        { "noresponse", 0, 0, 'r' },
        { "file",		1, 0, 'f' },
        { "timeout",	1, 0, 't' },
        { 0, 			0, 0,  0  }
    };

    opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                      &option_index);
    while (opt != -1)
    {
        switch (opt)
        {
        case 'v':
            PrintVersion();
            exit(0);
            break;

        case 'h':
            Usage(argv[0]);
            break;

        case 'c':
            if (strchr(optarg, ':'))
            {
                memset(RemoteHost, 0, sizeof(RemoteHost));
                strncpy(RemoteHost, optarg, strchr(optarg, ':') - optarg);
                RemotePort = atoi(strchr(optarg, ':') + 1);
                if ((strlen(RemoteHost) == 0) || (RemotePort <= 0))
                {
                    printf("Invalid host/port to connect to daemon\n");
                    exit(0);
                }
            }
            else
            {
                printf("Syntax: -c <host>:<port>\n");
                exit(0);
            }
            break;
            
        case 'e':
            Echo = TRUE;
            break;

        case 'p':
            RemotePort = atoi(optarg);
            break;
            
        case 'q':
            Quiet = TRUE;
            break;
            
        case 'r':
            NoResponse = TRUE;
            break;
            
        case 'f':
            strcpy(FileToRead, optarg);
            if (freopen(optarg, "r", stdin) != stdin)
            {
                fprintf(stderr, "Couldn't read file \"%s\": %s",
                        optarg, strerror(errno));
                exit(1);
            }
            break;

        case 't':
            Timeout = atoi(optarg);
            break;
        }
        opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                          &option_index);        
    }
    
    if (optind < argc)
        return argv[optind];
    else
       return NULL;
}

int main(int argc, char *argv[])
{
    int fd, rc;
    BOOL last_command = TRUE;
    char buffer[32768], *command;

    command = ReadCmdLine(argc, argv);

    if (strcmp(RemoteHost, "") && (RemotePort > 0))
    {
        sprintf(buffer, "Enter password for daemon on %s: ", RemoteHost);
        fd = DX_ConnectRemoteClient(RemoteHost, RemotePort, getpass(buffer),
                                    "darxcmd");
    }
    else
    {
        fd = DX_ConnectClient("darxcmd");
    }
    
    if (fd < 0)
    {
        fprintf(stderr, "Couldn't connect to daemon: %s\n",
                strerror(DX_errno));
        return 1;
    }

    memset(buffer, 0, sizeof(buffer));
    if (!command || (!strcmp(command, "-")))
    {
        if (fgets(buffer, sizeof(buffer), stdin) != NULL)
        {
            last_command = FALSE;
        }
    }
    else
    {
        strncpy(buffer, command, sizeof(buffer));
    }

    do {
        if (buffer[strlen(buffer) - 1] != '\n')
            strcat(buffer, "\n");
        // skip it if it's an empty command
        if (!strcmp(buffer, "\n"))
            break;
        // Write (and echo to screen?) the command
        if (write(fd, buffer, strlen(buffer)) < 0)
            perror("write");
        else if (Echo)
            printf("%s", buffer);
    
        if (!NoResponse)
        {
            do {
                rc = DX_GetResponse(fd, buffer, sizeof(buffer) - 1, Timeout);
                if (rc == DX_LIB_OK)
                {
                    if (!Quiet)
                        printf("%s", buffer);    
                }
                else if (rc == DX_LIB_TIMEOUT)
                {
                    printf("There was no response within %d second%s.\n",
                           Timeout, (Timeout == 1) ? "" : "s");
                }
                else if (rc == DX_LIB_MORE_DATA)
                {
                    buffer[sizeof(buffer) - 1] = '\0';
                    if (!Quiet)
                        printf("%s", buffer);
                }
                else
                {
                    printf("Couldn't get the response: %s",
                           strerror(DX_errno));
                }
            } while (rc == DX_LIB_MORE_DATA);
        }
        
        memset(buffer, 0, sizeof(buffer));
        if (!last_command)
        {
            if (fgets(buffer, sizeof(buffer), stdin) == NULL)
                last_command = TRUE;
            else if (!strcmp(buffer, "\n"))
                last_command = TRUE;
        }
    } while (!last_command);
    
    DX_DisconnectClient(fd);
    
    return 0;
}
