// Darxite-based Web browser

#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <getopt.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/stat.h>
#include <darxite.h>
#include <gtk/gtk.h>
#include <gtk-xmhtml/gtk-xmhtml.h>

#define TEMP_FILE "/tmp/nightscape.tmp"
#define VERSION "0.1"
#define OPTION_STRING	"vhc:"
#define AUTHORS "Ashley Montanaro"

enum {
    NOT_WAITING,
    WAIT_GET_RESPONSE,
    WAIT_DATA,
    WAIT_CANCEL_RESPONSE
} WaitState;

GtkWidget *MainWindow, *MainVbox;
GtkWidget *UrlEntry, *UrlGo, *Html;
GtkWidget *StatusBar, *SaveDialogue;

char DaemonHost[256];
char ThisUrl[256], ReqdUrl[256];
int DaemonFd, DaemonPort, PageLen;
int RespStage;
char *PageBuf;

void SetStatus(const char *string, ...);
void ExpandUrl(char *url, int url_max);

void PrintVersion(void)
{
    printf("Nightscape v%s, release %s (%s) by %s\n", VERSION,
           RELEASE_VER, RELEASE_NAME, AUTHORS);
}

void Usage(char *prog_name)
{
    printf("Usage: %s [OPTIONS] command\n"
           "where options are:\n"
           "-v, --version\t\t\tShow version and exit\n"
           "-h, --help\t\t\tShow some usage information\n"
           "-c<host>:<port> --connect<host>\tConnect to <host> on <port>\n",
           prog_name);
    exit(0);
}

char *ReadCmdLine(int argc, char **argv)
{
    char opt;
    int option_index;

    static struct option long_options[] = {
        { "version", 	0, 0, 'v' },
        { "help",		0, 0, 'h' },
        { "connect",	1, 0, 'c' },
        { 0, 			0, 0,  0  }
    };
    
    opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                      &option_index);
    while (opt != -1)
    {
        switch (opt)
        {
        case 'v':
            PrintVersion();
            exit(0);
            break;

        case 'h':
            Usage(argv[0]);
            break;

        case 'c':
            if (strchr(optarg, ':'))
            {
                memset(DaemonHost, 0, sizeof(DaemonHost));
                strncpy(DaemonHost, optarg, strchr(optarg, ':') - optarg);
                DaemonPort = atoi(strchr(optarg, ':') + 1);
                if ((strlen(DaemonHost) == 0) || (DaemonPort <= 0))
                {
                    printf("Invalid host/port to connect to daemon\n");
                    exit(0);
                }
            }
            else
            {
                printf("Syntax: -c <host>:<port>\n");
                exit(0);
            }
            break;
        }
        opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                          &option_index);        
    }
    
    if (optind < argc)
        return argv[optind];
    else
       return NULL;
}

void close_window(void)
{
    gtk_main_quit();
}

void save_ok(void)
{
    char file_buf[256];
    char *filename =
        gtk_file_selection_get_filename(GTK_FILE_SELECTION(SaveDialogue));
    FILE *file;
    int i;

    strcpy(file_buf, filename);
    gtk_widget_destroy(SaveDialogue);
    file = fopen(file_buf, "w");
    if (!file)
    {
        SetStatus("Couldn't open file: %s", strerror(errno));
        return;
    }
    for (i = 0; i < PageLen; i++)
        fputc(PageBuf[i], file);
    fclose(file);
    SetStatus("File \"%s\" saved.", file_buf);
}

void save_page(void)
{
    SaveDialogue = gtk_file_selection_new("Save As...");
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(SaveDialogue)->ok_button),
                       "clicked", GTK_SIGNAL_FUNC(save_ok), NULL);
    gtk_widget_show(SaveDialogue);
}

void SetStatus(const char *string, ...)
{
    va_list ap;
    char status_text[256];
    
    if (string == NULL)
        return;
    
    va_start(ap, string);
    vsnprintf(status_text, sizeof(status_text) - 1, string, ap);
    status_text[2047] = '\0';
    if (status_text[strlen(status_text) - 1] == '\n')
        status_text[strlen(status_text) - 1] = '\0';
    gtk_label_set(GTK_LABEL(StatusBar), status_text);
    va_end(ap);
}

void ExpandUrl(char *link, int url_max)
{
    char buffer[256], lastpath[256];
    char protocol[10], server[256], path[256];

    if (strstr(link, "://")) // fully-qualified url
        return;
    DX_ParseUrl(ThisUrl, protocol, server, sizeof(server), path, sizeof(path));
    memset(lastpath, 0, sizeof(lastpath));
    strncpy(lastpath, path, strrchr(path, '/') - path);
    if (strlen(lastpath) == 0)
        strcpy(lastpath, "/");
    if (*link == '/') // if it's a fully-qualified path
        sprintf(buffer, "%s://%s%s", protocol, server, link);
    else if (path[strlen(path) - 1] == '/') // current path ends in /
        sprintf(buffer, "%s://%s%s%s", protocol, server, path, link);
    else 
        sprintf(buffer, "%s://%s%s/%s", protocol, server, lastpath, link);

    if (strchr(buffer, '#'))
        *strchr(buffer, '#') = '\0';
    strncpy(link, buffer, url_max);
    return;
}

void get_url(const char *url)
{
    char buffer[512], buffer2[256];

    free(PageBuf);
    remove(TEMP_FILE);
    SetStatus("Getting %s...", url);
    if (strstr(url, "://") != NULL) // protocol specified
        strcpy(buffer2, url);
    else if ((*url == '/') || (*url == '~'))
        sprintf(buffer2, "file://%s", url);
    else
        sprintf(buffer2, "http://%s", url);
    sprintf(buffer, "Get \"%s\" | " TEMP_FILE " | | | n\n", buffer2);
    write(DaemonFd, buffer, strlen(buffer));
    strcpy(ReqdUrl, buffer2);
    RespStage = WAIT_GET_RESPONSE;
}

void go_clicked(void)
{
    char *url, buffer[256];

    // if we want to stop the transfer
    if (RespStage == WAIT_DATA)
    {
        sprintf(buffer, "Cancel %s\n", ReqdUrl);
        write(DaemonFd, buffer, strlen(buffer));
        RespStage = WAIT_CANCEL_RESPONSE;
    }
    else
    {
        url = gtk_entry_get_text(GTK_ENTRY(UrlEntry));
        get_url(url);
    }
}

void link_clicked(void)
{
    char link[256];

    strcpy(link, GTK_XMHTML(Html)->html.selected->href);
    ExpandUrl(link, sizeof(link));
    get_url(link);
}

void link_over(void)
{
    /*char buffer[256];

    printf("%d\n", GTK_XMHTML(Html)->html.selected);
    strcpy(buffer, GTK_XMHTML(Html)->html.selected->href);
    ExpandUrl(buffer, sizeof(buffer));
    SetStatus(buffer);*/
}

void got_data(gpointer data, gint source, GdkInputCondition condition)
{
    char c, buffer[256];
    struct stat stat_buf;
    int rc, i = 0;
    FILE *file;
    
    rc = DX_GetResponse(DaemonFd, buffer, sizeof(buffer), 10);
    
    switch (RespStage)
    {        
    case WAIT_GET_RESPONSE:
        if ((rc != DX_LIB_OK) || (atoi(buffer) > 900))
        {
            SetStatus(buffer);
            RespStage = NOT_WAITING;
            return;
        }
        SetStatus("Downloading %s...", ReqdUrl);
        gtk_label_set(GTK_LABEL(GTK_BIN(UrlGo)->child), "Stop");
        RespStage = WAIT_DATA;
        break;
        
    case WAIT_DATA:
        RespStage = NOT_WAITING;
        if ((rc != DX_LIB_OK) || (atoi(buffer) > 900))
        {
            SetStatus(buffer);
            return;
        }
        if (stat(TEMP_FILE, &stat_buf) == -1)
        {
            SetStatus("Output file doesn't seem to exist...");
            return;
        }
        PageLen = stat_buf.st_size;
        PageBuf = malloc(PageLen);
        file = fopen(TEMP_FILE, "r");
        memset(PageBuf, 0, PageLen);
        while ((c = fgetc(file)) > 0)
            PageBuf[i++] = c;
        fclose(file);
        gtk_xmhtml_source(GTK_XMHTML(Html), PageBuf);
        sscanf(buffer, "File %s complete", ThisUrl);
        gtk_entry_set_text(GTK_ENTRY(UrlEntry), ThisUrl);
        gtk_label_set(GTK_LABEL(GTK_BIN(UrlGo)->child), "Go");
        SetStatus("Done.");
        break;
        
    case WAIT_CANCEL_RESPONSE:
        SetStatus(buffer);
        RespStage = NOT_WAITING;
        gtk_label_set(GTK_LABEL(GTK_BIN(UrlGo)->child), "Go");
        break;
        
    default:
        SetStatus("Unexpected response: %s!", buffer);
        break;
    }
}

void CreateMenus(void)
{
    GtkWidget *menu_bar, *menu, *menu_item;

    menu_bar = gtk_menu_bar_new();
    gtk_box_pack_start(GTK_BOX(MainVbox), menu_bar, FALSE, FALSE, 0);
    
    // Create "File" menu
    menu = gtk_menu_new();
    
    // Add "Save As" menu item
    menu_item = gtk_menu_item_new_with_label("Save As...");
    gtk_menu_append(GTK_MENU(menu), menu_item);
    gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
                       GTK_SIGNAL_FUNC(save_page), NULL);
    gtk_widget_show(menu_item);
    
    // Add separator
    menu_item = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(menu), menu_item);
    gtk_widget_set_sensitive(menu_item, FALSE);
    gtk_widget_show(menu_item);
    
    // Add "Exit" menu item
    menu_item = gtk_menu_item_new_with_label("Exit");
    gtk_menu_append(GTK_MENU(menu), menu_item);
    gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
                       GTK_SIGNAL_FUNC(close_window), NULL);
    gtk_widget_show(menu_item);
    
    // Create "File" label
    menu_item = gtk_menu_item_new_with_label("File");
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), menu);
    gtk_widget_show(menu_item);
    
    gtk_menu_bar_append(GTK_MENU_BAR(menu_bar), menu_item);
    gtk_widget_show(menu_bar);
}

int main(int argc, char *argv[])
{
    GtkWidget *hbox, *label;
    char buffer[256];

    gtk_init(&argc, &argv);
    ReadCmdLine(argc, argv);
    
    if (strcmp(DaemonHost, "") && (DaemonPort > 0))
    {
        sprintf(buffer, "Enter password for daemon on %s: ", DaemonHost);
        DaemonFd = DX_ConnectRemoteClient(DaemonHost, DaemonPort,
                                          getpass(buffer), "Nightscape");
    }
    else
    {
        DaemonFd = DX_ConnectClient("Nightscape");
    }
    
    if (DaemonFd < 0)
    {
        fprintf(stderr, "Couldn't connect to daemon: %s\n",
                strerror(DX_errno));
        return 1;
    }
    
    MainWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(MainWindow), "Nightscape");
    gtk_widget_set_usize(MainWindow, 500, 500);
    // allow the window to grow or shrink automatically
    gtk_window_set_policy(GTK_WINDOW(MainWindow), TRUE, TRUE, TRUE);
    gtk_window_position(GTK_WINDOW(MainWindow), GTK_WIN_POS_CENTER);
    //gtk_container_border_width(GTK_CONTAINER(MainWindow), 10);
    gtk_signal_connect (GTK_OBJECT (MainWindow), "delete_event",
                        GTK_SIGNAL_FUNC(close_window), NULL);

    MainVbox = gtk_vbox_new(FALSE, 2);
    gtk_container_add(GTK_CONTAINER(MainWindow), MainVbox);
    gtk_widget_show(MainVbox);

    CreateMenus();
    
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(MainVbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show(hbox);

    label = gtk_label_new("Location:");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
    gtk_widget_show(label);
    
    UrlEntry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox), UrlEntry, TRUE, TRUE, 0);
    gtk_widget_show(UrlEntry);
    
    UrlGo = gtk_button_new_with_label("Go");
    gtk_signal_connect(GTK_OBJECT(UrlGo), "clicked", go_clicked, NULL);
    gtk_signal_connect_object(GTK_OBJECT(UrlEntry), "activate",
                              GTK_SIGNAL_FUNC(gtk_button_clicked),
                              GTK_OBJECT(UrlGo));
    gtk_box_pack_start(GTK_BOX(hbox), UrlGo, FALSE, FALSE, 5);
    gtk_widget_show(UrlGo);
    
    Html = gtk_xmhtml_new();
    gtk_box_pack_start(GTK_BOX(MainVbox), Html, TRUE, TRUE, 0);
    gtk_xmhtml_set_anchor_buttons(GTK_XMHTML(Html), FALSE);
    gtk_xmhtml_set_anchor_underline_type(GTK_XMHTML(Html), 1);
    gtk_signal_connect(GTK_OBJECT(Html), "activate", link_clicked, NULL);
    gtk_signal_connect(GTK_OBJECT(Html), "anchor_track", link_over, NULL);
    //gtk_xmhtml_set_event_proc(GTK_XMHTML(Html), link_clicked);
    gtk_xmhtml_set_allow_images(GTK_XMHTML(Html), FALSE);
    gtk_widget_show(Html);
    
    StatusBar = gtk_label_new("Welcome to Nightscape " VERSION " by "
                              AUTHORS ", built for "
                              "Darxite " RELEASE_VER " (" RELEASE_NAME ").");
    gtk_label_set_justify(GTK_LABEL(StatusBar), GTK_JUSTIFY_LEFT);
    gtk_box_pack_start(GTK_BOX(MainVbox), StatusBar, FALSE, FALSE, 0);
    gtk_widget_show(StatusBar);
    
    // use async input for responses from the daemon
    gdk_input_add(DaemonFd, GDK_INPUT_READ, got_data, NULL);
    gtk_widget_show(MainWindow);
    gtk_main();
    return 0;
}
