// File dealing with FTP commands

#include <grp.h>

#include "commands.h"

void nospace(char *str)
{
    if (str[strlen(str) - 1] == ' ')
        str[strlen(str) - 1] = '\0';
}

char *Completion(char *text, int state)
{
    static char *matches[256];
    static int match_count = 0;
    char cmd[256];
    int i;
    
    if (!rl_line_buffer || (strlen(rl_line_buffer) == 0))
        return NULL;

    // do we want remote or local completion?
    if (strchr(rl_line_buffer, ' '))
    {
        memset(cmd, 0, sizeof(cmd));
        strncpy(cmd, rl_line_buffer,
                strchr(rl_line_buffer, ' ') - rl_line_buffer);
        //printf("*%s**%s*\n", text, text + strlen(cmd) + 1);
        if (!strcasecmp(cmd, "lls") || !strcasecmp(cmd, "lcd") ||
            !strcasecmp(cmd, "lrm") || !strcasecmp(cmd, "lrename") ||
            !strcasecmp(cmd, "lrmdir") || !strcasecmp(cmd, "lmkdir") ||
            !strcasecmp(cmd, "put"))
        {
            return filename_completion_function(text, state);
        }
    }
    
    // from here on, do remote completion
    if (!LsCacheOk)
        return NULL;
    if (state == 0)
    {
        match_count = 0;
        for (i = 0; i < LsCacheCount; i++)
        {
            if (!strncmp(LsCache[i], text, strlen(text)))
                matches[match_count++] = strdup(LsCache[i]);
        }
    }
    if (state < match_count)
        return matches[state];
    else
        return NULL;
}

int GetResponse(BOOL show_response)
{
    int rc;

    while ((rc = DX_GetResponse(DaemonFd, RespBuf,
                                sizeof(RespBuf) - 1, 30))
           != DX_LIB_ERROR)
    {
        if (rc == DX_LIB_OK)
        {
            if (show_response)
                printf(RespBuf);
            return DX_LIB_OK;
        }
        else if (rc == DX_LIB_MORE_DATA)
        {
            if (show_response)
            {
                RespBuf[sizeof(RespBuf) - 1] = '\0';
                printf(RespBuf);
            }
            return DX_LIB_MORE_DATA;
        }
        else if (rc == DX_LIB_TIMEOUT)
        {
            if (show_response)
                printf("There was no response from the daemon");
            return DX_LIB_TIMEOUT;
        }
    }
    sprintf(RespBuf, "Couldn't get the response: %s", strerror(DX_errno));
    if (show_response)
        printf(RespBuf);
    return DX_ERROR;
}

int DaemonWrite(BOOL show_response, const char *string, ...)
{
    va_list ap;
    char buffer[4096];

    if (string == NULL)
        return DX_LIB_ERROR;
    
    va_start(ap, string);
    vsnprintf(buffer, sizeof(buffer) - 1, string, ap);
    strcat(buffer, "\n");
    if (write(DaemonFd, buffer, strlen(buffer)) < 0)
        return DX_LIB_ERROR;
    else
        return GetResponse(show_response);
}

BOOL OpenUrl(char *url, BOOL anon_login)
{
    char *resp;
    char protocol[10], server[256], path[256];
    char buffer[256];
    int i;
    
    if (!url)
        return FALSE;

    if ((DX_ParseUrl(url, protocol, server, sizeof(server),
                    path, sizeof(path)) != DX_LIB_OK) ||
        strcasecmp(protocol, "ftp"))
    {
        printf("Invalid FTP URL \"%s\"\n", url);
        return FALSE;
    }
    if (LoggedIn)
    {
        LoggedIn = FALSE;
        DaemonWrite(TRUE, "close");
    }
    for (i = 0; i < ServerCount; i++)
    {
        if (ServAlias[i] && !strcasecmp(ServAlias[i], server))
        {
            strcpy(UserName, ServLogin[i]);
            strcpy(Password, ServPassword[i]);
            anon_login = FALSE;
            break;
        }
    }
    
    DaemonWrite(TRUE, "open %s", server);
    if (atoi(RespBuf) >= 400)
        return FALSE;
    
    // get an actual server name to correspond to the alias we just opened
    DaemonWrite(FALSE, "ServerForAlias %s", server);
    if (atoi(RespBuf) == DX_ERROR)
    {
        printf("Alias \"%s\" appears not to exist\n", server);
        strcpy(Server, server);
    }
    else
    {
        // strip trailing newline
        RespBuf[strlen(RespBuf) - 1] = '\0';
        strcpy(Server, RespBuf + 4);
    }
    strcpy(Path, "/");
    if (anon_login)
    {
        DaemonWrite(TRUE, "ftp user anonymous");
        if (atoi(RespBuf) >= 400)
        {
            LoggedIn = FALSE;
            DaemonWrite(TRUE, "close");
            return FALSE;
        }
        strcpy(UserName, "anonymous");
        strcpy(Password, DX_EMailAddress);
        DaemonWrite(TRUE, "ftp pass %s", DX_EMailAddress);
        if (atoi(RespBuf) >= 400)
        {
            LoggedIn = FALSE;
            DaemonWrite(TRUE, "close");
            return FALSE;
        }
    }
    else
    {
        if (!*UserName)
        {
            printf("User name for %s: ", server);
            resp = fgets(buffer, sizeof(buffer), stdin);
            if (!resp || (strlen(resp) <= 1))
            {
                LoggedIn = FALSE;
                DaemonWrite(TRUE, "close");
                return FALSE;
            }
            buffer[strlen(buffer) - 1] = '\0';
            strcpy(UserName, buffer);
        }
        DaemonWrite(TRUE, "ftp user %s", UserName);
        if (!strlen(RespBuf) || (atoi(RespBuf) >= 400))
        {
            LoggedIn = FALSE;
            DaemonWrite(TRUE, "close");
            return FALSE;
        }
        if (!*Password)
        {
            sprintf(buffer, "Password for %s: ", server);
            resp = getpass(buffer);
            if (!resp)
            {
                LoggedIn = FALSE;
                DaemonWrite(TRUE, "close");
                return FALSE;
            }
            strcpy(Password, resp);
        }
        DaemonWrite(TRUE, "ftp pass %s", Password);
        if (atoi(RespBuf) >= 400)
        {
            LoggedIn = FALSE;
            DaemonWrite(TRUE, "close");
            return FALSE;
        }
    }
    LoggedIn = TRUE;
    if (strcmp(path, "/"))
        Cwd(path);
    else
        Pwd();
    LsCacheOk = FALSE;
    return TRUE;
}

BOOL Cwd(char *path)
{
    if (!path)
        return FALSE;
    DaemonWrite(FALSE, "ftp cwd %s", path);
    if (atoi(RespBuf) >= 400)
    {
        printf(RespBuf);
        return FALSE;
    }
    
    LsCacheOk = FALSE;
    return Pwd();
}

BOOL Pwd(void)
{
    char buffer[256], *buf_ptr;
    
    DaemonWrite(FALSE, "ftp pwd");
    if (atoi(RespBuf) >= 400)
    {
        printf(RespBuf);
        return FALSE;
    }
    strcpy(buffer, RespBuf);
    buf_ptr = strchr(buffer, '"');
    while (*(++buf_ptr) != '"')
        ;
    *buf_ptr = '\0';
    strcpy(Path, strchr(buffer, '"') + 1);
    if (Path[strlen(Path) - 1] != '/')
        strcat(Path, "/");
    return TRUE;
}

void RemoteLs(const char *path, const char *options)
{
    int i, new_rc, rc;
    char *line = NULL, *last_line = NULL, *buf_ptr, buffer[4096 + 256];
    char buffer2[256];
    BOOL show_resp;

    // if it's not a list of the current dir, we don't want to cache it
    if (strcmp(path, "."))
    {
        if (strchr(options, 'l'))
        {
            show_resp = TRUE;
            new_rc = DaemonWrite(TRUE, "LIST %s", path);
        }
        else
        {
            show_resp = FALSE;
            new_rc = DaemonWrite(FALSE, "LIST %s", path);
        }
    }
    else
    {
        if (strchr(options, 'l'))
        {
            show_resp = TRUE;
            new_rc = DaemonWrite(TRUE, "LIST");
        }
        else
        {
            show_resp = FALSE;
            new_rc = DaemonWrite(FALSE, "LIST");
        }
        for (i = 0; i < LsCacheCount; i++)
        {
            free(LsCache[i]);
            LsCache[i] = NULL;
        }
        LsCacheCount = 0;
        LsCacheOk = FALSE;
    }
    if (atoi(RespBuf) > 400)
        return;

    memset(buffer2, 0, sizeof(buffer2));
    do {
        rc = new_rc;
        if ((rc == DX_LIB_ERROR) || (rc == DX_LIB_TIMEOUT))
            return;
        //printf("********************\n%s\n********************\n", RespBuf);
        memset(buffer, 0, sizeof(buffer));
        sprintf(buffer, "%s%s", buffer2, RespBuf);
        last_line = NULL;
        line = strtok(buffer, "\r\n");
        while (line)
        {
            //printf("L: ***%s***", line);
            buf_ptr = line;
            for (i = 0; i < 8; i++)
            {
                buf_ptr = strchr(buf_ptr, ' ');
                if (!buf_ptr)
                    break;
                while (*buf_ptr == ' ')
                    buf_ptr++;
            }
            if (buf_ptr)
            {
                memset(buffer2, 0, sizeof(buffer2));
                // if it's a symlink
                if ((*line == 'l') && strstr(buf_ptr, " -> "))
                {
                    strncpy(buffer2, buf_ptr,strstr(buf_ptr," -> ") - buf_ptr);
                    strcat(buffer2, "@");
                }
                else if (*line == 'd')
                {
                    sprintf(buffer2, "%s/", buf_ptr);
                }
                else
                {
                    strcpy(buffer2, buf_ptr);
                }
                // if it's not a verbose list, print it
                if (!strchr(options, 'l'))
                    printf("%s\n", buffer2);
                if (!strcmp(path, "."))
                    LsCache[LsCacheCount++] = strdup(buffer2);
            }
            last_line = line;
            line = strtok(NULL, "\r\n");
        }
        // allow for weird case where the last character in the response is
        // a carriage return
        //printf("********************\n%s\n********************\n", RespBuf);
        if ((last_line != NULL) && (RespBuf[sizeof(RespBuf) - 2] != '\n'))
            strcpy(buffer2, last_line);
        else
            memset(buffer2, 0, sizeof(buffer2));
        if (rc == DX_LIB_MORE_DATA)
            new_rc = GetResponse(show_resp);
    } while (rc == DX_LIB_MORE_DATA);
    if (!strcmp(path, "."))
        LsCacheOk = TRUE;
}

/* used in LocalLs for scandir() */
static int one(const struct dirent *unused)
{
    return 1;
}

void LocalLs(const char *path, BOOL verbose, BOOL show_hidden)
{
    struct dirent **eps;
    struct stat stat_buf;
    char size[10], name[256], lname[256];
    char nlinks[10], uid[50], gid[50];
    char date[15], perms[15];
    mode_t mode;
    struct tm *time_ptr;
    int n, cnt;
    BOOL is_dir;

    if (chdir(path) != 0)
    {
        printf("Couldn't list: %s\n", strerror(errno));
        return;
    }
    n = scandir(".", &eps, one, alphasort);
    if (n >= 0)
    {
        for (cnt = 0; cnt < n; cnt++)
        {
            // for symlinks, check whether what they point to is a dir
            stat(eps[cnt]->d_name, &stat_buf);
            sprintf(size, "%d", (int)stat_buf.st_size);
            if (S_ISDIR(stat_buf.st_mode))
                is_dir = TRUE;
            else
                is_dir = FALSE;
            lstat(eps[cnt]->d_name, &stat_buf);
            mode = stat_buf.st_mode;
            if (S_ISLNK(mode))
            {
                memset(lname, 0, sizeof(lname));
                readlink(eps[cnt]->d_name, lname, sizeof(lname));
                sprintf(name, "%s -> %s", eps[cnt]->d_name, lname);
            }
            else
            {
                strcpy(name, eps[cnt]->d_name);
            }
            free(eps[cnt]);
            sprintf(nlinks, "%d", stat_buf.st_nlink);
            strcpy(uid, getpwuid(stat_buf.st_uid)->pw_name);
            strcpy(gid, getgrgid(stat_buf.st_gid)->gr_name);
            time_ptr = localtime(&stat_buf.st_mtime);
            // if it's very old then put the year, not the time in
            if (time(NULL) - stat_buf.st_mtime < (60 * 60 * 24 * 365))
                strftime(date, sizeof(date), "%b %d %H:%M", time_ptr);
            else
                strftime(date, sizeof(date), "%b %d  %Y", time_ptr);
            strcpy(perms, "----------");
            if (S_ISLNK(mode))
                perms[0] = 'l';
            else if (S_ISDIR(mode))
                perms[0] = 'd';
            else if (S_ISSOCK(mode))
                perms[0] = 's';
            if (S_IRUSR & mode) perms[1] = 'r';
            if (S_IWUSR & mode) perms[2] = 'w';
            if (S_IXUSR & mode) perms[3] = 'x';
            if (S_IRGRP & mode) perms[4] = 'r';
            if (S_IWGRP & mode) perms[5] = 'w';
            if (S_IXGRP & mode) perms[6] = 'x';
            if (S_IROTH & mode) perms[7] = 'r';
            if (S_IWOTH & mode) perms[8] = 'w';
            if (S_IXOTH & mode) perms[9] = 'x';
            if (is_dir)
                strcat(name, "/");

            if ((strcmp(name, "../") && strcmp(name, "./") &&
                (*name != '.')) || show_hidden)
            {
                if (verbose)
                    printf("%s %3s %-8s %-8s %8s %s %s\n",
                           perms, nlinks, uid, gid, size, date, name);
                else
                    printf("%s\n", name);
            }
        }
    }
    chdir(LocalDir);
}
