/*
 * Copyright (C) 1998  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <errno.h>
#include "compat/strerror.h"
#include "connection.h"
#include "bn_type.h"
#include "queue.h"
#include "packet.h"
#include "bot_protocol.h"
#include "bnet_protocol.h"
#include "field_sizes.h"
#include "eventlog.h"
#include "list.h"
#include "util.h"
#include "version.h"
#include "account.h"
#include "game.h"
#include "channel.h"
#include "message.h"


static char * message_format_line(t_connection const * c, char const * in);


/* make sure none of the expanded format symbols is longer than this (with null) */
#define MAX_INC 64

static char * message_format_line(t_connection const * c, char const * in)
{
    char *       out;
    unsigned int inpos;
    unsigned int outpos;
    unsigned int outlen=MAX_INC;
    
    if (!(out = malloc(outlen+1)))
    {
	pfree((void *)in,strlen(in)+1); /* avoid warning */
	return NULL;
    }
    
    out[0] = 'I';
    for (inpos=0,outpos=1; inpos<strlen(in); inpos++)
    {
        if (in[inpos]!='%')
	{
	    out[outpos] = in[inpos];
	    outpos += 1;
	}
        else
	    switch (in[++inpos])
	    {
	    case '%':
		out[outpos++] = '%';
		break;
		
	    case 'a':
		sprintf(&out[outpos],"%d",accountlist_get_length());
		outpos += strlen(&out[outpos]);
		break;
		
	    case 'c':
		sprintf(&out[outpos],"%d",channellist_get_length());
		outpos += strlen(&out[outpos]);
		break;
		
	    case 'g':
		sprintf(&out[outpos],"%d",gamelist_get_length());
		outpos += strlen(&out[outpos]);
		break;
		
	    case 'h':
    		if (gethostname(&out[outpos],MAX_INC)<0)
    		{
		    eventlog(eventlog_level_error,"message_format_line","could not get hostname (gethostname: %s)",strerror(errno));
		    strcpy(&out[outpos],"localhost"); /* not much else you can do */
    		}  
		outpos += strlen(&out[outpos]);
		break;
		
	    case 'i':
		sprintf(&out[outpos],"#%06u",conn_get_userid(c));
		out[outpos+USER_NAME_LEN] = '\0';
		outpos += strlen(&out[outpos]);
		break;
		
	    case 'l':
	        {
		    char const * tname;
		    
		    strncpy(&out[outpos],(tname = conn_get_username(c)),USER_NAME_LEN);
		    conn_unget_username(tname);
		}
		out[outpos+USER_NAME_LEN] = '\0';
		outpos += strlen(&out[outpos]);
		break;
		
	    case 'u':
		sprintf(&out[outpos],"%d",connlist_login_get_length());
		outpos += strlen(&out[outpos]);
		break;
		
	    case 'v':
		strcpy(&out[outpos],BNETD_VERSION);
		outpos += strlen(&out[outpos]);
		break;
		
	    case 'B': /* BROADCAST */
		out[0] = 'B';
		break;
		
	    case 'E': /* ERROR */
		out[0] = 'E';
		break;
		
	    case 'I': /* INFO */
		out[0] = 'I';
		break;
		
	    case 'M': /* MESSAGE */
		out[0] = 'M';
		break;
		
	    case 'T': /* EMOTE */
		out[0] = 'T';
		break;
		
	    case 'W': /* WARN */
		out[0] = 'W';
		break;
		
	    default:
		eventlog(eventlog_level_warn,"message_format_line","bad formatter \"%%%c\"",in[inpos-1]);
	    }
	
	if ((outpos+MAX_INC)>=outlen)
	{
	    char * newout;
	    
	    outlen += MAX_INC;
	    if (!(newout = realloc(out,outlen)))
	    {
		pfree((void *)in,strlen(in)+1); /* avoid warning */
		pfree(out,outlen-MAX_INC);
		return NULL;
	    }
	    out = newout;
	}
    }
    out[outpos] = '\0';
    
    pfree((void *)in,strlen(in)+1); /* avoid warning */
    
    return out;
}


extern int message_bot_format(t_packet * packet, int type, t_connection * me, char const * text)
{
    char         msgtemp[MAX_MESSAGE_LEN+USER_NAME_LEN+64];
    char const * tname=NULL;
    
    if (!packet)
    {
	eventlog(eventlog_level_error,"message_bot_format","got NULL packet");
	return -1;
    }
    
    switch (type)
    {
    case 0:
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL text for UNIQUENAME");
	    return -1;
	}
        sprintf(msgtemp,"%u %s %s\r\n",EID_UNIQUENAME,"NAME",text);
	break;
    case MT_ADD:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL connection for SHOWUSER");
	    return -1;
	}
	sprintf(msgtemp,"%u %s %s %04x [%s]\r\n",EID_SHOWUSER,"USER",(tname = conn_get_username(me)),conn_get_flags(me),conn_get_clienttag(me));
	conn_unget_username(tname);
	break;
    case MT_JOIN:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL connection for JOIN");
	    return -1;
	}
	sprintf(msgtemp,"%u %s %s %04x [%s]\r\n",EID_JOIN,"JOIN",(tname = conn_get_username(me)),conn_get_flags(me),conn_get_clienttag(me));
	conn_unget_username(tname);
	break;
    case MT_PART:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL connection for LEAVE");
	    return -1;
	}
	sprintf(msgtemp,"%u %s %s %04x\r\n",EID_LEAVE,"LEAVE",(tname = conn_get_username(me)),conn_get_flags(me));
	conn_unget_username(tname);
	break;
    case MT_WHISPER:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL connection for WHISPER");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL text for WHISPER");
	    return -1;
	}
	sprintf(msgtemp,"%u %s %s %04x \"%s\"\r\n",EID_WHISPER,"WHISPER",(tname = conn_get_username(me)),conn_get_flags(me),text);
	conn_unget_username(tname);
	break;
    case MT_MESSAGE:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL connection for TALK");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL text for TALK");
	    return -1;
	}
	sprintf(msgtemp,"%u %s %s %04x \"%s\"\r\n",EID_TALK,"TALK",(tname = conn_get_username(me)),conn_get_flags(me),text);
	conn_unget_username(tname);
	break;
    case MT_BROADCAST:
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL text for BROADCAST");
	    return -1;
	}
	sprintf(msgtemp,"%u %s \"%s\"\r\n",EID_BROADCAST,"_",text); /* FIXME */
	break;
    case MT_JOINING:
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL text for CHANNEL");
	    return -1;
	}
	sprintf(msgtemp,"%u %s \"%s\"\r\n",EID_CHANNEL,"CHANNEL",text);
	break;
    case MT_USERFLAGS:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL connection for USERFLAGS");
	    return -1;
	}
	sprintf(msgtemp,"%u %s %s %04x\r\n",EID_USERFLAGS,"USER",(tname = conn_get_username(me)),conn_get_flags(me));
	conn_unget_username(tname);
	break;
    case MT_WHISPERACK:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL connection for WHISPERSENT");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL text for WHISPERSENT");
	    return -1;
	}
	sprintf(msgtemp,"%u %s %s %04x \"%s\"\r\n",EID_WHISPERSENT,"WHISPER",(tname = conn_get_username(me)),conn_get_flags(me),text);
	conn_unget_username(tname);
	break;
    case MT_CHANNELFULL:
	sprintf(msgtemp,"%u \r\n",EID_CHANNELFULL); /* FIXME */
	break;
    case MT_CHANNELDOESNOTEXIST:
	sprintf(msgtemp,"%u \r\n",EID_CHANNELDOESNOTEXIST); /* FIXME */
	break;
    case MT_CHANNELRESTRICTED:
	sprintf(msgtemp,"%u \r\n",EID_CHANNELRESTRICTED); /* FIXME */
	break;
    case MT_WARN:
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL text for INFO");
	    return -1;
	}
	sprintf(msgtemp,"%u %s \"%s\"\r\n",EID_INFO,"INFO",text);
	break;
    case MT_ERROR:
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL text for ERROR");
	    return -1;
	}
	sprintf(msgtemp,"%u %s \"%s\"\r\n",EID_ERROR,"ERROR",text);
	break;
    case MT_EMOTE:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL connection for EMOTE");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_bot_format","got NULL text for EMOTE");
	    return -1;
	}
	sprintf(msgtemp,"%u %s %s %04x \"%s\"\r\n",EID_EMOTE,"EMOTE",(tname = conn_get_username(me)),conn_get_flags(me),text);
	conn_unget_username(tname);
	break;
    default:
	eventlog(eventlog_level_error,"message_bot_format","got bad message type %d",type);
	return -1;
    }
    
    return packet_append_data(packet,msgtemp,strlen(msgtemp));
}


extern int message_normal_format(t_packet * packet, int type, t_connection * me, char const * text)
{
    char const * tname=NULL;
    
    if (!packet)
    {
	eventlog(eventlog_level_error,"message_normal_format","got NULL packet");
	return -1;
    }
    
    packet_set_size(packet,sizeof(t_server_message));
    packet_set_type(packet,SERVER_MESSAGE);
    bn_int_set(&packet->u.server_message.type,type);
    bn_int_set(&packet->u.server_message.unknown1,0);
    bn_int_set(&packet->u.server_message.unknown2,0);
    bn_int_set(&packet->u.server_message.unknown3,0);
    
    switch (type)
    {
    case MT_ADD:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for ADD");
	    return -1;
	}
	
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,conn_get_playerinfo(me,conn_get_clienttag(me)));
	break;
    case MT_JOIN:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for JOIN");
	    return -1;
	}
	
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,conn_get_playerinfo(me,conn_get_clienttag(me))); /* FIXME: should we just send "" here? */
	break;
    case MT_PART:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for PART");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,"");
	break;
    case MT_WHISPER:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for WHISPER");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL text for WHISPER");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,text);
	break;
    case MT_MESSAGE:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for MESSAGE");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL text for MESSAGE");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,text);
	break;
    case MT_BROADCAST:
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL text for BROADCAST");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,text);
	break;
    case MT_JOINING:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for JOINING");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL text for JOINING");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,0);
	bn_int_set(&packet->u.server_message.latency,0);
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,text);
	break;
    case MT_USERFLAGS:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for USERFLAGS");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,"");
	break;
    case MT_WHISPERACK:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for WHISPERACK");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL text for WHISPERACK");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,text);
	break;
    case MT_CHANNELFULL: /* FIXME */
	bn_int_set(&packet->u.server_message.flags,0);
	bn_int_set(&packet->u.server_message.latency,0);
	packet_append_string(packet,"");
	packet_append_string(packet,"");
	break;
    case MT_CHANNELDOESNOTEXIST:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for CHANNELDOESNOTEXIST");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL text for CHANNELDOESNOTEXIST");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,text);
	break;
    case MT_CHANNELRESTRICTED: /* FIXME */
	bn_int_set(&packet->u.server_message.flags,0);
	bn_int_set(&packet->u.server_message.latency,0);
	packet_append_string(packet,"");
	packet_append_string(packet,"");
	break;
    case MT_WARN:
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL text for WARN");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,0);
	bn_int_set(&packet->u.server_message.latency,0);
	packet_append_string(packet,"");
	packet_append_string(packet,text);
	break;
    case MT_ERROR:
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL text for ERROR");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,0);
	bn_int_set(&packet->u.server_message.latency,0);
	packet_append_string(packet,"");
	packet_append_string(packet,text);
	break;
    case MT_EMOTE:
	if (!me)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL connection for EMOTE");
	    return -1;
	}
	if (!text)
	{
	    eventlog(eventlog_level_error,"message_normal_format","got NULL text for EMOTE");
	    return -1;
	}
	bn_int_set(&packet->u.server_message.flags,conn_get_flags(me));
	bn_int_set(&packet->u.server_message.latency,conn_get_latency(me));
	packet_append_string(packet,(tname = conn_get_username(me)));
	conn_unget_username(tname);
	packet_append_string(packet,text);
	break;
    default:
	eventlog(eventlog_level_error,"message_normal_format","got bad message type %d",type);
	return -1;
    }
    
    return 0;
}
    

extern void message_send(t_connection * dst, int type, t_connection * src, char const * text)
{
    t_packet *   packet;
    char const * tname=NULL;
    
    if (!dst)
    {
	eventlog(eventlog_level_error,"message_send","got NULL dst connection");
	return;
    }
    if (!src)
    {
	eventlog(eventlog_level_error,"message_send","got NULL src connection");
	return;
    }
    
    if (conn_check_ignoring(dst,(tname = conn_get_username(src)))!=1)
    {
	conn_unget_username(tname);
 	switch (conn_get_class(dst))
	{
	case conn_class_bot:
	    if (!(packet = packet_create(packet_class_raw)))
	    {
		eventlog(eventlog_level_error,"message_send","could not create packet");
		return;
	    }
	    message_bot_format(packet,type,src,text);
	    queue_push_packet(conn_get_out_queue(dst),packet);
	    packet_del_ref(packet);
	    break;
	case conn_class_normal:
	    if (!(packet = packet_create(packet_class_normal)))
	    {
		eventlog(eventlog_level_error,"message_send","could not create packet");
		return;
	    }
	    message_normal_format(packet,type,src,text);
	    queue_push_packet(conn_get_out_queue(dst),packet);
	    packet_del_ref(packet);
	    break;
	default:
	    eventlog(eventlog_level_error,"message_send","bad connection class %d",(int)conn_get_class(dst));
	}
    }
    else
	conn_unget_username(tname);
}


extern void message_send_all(int type, t_connection * src, char const * text)
{
    t_connection *         c;
    t_packet *             packet;
    t_packet *             bpacket;
    t_list const * const * save;
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	eventlog(eventlog_level_error,"message_send_all","could not create packet");
	return;
    }
    if (!(bpacket = packet_create(packet_class_raw)))
    {
	eventlog(eventlog_level_error,"message_send_all","could not create bpacket");
	return;
    }
    
    if (message_normal_format(packet,type,src,text)<0)
    {
	eventlog(eventlog_level_error,"message_send_all","could not format message");
	packet_del_ref(packet);
	packet_del_ref(bpacket);
	return;
    }
    if (message_bot_format(bpacket,type,src,text)<0)
    {
	eventlog(eventlog_level_error,"message_send_all","could not format message");
	packet_del_ref(packet);
	packet_del_ref(bpacket);
	return;
    }
    
    for (c=connlist_get_first(&save); c; c=connlist_get_next(&save))
 	switch (conn_get_class(c))
	{
	case conn_class_bot:
            queue_push_packet(conn_get_out_queue(c),bpacket);
	    break;
	case conn_class_normal:
            queue_push_packet(conn_get_out_queue(c),packet);
	    break;
	default: /* skip other classes */
	    break;
	}
    
    packet_del_ref(packet);
    packet_del_ref(bpacket);
}


extern void message_send_file(t_connection * c, FILE * fd)
{
    char * buff;
    
    if (!c)
    {
	eventlog(eventlog_level_error,"message_send_file","got NULL connection");
	return;
    }
    if (!fd)
    {
	eventlog(eventlog_level_error,"message_send_file","got NULL fd");
	return;
    }
    
    while ((buff = file_get_line(fd)))
	if ((buff = message_format_line(c,buff)))
	{
	    if (strlen(buff)>MAX_MESSAGE_LEN)
		buff[MAX_MESSAGE_LEN] = '\0'; /* now truncate to max size */
	    
	    /* empty messages can crash the clients */
	    switch (buff[0])
	    {
	    case 'B':
		if (buff[1]!='\0')
		    message_send(c,MT_BROADCAST,c,&buff[1]);
		else
		    message_send(c,MT_BROADCAST,c," ");
		break;
	    case 'E':
		if (buff[1]!='\0')
		    message_send(c,MT_ERROR,c,&buff[1]);
		else
		    message_send(c,MT_ERROR,c," ");
		break;
	    case 'M':
		if (buff[1]!='\0')
		    message_send(c,MT_MESSAGE,c,&buff[1]);
		else
		    message_send(c,MT_MESSAGE,c," ");
		break;
	    case 'T':
		if (buff[1]!='\0')
		    message_send(c,MT_EMOTE,c,&buff[1]);
		else
		    message_send(c,MT_EMOTE,c," ");
		break;
	    case 'I':
	    case 'W':
		if (buff[1]!='\0')
		    message_send(c,MT_WARN,c,&buff[1]);
		else
		    message_send(c,MT_WARN,c," ");
		break;
	    default:
		eventlog(eventlog_level_error,"conn_send_welcome","unknown message type '%c'",buff[0]);
	    }
	    pfree(buff,strlen(buff)+1);
	}
}
