/*
 * Copyright (C) 1998  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * Copyright (C) 1999  Gediminas (gediminas_lt@mailexcite.com)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "compat/strdup.h"
#include "compat/strtoul.h"
#include <ctype.h>
#include <errno.h>
#include "compat/strerror.h"
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include "compat/strftime.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include "compat/netinet_in.h"
#include <arpa/inet.h>
#include "compat/inet_ntoa.h"
#include "message.h"
#include "tag.h"
#include "connection.h"
#include "channel.h"
#include "game.h"
#include "util.h"
#include "version.h"
#include "account.h"
#include "server.h"
#include "prefs.h"
#include "eventlog.h"
#include "ladder.h"
#include "timer.h"
#include "command.h"


static char const * bnclass_get_str(unsigned int class);
static void do_whisper(t_connection * user_c, char const * dest, char const * text);
static void do_whois(t_connection * c, char const * dest);
static void user_timer_cb(t_connection * c, time_t now, t_timer_data str);


static char const * bnclass_get_str(unsigned int class)
{
    switch (class)
    {
    case PLAYERINFO_DRTL_CLASS_WARRIOR:
	return "warrior";
    case PLAYERINFO_DRTL_CLASS_ROGUE:
	return "rogue";
    case PLAYERINFO_DRTL_CLASS_SORCERER:
	return "sorcerer";
    default:
	return "unknown";
    }
}


static void do_whisper(t_connection * user_c, char const * dest, char const * text)
{
    t_connection * dest_c;
    char           temp[MAX_MESSAGE_LEN];
    char const *   tname;
    
    if (!(dest_c = connlist_find_connection(dest)))
    {
	message_send(user_c,MT_ERROR,user_c,"That user is not logged on.");
	return;
    }
    
    if (conn_get_dndstr(dest_c))
    {
        sprintf(temp,"%.64s is unavaliable (%.128s)",(tname = conn_get_username(dest_c)),conn_get_dndstr(dest_c));
	conn_unget_username(tname);
        message_send(user_c,MT_WARN,user_c,temp);
        return;
    }
    
    message_send(user_c,MT_WHISPERACK,dest_c,text);
    
    if (conn_get_awaystr(dest_c))
    {
        sprintf(temp,"%.64s is away (%.128s)",(tname = conn_get_username(dest_c)),conn_get_awaystr(dest_c));
	conn_unget_username(tname);
        message_send(user_c,MT_WARN,user_c,temp);
    }
    
    message_send(dest_c,MT_WHISPER,user_c,text);
}


static void do_whois(t_connection * c, char const * dest)
{
    t_connection * dest_c;
    char           namepart[128];
    char           temp[MAX_MESSAGE_LEN];
    char const *   verb;
    t_game const * game;
    
    if (!(dest_c = connlist_find_connection(dest)))
    {
	message_send(c,MT_ERROR,c,"That user is not logged on.");
	return;
    }
    
    if (c==dest_c)
    {
	strcpy(namepart,"You");
	verb = "are";
    }
    else
    {
	char const * tname;
	
	sprintf(namepart,"%.64s",(tname = conn_get_username(dest_c)));
	conn_unget_username(tname);
	verb = "is";
    }
    
    if ((game = conn_get_game(dest_c)))
    {
	if (strcmp(game_get_pass(game),"")==0)
	    sprintf(temp,"%s %s logged on from account #%06u, and %s currently in game \"%.64s\".",
		    namepart,
		    verb,
		    conn_get_userid(dest_c),
		    verb,
		    game_get_name(conn_get_game(dest_c)));
	else
	    sprintf(temp,"%s %s logged on from account #%06u, and %s currently in private game \"%.64s\".",
		    namepart,
		    verb,
		    conn_get_userid(dest_c),
		    verb,
		    game_get_name(conn_get_game(dest_c)));
    }
    else if (conn_get_channel(dest_c))
        sprintf(temp,"%s %s logged on from account #%06u, and %s currently in the channel \"%.64s\".",
		namepart,
		verb,
		conn_get_userid(dest_c),
		verb,
		channel_get_name(conn_get_channel(dest_c)));
    else
	sprintf(temp,"%s %s logged on from account #%06u.",
		namepart,
		verb,
		conn_get_userid(dest_c));
    message_send(c,MT_WARN,c,temp);
    if (game && strcmp(game_get_pass(game),"")!=0)
	message_send(c,MT_WARN,c,"(This game is password protected.)");
    
    if (conn_get_dndstr(dest_c))
    {
        sprintf(temp,"%s %s refusing messages (%.128s)",
		namepart,
		verb,
		conn_get_dndstr(dest_c));
	message_send(c,MT_WARN,c,temp);
    }
    else
        if (conn_get_awaystr(dest_c))
        {
            sprintf(temp,"%s away (%.128s)",
		    namepart,
		    conn_get_awaystr(dest_c));
	    message_send(c,MT_WARN,c,temp);
        }
}


static void user_timer_cb(t_connection * c, time_t now, t_timer_data str)
{
    if (!c)
    {
	eventlog(eventlog_level_error,"user_timer_cb","got NULL connection");
	return;
    }
    if (!str.p)
    {
	eventlog(eventlog_level_error,"user_timer_cb","got NULL str");
	return;
    }
    
    message_send(c,MT_WARN,c,str.p);
    pfree(str.p,strlen(str.p)+1);
}


extern int handle_command(t_connection * c,  char const * text)
{
    if (strstart(text,"/me")==0)
    {
	unsigned int i;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++); /* skip spaces */
	
	channel_message_send(conn_get_channel(c),MT_EMOTE,c,&text[i]);
	return 0;
    }
    if (strstart(text,"/msg")==0 ||
	strstart(text,"/whisper")==0 ||
	strstart(text,"/w")==0 ||
	strstart(text,"/m")==0)
    {
	char         dest[USER_NAME_LEN];
	unsigned int i,j;
	
	dest[0] = '\0';
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	for (j=0; text[i]!=' ' && text[i]!='\0'; i++,j++) /* get dest */
	    if (j<USER_NAME_LEN-1) dest[j] = text[i];
	dest[j] = '\0';
	
	if (text[i]=='\0')
	{
	    message_send(c,MT_ERROR,c,"What do you want to say?");
	    return 0;
	}
	
	for (; text[i]==' '; i++);
	do_whisper(c,dest,&text[i]);
	
	return 0;
    }
    if (strstart(text,"/status")==0 || strstart(text,"/users")==0)
    {
	char msgtemp[MAX_MESSAGE_LEN];
	
	sprintf(msgtemp,"There are currently %d users online, in %d games and %d channels.",
		connlist_login_get_length(),
		gamelist_get_length(),
		channellist_get_length());
	message_send(c,MT_WARN,c,msgtemp);
	
	return 0;
    }
    if (strstart(text,"/who")==0)
    {
	char                 msgtemp[MAX_MESSAGE_LEN];
	t_connection const * conn;
	t_channel const *    chan;
	unsigned int         i;
	char const *         tname;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (!(chan = channellist_find_channel(&text[i])))
	{
	    message_send(c,MT_ERROR,c,"That channel does not exist.");
	    message_send(c,MT_ERROR,c,"(If you are trying to search for a user, use the /whois command.)");
	    return 0;
	}
	
	sprintf(msgtemp,"Users in channel %.64s:",&text[i]);
	i = strlen(msgtemp);
	for (conn=channel_get_first(chan); conn; conn=channel_get_next())
	{
	    if (i+strlen((tname = conn_get_username(conn)))+2>MAX_MESSAGE_LEN) /* " ", name, '\0' */
	    {
		message_send(c,MT_WARN,c,msgtemp);
		i = 0;
	    }
	    sprintf(&msgtemp[i]," %s",tname);
	    conn_unget_username(tname);
	    i += strlen(&msgtemp[i]);
	}
	if (i>0)
	    message_send(c,MT_WARN,c,msgtemp);
	
	return 0;
    }
    if (strstart(text,"/whois")==0 || strstart(text,"/whereis")==0)
    {
	unsigned int i;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	do_whois(c,&text[i]);
	
	return 0;
    }
    if (strstart(text,"/announce")==0)
    {
	unsigned int i;
	char         msgtemp[MAX_MESSAGE_LEN];
	char const * tname;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (account_get_auth_announce(conn_get_account(c))!=1) /* default to false */
	{
	    message_send(c,MT_WARN,c,"You do not have permission to use this command.");
	    return 0;
	}
	
	sprintf(msgtemp,"Announcement from %.64s: %.128s",(tname = conn_get_username(c)),&text[i]);
	conn_unget_username(tname);
	message_send_all(MT_BROADCAST,c,msgtemp);
	
	return 0;
    }
    if (strstart(text,"/version")==0)
    {
	message_send(c,MT_WARN,c,"BNETD " BNETD_VERSION);
	
	return 0;
    }
    if (strstart(text,"/copyright")==0 || strstart(text,"/warranty")==0 || strstart(text,"/license")==0)
    {
	static char const * const info[] =
	{
	    " Copyright (C) 1998,1999  Mark Baysinger (mbaysing@ucsd.edu)",
	    " Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)",
	    " Copyright (C) 1999  Rob Crittenden (rcrit@greyoak.com)",
	    " Copyright (C) 1999  Descolada (dyn1-tnt9-237.chicago.il.ameritech.net)",
	    " Copyright (C) 1999  Gediminas (gediminas_lt@mailexcite.com)",
	    " ",
	    " This program is free software; you can redistribute it and/or",
	    " modify it under the terms of the GNU General Public License",
	    " as published by the Free Software Foundation; either version 2",
	    " of the License, or (at your option) any later version.",
	    " ",
	    " This program is distributed in the hope that it will be useful,",
	    " but WITHOUT ANY WARRANTY; without even the implied warranty of",
	    " MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the",
	    " GNU General Public License for more details.",
	    " ",
	    " You should have received a copy of the GNU General Public License",
	    " along with this program; if not, write to the Free Software",
	    " Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.",
	    NULL
	};
	unsigned int i;
	
	for (i=0; info[i]; i++)
		message_send(c,MT_WARN,c,info[i]);
	
	return 0;
    }
    if (strstart(text,"/uptime")==0)
    {
	char         msgtemp[MAX_MESSAGE_LEN];
	unsigned int ut;
	unsigned int d,h,m;
	
	ut = server_get_uptime();
	d = ut/(24*60*60);
	ut %= (24*60*60);
	h = ut/(60*60);
	ut %= (60*60);
	m = ut/(60);	
	
	sprintf(msgtemp,"Uptime: %d days %d hours %d minutes",d,h,m);
	message_send(c,MT_WARN,c,msgtemp);
	
	return 0;
    }
    if (strstart(text,"/stats")==0 || strstart(text,"/astat")==0)
    {
	char         dest[USER_NAME_LEN];
	char         msgtemp[MAX_MESSAGE_LEN];
	unsigned int i,j;
	t_account *  account;
	char const * clienttag;
	char const * tname;
	
	dest[0] = '\0';
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	for (j=0; text[i]!=' ' && text[i]!='\0'; i++,j++) /* get dest */
	    if (j<USER_NAME_LEN-1) dest[j] = text[i];
	dest[j] = '\0';
	for (; text[i]==' '; i++);
	
	if (dest[0]=='\0') /* if no argument, stat ourself */
	{
	    strcpy(dest,(tname = conn_get_username(c)));
	    conn_unget_username(tname);
	}
	
	if (!(account = accountlist_find_account(dest)))
	{
	    message_send(c,MT_ERROR,c,"Invalid user.");
	    return 0;
	}
	
	if (text[i]!='\0')
	    clienttag = &text[i];
	else if (!(clienttag = conn_get_clienttag(c)))
	{
	    message_send(c,MT_ERROR,c,"Unable to determine client game.");
	    return 0;
	}
	
	if (strlen(clienttag)!=4)
	{
	    sprintf(msgtemp,"Invalid client game tag \"%.32s\".",clienttag);
	    message_send(c,MT_ERROR,c,msgtemp);
	    message_send(c,MT_ERROR,c,"Example: /stats joe STAR");
	    return 0;
	}
	
	if (strcasecmp(clienttag,CLIENTTAG_BNCHATBOT)==0)
	{
	    message_send(c,MT_ERROR,c,"This game does not support win/loss records.");
	    return 0;
	}
	else if (strcasecmp(clienttag,CLIENTTAG_DIABLORTL)==0 ||
	         strcasecmp(clienttag,CLIENTTAG_DIABLOSHR)==0)
	{
	    sprintf(msgtemp,"%.64s's record:",(tname = account_get_name(account)));
	    account_unget_name(tname);
	    message_send(c,MT_WARN,c,msgtemp);
	    
	    sprintf(msgtemp,"level: %u",account_get_normal_level(account,clienttag));
	    message_send(c,MT_WARN,c,msgtemp);
	    sprintf(msgtemp,"class: %.16s",bnclass_get_str(account_get_normal_class(account,clienttag)));
	    message_send(c,MT_WARN,c,msgtemp);
	    sprintf(msgtemp,"stats: %u str  %u mag  %u dex  %u vit  %u gld",
		    account_get_normal_strength(account,clienttag),
		    account_get_normal_magic(account,clienttag),
		    account_get_normal_dexterity(account,clienttag),
		    account_get_normal_vitality(account,clienttag),
		    account_get_normal_gold(account,clienttag));
	    message_send(c,MT_WARN,c,msgtemp);
	    sprintf(msgtemp,"Diablo kills: %u",account_get_normal_diablo_kills(account,clienttag));
	    message_send(c,MT_WARN,c,msgtemp);
	}
	else
	{
	    sprintf(msgtemp,"%.64s's record:",(tname = account_get_name(account)));
	    account_unget_name(tname);
	    message_send(c,MT_WARN,c,msgtemp);
	    
	    sprintf(msgtemp,"Normal games: %d-%d-%d",
		    account_get_normal_wins(account,clienttag),
		    account_get_normal_losses(account,clienttag),
		    account_get_normal_disconnects(account,clienttag));
	    message_send(c,MT_WARN,c,msgtemp);
	    
	    if (account_get_ladder_rating(account,clienttag)>0)
		sprintf(msgtemp,"Ladder games: %d-%d-%d (rating %d)",
			account_get_ladder_wins(account,clienttag),
			account_get_ladder_losses(account,clienttag),
			account_get_ladder_disconnects(account,clienttag),
			account_get_ladder_rating(account,clienttag));
	    else
		strcpy(msgtemp,"Ladder games: 0-0-0");
	    message_send(c,MT_WARN,c,msgtemp);
	}
	
	return 0;
    }
    if (strstart(text,"/time")==0)
    {
	char        msgtemp[MAX_MESSAGE_LEN];
	time_t      now;
	struct tm * gmnow;
	
	now = time(NULL);
	gmnow = gmtime(&now);
	
	/* Battle.net time: Wed Jun 23 15:15:29 */
	strftime(msgtemp,MAX_MESSAGE_LEN,"BNETD time: %a %b %d %H:%M:%S GMT",gmnow);
	message_send(c,MT_WARN,c,msgtemp);
	if (conn_get_class(c)==conn_class_normal)
	{
	    strftime(msgtemp,MAX_MESSAGE_LEN,"Your local time: %a %b %d %H:%M:%S",gmnow);
	    message_send(c,MT_WARN,c,msgtemp);
	}
	
	return 0;
    }
    if (strstart(text,"/channel")==0 || strstart(text,"/join")==0)
    {
	unsigned int i;
        
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
        
        if (strlen(&text[i])<1)
	{
	    message_send(c,MT_ERROR,c,"Please specify a channel.");
	    return 0;
	}
        
        if (conn_set_channel(c,&text[i])<0)
            conn_set_channel(c,CHANNEL_NAME_BANNED); /* should not fail */
        
	return 0;
    }
    if (strstart(text,"/rejoin")==0)
    {
	char const * temp;
	char const * chname;
	
        if (!conn_get_channel(c))
	{
	    message_send(c,MT_ERROR,c,"You are not in a channel.");
	    return 0;
	}
        
	/* we need to copy the channel name because we might remove the
	   last person from the channel and cause the string to be freed */
	if (!(temp = channel_get_name(conn_get_channel(c))))
	    chname = NULL;
	else
	    chname = strdup(temp); /* ok if NULL */
	
        if (conn_set_channel(c,chname)<0)
            conn_set_channel(c,CHANNEL_NAME_BANNED); /* should not fail */
        
	if (chname)
	    pfree((void *)chname,strlen(chname)+1); /* avoid warning */
	
	return 0;
    }
    if (strstart(text,"/away")==0)
    {
	unsigned int i;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0') /* back to normal */
	    message_send(c,MT_WARN,c,"You are no longer marked as away.");
	else
	    message_send(c,MT_WARN,c,"You are now marked as being away.");
	
	conn_set_awaystr(c,&text[i]);
	
	return 0;
    }
    if (strstart(text,"/dnd")==0)
    {
	unsigned int i;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0') /* back to normal */
	    message_send(c,MT_WARN,c,"Do Not Disturb mode cancelled.");
	else
	    message_send(c,MT_WARN,c,"Do Not Disturb mode engaged.");
	
	conn_set_dndstr(c,&text[i]);
	
	return 0;
    }
    if (strstart(text,"/ignore")==0 || strstart(text,"/squelch")==0)
    {
	char         msgtemp[MAX_MESSAGE_LEN];
	unsigned int i;
	t_account *  account;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
	{
	    message_send(c,MT_WARN,c,"Who do you want to ignore?");
	    return 0;
	}
	
	if (!(account = accountlist_find_account(&text[i])))
	{
	    message_send(c,MT_ERROR,c,"No such user.");
	    return 0;
	}
	
	if (conn_get_account(c)==account)
	{
	    message_send(c,MT_ERROR,c,"You can't squelch yourself.");
	    return 0;
	}
	
	if (conn_add_ignore(c,account)<0)
	    message_send(c,MT_ERROR,c,"Could not squelch user.");
	else
	{
	    char const * tname;
	    
	    sprintf(msgtemp,"%-.20s has been squelched.",(tname = account_get_name(account)));
	    account_unget_name(tname);
	    message_send(c,MT_WARN,c,msgtemp);
	}
	
	return 0;
    }
    if (strstart(text,"/unignore")==0 || strstart(text,"/unsquelch")==0)
    {
	unsigned int      i;
        t_account const * account;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
        {
	    message_send(c,MT_WARN,c,"Who don't you want to ignore?");
	    return 0;
	}
	
        if (!(account = accountlist_find_account(&text[i])))
	{
	    message_send(c,MT_WARN,c,"No such user.");
	    return 0;
	}
	
	if (conn_del_ignore(c,account)<0)
	    message_send(c,MT_WARN,c,"User was not being ignored.");
	else
	    message_send(c,MT_WARN,c,"No longer ignoring.");
	
	return 0;
    }
    if (strstart(text,"/designate")==0)
    {
	char           msgtemp[MAX_MESSAGE_LEN];
	unsigned int   i;
        t_connection * noc;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
        {
	    message_send(c,MT_WARN,c,"Who do you want to designate?");
	    return 0;
	}
	
        if (!conn_get_channel(c))
        { 
            message_send(c,MT_ERROR,c,"This command can only be used inside a channel.");
	    return 0;
        }
	if (account_get_auth_admin(conn_get_account(c))!=1 && /* default to false */
            channel_get_operator(conn_get_channel(c))!=c)
        {
            message_send(c,MT_ERROR,c,"You are not a channel operator.");
	    return 0;
        }
        if (!(noc = connlist_find_connection(&text[i])))
        {
            message_send(c,MT_ERROR,c,"That user is not logged in.");
	    return 0;
        }
        if (conn_get_channel(noc)!=conn_get_channel(c))
        {
            message_send(c,MT_ERROR,c,"That user is not in this channel.");
            return 0;
        }
        
        if (channel_set_next_operator(conn_get_channel(c),noc)<0)
            message_send(c,MT_ERROR,c,"Unable to designate that user.");
        else
        {
	    char const * tname;
	    
            sprintf(msgtemp,"%s will be the new operator when you resign.",(tname = conn_get_username(noc)));
	    conn_unget_username(tname);
            message_send(c,MT_WARN,c,msgtemp);
        }
	
        return 0;
    }
    if (strstart(text,"/resign")==0)
    {
        if (!conn_get_channel(c))
        { 
            message_send(c,MT_ERROR,c,"This command can only be used inside a channel.");
	    return 0;
        }
        if (channel_get_operator(conn_get_channel(c))!=c)
        { 
            message_send(c,MT_ERROR,c,"This command is reserved for channel operators.");
	    return 0;
        }
        
        if (channel_choose_operator(conn_get_channel(c),NULL)<0)
            message_send(c,MT_ERROR,c,"You are unable to resign.");
        else
            message_send(c,MT_WARN,c,"You are no longer the operator.");
        
	return 0;
    }
    if (strstart(text,"/kick")==0)
    {
	unsigned int   i;
        t_connection * kuc;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
        {
	    message_send(c,MT_WARN,c,"Who do you want to kick off the channel?");
	    return 0;
	}
	
        if (!conn_get_channel(c))
        { 
            message_send(c,MT_ERROR,c,"This command can only be used inside a channel.");
	    return 0;
        }
	if (account_get_auth_admin(conn_get_account(c))!=1 && /* default to false */
            channel_get_operator(conn_get_channel(c))!=c)
        { 
            message_send(c,MT_ERROR,c,"You are not a channel operator.");
	    return 0;
        }
        if (!(kuc = connlist_find_connection(&text[i])))
        {
            message_send(c,MT_ERROR,c,"That user is not logged in.");
	    return 0;
        }
        if (conn_get_channel(kuc)!=conn_get_channel(c))
        {
            message_send(c,MT_ERROR,c,"That user is not in this channel.");
            return 0;
        }
        
        conn_set_channel(kuc,CHANNEL_NAME_KICKED); /* should not fail */
        
        return 0;
    }
    if (strstart(text,"/ban")==0)
    {
	char           msgtemp[MAX_MESSAGE_LEN];
	unsigned int   i;
        t_connection * buc;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
        {
	    message_send(c,MT_WARN,c,"Who do you want to ban from the channel?");
	    return 0;
	}
	
        if (!conn_get_channel(c))
        { 
            message_send(c,MT_ERROR,c,"This command can only be used inside a channel.");
	    return 0;
        }
	if (account_get_auth_admin(conn_get_account(c))!=1 && /* default to false */
            channel_get_operator(conn_get_channel(c))!=c)
        { 
            message_send(c,MT_ERROR,c,"You are not a channel operator.");
	    return 0;
        }
        
        if (channel_ban_user(conn_get_channel(c),&text[i])<0)
            message_send(c,MT_ERROR,c,"Unable to ban user.");
        else
        {
            sprintf(msgtemp,"%s is now banned from this channel.",&text[i]);
            message_send(c,MT_WARN,c,msgtemp);
        }
        if ((buc = connlist_find_connection(&text[i])) &&
            conn_get_channel(buc)==conn_get_channel(c))
            conn_set_channel(buc,CHANNEL_NAME_BANNED);
	
        return 0;
    }
    if (strstart(text,"/unban")==0)
    {
	char         msgtemp[MAX_MESSAGE_LEN];
	unsigned int i;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
        {
	    message_send(c,MT_WARN,c,"Who do you want to unban from the channel?");
	    return 0;
	}
	
        if (!conn_get_channel(c))
        { 
            message_send(c,MT_ERROR,c,"This command can only be used inside a channel.");
	    return 0;
        }
	if (account_get_auth_admin(conn_get_account(c))!=1 && /* default to false */
            channel_get_operator(conn_get_channel(c))!=c)
        { 
            message_send(c,MT_ERROR,c,"You are not a channel operator.");
	    return 0;
        }
        
        if (channel_unban_user(conn_get_channel(c),&text[i])<0)
            message_send(c,MT_ERROR,c,"That user is not banned.");
        else
        {
            sprintf(msgtemp,"%s is no longer banned from this channel.",&text[i]);
            message_send(c,MT_WARN,c,msgtemp);
        }
	
        return 0;
    }
    
    if (prefs_get_extra_commands()==0)
    {
	message_send(c,MT_ERROR,c,"Unknown command.");
	eventlog(eventlog_level_debug,"handle_command","got unknown standard command \"%s\"",text);
	return 0;
    }
    
    /*******************************************************************/
    
    if (strstart(text,"/ann")==0) /* same as /announce */
    {
	unsigned int i;
	char         msgtemp[MAX_MESSAGE_LEN];
	char const * tname;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (account_get_auth_announce(conn_get_account(c))!=1) /* default to false */
	{
	    message_send(c,MT_WARN,c,"You do not have permission to use this command.");
	    return 0;
	}
	
	sprintf(msgtemp,"Announcement from %.64s: %.128s",(tname = conn_get_username(c)),&text[i]);
	conn_unget_username(tname);
	message_send_all(MT_BROADCAST,c,msgtemp);
	
	return 0;
    }
    if (strstart(text,"/watch")==0)
    {
	unsigned int i;
	t_account *  account;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
        {
	    message_send(c,MT_WARN,c,"Who do you want to watch?");
	    return 0;
	}
	if (!(account = accountlist_find_account(&text[i])))
	{
	    message_send(c,MT_WARN,c,"That user does not exist.");
	    return 0;
	}
	
        conn_add_watch(c,account); /* FIXME: adds all events for now */
	
        return 0;
    }
    if (strstart(text,"/unwatch")==0)
    {
	unsigned int i;
	t_account *  account;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
        {
	    message_send(c,MT_WARN,c,"Who do you want to unwatch?");
	    return 0;
	}
	if (!(account = accountlist_find_account(&text[i])))
	{
	    message_send(c,MT_WARN,c,"That user does not exist.");
	    return 0;
	}
	
        if (conn_del_watch(c,account)<0) /* FIXME: deletes all events for now */
	{
	    message_send(c,MT_WARN,c,"You are not watching that user.");
	    return 0;
	}
	
        return 0;
    }
    if (strstart(text,"/watchall")==0)
    {
        conn_add_watch(c,NULL); /* FIXME: adds all events for now */
	
        return 0;
    }
    if (strstart(text,"/unwatchall")==0)
    {
        if (conn_del_watch(c,NULL)<0) /* FIXME: deletes all events for now */
	{
	    message_send(c,MT_WARN,c,"You are not watching all users.");
	    return 0;
	}
	
        return 0;
    }
    if (strstart(text,"/lusers")==0)
    {
	char         msgtemp[MAX_MESSAGE_LEN];
	char const * curr;
	unsigned int i;
	
	if (!conn_get_channel(c))
        { 
            message_send(c,MT_ERROR,c,"This command can only be used inside a channel.");
	    return 0;
        }
	
	strcpy(msgtemp,"Banned users:");
	i = strlen(msgtemp);
	for (curr=channel_banlist_get_first(conn_get_channel(c)); curr; curr=channel_banlist_get_next())
	{
	    if (i+strlen(curr)+2>MAX_MESSAGE_LEN) /* " ", name, '\0' */
	    {
		message_send(c,MT_WARN,c,msgtemp);
		i = 0;
	    }
	    sprintf(&msgtemp[i]," %s",curr);
	    i += strlen(&msgtemp[i]);
	}
	if (i>0)
	    message_send(c,MT_WARN,c,msgtemp);
	
	return 0;
    }
    if (strstart(text,"/news")==0)
    {
	char const * filename;
	FILE *       fd;
	
	if ((filename = prefs_get_newsfile()))
	    if ((fd = fopen(filename,"r")))
	    {
		message_send_file(c,fd);
		fclose(fd);
	    }
	    else
	    {
		eventlog(eventlog_level_error,"handle_command","could not open news file \"%s\" for reading (fopen: %s)",filename,strerror(errno));
                message_send(c,MT_WARN,c,"No news today.");
	    }
	else
            message_send(c,MT_WARN,c,"No news today.");
	
	return 0;
    }
    if (strstart(text,"/games")==0)
    {
	unsigned int           i;
	char                   msgtemp[MAX_MESSAGE_LEN];
        t_game const *         curr;
	char const *           tag;
	t_list const * const * save;
        
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
	{
	    tag = conn_get_clienttag(c);
	    message_send(c,MT_WARN,c,"Currently accessable games:");
	}
	else if (strcmp(&text[i],"all")==0)
	{
	    tag = NULL;
	    message_send(c,MT_WARN,c,"All current games:");
	}
	else
	{
	    tag = &text[i];
	    message_send(c,MT_WARN,c,"Current games of that type:");
	}
	 
	sprintf(msgtemp," ------name------ p -status- --------type--------- count --------addr--------");
	message_send(c,MT_WARN,c,msgtemp);
        for (curr=gamelist_get_first(&save); curr; curr=gamelist_get_next(&save))
	    if ((!tag || !prefs_get_hide_pass_games() || strcmp(game_get_pass(curr),"")==0) &&
		(!tag || strcasecmp(game_get_clienttag(curr),tag)==0))
		{
		    struct in_addr gaddr;
		    
		    gaddr.s_addr = htonl(game_get_addr(curr));
        	    sprintf(msgtemp," %-16.16s %1.1s %-8.8s %-21.21s %5u %s:%hu",
			    game_get_name(curr),
			    strcmp(game_get_pass(curr),"")==0 ? "n":"y",
			    game_status_get_str(game_get_status(curr)),
			    game_type_get_str(game_get_type(curr)),
			    game_get_ref(curr),
			    inet_ntoa(gaddr),game_get_port(curr));
                    message_send(c,MT_WARN,c,msgtemp);
		}
        
	return 0;
    }
    if (strstart(text,"/channels")==0 || strstart(text,"/chs")==0)
    {
	unsigned int      i;
	char              msgtemp[MAX_MESSAGE_LEN];
        t_channel const * curr;
	t_connection *    opr;
	char const *      oprname;
	char const *      tag;
        
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
	{
	    tag = conn_get_clienttag(c);
	    message_send(c,MT_WARN,c,"Currently accessable channels:");
	}
	else if (strcmp(&text[i],"all")==0)
	{
	    tag = NULL;
	    message_send(c,MT_WARN,c,"All current channels:");
	}
	else
	{
	    tag = &text[i];
	    message_send(c,MT_WARN,c,"Current channels of that type:");
	}
	
	sprintf(msgtemp," ----------name---------- users ----operator----");
	message_send(c,MT_WARN,c,msgtemp);
        for (curr=channellist_get_first(); curr; curr=channellist_get_next())
	    if ((!tag || !prefs_get_hide_temp_channels() || channel_get_permanent(curr)) &&
		(!tag || !channel_get_clienttag(curr) ||
		 strcasecmp(channel_get_clienttag(curr),tag)==0))
	    {
		if ((opr = channel_get_operator(curr)))
		    oprname = conn_get_username(opr);
		else
		    oprname = NULL;
        	sprintf(msgtemp," %-24.24s %5u %-16.16s",
			channel_get_name(curr),
			channel_get_length(curr),
			oprname?oprname:"");
		if (oprname)
		    conn_unget_username(oprname);
        	message_send(c,MT_WARN,c,msgtemp);
            }
        
	return 0;
    }
    if (strstart(text,"/connections")==0 || strstart(text,"/con")==0)
    {
	char                   msgtemp[MAX_MESSAGE_LEN];
	t_list const * const * save;
	t_connection *         curr;
	char                   name[19];
        
#ifndef CONNDEBUG
	if (account_get_auth_admin(conn_get_account(c))!=1) /* default to false */
        { 
            message_send(c,MT_ERROR,c,"This command is reserved for admins.");
	    return 0;
        }
#endif
	
        message_send(c,MT_WARN,c,"Current connections:");
	sprintf(msgtemp," -#- -class ----state--- tag- -----name------ -session-- -flag- -lat-- c g ---------addr--------");
	message_send(c,MT_WARN,c,msgtemp);
        for (curr=connlist_get_first(&save); curr; curr=connlist_get_next(&save))
        {
	    struct in_addr addr;

	    if (conn_get_account(curr))
	    {
		char const * tname;
		
		sprintf(name,"\"%.16s\"",(tname = conn_get_username(curr)));
		conn_unget_username(tname);
	    }
	    else
		strcpy(name,"(none)");
	    
	    addr.s_addr = htonl(conn_get_addr(curr));
            sprintf(msgtemp," %3u %-6.6s %-12.12s %4.4s %-15.15s 0x%08x 0x%04x 0x%04x %1d %1d %s:%hu",
		    conn_get_socket(curr),
		    conn_class_get_str(conn_get_class(curr)),
		    conn_state_get_str(conn_get_state(curr)),
		    conn_get_clienttag(curr),
		    name,
		    conn_get_sessionkey(curr),
		    conn_get_flags(curr),
		    conn_get_latency(curr),
		    conn_get_channel(curr)!=NULL,
		    conn_get_game(curr)!=NULL,
		    inet_ntoa(addr),conn_get_port(curr));
            message_send(c,MT_WARN,c,msgtemp);
        }
        
	return 0;
    }
    if (strstart(text,"/finger")==0)
    {
	char           dest[USER_NAME_LEN];
	char           msgtemp[MAX_MESSAGE_LEN];
	unsigned int   i,j;
	t_account *    account;
        t_connection * conn;
	char const *   host;
	char *         tok;
	char const *   tname;
	char const *   tsex;
	char const *   tloc;
	char const *   tage;
	char const *   thost;
	char const *   tdesc;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	dest[0] = '\0';
	for (j=0; text[i]!=' ' && text[i]!='\0'; i++,j++) /* get dest */
	    if (j<USER_NAME_LEN-1) dest[j] = text[i];
	dest[j] = '\0';
	
	for (; text[i]==' '; i++);
	
	if (!(account = accountlist_find_account(dest)))
	{
	    message_send(c,MT_ERROR,c,"Invalid user.");
	    return 0;
	}
	
        sprintf(msgtemp,"Login: %-16.16s #%06u Sex: %.16s",
		(tname = account_get_name(account)),
		account_get_uid(account),
		(tsex = account_get_sex(account)));
	account_unget_name(tname);
	account_unget_sex(tsex);
        message_send(c,MT_WARN,c,msgtemp);
	
        sprintf(msgtemp,"Location: %-21.21s Age: %.16s",
		(tloc = account_get_loc(account)),
		(tage = account_get_age(account)));
	account_unget_loc(tloc);
	account_unget_age(tage);
        message_send(c,MT_WARN,c,msgtemp);
	
        if (!(host=thost = account_get_ll_host(account)))
	    host = "unknown";
        {
            time_t then;
            
            then = account_get_ll_time(account);
	    if (!(conn = connlist_find_connection(dest)))
		strftime(msgtemp,MAX_MESSAGE_LEN,"Last login %a %b %d %H:%M from ",localtime(&then));
	    else
		strftime(msgtemp,MAX_MESSAGE_LEN,"On since %a %b %d %H:%M from ",localtime(&then));
        }
	strncat(msgtemp,host,32);
	message_send(c,MT_WARN,c,msgtemp);
	if (thost)
	    account_unget_ll_host(thost);
        
        strncpy(msgtemp,(tdesc = account_get_desc(account)),MAX_MESSAGE_LEN);
	msgtemp[MAX_MESSAGE_LEN-1] = '\0';
	account_unget_desc(tdesc);
        tok = strtok(msgtemp,"\r\n");
        while (tok)
        {
	    if (tok[0]=='\0') /* empty messages crash some clients */
        	message_send(c,MT_WARN,c," ");
	    else
		message_send(c,MT_WARN,c,tok);
            tok = strtok(NULL,"\r\n");
        }
        message_send(c,MT_WARN,c," ");
        
	return 0;
    }
    if (strstart(text,"/operator")==0)
    {
	char                 msgtemp[MAX_MESSAGE_LEN];
	t_connection const * opr;
	
        if (!conn_get_channel(c))
        { 
            message_send(c,MT_ERROR,c,"This command can only be used inside a channel.");
	    return 0;
        }
	
	if (!(opr = channel_get_operator(conn_get_channel(c))))
	    strcpy(msgtemp,"There is no operator.");
	else
	{
	    char const * tname;
	    
	    sprintf(msgtemp,"%.64s is the operator.",(tname = conn_get_username(opr)));
	    conn_unget_username(tname);
	}
        message_send(c,MT_WARN,c,msgtemp);
        return 0;
    }
    if (strstart(text,"/logout")==0 || strstart(text,"/quit")==0)
    {
	message_send(c,MT_WARN,c,"Connection closed.");
        conn_set_state(c,conn_state_destroy);
        return 0;
    }
    if (strstart(text,"/kill")==0)
    {
	unsigned int   i;
	t_connection * user;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
	{
	    message_send(c,MT_ERROR,c,"Which user do you want to kill?");
	    return 0;
	}
        if (!(user = connlist_find_connection(&text[i])))
	{
	    message_send(c,MT_ERROR,c,"That user is not logged in?");
	    return 0;
	}
	if (account_get_auth_admin(conn_get_account(c))!=1) /* default to false */
        { 
            message_send(c,MT_ERROR,c,"This command is reserved for admins.");
	    return 0;
        }
	message_send(user,MT_WARN,user,"Connection closed by admin.");
        conn_set_state(user,conn_state_destroy);
        return 0;
    }
    if (strstart(text,"/killsession")==0)
    {
	unsigned int   i;
	t_connection * user;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
	{
	    message_send(c,MT_ERROR,c,"Which session do you want to kill?");
	    return 0;
	}
	if (!isxdigit((int)text[i]))
	{
	    message_send(c,MT_ERROR,c,"That is not a valid session.");
	    return 0;
	}
	if (!(user = connlist_find_connection_by_sessionkey((unsigned int)strtoul(&text[i],NULL,16))))
	{
            message_send(c,MT_ERROR,c,"That session does not exist.");
	    return 0;
	}
	if (account_get_auth_admin(conn_get_account(c))!=1) /* default to false */
        { 
            message_send(c,MT_ERROR,c,"This command is reserved for admins.");
	    return 0;
        }
	message_send(user,MT_WARN,user,"Connection closed by admin.");
        conn_set_state(user,conn_state_destroy);
        return 0;
    }
    if (strstart(text,"/gameinfo")==0)
    {
	unsigned int   i;
	char           msgtemp[MAX_MESSAGE_LEN];
	t_game const * game;
	struct in_addr gaddr;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
	{
	    message_send(c,MT_ERROR,c,"Which game do you want info for?");
	    return 0;
	}
        if (!(game = gamelist_find_game(&text[i],game_type_all)))
	{
	    message_send(c,MT_ERROR,c,"That game does not exist.");
	    return 0;
	}
	
	gaddr.s_addr = htonl(game_get_addr(game));
	
        sprintf(msgtemp,"Name: %-20.20s    ID: %06u (%s)",game_get_name(game),game_get_id(game),strcmp(game_get_pass(game),"")==0?"public":"private");
        message_send(c,MT_WARN,c,msgtemp);
        sprintf(msgtemp,"Owner: %-20.20s  (%s:%hu)","???",inet_ntoa(gaddr),game_get_port(game));
        message_send(c,MT_WARN,c,msgtemp);
        sprintf(msgtemp,"Client: %4s (%u)",game_get_clienttag(game),game_get_version(game));
	message_send(c,MT_WARN,c,msgtemp);
	sprintf(msgtemp,"Created: %s  Started: %s","","");
        message_send(c,MT_WARN,c,msgtemp);
	sprintf(msgtemp,"Status: %s",game_status_get_str(game_get_status(game)));
	message_send(c,MT_WARN,c,msgtemp);
        sprintf(msgtemp,"Type: %-20.20s",game_type_get_str(game_get_type(game)));
	message_send(c,MT_WARN,c,msgtemp);
        sprintf(msgtemp,"Map: %-20.20s","???");
	message_send(c,MT_WARN,c,msgtemp);
        sprintf(msgtemp,"Players: %u/%u",game_get_ref(game),game_get_count(game));
	message_send(c,MT_WARN,c,msgtemp);
	
	return 0;
    }
    if (strstart(text,"/ladderactivate")==0)
    {
	if (account_get_auth_admin(conn_get_account(c))!=1) /* default to false */
        { 
            message_send(c,MT_ERROR,c,"This command is reserved for admins.");
	    return 0;
        }
	ladder_make_all_active();
	message_send(c,MT_WARN,c,"Updated active scores on STAR and SEXP ladders.");
	return 0;
    }
    if (strstart(text,"/ladderinfo")==0)
    {
	unsigned int  i;
	char          msgtemp[MAX_MESSAGE_LEN];
	unsigned int  rank;
	t_account *   account;
	char const *  tname;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	if (text[i]=='\0')
	{
	    message_send(c,MT_ERROR,c,"Which rank do you want ladder info for?");
	    return 0;
	}
	if (str_to_uint(&text[i],&rank)<0 || rank<1)
	{
	    message_send(c,MT_ERROR,c,"Invalid rank.");
	    return 0;
	}
	
        if ((account = ladder_get_account_by_rank(rank,ladder_sort_highestrated,ladder_time_active,CLIENTTAG_STARCRAFT)))
	{
	    sprintf(msgtemp,"Starcraft active  %5u: %-20.20s %u/%u/%u rating %u",
		    rank,
		    (tname = account_get_name(account)),
		    account_get_ladder_active_wins(account,CLIENTTAG_STARCRAFT),
		    account_get_ladder_active_losses(account,CLIENTTAG_STARCRAFT),
		    account_get_ladder_active_disconnects(account,CLIENTTAG_STARCRAFT),
		    account_get_ladder_active_rating(account,CLIENTTAG_STARCRAFT));
	    account_unget_name(tname);
	}
	else
	    sprintf(msgtemp,"Starcraft active  %5u: <none>",rank);
	message_send(c,MT_WARN,c,msgtemp);
	
        if ((account = ladder_get_account_by_rank(rank,ladder_sort_highestrated,ladder_time_current,CLIENTTAG_STARCRAFT)))
	{
	    sprintf(msgtemp,"Starcraft current %5u: %-20.20s %u/%u/%u rating %u",
		    rank,
		    (tname = account_get_name(account)),
		    account_get_ladder_wins(account,CLIENTTAG_STARCRAFT),
		    account_get_ladder_losses(account,CLIENTTAG_STARCRAFT),
		    account_get_ladder_disconnects(account,CLIENTTAG_STARCRAFT),
		    account_get_ladder_rating(account,CLIENTTAG_STARCRAFT));
	    account_unget_name(tname);
	}
	else
	    sprintf(msgtemp,"Starcraft current %5u: <none>",rank);
	message_send(c,MT_WARN,c,msgtemp);
	
        if ((account = ladder_get_account_by_rank(rank,ladder_sort_highestrated,ladder_time_active,CLIENTTAG_BROODWARS)))
	{
	    sprintf(msgtemp,"Brood War active  %5u: %-20.20s %u/%u/%u rating %u",
		    rank,
		    (tname = account_get_name(account)),
		    account_get_ladder_active_wins(account,CLIENTTAG_BROODWARS),
		    account_get_ladder_active_losses(account,CLIENTTAG_BROODWARS),
		    account_get_ladder_active_disconnects(account,CLIENTTAG_BROODWARS),
		    account_get_ladder_active_rating(account,CLIENTTAG_BROODWARS));
	    account_unget_name(tname);
	}
	else
	    sprintf(msgtemp,"Brood War active  %5u: <none>",rank);
	message_send(c,MT_WARN,c,msgtemp);
	
        if ((account = ladder_get_account_by_rank(rank,ladder_sort_highestrated,ladder_time_current,CLIENTTAG_BROODWARS)))
	{
	    sprintf(msgtemp,"Brood War current %5u: %-20.20s %u/%u/%u rating %u",
		    rank,
		    (tname = account_get_name(account)),
		    account_get_ladder_wins(account,CLIENTTAG_BROODWARS),
		    account_get_ladder_losses(account,CLIENTTAG_BROODWARS),
		    account_get_ladder_disconnects(account,CLIENTTAG_BROODWARS),
		    account_get_ladder_rating(account,CLIENTTAG_BROODWARS));
	    account_unget_name(tname);
	}
	else
	    sprintf(msgtemp,"Brood War current %5u: <none>",rank);
	message_send(c,MT_WARN,c,msgtemp);
	
	return 0;
    }
    if (strstart(text,"/timer")==0)
    {
	unsigned int i,j;
	unsigned int delta;
	char         deltastr[64];
	t_timer_data data;
	
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	
	deltastr[0] = '\0';
	if (text[i]=='/') /* skip / in front of command (if present) */
	    i++;
	for (j=0; text[i]!=' ' && text[i]!='\0'; i++,j++) /* get comm */
	    if (j<64-1) deltastr[j] = text[i];
	deltastr[j] = '\0';
	
	for (; text[i]==' '; i++);
	
	if (strlen(deltastr)<1)
	{
	    message_send(c,MT_ERROR,c,"How long do you want the timer to last?");
	    return 0;
	}
	if (str_to_uint(deltastr,&delta)<0)
	{
	    message_send(c,MT_ERROR,c,"Invalid duration.");
	    return 0;
	}
	
	if (strlen(&text[i])<0)
	    data.p = strdup("Your timer has expired.");
	else
	    data.p = strdup(&text[i]);
	
	if (timerlist_add_timer(c,time(NULL)+(time_t)delta,user_timer_cb,data)<0)
	{
	    eventlog(eventlog_level_error,"handle_command","could not add timer");
	    pfree(data.p,strlen(data.p)+1);
	    message_send(c,MT_ERROR,c,"Could not set timer.");
	}
	
	return 0;
    }
    if (strstart(text,"/help")==0)
    {
	unsigned int i,j;
	char         comm[MAX_COMMAND_LEN];
	
	comm[0] = '\0';
	for (i=0; text[i]!=' ' && text[i]!='\0'; i++); /* skip command */
	for (; text[i]==' '; i++);
	if (text[i]=='/') /* skip / in front of command (if present) */
	    i++;
	for (j=0; text[i]!=' ' && text[i]!='\0'; i++,j++) /* get comm */
	    if (j<MAX_COMMAND_LEN-1) comm[j] = text[i];
	comm[j] = '\0';
	
	if (strlen(comm)<1)
	{
	    message_send(c,MT_WARN,c,"Chat commands:");
	    message_send(c,MT_WARN,c," /me");
	    message_send(c,MT_WARN,c," /msg /whisper /w /m");
	    message_send(c,MT_WARN,c," /status /users");
	    message_send(c,MT_WARN,c," /who");
	    message_send(c,MT_WARN,c," /whois /whereis");
	    message_send(c,MT_WARN,c," /announce /ann");
	    message_send(c,MT_WARN,c," /stats /astat");
	    message_send(c,MT_WARN,c," /time");
	    message_send(c,MT_WARN,c," /channel /join");
	    message_send(c,MT_WARN,c," /rejoin");
	    message_send(c,MT_WARN,c," /away");
	    message_send(c,MT_WARN,c," /dnd");
	    message_send(c,MT_WARN,c," /ignore /squelch");
	    message_send(c,MT_WARN,c," /unignore /unsquelch");
	    message_send(c,MT_WARN,c," /designate");
	    message_send(c,MT_WARN,c," /resign");
	    message_send(c,MT_WARN,c," /kick");
	    message_send(c,MT_WARN,c," /ban");
	    message_send(c,MT_WARN,c," /unban");
	    message_send(c,MT_WARN,c," /uptime");
	    message_send(c,MT_WARN,c," /version");
            /****/
	    message_send(c,MT_WARN,c," /watch");
	    message_send(c,MT_WARN,c," /unwatch");
	    message_send(c,MT_WARN,c," /watchall");
	    message_send(c,MT_WARN,c," /unwatchall");
	    message_send(c,MT_WARN,c," /lusers");
	    message_send(c,MT_WARN,c," /games");
	    message_send(c,MT_WARN,c," /channels /chs");
	    message_send(c,MT_WARN,c," /connections /con");
	    message_send(c,MT_WARN,c," /finger");
	    message_send(c,MT_WARN,c," /operator");
	    message_send(c,MT_WARN,c," /copyright /warranty /license");
	    message_send(c,MT_WARN,c," /news");
	    message_send(c,MT_WARN,c," /logout /quit");
	    message_send(c,MT_WARN,c," /kill");
	    message_send(c,MT_WARN,c," /killsession");
	    message_send(c,MT_WARN,c," /gameinfo");
	    message_send(c,MT_WARN,c," /ladderactivate");
	    message_send(c,MT_WARN,c," /ladderinfo");
	    message_send(c,MT_WARN,c," /timer");
	    message_send(c,MT_WARN,c," /help");
	    
	    return 0;
	}
	
	{
	    char const * helpstr;
	    
	    if (strcasecmp(comm,"whois")==0 ||
		strcasecmp(comm,"whereis")==0)
		helpstr = " /whois, /whereis - looks up some basic information on a user, including their account number and their current location";
	    else if (strcasecmp(comm,"whisper")==0 ||
		     strcasecmp(comm,"msg")==0 ||
		     strcasecmp(comm,"m")==0 ||
		     strcasecmp(comm,"w")==0)
		helpstr = " /msg, /whisper, /m, /w - sends a private message to a user online regardless of the channel they are in";
	    else if (strcasecmp(comm,"squelch")==0 ||
		     strcasecmp(comm,"ignore")==0)
		helpstr = " /ignore, /squelch - ignores any future messages from that user effectively muting that user to you";
            else if (strcasecmp(comm,"unsquelch")==0 ||
		     strcasecmp(comm,"unignore")==0)
		helpstr = " /unignore, /unsquelch - allows a previously squelched user to talk to you normally";
            else if (strcasecmp(comm,"away")==0)
		helpstr = " /away - displays a message that you are away from your keyboard whenever someone whispers to you or does a /whois command on you";
            else if (strcasecmp(comm,"dnd")==0)
		helpstr = " /dnd - prevents ALL whispers from displaying to your screen";
            else if (strcasecmp(comm,"who")==0)
		helpstr = " /who - displays a list of all users in the given channel";
            else if (strcasecmp(comm,"stats")==0 ||
		     strcasecmp(comm,"astat")==0)
		helpstr = " /stats /astat - displays a player's game record";
            else if (strcasecmp(comm,"users")==0 ||
		     strcasecmp(comm,"status")==0)
		helpstr = " /status, /users - displays the number of users currently on Battle.net";
            else if (strcasecmp(comm,"time")==0)
		helpstr = " /time - displays the current Battle.net time";
	    else if (strcasecmp(comm,"channel")==0 ||
		     strcasecmp(comm,"join")==0)
		helpstr = " /channel /join - switches to a channel without having to use the channel join screen";
	    else if (strcasecmp(comm,"/rejoin")==0)
		helpstr = " /rejoin - join the channel you are already in";
	    else if (strcasecmp(comm,"me")==0)
		helpstr = " /me - displays emotion text, as in 'tN is rolling on the floor laughing'";
            else if (strcasecmp(comm,"designate")==0)
		helpstr = " /designate - selects user as the next channel operator when you step down, either by resigning or by leaving the channel";
            else if (strcasecmp(comm,"resign")==0)
		helpstr = " /resign - lets you step down as channel operator and become a normal user again";
            else if (strcasecmp(comm,"kick")==0)
		helpstr = " /kick - kicks a user out of the channel";
            else if (strcasecmp(comm,"ban")==0)
		helpstr = " /ban - kicks a user out of the channel and prevents them from returning until the current operator steps down or leaves the channel";
            else if (strcasecmp(comm,"unban")==0)
		helpstr = " /unban - allows a banned user to return to the channel";
            else if (strcasecmp(comm,"uptime")==0)
	        helpstr = " /uptime - show the amount of time the server has been running";
            else if (strcasecmp(comm,"version")==0)
	        helpstr = " /version - shows the version of Bnetd";
	    /************** bnetd specific ********************/
            else if (strcasecmp(comm,"lusers")==0)
		helpstr = " /lusers - shows list of banned players in this channel";           
            else if (strcasecmp(comm,"games")==0)
		helpstr = " /games - displays current game list";
            else if (strcasecmp(comm,"channels")==0 || strcasecmp(comm,"chs")==0)
		helpstr = " /channels /chs - displays current channel list";
            else if (strcasecmp(comm,"connections")==0 || strcasecmp(comm,"con")==0)
		helpstr = " /connections /con - displays current connections";
            else if (strcasecmp(comm,"finger")==0)
		helpstr = " /finger - displays information about player";
            else if (strcasecmp(comm,"operator")==0)
		helpstr = " /operator - shows who is the operator of this channel";
            else if (strcasecmp(comm,"announce")==0 || strcasecmp(comm,"ann")==0)
	        helpstr = " /announce /ann - announces a message in all channels";
            else if (strcasecmp(comm,"copyright")==0)
	        helpstr = " /copyright /warranty /license - shows the legal information for Bnetd";
	    else if (strcasecmp(comm,"news")==0)
		helpstr = " /news - show some system-specific news";
            else if (strcasecmp(comm,"logout")==0 || strcasecmp(comm,"quit")==0)
	        helpstr = " /logout /quit - closes your connection to the server";
            else if (strcasecmp(comm,"kill")==0)
	        helpstr = " /kill - close a player's connection to the server";
            else if (strcasecmp(comm,"killsession")==0)
	        helpstr = " /killsession - close a connection to the server";
            else if (strcasecmp(comm,"watch")==0)
	        helpstr = " /watch - sends notification when a player logs or out or joins a game";
            else if (strcasecmp(comm,"unwatch")==0)
	        helpstr = " /unwatch - stops sending notification about a player";
            else if (strcasecmp(comm,"watchall")==0)
	        helpstr = " /watchall - sends notification when any player logs in or out";
            else if (strcasecmp(comm,"unwatchall")==0)
	        helpstr = " /unwatchall - stops sending notification about all players";
            else if (strcasecmp(comm,"gameinfo")==0)
	        helpstr = " /gameinfo - prints information about a game";
            else if (strcasecmp(comm,"ladderactivate")==0)
	        helpstr = " /ladderactivate - move current ladder stats to active ladder stats";
            else if (strcasecmp(comm,"ladderinfo")==0)
	        helpstr = " /ladderinfo - display ladder info for given rank";
            else if (strcasecmp(comm,"timer")==0)
	        helpstr = " /timer - send notification after a period of time";
            else if (strcasecmp(comm,"help")==0)
	        helpstr = " /help - does this";
	    else
	    {
		message_send(c,MT_ERROR,c," no help avaliable for that command");
		return 0;
	    }
	    
	    message_send(c,MT_WARN,c,helpstr);
	}
	
	return 0;
    }
    
    message_send(c,MT_ERROR,c,"Unknown command.");
    eventlog(eventlog_level_debug,"handle_command","got unknown command \"%s\"",text);
    return 0;
}
