/*
 * Copyright (C) 1998  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stddef.h>
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <errno.h>
#include "compat/strerror.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "compat/netinet_in.h"
#include <arpa/inet.h>
#include "packet.h"
#include "eventlog.h"
#include "network.h"


extern int net_recv_packet(int sock, t_packet * packet, unsigned int * currsize)
{
    int            addlen;
    unsigned int   header_size;
    t_packet_class class;
    
    if (!packet)
    {
	eventlog(eventlog_level_error,"net_recv_packet","[%d] got NULL packet (closing connection)",sock);
	return -1;
    }
    if (!currsize)
    {
	eventlog(eventlog_level_error,"net_recv_packet","[%d] got NULL currsize (closing connection)",sock);
	return -1;
    }
    
    class = packet_get_class(packet);
    
    switch (class)
    {
    case packet_class_normal:
	header_size = sizeof(t_normal_header);
	if (*currsize<1)
	    packet_set_size(packet,0);
	break;
    case packet_class_file:
	header_size = sizeof(t_file_header);
	if (*currsize<1)
	    packet_set_size(packet,0);
	break;
    case packet_class_raw:
	header_size = 0;
	break;
    default:
	eventlog(eventlog_level_error,"net_recv_packet","[%d] packet has bad class (closing connection)",sock);
	return -1;
    }
    
    if (*currsize<header_size)
    {
#ifdef HAVE_RECV
	addlen = recv(sock,
		      packet_get_raw_data_build(packet,*currsize),
		      header_size-*currsize,
		      0);
#else
	addlen = recvfrom(sock,
			  packet_get_raw_data_build(packet,*currsize),
			  header_size-*currsize,
			  0,NULL,NULL);
#endif
    }
    else
    {
	unsigned int total_size=packet_get_size(packet);
	
	if (total_size<header_size)
	{
	    eventlog(eventlog_level_warn,"net_recv_packet","[%d] corrupted packet received (total_size=%u currsize=%u) (closing connection)",sock,total_size,*currsize);
	    return -1;
	}
	
#ifdef HAVE_RECV
	addlen = recv(sock,
		      packet_get_raw_data_build(packet,*currsize),
		      total_size-*currsize,
		      0);
#else
	addlen = recvfrom(sock,
			  packet_get_raw_data_build(packet,*currsize),
			  total_size-*currsize,
			  0,NULL,NULL);
#endif
    }
    
    if (addlen==0)
    {
	eventlog(eventlog_level_debug,"net_recv_packet","[%d] remote host closed connection",sock);
	return -1;
    }
    if (addlen<0)
    {
        if (errno==EINTR || errno==EAGAIN || errno==EWOULDBLOCK) /* try later */
            return 0;
	if (errno==ENOTCONN || errno==ECONNRESET)
	{
/*	    eventlog(eventlog_level_debug,"net_recv_packet","[%d] remote host closed connection (recvfrom: %s)",sock,strerror(errno)); */
	    return -1; /* close connection, but no message */
	}
	eventlog(eventlog_level_error,"net_recv_packet","[%d] receive error (closing connection) (recvfrom: %s)",sock,strerror(errno));
	return -1;
    }
    
    *currsize += addlen;
    
    if (*currsize>=header_size && *currsize==packet_get_size(packet))
	return 1;
    
    return 0;
}


extern int net_send_packet(int sock, t_packet const * packet, unsigned int * currsize)
{
    unsigned int size;
    int          addlen;
    
    if (!packet)
    {
	eventlog(eventlog_level_error,"net_send_packet","[%d] got NULL packet (closing connection)",sock);
	return -1;
    }
    if (!currsize)
    {
	eventlog(eventlog_level_error,"net_send_packet","[%d] got NULL currsize (closing connection)",sock);
	return -1;
    }
    
    if ((size = packet_get_size(packet))<1)
    {
	eventlog(eventlog_level_error,"net_send_packet","[%d] packet to send is empty (skipping it)",sock);
	*currsize = 0;
	return 1;
    }
    
    addlen = sendto(sock,
		    packet_get_raw_data_const(packet,*currsize),
		    size-*currsize,
		    0,NULL,0);
    
    if (addlen<0)
    {
	if (errno==EINTR || errno==EAGAIN || errno==EWOULDBLOCK)
	    return 0; /* try again later */
	
	eventlog(eventlog_level_error,"net_send_packet","[%d] could not send data (closing connection) (sendto: %s)",sock,strerror(errno));
	return -1;
    }
    
    if (addlen==0)
    {
	eventlog(eventlog_level_error,"net_send_packet","[%d] no data sent (closing connection)",sock);
	return -1;
    }
    
    *currsize += addlen;
    
    /* sent all data in this packet? */
    if (size==*currsize)
    {
	*currsize = 0;
	return 1;
    }
    
    return 0;
}
