/*
 * Copyright (C) 1998  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#define GAME_INTERNAL_ACCESS
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "compat/strdup.h"
#include <errno.h>
#include "compat/strerror.h"
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include "compat/netinet_in.h"
#include <arpa/inet.h>
#include "compat/inet_aton.h"
#include "compat/inet_ntoa.h"
#include "eventlog.h"
#include "prefs.h"
#include "connection.h"
#include "account.h"
#include "ladder.h"
#include "bnettime.h"
#include "util.h"
#include "list.h"
#include "game.h"


static t_list * gamelist_head=NULL;
static int totalcount=0;


static void game_trans_net(unsigned int * addr, unsigned short * port);
static void game_choose_host(t_game * game);
static void game_destroy(t_game const * game);
static int game_report(t_game * game);


static void game_trans_net(unsigned int * addr, unsigned short * port)
{
    struct in_addr curraddr;
    char *         temp;
    char *         laddr;
    char *         raddr;
    char *         rport;
    char *         tok;
    
    if (!(temp = strdup(prefs_get_gametrans())))
    {
	eventlog(eventlog_level_error,"game_trans_net","could not allocate memory for temp");
	return;
    }
    
    tok = strtok(temp,","); /* strtok modifies the string it is passed */
    while (tok)
    {
	if (!(laddr = malloc(strlen(tok)+1)))
	{
	    eventlog(eventlog_level_error,"game_trans_net","could not allocate memory for laddr");
	    pfree(temp,strlen(prefs_get_gametrans())+1);
	    return;
	}
	if (!(raddr = malloc(strlen(tok)+1)))
	{
	    eventlog(eventlog_level_error,"game_trans_net","could not allocate memory for raddr");
	    pfree(laddr,strlen(tok)+1);
	    pfree(temp,strlen(prefs_get_gametrans())+1);
	    return;
	}
	if (!(rport = malloc(strlen(tok)+1)))
	{
	    eventlog(eventlog_level_error,"game_trans_net","could not allocate memory for rport");
	    pfree(raddr,strlen(tok)+1);
	    pfree(laddr,strlen(tok)+1);
	    pfree(temp,strlen(prefs_get_gametrans())+1);
	    return;
	}
	
	/* example: "12.34.56.78@11.22.33.44:8000" */
	if (sscanf(tok,"%[0123456789.]@%[0123456789.]:%[0123456789]",laddr,raddr,rport)!=3)
	{
	    eventlog(eventlog_level_error,"game_trans_net","bad gametrans format in config file");
	    pfree(rport,strlen(tok)+1);
	    pfree(raddr,strlen(tok)+1);
	    pfree(laddr,strlen(tok)+1);
	    pfree(temp,strlen(prefs_get_gametrans())+1);
	    return;
	}
	if (inet_aton(laddr,&curraddr)==0)
	{
	    eventlog(eventlog_level_error,"game_trans_net","invalid left-hand address \"%s\" for gametrans in config file",laddr);
	    pfree(rport,strlen(tok)+1);
	    pfree(raddr,strlen(tok)+1);
	    pfree(laddr,strlen(tok)+1);
	    pfree(temp,strlen(prefs_get_gametrans())+1);
	    return;
	}
	if (*addr==ntohl(curraddr.s_addr))
	{
	    if (inet_aton(raddr,&curraddr)==0)
	    {
		eventlog(eventlog_level_error,"game_trans_net","invalid right-hand address \"%s\" for gametrans in config file",raddr);
		pfree(rport,strlen(tok)+1);
		pfree(raddr,strlen(tok)+1);
		pfree(laddr,strlen(tok)+1);
		pfree(temp,strlen(prefs_get_gametrans())+1);
		return;
	    }
	    *addr = ntohl(curraddr.s_addr);
	    if (str_to_ushort(rport,port)<0)
	    {
		eventlog(eventlog_level_error,"game_trans_net","invalid right-hand port \"%s\" for gametrans in config file",rport);
		pfree(rport,strlen(tok)+1);
		pfree(raddr,strlen(tok)+1);
		pfree(laddr,strlen(tok)+1);
		pfree(temp,strlen(prefs_get_gametrans())+1);
		return;
	    }
	    pfree(rport,strlen(tok)+1);
	    pfree(raddr,strlen(tok)+1);
	    pfree(laddr,strlen(tok)+1);
	    pfree(temp,strlen(prefs_get_gametrans())+1);
	    return;
	}
	
	pfree(rport,strlen(tok)+1);
	pfree(raddr,strlen(tok)+1);
	pfree(laddr,strlen(tok)+1);
	tok = strtok(NULL,",");
    }
    
    pfree(temp,strlen(prefs_get_gametrans())+1);
}


static void game_choose_host(t_game * game)
{
    unsigned int i;
    
    if (game->count<1)
    {
	eventlog(eventlog_level_error,"game_choose_host","game has had no connections?!");
	return;
    }
    if (!game->connections)
    {
	eventlog(eventlog_level_error,"game_choose_host","game has NULL connections array");
	return;
    }
    
    for (i=0; i<game->count; i++)
	if (game->connections[i])
	{
	    game->addr = conn_get_game_addr(game->connections[i]);
	    game->port = conn_get_game_port(game->connections[i]);
	    return;
	}
    eventlog(eventlog_level_warn,"game_choose_host","no valid connections found");
}


extern char const * game_type_get_str(t_game_type type)
{
    switch (type)
    {
    case game_type_none:
	return "NONE";
	
    case game_type_melee:
	return "melee";
	
    case game_type_ffa:
	return "free for all";
	
    case game_type_oneonone:
	return "one on one";
	
    case game_type_ctf:
	return "capture the flag";
	
    case game_type_greed:
	return "greed";
	
    case game_type_slaughter:
	return "slaughter";
	
    case game_type_sdeath:
	return "sudden death";
	
    case game_type_ladder:
	return "ladder";
	
    case game_type_mapset:
	return "mapset";
	
    case game_type_teammelee:
	return "team melee";
	
    case game_type_teamffa:
	return "team free for all";
	
    case game_type_teamctf:
	return "team capture the flag";
	
    case game_type_pgl:
	return "PGL";
	
    case game_type_diablo:
	return "Diablo";
	
    case game_type_all:
    default:
	return "UNKNOWN";
    }
}


extern char const * game_status_get_str(t_game_status status)
{
    switch (status)
    {
    case game_status_started:
	return "started";
	
    case game_status_full:
	return "full";
	
    case game_status_open:
	return "open";
	
    case game_status_done:
	return "done";
	
    default:
	return "UNKNOWN";
    }
}


extern char const * game_result_get_str(t_game_result result)
{
    switch (result)
    {
    case game_result_none:
        return "NONE";

    case game_result_win:
        return "WIN";

    case game_result_loss:
        return "LOSS";

    case game_result_draw:
        return "DRAW";

    case game_result_disconnect:
        return "DISCONNECT";

    default:
        return "UNKNOWN";
    }
}


extern t_game * game_create(char const * name, char const * pass, char const * info, t_game_type type, int version, char const * clienttag)
{
    t_game * game;
    
    if (!name)
    {
	eventlog(eventlog_level_info,"game_create","got NULL game name");
	return NULL;
    }
    if (!pass)
    {
	eventlog(eventlog_level_info,"game_create","got NULL game pass");
	return NULL;
    }
    if (!info)
    {
	eventlog(eventlog_level_info,"game_create","got NULL game info");
	return NULL;
    }
    
    if (gamelist_find_game(name,game_type_all))
    {
	eventlog(eventlog_level_info,"game_create","game \"%s\" not created because it already exists",name);
	return NULL; /* already have a game by that name */
    }
    
    if (!(game = malloc(sizeof(t_game))))
    {
	eventlog(eventlog_level_error,"game_create","could not allocate memory for game");
	return NULL;
    }
    
    if (!(game->name = strdup(name)))
    {
	pfree(game,sizeof(t_game));
	return NULL;
    }
    if (!(game->pass = strdup(pass)))
    {
	pfree((void *)game->name,strlen(game->name)+1); /* avoid warning */
	pfree(game,sizeof(t_game));
	return NULL;
    }
    if (!(game->info = strdup(info)))
    {
	pfree((void *)game->pass,strlen(game->pass)+1); /* avoid warning */
	pfree((void *)game->name,strlen(game->name)+1); /* avoid warning */
	pfree(game,sizeof(t_game));
	return NULL;
    }
    if (!(game->clienttag = strdup(clienttag)))
    {
	eventlog(eventlog_level_error,"game_create","could not allocate memory for game->clienttag");
	pfree((void *)game->info,strlen(game->info)+1); /* avoid warning */
	pfree((void *)game->pass,strlen(game->pass)+1); /* avoid warning */
	pfree((void *)game->name,strlen(game->name)+1); /* avoid warning */
	pfree(game,sizeof(t_game));
	return NULL;
    }
    game->type          = type;
    game->addr          = 0; /* will be set by first player */
    game->port          = 0;
    game->version       = version;
    game->status        = game_status_open;
    game->id            = ++totalcount;
    game->ref           = 0;
    game->count         = 0;
    game->connections   = NULL;
    game->players       = NULL;
    game->results       = NULL;
    game->report_heads  = NULL;
    game->report_bodies = NULL;
    game->create_time   = (unsigned int)time(NULL);
    game->start_time    = 0;
    game->bad           = 0;
    
    if (list_prepend_item(&gamelist_head,game)<0)
    {
	eventlog(eventlog_level_error,"game_create","could not insert game");
	pfree((void *)game->clienttag,strlen(game->clienttag)+1); /* avoid warning */
	pfree((void *)game->info,strlen(game->info)+1); /* avoid warning */
	pfree((void *)game->pass,strlen(game->pass)+1); /* avoid warning */
	pfree((void *)game->name,strlen(game->name)+1); /* avoid warning */
	pfree(game,sizeof(t_game));
	return NULL;
    }
    
    eventlog(eventlog_level_info,"game_create","game \"%s\" (pass \"%s\") type %hu version %d created",name,pass,(unsigned short)type,version);
    
    return game;
}


static void game_destroy(t_game const * game)
{
    unsigned int i;
    
    if (!game)
    {
	eventlog(eventlog_level_error,"game_destroy","got NULL game");
	return;
    }
    
    if (list_remove_item(&gamelist_head,game)<0)
    {
	eventlog(eventlog_level_error,"game_destroy","could not find game \"%s\" in list",game_get_name(game));
        return;
    }
    
    eventlog(eventlog_level_debug,"game_destroy","game \"%s\" (count=%u ref=%u) removed from list...",game_get_name(game),game->count,game->ref);
    
    for (i=0; i<game->count; i++)
    {
	if (game->report_bodies && game->report_bodies[i])
	    pfree((void *)game->report_bodies[i],strlen(game->report_bodies[i])+1); /* avoid warning */
	if (game->report_heads && game->report_heads[i])
	    pfree((void *)game->report_heads[i],strlen(game->report_heads[i])+1); /* avoid warning */
    }
    if (game->report_bodies)
	pfree((void *)game->report_bodies,game->count*sizeof(char const *)); /* avoid warning */
    if (game->report_heads)
	pfree((void *)game->report_heads,game->count*sizeof(char const *)); /* avoid warning */
    if (game->results)
	pfree((void *)game->results,game->count*sizeof(t_game_result)); /* avoid warning */
    if (game->connections)
	pfree((void *)game->connections,game->count*sizeof(t_connection *)); /* avoid warning */
    if (game->players)
	pfree((void *)game->players,game->count*sizeof(t_account *)); /* avoid warning */
    pfree((void *)game->clienttag,strlen(game->clienttag)+1); /* avoid warning */
    pfree((void *)game->info,strlen(game->info)+1); /* avoid warning */
    pfree((void *)game->pass,strlen(game->pass)+1); /* avoid warning */
    pfree((void *)game->name,strlen(game->name)+1); /* avoid warning */
    pfree((void *)game,sizeof(t_game)); /* avoid warning */
    
    eventlog(eventlog_level_info,"game_destroy","game deleted");
    
    return;
}


static int game_report(t_game * game)
{
    FILE *          fp;
    char *          realname;
    char *          tempname;
    char const *    tname;
    unsigned int    i;
    unsigned int    realcount;
    time_t          now=time(NULL);
    t_ladder_info * ladder_info=NULL;
    
    if (!game)
    {
	eventlog(eventlog_level_error,"game_report","got NULL game");
	return -1;
    }
    if (!game->clienttag || strlen(game->clienttag)!=4)
    {
	eventlog(eventlog_level_error,"game_report","got bad clienttag");
	return -1;
    }
    if (!game->players)
    {
	eventlog(eventlog_level_error,"game_report","player array is NULL");
	return -1;
    }
    if (!game->results)
    {
	eventlog(eventlog_level_error,"game_report","results array is NULL");
	return -1;
    }
    
    /* "compact" the game; move all the real players to the top... */
    realcount = 0;
    for (i=0; i<game->count; i++)
    {
	if (!game->players[i])
	{
	    eventlog(eventlog_level_error,"game_report","player slot %u has NULL account",i);
	    continue;
	}
	
	if (game->results[i]!=game_result_none)
	{
	    game->players[realcount]       = game->players[i];
	    game->results[realcount]       = game->results[i];
	    game->report_heads[realcount]  = game->report_heads[i];
	    game->report_bodies[realcount] = game->report_bodies[i];
	    realcount++;
	}
    }
    eventlog(eventlog_level_debug,"game_report","realcount=%d count=%u",realcount,game->count);
    
    /* nuke duplicate players after the real players */
    for (i=realcount; i<game->count; i++)
    {
	game->players[i]       = NULL;
        game->results[i]       = game_result_none;
	game->report_heads[i]  = NULL;
	game->report_bodies[i] = NULL;
    }
    
    if (realcount<1)
    {
	eventlog(eventlog_level_error,"game_report","no results for any player, ignoring game");
	return 0;
    }
    
    if (!game->bad)
	if (game_get_type(game)==game_type_ladder)
	{
	    for (i=0; i<realcount; i++)
	    {
		eventlog(eventlog_level_debug,"game_report","realplayer %u result=%u",i+1,(unsigned int)game->results[i]);
		
		ladder_init_account(game->players[i],game->clienttag);
		
		switch (game->results[i])
		{
		case game_result_win:
		    account_inc_ladder_wins(game->players[i],game->clienttag);
		    break;
		case game_result_loss:
		    account_inc_ladder_losses(game->players[i],game->clienttag);
		    break;
		case game_result_draw:
		    account_inc_ladder_draws(game->players[i],game->clienttag);
		    break;
		case game_result_disconnect:
		    account_inc_ladder_disconnects(game->players[i],game->clienttag);
		default:
		    eventlog(eventlog_level_error,"game_report","bad game realplayer results[%u] = %u",i,game->results[i]);
		    account_inc_ladder_disconnects(game->players[i],game->clienttag);
		}
		account_set_ladder_last_result(game->players[i],game->clienttag,game_result_get_str(game->results[i]));
		account_set_ladder_last_time(game->players[i],game->clienttag,bnettime());
	    }
	    
	    if (!(ladder_info = malloc(sizeof(t_ladder_info)*realcount)))
		eventlog(eventlog_level_error,"game_report","unable to allocate memory for ladder_info, ladder ratings will not be updated");
	    else
		if (ladder_update(game->clienttag,realcount,game->players,game->results,ladder_info)<0)
		{
		    eventlog(eventlog_level_error,"game_report","unable to update ladder stats");
		    pfree(ladder_info,sizeof(t_ladder_info)*realcount);
		    ladder_info = NULL;
		}
	}
	else
	    for (i=0; i<realcount; i++)
	    {
		switch (game->results[i])
		{
		case game_result_win:
		    account_inc_normal_wins(game->players[i],game->clienttag);
		    break;
		case game_result_loss:
		    account_inc_normal_losses(game->players[i],game->clienttag);
		    break;
		case game_result_draw:
		    account_inc_normal_draws(game->players[i],game->clienttag);
		    break;
		case game_result_disconnect:
		default:
		    account_inc_normal_disconnects(game->players[i],game->clienttag);
		}
		account_set_normal_last_result(game->players[i],game->clienttag,game_result_get_str(game->results[i]));
		account_set_normal_last_time(game->players[i],game->clienttag,bnettime());
	    }
    
    if (game_get_type(game)!=game_type_ladder && prefs_get_report_all_games()!=1)
    {
	eventlog(eventlog_level_debug,"game_report","not reporting normal games");
	return 0;
    }
    
    {
	struct tm * tmval;
	char        dstr[64];
	
	if (!(tmval = gmtime(&now)))
	    dstr[0] = '\0';
	else
	    sprintf(dstr,"%04d%02d%02d%02d%02d%02d",
		    1900+tmval->tm_year,
		    tmval->tm_mon,
		    tmval->tm_mday,
		    tmval->tm_hour,
		    tmval->tm_min,
		    tmval->tm_sec);
	
	if (!(tempname = malloc(strlen(prefs_get_reportdir())+1+1+5+1+2+1+strlen(dstr)+1+6+1)))
	{
	    eventlog(eventlog_level_error,"game_report","could not allocate memory for tempname");
	    if (ladder_info)
		pfree(ladder_info,sizeof(t_ladder_info)*realcount);
	    return -1;
	}
	sprintf(tempname,"%s/.bnetd-gr_%s_%06u",prefs_get_reportdir(),dstr,game->id);
	if (!(realname = malloc(strlen(prefs_get_reportdir())+1+2+1+strlen(dstr)+1+6+1)))
	{
	    eventlog(eventlog_level_error,"game_report","could not allocate memory for realname");
	    pfree(tempname,strlen(tempname)+1);
	    if (ladder_info)
		pfree(ladder_info,sizeof(t_ladder_info)*realcount);
	    return -1;
	}
	sprintf(realname,"%s/gr_%s_%06u",prefs_get_reportdir(),dstr,game->id);
    }
    
    if (!(fp = fopen(tempname,"w")))
    {
	eventlog(eventlog_level_error,"game_report","could not open report file \"%s\" for writing (fopen: %s)",tempname,strerror(errno));
	if (ladder_info)
	    pfree(ladder_info,sizeof(t_ladder_info)*realcount);
	pfree(realname,strlen(realname)+1);
	pfree(tempname,strlen(tempname)+1);
	return -1;
    }
    
    if (game->bad)
	fprintf(fp,"[ game results ignored due to inconsistancies ]\n\n");
    fprintf(fp,"name=\"%s\" id=%06u\n",
	    game->name,
	    game->id);
    fprintf(fp,"clienttag=%4s type=\"%s\"\n",
	    game->clienttag,
	    game_type_get_str(game->type));
    fprintf(fp,"created=%u started=%u ended=%lu\n",
	    game->create_time,
	    game->start_time,
	    (unsigned long)now);
    {
	struct in_addr gaddr;
	
	gaddr.s_addr = htonl(game->addr); 
	fprintf(fp,"host=%s\n\n\n",inet_ntoa(gaddr));
    }
    
    if (ladder_info)
	for (i=0; i<realcount; i++)
	{
	    tname = account_get_name(game->players[i]);
	    fprintf(fp,"%-16s %-8s rating=%u [#%04u]  prob=%4.1f%%  K=%2u  adj=%+d\n",
		    tname,
		    game_result_get_str(game->results[i]),
		    ladder_info[i].oldrating,
		    ladder_info[i].oldrank,
		    ladder_info[i].prob*100.0,
		    ladder_info[i].k,
		    ladder_info[i].adj);
	    account_unget_name(tname);
	}
    else
	for (i=0; i<realcount; i++)
	{
	    tname = account_get_name(game->players[i]);
	    fprintf(fp,"%-16s %-8s\n",
		    tname,
		    game_result_get_str(game->results[i]));
	    account_unget_name(tname);
	}
    fprintf(fp,"\n\n");
    
    if (ladder_info)
	pfree(ladder_info,sizeof(t_ladder_info)*realcount);
    
    for (i=0; i<realcount; i++)
    {
	if (game->report_heads[i])
	    fprintf(fp,"%s\n",game->report_heads[i]);
	else
	{
	    tname = account_get_name(game->players[i]);
	    fprintf(fp,"[ game report header not avaliable for player %u (\"%s\") ]\n",i+1,tname);
	    account_unget_name(tname);
	}
	if (game->report_bodies[i])
	    fprintf(fp,"%s\n",game->report_bodies[i]);
	else
	{
	    tname = account_get_name(game->players[i]);
	    fprintf(fp,"[ game report body not avaliable for player %u (\"%s\") ]\n\n",i+1,tname);
	    account_unget_name(tname);
	}
    }
    fprintf(fp,"\n\n");
    
    for (i=0; i<realcount; i++)
    {
	tname = account_get_name(game->players[i]);
	fprintf(fp,"%s's normal record is now %u/%u/%u (%u draws)\n",
		tname,
		account_get_normal_wins(game->players[i],game->clienttag),
		account_get_normal_losses(game->players[i],game->clienttag),
		account_get_normal_disconnects(game->players[i],game->clienttag),
		account_get_normal_draws(game->players[i],game->clienttag));
	account_unget_name(tname);
    }
    fprintf(fp,"\n");
    for (i=0; i<realcount; i++)
    {
	tname = account_get_name(game->players[i]);
	fprintf(fp,"%s's ladder record is now %u/%u/%u (rating %u [#%05u]) (%u draws)\n",
		tname,
		account_get_ladder_wins(game->players[i],game->clienttag),
		account_get_ladder_losses(game->players[i],game->clienttag),
		account_get_ladder_disconnects(game->players[i],game->clienttag),
		account_get_ladder_rating(game->players[i],game->clienttag),
		account_get_ladder_rank(game->players[i],game->clienttag),
		account_get_ladder_draws(game->players[i],game->clienttag));
	account_unget_name(tname);
    }
    fprintf(fp,"\nThis game lasted %u minutes (elapsed).\n",((unsigned int)now-game->start_time)/60);
    
    fclose(fp);
    
    if (rename(tempname,realname)<0)
    {
	eventlog(eventlog_level_error,"game_report","could not rename report to \"%s\" (rename: \"%s\")",realname,strerror(errno));
	pfree(realname,strlen(realname)+1);
	pfree(tempname,strlen(tempname)+1);
	return -1;
    }
    
    eventlog(eventlog_level_debug,"game_report","game report saved as \"%s\"",realname);
    pfree(realname,strlen(realname)+1);
    pfree(tempname,strlen(tempname)+1);
    return 0;
}


extern unsigned int game_get_id(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_id","got NULL game");
        return 0;
    }
    return game->id;
}


extern char const * game_get_name(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_name","got NULL game");
        return NULL;
    }
    return game->name;
}


extern t_game_type game_get_type(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_type","got NULL game");
        return 0;
    }
    return game->type;
}


extern char const * game_get_pass(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_pass","got NULL game");
        return NULL;
    }
    return game->pass;
}


extern char const * game_get_info(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_info","got NULL game");
        return NULL;
    }
    return game->info;
}


extern unsigned int game_get_version(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_version","got NULL game");
        return 0;
    }
    return game->version;
}


extern unsigned int game_get_ref(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_ref","got NULL game");
        return 0;
    }
    return game->ref;
}


extern unsigned int game_get_count(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_count","got NULL game");
        return 0;
    }
    return game->count;
}


extern void game_set_status(t_game * game, t_game_status status)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_set_status","got NULL game");
        return;
    }
    if (status==game_status_started && game->start_time==0)
	game->start_time = (unsigned int)time(NULL);
    game->status = status;
}


extern t_game_status game_get_status(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_status","got NULL game");
        return 0;
    }
    return game->status;
}


extern unsigned short game_get_port(t_game const * game)
{
    unsigned int   addr;
    unsigned short port;
    
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_port","got NULL game");
        return 0;
    }
    
    addr = game->addr; /* host byte order */
    port = prefs_get_gameport();
    game_trans_net(&addr,&port);
    
    return port;
}


extern int game_set_port(t_game * game, unsigned int port)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_set_port","got NULL game");
        return -1;
    }
    
    game->port = port;
    
    return 0;
}


extern unsigned int game_get_addr(t_game const * game)
{
    unsigned int   addr;
    unsigned short port;
    
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_addr","got NULL game");
        return 0;
    }
    addr = game->addr; /* host byte order */
    port = game->port; /* host byte order */
    game_trans_net(&addr,&port);
    return addr;
}


extern int game_set_addr(t_game * game, unsigned int addr)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_set_addr","got NULL game");
        return -1;
    }
    
    game->addr = addr;
    
    return 0;
}


extern unsigned int game_get_latency(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_latency","got NULL game");
        return 0;
    }
    if (game->ref<1)
    {
	eventlog(eventlog_level_error,"game_get_latency","game \"%s\" has no players",game->name);
	return 0;
    }
    if (!game->players)
    {
	eventlog(eventlog_level_error,"game_get_latency","game \"%s\" has NULL players array (ref=%u)",game->name,game->ref);
	return 0;
    }
    if (!game->players[0])
    {
	eventlog(eventlog_level_error,"game_get_latency","game \"%s\" has NULL players[0] entry (ref=%u)",game->name,game->ref);
	return 0;
    }
    
    return 0; /* conn_get_latency(game->players[0]); */
}


extern char const * game_get_clienttag(t_game const * game)
{
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_clienttag","got NULL game");
        return NULL;
    }
    return game->clienttag;
}


extern int game_add_player(t_game * game, char const * pass, int version, t_connection * c)
{
    t_connection * * tempc;
    t_account * *    tempp;
    t_game_result *  tempr;
    char const * *   temprh;
    char const * *   temprb;
    
    if (!game)
    {
	eventlog(eventlog_level_error,"game_add_player","got NULL game");
        return -1;
    }
    if (!pass)
    {
	eventlog(eventlog_level_error,"game_add_player","got NULL password");
	return -1;
    }
    if (version!=GAME_VERSION_UNKNOWN && version!=GAME_VERSION_1 && version!=GAME_VERSION_3 && version!=GAME_VERSION_4)
    {
	eventlog(eventlog_level_error,"game_add_player","got bad game version %d",version);
	return -1;
    }
    if (!c)
    {
	eventlog(eventlog_level_error,"game_add_player","got NULL connection");
        return -1;
    }
    if (game->type==game_type_ladder && account_get_normal_wins(conn_get_account(c),conn_get_clienttag(c))<10)
    {
	eventlog(eventlog_level_error,"game_add_player","can not join ladder game without 10 normal wins");
	return -1;
    }
    
    {
	char const * gt;
	
	if (!(gt = game_get_clienttag(game)))
	{
	    eventlog(eventlog_level_error,"game_add_player","could not get clienttag for game");
	    return -1;
	}
	if (strcmp(conn_get_clienttag(c),gt)!=0)
	{
	    eventlog(eventlog_level_error,"game_add_player","player clienttag (\"%s\") does not match game clienttag (\"%s\")",conn_get_clienttag(c),gt);
	    return -1;
	}
    }
    
    if (strcasecmp(game->pass,pass)!=0)
	return -1;
    
    if (!game->connections) /* some realloc()s are broken */
    {
	if (!(tempc = malloc((game->count+1)*sizeof(t_connection *))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game connections");
	    return -1;
	}
    }
    else
    {
	if (!(tempc = realloc(game->connections,(game->count+1)*sizeof(t_connection *))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game connections");
	    return -1;
	}
    }
    game->connections = tempc;
    
    if (!game->players) /* some realloc()s are broken */
    {
	if (!(tempp = malloc((game->count+1)*sizeof(t_account *))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game players");
	    return -1;
	}
    }
    else
    {
	if (!(tempp = realloc(game->players,(game->count+1)*sizeof(t_account *))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game players");
	    return -1;
	}
    }
    game->players = tempp;
    
    if (!game->results) /* some realloc()s are broken */
    {
	if (!(tempr = malloc((game->count+1)*sizeof(t_game_result))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game results");
	    return -1;
	}
    }
    else
    {
	if (!(tempr = realloc(game->results,(game->count+1)*sizeof(t_game_result))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game results");
	    return -1;
	}
    }
    game->results = tempr;
    
    if (!game->report_heads) /* some realloc()s are broken */
    {
	if (!(temprh = malloc((game->count+1)*sizeof(char const *))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game report headers");
	    return -1;
	}
    }
    else
    {
	if (!(temprh = realloc(game->report_heads,(game->count+1)*sizeof(char const *))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game report headers");
	    return -1;
	}
    }
    game->report_heads = temprh;
    
    if (!game->report_bodies) /* some realloc()s are broken */
    {
	if (!(temprb = malloc((game->count+1)*sizeof(char const *))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game report bodies");
	    return -1;
	}
    }
    else
    {
	if (!(temprb = realloc(game->report_bodies,(game->count+1)*sizeof(char const *))))
	{
	    eventlog(eventlog_level_error,"game_add_player","unable to allocate memory for game report bodies");
	    return -1;
	}
    }
    game->report_bodies = temprb;
    
    game->connections[game->count]   = c;
    game->players[game->count]       = conn_get_account(c);
    game->results[game->count]       = game_result_none;
    game->report_heads[game->count]  = NULL;
    game->report_bodies[game->count] = NULL;
    
    game->count++;
    game->ref++;
    
    if (game->version!=version)
    {
	char const * tname;
	
	eventlog(eventlog_level_error,"game_add_player","player \"%s\" client \"%s\" version %u joining game version %u (count=%u ref=%u)",(tname = account_get_name(conn_get_account(c))),conn_get_clienttag(c),version,game->version,game->count,game->ref);
	account_unget_name(tname);
    }
    
    game_choose_host(game);
    
    return 0;
}


extern int game_del_player(t_game * game, t_connection * c)
{
    char const * tname;
    unsigned int i;
    t_account *  account;
    
    if (!game)
    {
	eventlog(eventlog_level_error,"game_del_player","got NULL game");
        return -1;
    }
    if (!c)
    {
	eventlog(eventlog_level_error,"game_del_player","got NULL connection");
	return -1;
    }
    if (!game->players)
    {
	eventlog(eventlog_level_error,"game_del_player","player array is NULL");
	return -1;
    }
    if (!game->results)
    {
	eventlog(eventlog_level_error,"game_del_player","results array is NULL");
	return -1;
    }
    
    account = conn_get_account(c);
    
    for (i=0; i<game->count; i++)
	if (game->players[i]==account && game->connections[i])
	{
	    eventlog(eventlog_level_debug,"game_del_player","removing player #%u \"%s\" from \"%s\", %u players left",i,(tname = account_get_name(account)),game_get_name(game),game->ref-1);
	    game->connections[i] = NULL;
	    
	    if (game->results[i]!=game_result_win &&
		game->results[i]!=game_result_loss &&
		game->results[i]!=game_result_draw &&
		game->results[i]!=game_result_disconnect &&
	        game->results[i]!=game_result_none)
		eventlog(eventlog_level_error,"game_del_player","player \"%s\" leaving with bad result %u",tname,game->results[i]);
	    account_unget_name(tname);
	    
	    eventlog(eventlog_level_debug,"game_del_player","player deleted...");
	    
	    if (game->ref<2)
	    {
	        eventlog(eventlog_level_debug,"game_del_player","no more players, reporting game");
		game_report(game);
	        eventlog(eventlog_level_debug,"game_del_player","no more players, destroying game");
		game_destroy(game);
	        return 0;
	    }
	    
	    game->ref--;
	    
	    game_choose_host(game);
	    
	    return 0;
	}
    
    eventlog(eventlog_level_error,"game_del_player","player \"%s\" was not in the game",(tname = account_get_name(account)));
    account_unget_name(tname);
    return -1;
}


extern int game_check_result(t_game * game, t_account * account, t_game_result result)
{
    unsigned int pos;
    char const * tname;
    
    if (!game)
    {
	eventlog(eventlog_level_error,"game_check_result","got NULL game");
	return -1;
    }
    if (!account)
    {
	eventlog(eventlog_level_error,"game_check_result","got NULL account");
	return -1;
    }
    if (!game->players)
    {
	eventlog(eventlog_level_error,"game_check_result","player array is NULL");
	return -1;
    }
    if (!game->results)
    {
	eventlog(eventlog_level_error,"game_check_result","results array is NULL");
	return -1;
    }
    if (result!=game_result_win &&
	result!=game_result_loss &&
	result!=game_result_draw &&
	result!=game_result_disconnect &&
	result!=game_result_playing)
    {
	eventlog(eventlog_level_error,"game_check_result","got bad result %u for player \"%s\"",(unsigned int)result,(tname = account_get_name(account)));
	account_unget_name(tname);
	return -1;
    }
    
    {
	unsigned int i;
	
	pos = game->count;
	for (i=0; i<game->count; i++)
	    if (game->players[i]==account)
		pos = i;
    }
    if (pos==game->count)
    {
	eventlog(eventlog_level_error,"game_check_result","could not find player \"%s\" to check result",(tname = account_get_name(account)));
	account_unget_name(tname);
	game->bad = 1;
	return 0; /* return success even though it didn't check */
    }
    
    if (game->results[pos]==result)
    {
	eventlog(eventlog_level_debug,"game_check_result","result %u agrees with previous for player in slot %u",(unsigned int)result,pos);
	return 0;
    }
    if (game->results[pos]==game_result_none ||
	game->results[pos]==game_result_playing)
    {
	if (result==game_result_playing)
	    eventlog(eventlog_level_debug,"game_check_result","player in slot %u reported as \"still playing\", assuming that is right",pos);
	game->results[pos] = result;
	eventlog(eventlog_level_debug,"game_check_result","result %u initially obtained for player in slot %u",(unsigned int)result,pos);
	return 0;
    }
    
    eventlog(eventlog_level_error,"game_check_result","got inconsistant results for player in slot %u (\"%s\") previous=%u current=%u",pos,(tname = account_get_name(account)),(unsigned int)game->results[pos],(unsigned int)result);
    account_unget_name(tname);
    game->bad = 1;
    return 0; /* return success even though it didn't check */
}


extern int game_set_result(t_game * game, t_account * account, t_game_result result, char const * rephead, char const * repbody)
{
    unsigned int pos;
    char const * tname;
    
    if (!game)
    {
	eventlog(eventlog_level_error,"game_set_result","got NULL game");
	return -1;
    }
    if (!account)
    {
	eventlog(eventlog_level_error,"game_set_result","got NULL account");
	return -1;
    }
    if (!game->players)
    {
	eventlog(eventlog_level_error,"game_set_result","player array is NULL");
	return -1;
    }
    if (!game->results)
    {
	eventlog(eventlog_level_error,"game_set_result","results array is NULL");
	return -1;
    }
    if (!game->report_heads)
    {
	eventlog(eventlog_level_error,"game_set_result","report_heads array is NULL");
	return -1;
    }
    if (!game->report_bodies)
    {
	eventlog(eventlog_level_error,"game_set_result","report_bodies array is NULL");
	return -1;
    }
    if (result!=game_result_win &&
	result!=game_result_loss &&
	result!=game_result_draw &&
	result!=game_result_disconnect)
    {
	eventlog(eventlog_level_error,"game_set_result","got bad result %u for player \"%s\"",(unsigned int)result,(tname = account_get_name(account)));
	account_unget_name(tname);
	return -1;
    }
    if (!rephead)
    {
	eventlog(eventlog_level_error,"game_set_result","report head is NULL");
	return -1;
    }
    if (!repbody)
    {
	eventlog(eventlog_level_error,"game_set_result","report body is NULL");
	return -1;
    }
    
    {
	unsigned int i;
	
	pos = game->count;
	for (i=0; i<game->count; i++)
	    if (game->players[i]==account)
		pos = i;
    }
    if (pos==game->count)
    {
	eventlog(eventlog_level_error,"game_set_result","could not find player \"%s\" to set result",(tname = account_get_name(account)));
	account_unget_name(tname);
	return -1;
    }
    
    if (!(game->report_heads[pos] = strdup(rephead)))
    {
	eventlog(eventlog_level_error,"game_set_result","could not allocate memory for report_heads in slot %u",pos);
	return -1;
    }
    if (!(game->report_bodies[pos] = strdup(repbody)))
    {
	eventlog(eventlog_level_error,"game_set_result","could not allocate memory for report_bodies in slot %u",pos);
	return -1;
    }
	
    if (game_check_result(game,account,result)<0)
    {
	eventlog(eventlog_level_error,"game_set_result","self-reported result did not check out with previous for player in slot %u",pos);
	return -1;
    }
    
    return 0;
}


extern t_game_result game_get_result(t_game * game, t_account * account)
{
    unsigned int pos;
    
    if (!game)
    {
	eventlog(eventlog_level_error,"game_get_result","got NULL game");
	return game_result_none;
    }
    if (!account)
    {
	eventlog(eventlog_level_error,"game_get_result","got NULL account");
	return game_result_none;
    }
    if (!game->players)
    {
	eventlog(eventlog_level_error,"game_get_result","player array is NULL");
	return game_result_none;
    }
    if (!game->results)
    {
	eventlog(eventlog_level_error,"game_get_result","results array is NULL");
	return game_result_none;
    }
    
    {
	unsigned int i;
	
	pos = game->count;
	for (i=0; i<game->count; i++)
	    if (game->players[i]==account)
		pos = i;
    }
    if (pos==game->count)
    {
	char const * tname;
	
	eventlog(eventlog_level_error,"game_get_result","could not find player \"%s\" to return result",(tname = account_get_name(account)));
	account_unget_name(tname);
	
	return game_result_none;
    }
    
    return game->results[pos];
}


extern void gamelist_init(void)
{
}


extern int gamelist_get_length(void)
{
    return list_get_length(gamelist_head);
}


extern t_game * gamelist_find_game(char const * name, t_game_type type)
{
    t_list const * const * save;
    t_game *               curr;
    
    for (curr=gamelist_get_first(&save); curr; curr=gamelist_get_next(&save))
        if ((type==game_type_all || curr->type==type) && strcasecmp(name,curr->name)==0)
            return curr;
    
    return NULL;
}


extern t_game * gamelist_get_first(t_list const * const * * save)
{
    void * game;
    
    if (!save)
    {
	eventlog(eventlog_level_error,"gamelist_get_first","got NULL save");
        return NULL;
    }
    
    *save = (t_list const * const *)&gamelist_head; /* avoid warning */
    
    if (!**save)
    {
        *save = NULL;
        return NULL;
    }
    game = list_get_item(**save);
    *save = list_get_next_const(**save);
    
    return game;
}


extern t_game * gamelist_get_next(t_list const * const * * save)
{
    void * game;
    
    if (!save)
    {
	eventlog(eventlog_level_error,"gamelist_get_next","got NULL save");
        return NULL;
    }
    
    if (!*save || !**save)
    {
        *save = NULL;
        return NULL;
    }
    game = list_get_item(**save);
    *save = list_get_next_const(**save);
    
    return game;
}


extern int gamelist_total_games(void)
{
    return totalcount;
}
