/*
 * Copyright (C) 1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#define ADDR_INTERNAL_ACCESS
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "compat/strdup.h"
#include <errno.h>
#include "compat/strerror.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "compat/netinet_in.h"
#include <arpa/inet.h>
#include "compat/inet_ntoa.h"
#include <netdb.h>
#include "eventlog.h"
#include "list.h"
#include "util.h"
#include "addr.h"


extern t_addr * addr_create_num(unsigned int ipaddr, unsigned short port)
{
    t_addr * temp;
    
    if (!(temp = malloc(sizeof(t_addr))))
    {
	eventlog(eventlog_level_error,"addr_create_num","unable to allocate memory for addr");
	return NULL;
    }
    
    temp->str    = NULL;
    temp->ip     = ipaddr;
    temp->port   = port;
    temp->data.p = NULL;
    
    return temp;
}


extern t_addr * addr_create_str(char const * str, unsigned int defipaddr, unsigned short defport)
{
    char *             tstr;
    t_addr *           temp;
    struct sockaddr_in tsa;
    struct hostent *   hp;
    char const *       hostname;
    char *             portstr;
    unsigned short     port;
    
    if (!str)
    {
	eventlog(eventlog_level_error,"addr_create_str","got NULL str");
	return NULL;
    }
    
    if (!(tstr = strdup(str)))
    {
	eventlog(eventlog_level_error,"addr_create_str","could not allocate memory for str");
	return NULL;
    }
    
    if ((portstr = strrchr(tstr,':')))
    {
	*portstr = '\0';
	portstr++;
	
	if (portstr[0]!='\0')
	{
	    if (str_to_ushort(portstr,&port)<0)
	    {
		eventlog(eventlog_level_error,"addr_create_str","could not convert \"%s\" to a port number",portstr);
		pfree(tstr,strlen(str)+1);
		return NULL;
	    }
	}
	else
	    port = defport;
    }
    else
	port = defport;
    
    memset(&tsa,0,sizeof(tsa));
    tsa.sin_family = AF_INET;
    tsa.sin_port = htons((unsigned short)0);
    
    if (tstr[0]!='\0')
    {
	if (!(hostname = strdup(tstr)))
	{
	    eventlog(eventlog_level_error,"addr_create_str","could not allocate memory for hostname");
	    pfree(tstr,strlen(str)+1);
	    return NULL;
	}
	
	hp = gethostbyname(hostname);
	if (!hp || !hp->h_addr_list)
	{
	    eventlog(eventlog_level_error,"addr_create_str","could not lookup host \"%s\"",hostname);
	    pfree((void *)hostname,strlen(hostname)+1); /* avoid warning */
	    pfree(tstr,strlen(str)+1);
	    return NULL;
    	}
	
	memcpy(&tsa.sin_addr,(struct in_addr *)hp->h_addr_list[0],sizeof(struct in_addr)); /* avoid warning */
    }
    else
    {
	tsa.sin_addr.s_addr = htonl(defipaddr);
	if (!(hostname = strdup(inet_ntoa(tsa.sin_addr))))
	{
	    eventlog(eventlog_level_error,"addr_create_str","could not allocate memory for hostname");
	    pfree(tstr,strlen(str)+1);
	    return NULL;
	}
    }
    
    pfree(tstr,strlen(str)+1);
    
    if (!(temp = malloc(sizeof(t_addr))))
    {
	eventlog(eventlog_level_error,"addr_create_str","unable to allocate memory for addr");
	pfree((void *)hostname,strlen(hostname)+1); /* avoid warning */
	return NULL;
    }
    
    temp->str    = hostname;
    temp->ip     = ntohl(tsa.sin_addr.s_addr);
    temp->port   = port;
    temp->data.p = NULL;
    
    return temp;
}


extern int addr_destroy(t_addr const * addr)
{
    if (!addr)
    {
	eventlog(eventlog_level_error,"addr_destroy","got NULL addr");
	return -1;
    }
    
    if (addr->str)
	pfree((void *)addr->str,strlen(addr->str)+1); /* avoid warning */
    pfree((void *)addr,sizeof(t_addr)); /* avoid warning */
    
    return 0;
}


/* hostname or IP */
extern int addr_get_host_str(t_addr const * addr, char * str, unsigned int len)
{
    char               temp[64];
    struct sockaddr_in tsa;
    
    if (!addr)
    {
	eventlog(eventlog_level_error,"addr_get_host_str","got NULL addr");
	return -1;
    }
    if (!str)
    {
	eventlog(eventlog_level_error,"addr_get_host_str","got NULL str");
	return -1;
    }
    if (len<2)
    {
	eventlog(eventlog_level_error,"addr_get_host_str","str too short");
	return -1;
    }
    
    if (addr->str)
    {
	strncpy(str,addr->str,len-1);
	str[len-1] = '\0';
	
	return 0;
    }
    
    memset(&tsa,0,sizeof(tsa));
    tsa.sin_family = AF_INET;
    tsa.sin_port = htons((unsigned short)0);
    tsa.sin_addr.s_addr = htonl(addr->ip);
    sprintf(temp,"%.32s:%hu",inet_ntoa(tsa.sin_addr),addr->port);
    strncpy(str,temp,len-1);
    str[len-1] = '\0';
    
    return 0;
}


/* IP:port */
extern int addr_get_addr_str(t_addr const * addr, char * str, unsigned int len)
{
    char               temp[64];
    struct sockaddr_in tsa;
    
    if (!addr)
    {
	eventlog(eventlog_level_error,"addr_get_addr_str","got NULL addr");
	return -1;
    }
    if (!str)
    {
	eventlog(eventlog_level_error,"addr_get_addr_str","got NULL str");
	return -1;
    }
    if (len<2)
    {
	eventlog(eventlog_level_error,"addr_get_addr_str","str too short");
	return -1;
    }
    
    memset(&tsa,0,sizeof(tsa));
    tsa.sin_family = AF_INET;
    tsa.sin_port = htons((unsigned short)0);
    tsa.sin_addr.s_addr = htonl(addr->ip);
    sprintf(temp,"%.32s:%hu",inet_ntoa(tsa.sin_addr),addr->port);
    strncpy(str,temp,len-1);
    str[len-1] = '\0';
    
    return 0;
}


extern unsigned int addr_get_ip(t_addr const * addr)
{
    if (!addr)
    {
	eventlog(eventlog_level_error,"addr_get_ip","got NULL addr");
	return 0;
    }
    
    return addr->ip;
}


extern unsigned short addr_get_port(t_addr const * addr)
{
    if (!addr)
    {
	eventlog(eventlog_level_error,"addr_get_port","got NULL addr");
	return 0;
    }
    
    return addr->port;
}


extern int addr_set_data(t_addr * addr, t_addr_data data)
{
    if (!addr)
    {
	eventlog(eventlog_level_error,"addr_set_data","got NULL addr");
	return -1;
    }
    
    addr->data = data;
    return 0;
}


extern t_addr_data addr_get_data(t_addr const * addr)
{
    t_addr_data tdata;
    
    if (!addr)
    {
	eventlog(eventlog_level_error,"addr_get_data","got NULL addr");
	tdata.p = NULL;
	return tdata;
    }
    
    return addr->data;
}


extern t_addrlist * addrlist_create(char const * str, unsigned int defipaddr, unsigned short defport)
{
    t_addrlist * addrlist;
    t_addr *     addr;
    char *       tstr;
    char *       tok;
    
    if (!str)
    {
	eventlog(eventlog_level_error,"addrlist_create_str","got NULL str");
	return NULL;
    }
    
    if (!(tstr = strdup(str)))
    {
	eventlog(eventlog_level_error,"addrlist_create","could not allocate memory for tstr");
	return NULL;
    }
    
    addrlist = NULL;
    for (tok=strtok(tstr,","); tok; tok=strtok(NULL,",")) /* strtok modifies the string it is passed */
    {
	if (!(addr = addr_create_str(tok,defipaddr,defport)))
	{
	    eventlog(eventlog_level_error,"addrlist_create","could not create addr");
	    addrlist_destroy(&addrlist);
	    pfree(tstr,strlen(str)+1);
	    return NULL;
	}
	if (list_append_item(&addrlist,addr)<0)
	{
	    eventlog(eventlog_level_error,"addrlist_create","could not add item to list");
	    addr_destroy(addr);
	    addrlist_destroy(&addrlist);
	    pfree(tstr,strlen(str)+1);
	    return NULL;
	}
    }
    
    pfree(tstr,strlen(str)+1);
    
    return addrlist;
}


extern int addrlist_destroy(t_addrlist * * addrlist)
{
    t_addr * addr;
    
    if (!addrlist)
    {
	eventlog(eventlog_level_error,"addrlist_destory","got NULL addrlist");
	return -1;
    }
    
    while (*addrlist)
    {
        if (!(addr = list_get_item(*addrlist)))
            eventlog(eventlog_level_error,"addrlist_destory","found NULL addr in list");
        else
            addr_destroy(addr);
        list_remove(addrlist);
    }
    
    return 0;
}


extern int addrlist_get_length(t_addrlist const * addrlist)
{
    return list_get_length(addrlist);
}


extern t_addr * addrlist_get_first(t_addrlist const * addrlist, t_list const * const * * save)
{
    void * addr;

    if (!save)
    {
        eventlog(eventlog_level_error,"virtconnlist_get_first","got NULL save");
        return NULL;
    }

    *save = (t_list const * const *)&addrlist; /* avoid warning */

    if (!**save)
    {
        *save = NULL;
        return NULL;
    }
    addr = list_get_item(**save);
    *save = list_get_next_const(**save);

    return addr;
}


extern t_addr * addrlist_get_next(t_list const * const * * save)
{
    void * addr;

    if (!save)
    {
        eventlog(eventlog_level_error,"addrlist_get_next","got NULL save");
        return NULL;
    }

    if (!*save || !**save)
    {
        *save = NULL;
        return NULL;
    }
    addr = list_get_item(**save);
    *save = list_get_next_const(**save);

    return addr;
}
