/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>
 
#include <gtk/gtk.h>
#include <pwd.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <fcntl.h>

#ifdef HAVE_GETTIMEOFDAY
#include <sys/time.h>
#else
#include <time.h>
#endif

#include "bezerk.h"
#include "util.h"
#include "servers.h"
#include "reply.h"
#include "send.h"
#include "debug.h"

/* #include "arrow_up.xpm" */
/* #include "arrow_down.xpm" */

extern GtkWidget *user_status, *channel_status;
extern char *conn_status_msg[];
extern GSList *aliases;
extern GSList *system_aliases;
extern GSList *messages;

gushort convert_color (unsigned c) {
  if (c==0) return(0);
  c *= 257;			/* scale from char to short */
/*   c = c + c/5 + 0x800;	 */	/* add brightness */
  return (c > 0xffff)? 0xffff : c;
}


void extract_color (GdkColor *color, unsigned red, unsigned green, unsigned blue) 
{
  color->red = convert_color(red);
  color->green = convert_color(green);
  color->blue =  convert_color(blue);
}

char *homedir;

void aliases_load()
{
  FILE *aliases_fd;
  char buff[BUFFLEN];
  Alias *alias;
  char *comm_name, *comm_value;

  bs_function_enter();

  g_snprintf(buff, BUFFLEN, "%s/aliases", homedir);
  if ( (aliases_fd = fopen(buff, "r")) == NULL) {
    bs_function_leave();
    return;
  }

  while (!feof(aliases_fd)) {
    fgets(buff, BUFFLEN, aliases_fd);
    if (*buff == '#') {
      continue;
    }
    if (strlen(buff)) {
      if (buff[strlen(buff)-1] == '\n') {
	buff[strlen(buff)-1] = '\0';
      }
      comm_name = strtok(buff, " \t");
      comm_value = strtok(NULL, "\0");
      if (comm_name && comm_value) {
	alias = g_malloc(sizeof(Alias));
	alias->name = g_strdup(comm_name);
	alias->value = g_strdup(comm_value);
	aliases = g_slist_append( aliases, alias);
      }
    }
  }
  fclose(aliases_fd);

  bs_function_leave();
  return;
}

void prefs_add_alias(char *name, char *value)
{
  FILE *aliases_fd;
  char buff[BUFFLEN];

  bs_function_enter();

  g_snprintf(buff, BUFFLEN, "%s/aliases", homedir);
  if ( (aliases_fd = fopen(buff, "a")) == NULL) {
    bs_function_leave();
    return;
  }

  fputs("\n", aliases_fd);
  fputs(name, aliases_fd);
  fputs(" ", aliases_fd);
  fputs(value, aliases_fd);

  fclose(aliases_fd);

  bs_function_leave();
  return;
}

void prefs_save_aliases()
{
  FILE *aliases_fd;
  Alias *alias;
  GSList *current_alias;
  char buff[BUFFLEN];

  bs_function_enter();

  g_snprintf(buff, BUFFLEN, "%s/aliases", homedir);
  if ( (aliases_fd = fopen(buff, "w")) == NULL) {
    bs_function_leave();
    return;
  }

  current_alias = aliases;
  while(current_alias) {
    alias = current_alias->data;
    fputs("\n", aliases_fd);
    fputs(alias->name, aliases_fd);
    fputs(" ", aliases_fd);
    fputs(alias->value, aliases_fd);
    current_alias = g_slist_next(current_alias);
  }

  fclose(aliases_fd);

  bs_function_leave();
  return;
}

void aliases_init()
{
  int uid;
  struct passwd *pwdent;
  char buff[BUFFLEN];

  /* Get the UID of the current user */
  uid = getuid();

 /* Get the password entry using the UID */
  if ((pwdent = getpwuid(uid)) == NULL) {
    g_error("Can't find password file entry");
  }

  g_snprintf(buff, BUFFLEN, "%s/.bezerk", pwdent->pw_dir);
  homedir = g_strdup(buff);

  aliases_load();
}

void set_channel_status(ChannelInfo *target_channel)
{
  char buff[BUFFLEN];

  bs_function_enter();

  gtk_statusbar_pop( GTK_STATUSBAR(target_channel->window->channel_status), 1);

  g_snprintf(buff, BUFFLEN, "%s%s%s%s", target_channel->name,
	  strlen(target_channel->mode) ? "(+" : "",
	  strlen(target_channel->mode) ? target_channel->mode : "",
	  strlen(target_channel->mode) ? ")" : ""
	  );

  gtk_statusbar_push( GTK_STATUSBAR(target_channel->window->channel_status), 1, buff);

  bs_function_leave();
  return;
}

void set_user_status(Connection *connection)
{
  char buff[BUFFLEN];
  GSList *list_entry;
  ChannelInfo *channel_info;

  BezChannelWindow *window_info;

  bs_function_enter();

  g_snprintf(buff, BUFFLEN, "%s%s%s%s [%s] %s%s%s", connection->nick,
	     strlen(connection->mode) ? "(+" : "",
	     strlen(connection->mode) ? connection->mode : "",
	     strlen(connection->mode) ? ")" : "",
	     conn_status_msg[connection->status],
	     strlen(connection->lag) ? "[Lag " : "",
	     strlen(connection->lag) ? connection->lag : "",
	     strlen(connection->lag) ? "]" : ""
	     );

  gtk_statusbar_pop( GTK_STATUSBAR(BEZ_CHANNEL_WINDOW(connection->console)->user_status), 1);
  gtk_statusbar_push( GTK_STATUSBAR(BEZ_CHANNEL_WINDOW(connection->console)->user_status), 1, buff);
 
  list_entry = connection->channels;

  while (list_entry) {
    channel_info = (ChannelInfo *) list_entry->data;
    window_info = channel_info->window;
    if ( window_info != BEZ_CHANNEL_WINDOW(connection->console) ) {
      gtk_statusbar_pop( GTK_STATUSBAR(window_info->user_status), 1);
      gtk_statusbar_push( GTK_STATUSBAR(window_info->user_status), 1, buff);
    }
    list_entry = g_slist_next(list_entry);
  }

  bs_function_leave();
  return;
}

void set_user_mode(Connection *connection, char mode, int adding)
{
  int found = FALSE;
  char *current;

  bs_function_enter();

  /* TODO: check this against the new set_channel_status code */
  if (adding && (strlen(connection->mode) < MODELEN) ) {
    connection->mode[strlen(connection->mode)] = mode;
  } else {
    current = connection->mode;
    while (strlen(current)) {
      if (!found) {
	if (*current==mode) {
	  found=TRUE;
	}
      } else {
	current[-1] = *current;
      
      }
      current++;
    }
    current[-1] = '\0';
  }
  bs_function_leave();
  return;
}

char *bs_strtok(char *string, char *delim)
{
  static char *current_position;
  char *current_start;
  static int atend=FALSE;

  bs_function_enter();

  if (string) {
    current_position = string;
    atend = FALSE;
  }

  if ((*current_position == '\0') || atend) {
    atend=TRUE;
    bs_function_leave();
    return(NULL);
  }

  /* Ignore any empty tokens, unlike standard strtok */
  while ((*current_position != '\0') && (strchr(delim, *current_position))) {
    current_position++;
  }

  if (*current_position == '\0') {
    bs_function_leave();
    return(NULL);
  }

  current_start = current_position;
  while ( (*current_position != '\0') && (!strchr(delim, *current_position)) ) {
    current_position++;
  }

  if (current_position[0] == '\0') {
    atend = TRUE;
  } else {
    current_position[0] = '\0';
    atend = FALSE;
    current_position++;
  }

  bs_function_leave();
  return(current_start);
}

#define NUM_RECALL_COMMANDS 50

void cmd_recall_add(char *command, BezBaseWindow *window)
{
  GList *command_entry;
  char *command_str;

  bs_function_enter();

  if (!command || !window) {
    bs_function_leave();
    return;
  }

  command_str = g_strdup(command);
  window->recall_commands = g_list_append(window->recall_commands, command_str);
  window->recall_current = NULL;

  if ( g_list_length(window->recall_commands) > NUM_RECALL_COMMANDS) {
    command_entry = g_list_first(window->recall_commands);
    command_str = command_entry->data;
    window->recall_commands = g_list_remove(window->recall_commands, command_str);
    g_free(command_str);
  }

  bs_function_leave();
  return;
}

char *cmd_recall_previous(BezBaseWindow *window)
{

  bs_function_enter();

  if (!window) {
    bs_function_leave();
    return(NULL);
  }

  if (!window->recall_current) {
    window->recall_current = g_list_last(window->recall_commands);
    bs_function_leave();
    if (window->recall_current) {
      return(window->recall_current->data);
    } else {
      return(NULL);
    }
  } else if ( g_list_previous(window->recall_current) ) {
    window->recall_current = g_list_previous(window->recall_current);
    bs_function_leave();
    return(window->recall_current->data);
  }
    
  bs_function_leave();
  return(NULL);
}

char *cmd_recall_next(BezBaseWindow *window)
{

  bs_function_enter();

  if (!window) {
    bs_function_leave();
    return(NULL);
  }

  if (!window->recall_current) {
    bs_function_leave();
    return(NULL);
  }

  if ( g_list_next(window->recall_current) ) {
    window->recall_current = g_list_next(window->recall_current);
    bs_function_leave();
    return(window->recall_current->data);
  } else {
    window->recall_current = NULL;
  }

  bs_function_leave();
  return(NULL);
}

#define TIMELEN 20

char *get_time_string()
{
#ifdef HAVE_GETTIMEOFDAY

  struct timeval tv ;
  char *buff;

  bs_function_enter();

  buff = (char *) g_malloc(TIMELEN);
  gettimeofday(&tv, NULL);
  g_snprintf(buff, TIMELEN, "%lu.%lu", tv.tv_sec, tv.tv_usec);

#else

  time_t seconds;
  char *buff;

  bs_function_enter();
  
  buff = (char *) g_malloc(TIMELEN);
  seconds = time(NULL);
  g_snprintf(buff, TIMELEN, "%lu", seconds);

#endif

  bs_function_leave();
  return(buff);
}

char *get_time_diff(char *from)
{

#ifdef HAVE_GETTIMEOFDAY

  struct timeval tv ;
  char *time_ptr, *buff;
  time_t sec, usec;

  bs_function_enter();

  gettimeofday(&tv, NULL);
  time_ptr = strtok(from, ".");
  sec = strtol(time_ptr, NULL, 10);
  sec = tv.tv_sec - sec;
  time_ptr = strtok(NULL, ".");
  if (time_ptr) {
    usec = strtol(time_ptr, NULL, 10);
    usec = tv.tv_usec - usec;
    if (usec < 0) {
      sec--;
      usec += 1000000;
    }
    usec /= 10000;
  } else { 
    usec = 0;
  }

  buff = g_malloc(TIMELEN);
  g_snprintf(buff, TIMELEN, "%li.%.2li", sec, usec);

#else

  time_t seconds;
  char *buff;

  bs_function_enter();
  
  seconds = time(NULL) - strtol(from, NULL, 10);
  buff = (char *) g_malloc(TIMELEN);
  g_snprintf(buff, TIMELEN, "%lu", seconds);

#endif

  
  bs_function_leave();
  return(buff);
}

GSList *split_args(char *string)
{
  GSList *args_list=NULL;
  char *arg;

  bs_function_enter();
  
  arg = strtok(string, " \t");
  while (arg) {
    args_list = g_slist_append(args_list, arg);
    arg = strtok(NULL, " \t");
  }

  bs_function_leave();
  return(args_list);
}

/* ------------------------------------------------------------
 * create_pixmap
 *
 *  Create a new pixmap widget from the given data
 *
 * Arguments:
 *	"data" - A pointer to the xpm data
 *      "window" - the window within who's context the pixmap will be created
 *      "background" - the colour for transparent parts of the pixmap
 *
 * Results:
 *	a pointer to the create pixmap widget
 *
 * Side effects:
 *	
 * ----------------------------------------------------------- */

GtkWidget *create_pixmap (gchar     **data,
			  GdkWindow *window,
			  GdkColor  *background)
{
  GtkWidget *wpixmap;
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  pixmap = gdk_pixmap_create_from_xpm_d (window, &mask,
                                       background,
                                       data);
  wpixmap = gtk_pixmap_new (pixmap, mask);

  return(wpixmap);
}

/* GtkWidget *create_arrow_button(GtkWidget *parent, GtkArrowType type) */
/* { */
/*   GtkWidget *button; */
/*   GtkWidget *arrow=NULL; */

/*   button = gtk_button_new (); */
/*   gtk_box_pack_start(GTK_BOX(parent), button, TRUE, TRUE, 0); */
/*   gtk_widget_realize(button); */
/*   switch(type) */
/*     { */
/*     case ARROW_UP: */
/*       arrow = create_pixmap (arrow_up_xpm, button->window, */
/* 		     &button->style->bg[GTK_STATE_NORMAL]); */
/*       break; */
/*     case ARROW_DOWN: */
/*       arrow = create_pixmap (arrow_down_xpm, button->window, */
/* 		     &button->style->bg[GTK_STATE_NORMAL]); */
/*       break; */
/*     } */
/*   gtk_container_add (GTK_CONTAINER (button), arrow); */
/*   gtk_widget_show (arrow); */

/*   bs_function_leave(); */
/*   return(button); */
/* } */
    
GtkWidget *create_labelled_widget(gchar     *handle,
				  gint       handle_width,
				  GtkWidget *widget,
				  gint       widget_width)
{
  GtkWidget *hbox;
  GtkWidget *label;

  hbox = gtk_hbox_new(FALSE, 0);
/*   gtk_container_border_width (GTK_CONTAINER (hbox), 4); */
  
  label = gtk_label_new(handle);
  if (handle_width > 0) {
    gtk_widget_set_usize (label, handle_width, -1);
  }
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
  gtk_widget_show (label);

  if (widget_width > 0) {
    gtk_widget_set_usize (widget, widget_width, -1);
  }
  gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, TRUE, 3);
  gtk_widget_show (widget);
  gtk_widget_show (hbox);
  
  return(hbox);
}
