/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */
 
#include <glib.h>
#include <gtk/gtk.h>
#include <string.h>

#include "irc.h"
#include "bezerk.h"
#include "servers.h"
#include "reply.h"
#include "msg_utils.h"
#include "util.h"
#include "debug.h"

extern GSList *connections;

extern messageData irc_message;
extern GSList *messages;

void set_members_list(ChannelInfo *channel);

int find_channel_entry(GSList *channel_list, char *channel)
{
  ChannelInfo *target_channel;
  GSList *glist_entry;
  int glist_num = 0;

  bs_function_enter();

  glist_entry = channel_list;

  while(glist_entry)  {
    target_channel = (ChannelInfo *) glist_entry->data;
    if (strcasecmp(channel, target_channel->name) == 0) {
      bs_function_leave();
      return(glist_num);
    }
    glist_entry = g_slist_next(glist_entry);
    glist_num++;
  }

  if (!glist_entry) {
    bs_function_leave();
    return(-1);
  }
  
  bs_function_leave();
  return(glist_num);
}

ChannelInfo *find_channel(GSList *channel_list, char *channel)
{

  ChannelInfo *target_channel;
  GSList *glist_entry;
  int glist_num;

  bs_function_enter();

  glist_num = find_channel_entry(channel_list, channel);

  if (glist_num >= 0) {
    glist_entry = g_slist_nth(channel_list, glist_num);
    target_channel = (ChannelInfo *) glist_entry->data;
    bs_function_leave();
    return(target_channel);
  }
  
  bs_function_leave();
  return(NULL);
}

void remove_members(GSList *members)
{
  GSList *glist_entry;

  bs_function_enter();

  glist_entry = members;
  while (glist_entry) {
    g_free(glist_entry->data);
    glist_entry = g_slist_next(glist_entry);
  }
  g_slist_free(members);
  members = NULL;

  bs_function_leave();
  return;
}

void channel_destroy(ChannelInfo *target_channel)
{
    if (target_channel->topic)
      g_free(target_channel->topic);
    if (target_channel->members)
      remove_members(target_channel->members);
    if (target_channel->name)
      g_free(target_channel->name);
}

GSList *remove_channel(GSList *channel_list, char *channel)
{
  ChannelInfo *target_channel;
  GSList *glist_entry;
  int glist_num;

  bs_function_enter();

  glist_num = find_channel_entry(channel_list, channel);

  if (glist_num >= 0) {
    glist_entry = g_slist_nth(channel_list, glist_num);
    target_channel = (ChannelInfo *) glist_entry->data;
    if ( target_channel->window ) {
      if ( target_channel == target_channel->window->current_channel ) {
	gtk_clist_clear( GTK_CLIST(target_channel->window->nick_clist) );
	target_channel->window->current_channel = NULL;
	gtk_window_set_title(GTK_WINDOW(target_channel->window->window), "Bezerk");
      }
      if (target_channel->window != BEZ_CHANNEL_WINDOW(target_channel->connection->console)) {
	gtk_widget_destroy(target_channel->window->window);
      } else {
	gtk_clist_remove(GTK_CLIST(target_channel->window->ch_clist), glist_num);
      }
    }
    if (target_channel->topic)
      g_free(target_channel->topic);
    if (target_channel->members)
      remove_members(target_channel->members);
    if (target_channel->name)
      g_free(target_channel->name);
    channel_list = g_slist_remove(channel_list, target_channel);
    g_free(target_channel);
  }
  
  bs_function_leave();
  return(channel_list);
}

void set_channel_mode(ChannelInfo *channel, char mode, int adding)
{
/*   int found = FALSE; */
  char *current;

  bs_function_enter();

  if (adding && !strchr(channel->mode, mode) && (strlen(channel->mode) < MODELEN-1) ) {
    channel->mode[strlen(channel->mode)] = mode;
    channel->mode[strlen(channel->mode)] = '\0';
  } else if ( !adding && (current = strchr(channel->mode, mode)) ) {
    do {
      current[0] = current[1];
      current++;
    } while (*current);
  }

/*   if (adding && (strlen(channel->mode) < MODELEN-1) ) { */
/*     channel->mode[strlen(channel->mode)] = mode; */
/*     channel->mode[strlen(channel->mode)+1] = '\0'; */
/*   } else { */
/*     current = channel->mode; */
/*     while (strlen(current)) { */
/*       if (!found) { */
/* 	if (*current==mode) { */
/* 	  found=TRUE; */
/* 	} */
/*       } else { */
/* 	current[-1] = *current; */
      
/*       } */
/*       current++; */
/*     } */
/*     current[-1] = '\0'; */
/*   } */

  bs_function_leave();
  return;
}

gint user_info_compare(UserInfo *a, UserInfo *b)
{

  if ( !a ) return(-1);
  if ( !b ) return(1);

  if ( a->status < b->status ) {
    return(1);
  } else if ( a->status > b->status ) {
    return(-1);
  } else {
    return(strcasecmp(a->nick, b->nick));
  }
}

void add_channel_member(ChannelInfo *channel, char* nick)
{
  UserInfo *user_info;

  bs_function_enter();

  user_info = (UserInfo *) g_malloc(sizeof(UserInfo));
  if ( nick[0] == '@' ) {
    user_info->status = USEROPER;
    user_info->nick = g_strdup(nick+1);
   } else if ( nick[0] == '+' ) {
    user_info->status = USERVOICE;
    user_info->nick = g_strdup(nick+1);
   } else {
    user_info->status = 0;
    user_info->nick = g_strdup(nick);
  }
  user_info->address = NULL;

  channel->members = g_slist_insert_sorted(channel->members, user_info, 
					   (GCompareFunc) user_info_compare);

  bs_function_leave();
  return;
}

int remove_channel_member(ChannelInfo *channel, char *nick)
{
  int member_num=0;
  GSList *glist_entry;
  UserInfo *user_info;

  bs_function_enter();

  glist_entry = channel->members;
  while(glist_entry) {
    user_info = (UserInfo *) glist_entry->data;
    if ( !strcasecmp(user_info->nick, nick) ) {
      /* Found entry */
      channel->members = g_slist_remove(channel->members, user_info);
      g_free(user_info->nick);
      if (user_info->address) {
	g_free(user_info->address);
      }
      bs_function_leave();
      return(member_num);
    }
    glist_entry = g_slist_next(glist_entry);
    ++member_num;
  }

  bs_function_leave();
  return(-1);
}

int is_channel_member(ChannelInfo *channel, char *nick)
{
  GSList *glist_entry;
  UserInfo *user_info;

  bs_function_enter();

  glist_entry = channel->members;
  while(glist_entry) {
    user_info = (UserInfo *) glist_entry->data;
    if ( !strcasecmp(user_info->nick, nick) ) {
      /* Found entry */
      bs_function_leave();
      return(TRUE);
    }
    glist_entry = g_slist_next(glist_entry);
  }

  bs_function_leave();
  return(FALSE);
}

int change_member_nick(ChannelInfo *channel, char *oldnick, char *newnick)
{
  int member_num=0;
  GSList *glist_entry;
  UserInfo *user_info;
  char *nick_tmp;

  bs_function_enter();

  glist_entry = channel->members;
  while(glist_entry) {
    user_info = (UserInfo *) glist_entry->data;
    if ( !strcasecmp(user_info->nick, oldnick) ) {
      /* Found entry */
      channel->members = g_slist_remove(channel->members, user_info);
      nick_tmp = g_strdup(newnick);
      g_free(user_info->nick);
      user_info->nick = nick_tmp;
      channel->members = g_slist_insert_sorted(channel->members, user_info, 
					       (GCompareFunc) user_info_compare);
      /* Assume it's only in the list once */
      bs_function_leave();
      return(member_num);
    }
    glist_entry = g_slist_next(glist_entry);
    member_num++;
  }

  bs_function_leave();
  return(-1);
}

void set_member_status( ChannelInfo *channel, char *nick, unsigned status)
{
  int member_num=0;
  GSList *glist_entry;
  UserInfo *user_info;
  char *nick_tmp;

  bs_function_enter();

  glist_entry = channel->members;
  while(glist_entry) {
    user_info = (UserInfo *) glist_entry->data;
    if ( !strcasecmp(user_info->nick, nick) ) {
      /* Found entry */
      user_info->status = status;
      /* Need to remove and re-add it to keep the order right */
      channel->members = g_slist_remove(channel->members, user_info);
      nick_tmp = g_strdup(nick);
      g_free(user_info->nick);
      user_info->nick = nick_tmp;
      channel->members = g_slist_insert_sorted(channel->members, user_info, 
					       (GCompareFunc) user_info_compare);
      /* Assume it's only in the list once */
      bs_function_leave();
      return;
    }
    glist_entry = g_slist_next(glist_entry);
    member_num++;
  }
  bs_function_leave();
  return;
}

void set_members_list( ChannelInfo *channel )
{
  GSList *glist_entry;
  UserInfo *user_info;
  char *buff;
  int width=1;

  bs_function_enter();

  if (!channel) {
    bs_function_leave();
    return;
  }
  
  /* TODO: check that this free's all the strings */
  gtk_clist_clear( GTK_CLIST(channel->window->nick_clist) );

  glist_entry = channel->members;

  gtk_clist_freeze (GTK_CLIST(channel->window->nick_clist));
  while (glist_entry) {
    user_info = glist_entry->data;
    buff = g_malloc(strlen(user_info->nick)+2);
    if (user_info->status == USEROPER) {
      sprintf(buff, "@%s", user_info->nick);
    } else if (user_info->status == USERVOICE) {
      sprintf(buff, "+%s", user_info->nick);
    } else {
      strcpy(buff, user_info->nick);
    }
    width = MAX(width, gdk_string_width(channel->window->nick_clist->style->font, buff));
    gtk_clist_append (GTK_CLIST (channel->window->nick_clist), (gchar **) &buff);
    glist_entry = g_slist_next(glist_entry);
  }
  gtk_clist_set_column_width(GTK_CLIST(channel->window->nick_clist), 0, width);
  gtk_clist_thaw (GTK_CLIST(channel->window->nick_clist));

  bs_function_leave();
  return;
}

/* void clean_up(BezChannelWindow *window) */
/* { */
/*   ChannelInfo *target_channel; */
/*   GSList *connection_entry; */
/*   GSList *channel_entry; */
/*   Connection *connection; */

/*   bs_function_enter(); */

/*   gtk_clist_clear(window->ch_list); */
/*   gtk_clist_clear(window->nick_clist); */

/*   bs_function_leave(); */
/*   return; */
/* } */

/* Need to remember that the channel members list is ordered */
char *complete_nick(ChannelInfo *channel, char* incomplete_nick)
{
  GSList *glist_entry;
  UserInfo *user_info;
  int nick_len;
  char *candidate=NULL;

  bs_function_enter();

  if ( (nick_len = strlen(incomplete_nick)) == 0 ) {
    bs_function_leave();
    return(NULL);
  }

  glist_entry = channel->members;
  while(glist_entry) {
    user_info = (UserInfo *) glist_entry->data;

    /* If it is a complete match then return */
    if ( !strcasecmp(user_info->nick, incomplete_nick) ) {
      bs_function_leave();
      return(NULL);
    }
    if ( !strncasecmp(user_info->nick, incomplete_nick, nick_len) && !candidate) {
      candidate = user_info->nick;
/*       bs_function_leave(); */
/*       return(user_info->nick); */
    }
    glist_entry = g_slist_next(glist_entry);
  }

  bs_function_leave();
  return(candidate);
}
