static char rcsid[] = "$Id: spike_parms5.c,v 1.5 1997/07/18 03:02:36 dhb Exp $";

/*
** $Log: spike_parms5.c,v $
** Revision 1.5  1997/07/18 03:02:36  dhb
** Fix for getopt problem; getopt(), optopt and optind are now
** G_getopt(), G_optopt and G_optind.
**
** Revision 1.4  1994/06/16 00:52:17  dhb
** Changed -restV and -abs_exp option names in spikeparms command to
** -rest and -absexp respectively.
**
** Revision 1.3  1994/06/16  00:46:11  dhb
** Fixed error in initopt() string in do_spike_parms().
**
** Revision 1.2  1993/02/24  16:54:48  dhb
** 1.4 to 2.0 command argument changes.
**
** Revision 1.1  1992/12/11  19:06:10  dhb
** Initial revision
**
*/

/****************************************************************
**                                                             **
**                        SPIKE_PARMS                          **
**                                                             **
**          By Upinder S. Bhalla, Oct 1990. Caltech            **
**                                                             **
** This file contains routines for extracting a number of      **
** parameters of a spike that may be useful for comparing      **
** different spike traces. The objective is to get a set of    **
** such parameters which will enable one to evaluate the       **
** validity of a neuronal simulation and to constrain the      **
** parameters used in the simulation.                          **
**
** New functions : Get ISI, PTP, spike width, inflections on spike
** Make robust for even single spikes
** Make sensible windows for valleys and peaks - auto windows
** Get single spike parms if cant do interspike stuff
** Handle faster than capacitive recovery from spikes
** Use either exp fit or original file for recovery and thresh info
** Use better curve interpolator than splines, so that noisy data is ok
** Display diffile even if the interpolation did not work - graceful death
**                                                             **
****************************************************************/
#include <stdio.h>
#include <math.h>
#include "header.h"

#define MAXPTS 10000
#define MAX_NEXP 2000
#define MAXSPIKES 50
#define RES MAXSPIKES-1
#define DEV MAXSPIKES-2
#define MAXPARMS 500
#define ENDT 10.0
#define RESTING_POTL -0.07
#define DEFAULTRANGE 0.002	/* 2 mV is the defeault range for peaks */
#define DEFAULTMAX -200	
#define DEFAULTMIN 200
#define TOL 0.0001
#define INIT -1
#define HI 1
#define LO 2
#define LCLIFF 3
#define RCLIFF 4
#define LLEDGE 5
#define RLEDGE 6
#define ZERO 7
#define MAXRESULTS 200

#define ISI_RES	0
#define PTP_RES	1
#define SW_RES	2

#ifndef EPSILON
#define EPSILON 1e-10
#endif

typedef struct parm_struct {
	int		parmtype;
	int		index;
	float	t,v,dv,d2v;
} Parm;

typedef struct result_struct {
	int		type;
	int		n;			/* number of samples of this param */
	char	*name;
	float	value;
	float	sdev;
} Result;

float	findmax();
float	findmin();
float	findmaxslope();
float	findminslope();
float	finddiffmin();
float	crossval();
float	zbrent();
char	*gen_name();
float	x[MAXPTS],y[MAXPTS],d2y[MAXPTS];
char	*expin;
float	v0 = 0.0,tau0 = 1.0;
float	v1 = 0.0,tau1 = 1.0;
float	globt,globv,dt,delta;
float	startt,endt;
int		npts = 0;
Parm	parms[MAXPARMS];
int		nparms;
Result	results[MAXRESULTS];
int		nresults;
Parm	*sumparms;
Parm	*sumsqparms;
float	peakest,valleyest;
float	peakrange;
int		nfeatures;
int		n_in_class[10];
char	*expfile,*diffile;

char* do_spike_parms(argc,argv)
	int argc;
	char	**argv;
{
	int		peak[MAXSPIKES],valley[MAXSPIKES];
	int		npeaks;
	int		plotno = 0;
	float	temp;
	float	e0,e1;
	int		i,k;
	int		abs_exp_flag = 0;
	float	restV = RESTING_POTL;
	char	*outputfile,*filemode,*notes;
	int	status;


	FILE	*fp,*fopen();

	v0 = 0.0,tau0 = 1.0;
	v1 = 0.0,tau1 = 1.0;
	startt = 0.0;
	endt = ENDT;
	nparms = 0;
	peakest = DEFAULTMAX;
	valleyest = DEFAULTMIN;
	peakrange = DEFAULTRANGE;
	nresults = SW_RES + 1;
	nfeatures = 0;
	npts = 0;
	for (i = 0 ; i < MAXRESULTS ; i++)
		results[i].value = results[i].sdev = results[i].n = 0;

	for(i = 0 ; i < 10 ; i++)
		n_in_class[i] = 0;
	outputfile = NULL;
	filemode = NULL;
	notes = NULL;
	expin = NULL;
	expfile = NULL;
	diffile = NULL;

	initopt(argc, argv, "file-name -plot plot-number -start time -end time -peak voltage -valley voltage -peakrange voltage -rest voltage -file output-file-name mode -notes notes -expin file-name -exp0 v0 tau0 -exp1 v1 tau1 -absexp -expfile file-name voltage -diffile file-name");
	while ((status = G_getopt(argc, argv)) == 1)
	  {
	    if (strcmp(G_optopt,"-plot") == 0) {
		    plotno = atoi(optargv[1]);
	    }
	    if (strcmp(G_optopt,"-start") == 0) {
		    startt = Atof(optargv[1]);
	    }
	    if (strcmp(G_optopt,"-end") == 0) {
		    endt = Atof(optargv[1]);
	    }
	    if (strcmp(G_optopt,"-peak") == 0) {
		    peakest = Atof(optargv[1]);
	    }
	    if (strcmp(G_optopt,"-valley") == 0) {
		    valleyest = Atof(optargv[1]);
	    }
	    if (strcmp(G_optopt,"-peakrange") == 0) {
		    peakrange = Atof(optargv[1]);
	    }
	    if (strcmp(G_optopt,"-notes") == 0) {
		    notes = optargv[1];
	    }
	    if (strcmp(G_optopt,"-file") == 0) {
		    outputfile = optargv[1];
		    filemode = optargv[2];
	    }
	    if (strcmp(G_optopt,"-expin") == 0) {
		    expin = optargv[1];
	    }
	    if (strcmp(G_optopt,"-exp0") == 0) {
		    v0 = Atof(optargv[1]);
		    tau0 = Atof(optargv[2]);
	    }
	    if (strcmp(G_optopt,"-exp1") == 0) {
		    v1 = Atof(optargv[1]);
		    tau1 = Atof(optargv[2]);
	    }
	    if (strcmp(G_optopt,"-abs_exp") == 0) {
		    abs_exp_flag = 1;
	    }
	    if (strcmp(G_optopt,"-restV") == 0) {
		    restV = Atof(optargv[1]);
	    }
	    if (strcmp(G_optopt,"-expfile") == 0) {
		    expfile = optargv[1];
		    v0 = Atof(optargv[2]);
	    }
	    if (strcmp(G_optopt,"-diffile") == 0) {
		    diffile = optargv[1];
	    }
	  }

	if (status < 0) {
		printoptusage(argc, argv);
		printf("\n");
		printf("Valid file-modes : multifile onefile onefileappend\n");
		printf("The expin file or the precalculated exp terms are added to the interspike\n");
		printf("curves before analysis, if they are wanted.\n");
		printf("By default, v0 and v1 are scaled by ((v0+v1+RestV-MinV)/(v0+v1))\n");
		printf("In abs_exp mode the scaling is not done\n");
		printf("The output expfile contains the scaled exp decay curve.\n");
		printf("The diffile contains the difference plot between the spike and the exp decay.\n");
		return "failed";
	}

	if (!(smooth_plot(optargv[1],plotno)))	/* spline for now */
		return("failed");
	for(i = 1 ; i < (npts -2) ; i++)
		seek_features(i);	/* look for interesting points on curve */
	set_windows();	/* set windows for evaluating spikes */
	npeaks = match_spikes(); /* match spikes (if complete) and accumulate data
					** about interesting features */
	if (sum_exps())		/* Add on exp decay terms when relevant */
		exp_features();	/* seek-features for summed curve */
	eval_stats(npeaks);	/* Do statistical analysis of features found */
	write_output(outputfile,filemode,notes);

	return ""; /* success */
}


smooth_plot(filename,plotno)
	char	*filename;
	int		plotno;
{
	int	i;
	float 	t;
	if (!read_plot2(filename,plotno,&dt,y,MAXPTS,&npts,startt,endt)) {
		printf("Read plot failed\n");
		return(0);
	}
	delta = dt * 2.0;
	for (i = 0, t = startt ; i < npts ; i++, t+=dt)
		x[i] = t;
	endt = t - dt;
/*
** Precautionary measure to make sure I dont have any overflowing
** arrays
*/
	npts--;

/*
** Evaluate spline to get d2y/dx2 table
*/
	spline(x-1,y-1,npts,0.0,0.0,d2y);
	return(1);
}

seek_features(i)
	int i;
{
	float v0,v1,v2,v3;

	/* ensuring that we dont access invalid points on the graph */
	if (i < 1 || i > (npts - 3))
		return;
	
	v0 = y[i-1];
	v1 = y[i];
	v2 = y[i + 1];
	v3 = y[i + 2];

/* Zero crossing on rising phase */
	if (v0 < 0.0 && v1 >= 0.0) {
		parms[nparms].parmtype = ZERO;
		parms[nparms].index = i;
		if (splinroot(x,y,x[i]-delta,x[i]+delta,&(parms[nparms]),findmaxslope))
			nparms++;
		if (nparms >= MAXPARMS) return;
	}
/* Spike peak */
	if ( v0 < v1 && v2 <= v1) { 
		parms[nparms].parmtype = HI;
		parms[nparms].index = i;
		if (splinfit2(x,y,x[i]-delta,x[i]+delta,&(parms[nparms]),findmax))
		nparms++;
		if (nparms >= MAXPARMS) return;
	}

/* Spike trough */
	if ( v0 > v1 && v2 >= v1) { 
		parms[nparms].parmtype = LO;
		parms[nparms].index = i;
		if (splinfit2(x,y,x[i]-delta,x[i]+delta,&(parms[nparms]),findmin))
		nparms++;
		if (nparms >= MAXPARMS) return;
	}

/* Steepest rise point  */
	if ((v1-v0) < (v2-v1) && (v3-v2) < (v2-v1) && v0 < v3) {
		parms[nparms].parmtype = LCLIFF;
		parms[nparms].index = i;
		if (splinfit2(x,y,x[i]-delta,x[i]+delta,&(parms[nparms]),findmaxslope))
		nparms++;
		if (nparms >= MAXPARMS) return;
	}

/* Inflection on rising phase */
	if ((v1-v0) > (v2-v1) && (v3-v2) > (v2-v1) && v0 < v3) {
		parms[nparms].parmtype = LLEDGE;
		parms[nparms].index = i;
		if (splinfit2(x,y,x[i]-delta,x[i]+delta,&(parms[nparms]),findminslope))
		nparms++;
		if (nparms >= MAXPARMS) return;
	}

/* Steepest fall point  */
	if ((v1-v0) > (v2-v1) && (v3-v2) > (v2-v1) && v0 > v3) {
		parms[nparms].parmtype = RCLIFF;
		parms[nparms].index = i;
		if (splinfit2(x,y,x[i]-delta,x[i]+delta,&(parms[nparms]),findminslope))
		nparms++;
		if (nparms >= MAXPARMS) return;
	}

/* Inflection on falling phase */
	if ((v1-v0) < (v2-v1) && (v3-v2) < (v2-v1) && v0 > v3) {
		parms[nparms].parmtype = RLEDGE;
		parms[nparms].index = i;
		if (splinfit2(x,y,x[i]-delta,x[i]+delta,&(parms[nparms]),findmaxslope))
		nparms++;
		if (nparms >= MAXPARMS) return;
	}
}

set_windows()
{
int i;
/*
** Find the highest peak to set the upper window to
*/
	if (peakest == DEFAULTMAX) { /* peakest has not been set manually */
		for (i = 1; i < nparms ; i++) {
			if (parms[i].parmtype == HI) {
				if (peakest < parms[i].v)
					peakest = parms[i].v;
			}
		}
	}

/*
** Find the lowest valley to set the lower window to
*/
	if (valleyest == DEFAULTMIN) { /* valleyest has not been set manually */
		for (i = 1; i < nparms ; i++) {
			if (parms[i].parmtype == LO) {
				if (valleyest > parms[i].v)
					valleyest = parms[i].v;
			}
		}
	}
}

int match_spikes()
{
	int npeaks = 0;
	int i,j,k;
	int peaklist[MAXSPIKES];
	int	spike_with_most_features = 0;
	float	t0;
/*
** Get single spike parameters, and find peaks.
** This assumes that the entire spike, from thresh to peak to lowest
** trough, is present. Relevant params are
** Peak V, Trough V, PTP, trough time, Spike width (at 1/2 ht).
*/

	for(i = 0 ; i < nparms ; i++) {
		if (find_peak(peakest,valleyest,i)) {
			peaklist[npeaks] = i;
			npeaks++;
			if (npeaks >= MAXSPIKES) {
				printf("Stopping analysis at %d spikes. Should be plenty.\n",npeaks);
				break;
			}
		}
	}

/*
** Make list of results by scanning from first spike to second, or
** end of parm list.
*/
	if (npeaks == 0) {
		printf ("No peaks found within specified windows.\n");
		return(0);
	}
	printf("%d peaks were found\n",npeaks);

	if (npeaks == 1) {	/* We only have one peak, so just copy every
						** param into the stats list */
		sumparms = parms;
		sumsqparms = (Parm *) calloc(nparms,sizeof(Parm));
		for(i = 0; i < nparms ; i++) {
			sumsqparms[i].v += parms[i].v * parms[i].v;
			sumsqparms[i].t += parms[i].t * parms[i].t;
			sumsqparms[i].dv += parms[i].dv * parms[i].dv;
			sumsqparms[i].d2v += parms[i].d2v * parms[i].d2v;
		}
	} else if (npeaks > 1) {
/*
** Find the inter-spike interval : another special field.
*/
		for(i = 1; i < npeaks ; i++) {
			t0 = parms[peaklist[i]].t - parms[peaklist[i - 1]].t;
			results[ISI_RES].value += t0;
			results[ISI_RES].sdev += t0 * t0;
			results[ISI_RES].name = "ISI";
			results[ISI_RES].n += 1;
		}
/* We have a number of peaks. So we need
** to assign corresponding parms from 
** different peaks to each other */
		/* Find out the max number of features in a peak */
		nfeatures = 0;
		for (i = 1 ; i < npeaks; i++) {
			if (nfeatures < (peaklist[i] - peaklist[i-1])) {
				nfeatures = peaklist[i] - peaklist[i-1];
				spike_with_most_features = i - 1;
			}
		}
		if ((nfeatures - peaklist[1] - peaklist[0]) > 2)
		/* A lot of unmatched features */
			printf("Warning : the spike waveform appears to be highly variable. \n");
		/* allocate the feature averaging arrays */
		sumparms = (Parm *) calloc(nfeatures,sizeof(Parm));
		sumsqparms = (Parm *) calloc(nfeatures,sizeof(Parm));
		/* initialize sumparms. Need the type of feature to use later
		** to check for matching */
		t0 = parms[peaklist[spike_with_most_features]].t;
		for(i = 0, j = peaklist[spike_with_most_features];
			i < nfeatures ; i++,j++) {
			parms[j].t -= t0;
			sumparms[i] = parms[j];
			sumparms[i].t = parms[j].t;
			sumparms[i].index = 1;
			/* We use the index field of sumparms to count the number
			** of entries to be averaged over */
			sumsqparms[i].v = parms[j].v * parms[j].v;
			sumsqparms[i].t = parms[j].t * parms[j].t;
			sumsqparms[i].dv = parms[j].dv * parms[j].dv;
			sumsqparms[i].d2v = parms[j].d2v * parms[j].d2v;
		}
		/* match features for summing parms */
		for (k = 0 ; k < npeaks ; k++) {
			t0 = parms[peaklist[k]].t;
			for(i=0,j=peaklist[k];i<nfeatures;i++,j++) {
				if (k == spike_with_most_features) /* We have already done this */
					continue;
				if (parms[j].parmtype == sumparms[i].parmtype) {
				/* successful match */
					parms[j].t -= t0;
					sumparms[i].v += parms[j].v;
					sumparms[i].t += parms[j].t;
					sumparms[i].dv += parms[j].dv;
					sumparms[i].d2v += parms[j].d2v;
					sumparms[i].index += 1;
					sumsqparms[i].v += parms[j].v * parms[j].v;
					sumsqparms[i].t += parms[j].t * parms[j].t;
					sumsqparms[i].dv += parms[j].dv * parms[j].dv;
					sumsqparms[i].d2v += parms[j].d2v * parms[j].d2v;
				} else {
				/* Skip over this feature and try the next */
					j--;
				}
			}
		}
	}
	return(npeaks);
}

/*
** find_peak : This returns 1 when it finds a peak, and it also
** finds a few of the parameters associated with a peak.
*/
int find_peak(peakest,valleyest,thisparm)
	float	peakest,valleyest;
	int		thisparm;
{
	int i;
	int erflag = 0;
	float t1,t2;
	float temp;
	/* Look for peak */
	if (parms[thisparm].parmtype != HI) 
		return(0);
	/* Ensure it is within window */
	if ((peakest - peakrange) > parms[thisparm].v ||
		(peakest + peakrange) < parms[thisparm].v) 
		return(0);
	/* Look for valley */

	for (i = thisparm + 1; i < nparms ; i++) {
		/* Ensure that we havent gotten to another peak without finding
		** this valley */
		if (parms[i].parmtype == HI && (peakest - peakrange) <
			parms[i].v && (peakest + peakrange) > parms[i].v ) {
			printf("Hit next peak before finding valley : check valley window\n");
			return(0);
		}
		/* condition for finding valley */
		if (parms[i].parmtype == LO && (valleyest - peakrange) <
			parms[i].v && (valleyest + peakrange) > parms[i].v ) {
			break;
		}
	}
	if (i >= nparms) { /* we ran out of parms before the next valley */
		printf("Could not find any valleys : check window\n");
		return(0);
	}
	temp = parms[thisparm].v - parms[i].v;
	results[PTP_RES].value += temp;
	results[PTP_RES].sdev += temp * temp;
	results[PTP_RES].name = "PTP_V";
	results[PTP_RES].n += 1;

	/* Setting globv = half-ht of spike */
	globv = (parms[thisparm].v + parms[i].v)/2.0;
	t1 = zbrent(crossval,parms[thisparm].t - 2 * delta,
		parms[thisparm].t,TOL,&erflag);
	if (erflag)
		printf("Could not find spike width 1\n");
	t2 = zbrent(crossval,parms[thisparm].t,
		parms[thisparm].t + 2 * delta,TOL,&erflag);
	if (erflag)
		printf("Could not find spike width 2\n");
	temp = t2 - t1;
	results[SW_RES].value += temp;
	results[SW_RES].sdev += temp * temp;
	results[SW_RES].name = "SpikeWidth";
	results[SW_RES].n += 1;
	nresults = SW_RES + 1;
	return(1);
}

int sum_exps(restV)
	float restV;
{
	static float exptable[MAX_NEXP];
	int		nexptable;
	int		i,j,k;
	int		loindex = 0, hiindex = 0;
	float	xv;
	float	temp,mint,minv;
	FILE	*fexp,*fdiff,*fopen();

	if (expin && strlen(expin) > 0) {
		i =  read_plot2(expin,0,&temp,exptable,MAX_NEXP,&j,0.0,endt);
		for(j = 0 ; j < i ; k++,j++) {
			exptable[j] -= restV;
		}
		if (temp != dt) {
			printf("Error : dt of exponential decay is different from that of plot\n");
			return(0);
		}
	} else if (fabs(v0) > EPSILON) {
		for(j = 0 ; j < MAX_NEXP ; k++,j++) {
			xv = x[j];
			exptable[j] = v0*(1.0-exp((-xv)/tau0)) +
				v1*(1.0-exp((-xv)/tau1));
		}
	} else {
		return(0);
	}
	if (expfile)
		fexp = fopen(expfile,"w");
	else 
		fexp = NULL;
	if (diffile)
		fdiff = fopen(diffile,"w");
	else
		fdiff = NULL;

	for (i = 0 ; i < nparms ; i++) {
		loindex = troughindex(i);
		if (loindex == 0)
			break;
		mint = parms[loindex].t;
		minv = parms[loindex].v;
		hiindex = peakindex(i + 1);	
		if (hiindex) {
			nexptable = (hiindex  < (MAX_NEXP + loindex)) ?
				hiindex - loindex : MAX_NEXP + loindex;
			i = hiindex;
		} else {
			nexptable = (MAXPTS  < (MAX_NEXP + loindex)) ?
				MAXPTS - loindex : MAX_NEXP + loindex;
			i = nparms;
		}
		if (expin && strlen(expin) > 0) {
			temp = (v0 + restV - minv)/v0;
		} else if (fabs(v0) > EPSILON) {
			temp = (v0 + v1 + restV - minv)/(v0 + v1);
		}
		for(k = loindex,j = 0 ; j < nexptable ; k++,j++) {
			y[k] += exptable[j] * temp;
		}
		if (fexp) {
			for(k = loindex,j = 0 ; j < nexptable ; k++,j++) {
				fprintf(fexp,"%g %g\n",x[k],exptable[j]);
			}
		}
		if (fdiff) {
			for(k = loindex,j = 0 ; j < nexptable ; k++,j++) {
				fprintf(fdiff,"%g %g\n",x[k],y[k]);
			}
		}
	}
	if (fexp) 
		fclose(fexp);
	if (fdiff) 
		fclose(fdiff);
	return(1);
}

troughindex(thisparm)
	int thisparm;
{
	int i;

	for (i = thisparm ; i < nparms ; i++) {
		if (parms[i].parmtype == LO) {
			return(parms[i].index);
		}
	}
	return(0);
}

peakindex(thisparm)
	int thisparm;
{
	int i;

	for (i = thisparm ; i < nparms ; i++) {
		if (parms[i].parmtype == HI) {
			return(parms[i].index);
		}
	}
	return(0);
}

exp_features()
{
	int i,k;
	int	loindex,hiindex;

	for (i = 0 ; i < nparms ; i++) {
		loindex = troughindex(i);
		if (loindex == 0)
			break;
		hiindex = peakindex(i + 1);	
		for(k = loindex + 1 ; k < (hiindex - 2) ; k++) {
			seek_features(k);
		}
	}
}

eval_stats(npeaks)
	int npeaks;
{
	int i,j;
	float	fpeaks;
	float	val;

	/* initialising the name list */
	(void)gen_name(INIT,"INIT");
	for (i = 0; i < nfeatures ; i++) {
		if (nresults >= MAXRESULTS) break;
		results[nresults].value = sumparms[i].v;
		results[nresults].sdev = sumsqparms[i].v;
		results[nresults].name = gen_name(sumparms[i].parmtype,"V");
		results[nresults].n = sumparms[i].index;
		nresults++;

		if (nresults >= MAXRESULTS) break;
		results[nresults].value = sumparms[i].t;
		results[nresults].sdev = sumsqparms[i].t;
		results[nresults].name = gen_name(sumparms[i].parmtype,"T");
		results[nresults].n = sumparms[i].index;
		nresults++;

		if (nresults >= MAXRESULTS) break;
		results[nresults].value = sumparms[i].dv;
		results[nresults].sdev = sumsqparms[i].dv;
		results[nresults].name = gen_name(sumparms[i].parmtype,"DV");
		results[nresults].n = sumparms[i].index;
		nresults++;

		if (nresults >= MAXRESULTS) break;
		results[nresults].value = sumparms[i].d2v;
		results[nresults].sdev = sumsqparms[i].d2v;
		results[nresults].name = gen_name(sumparms[i].parmtype,"D2V");
		results[nresults].n = sumparms[i].index;
		nresults++;
		
	}
	if (nresults >= MAXRESULTS){
		printf("Too many results : possibly a bad waveform\n");
	}
	if (npeaks == 0)
		return(0);
	if (npeaks == 1) {
		for(j = 0 ; j < nresults ; j++)
			results[j].sdev = 0.0;
	}
	if (npeaks > 1) {
		for(j = 0; j < nresults ; j++) {
			fpeaks = results[j].n;
			val = results[j].value;
			if (fpeaks > 1.5 && (results[j].sdev > val*val/fpeaks))
				results[j].sdev = sqrt((results[j].sdev -
					val * val / fpeaks) / (fpeaks - 1.0));
			else
				results[j].sdev = 0.0;
			results[j].value = val / fpeaks;
		}
	}
}

char *
gen_name(parmtype,suffix)
	int	parmtype;
	char	*suffix;
{
	static char temp[40];
	char	*str;
	int parmno;
	static int	last_parmtype;

	/* Initialisation condition */
	if (parmtype == INIT) {
		last_parmtype = INIT;
		return(NULL);
	}
	/* incrementing the number of items in specified class */
	if (parmtype != last_parmtype) {
		n_in_class[parmtype] += 1;
		last_parmtype = parmtype;
	}
	parmno = n_in_class[parmtype];
	switch(parmtype) {
		case HI : sprintf(temp,"HI%d_%s",parmno,suffix);
			break;
		case LO :	sprintf(temp,"LO%d_%s",parmno,suffix);
			break;
		case LLEDGE :	sprintf(temp,"LL%d_%s",parmno,suffix);
			break;
		case RLEDGE :	sprintf(temp,"RL%d_%s",parmno,suffix);
			break;
		case LCLIFF :	sprintf(temp,"LC%d_%s",parmno,suffix);
			break;
		case RCLIFF :	sprintf(temp,"RC%d_%s",parmno,suffix);
			break;
		case ZERO :	sprintf(temp,"ZERO%d_%s",parmno,suffix);
			break;
		default :
			break;
	}
	str = (char *) malloc((strlen(temp) + 1) * sizeof(char));
	strcpy(str,temp);
	return(str);
}

write_output(filename,filemode,notes)
	char	*filename;
	char	*filemode;
	char	*notes;
{
	int i;
	static char	parmfile[200];
	FILE	*fp,*fopen();

	if (strcmp(filemode,"multifile") == 0) {
		if (!filename) {
			printf("No name specified for spike analysis file\n");
			return(0);
		}
			
		for (i = 0 ; i < nresults ; i++) {
			sprintf(parmfile,"%s.%s",filename,results[i].name);
			fp = fopen(parmfile,"a");
			if (notes)
				fprintf(fp,"%s	%g	%g\n",
					notes,results[i].value,results[i].sdev);
			else
				fprintf(fp,"%g	%g\n",
					results[i].value,results[i].sdev);
			fclose(fp);
		}
	}
	if (strcmp(filemode,"onefile") == 0) {
		fp = fopen(filename,"w");
		if (!fp) fp = stdout;
		if (notes)
			fprintf(fp,"\n%s\n",notes);
		for (i = 0 ; i < nresults ; i++) {
			fprintf(fp,"%-12s	%-12g	+/-	%-12g	(%1.4f %%)\n",
				results[i].name,results[i].value, results[i].sdev,
				results[i].sdev * 100.0/ results[i].value);
		}
		fprintf(fp,"\n");
		if (fp != stdout) fclose(fp);
	}
	if (strcmp(filemode,"onefileappend") == 0) {
		if (!filename) {
			printf("No name specified for spike analysis file\n");
			return(0);
		}
		fp = fopen(filename,"a");
		if (notes)
			fprintf(fp,"\n%s\n",notes);
		for (i = 0 ; i < nresults ; i++) {
			fprintf(fp,"%s	%g	",results[i].name,results[i].value);
			if ((i%3) == 2)
				fprintf(fp,"\n");
		}	
		fprintf(fp,"\n");
		fclose(fp);
	}
}


/*
** Upisplint is a version of splint which returns the first and
** second derivatives of the splined function as well. It is
** effectively doing linear interpolation in d2y, and the rest follows.
*/
int upisplint(xa,ya,y2a,n,x,y,dy,d2y)
float xa[],ya[],y2a[],x,*y,*dy,*d2y;
int n;
{
	int klo,khi,k;
	float h,b,a;
	float y2lo,y2hi;
	void nrerror();

	klo=1;
	khi=n;
	while (khi-klo > 1) {
		k=(khi+klo) >> 1;
		if (xa[k] > x) khi=k;
		else klo=k;
	}
	h=xa[khi]-xa[klo];
	if (h == 0.0) {
		nrerror("Bad XA input to routine SPLINT");
		return(0);
	}
	a=(xa[khi]-x)/h;
	b=(x-xa[klo])/h;
	y2lo=y2a[klo];
	y2hi=y2a[khi];
	*y=a*ya[klo]+b*ya[khi]+((a*a*a-a)*y2lo+(b*b*b-b)*y2hi)*(h*h)/6.0;
	*dy=(ya[khi]-ya[klo])/h-((3.0*a*a-1.0)*y2lo-(3.0*b*b-1.0)*y2hi)*h/6.0;
	*d2y=a*y2lo+b*y2hi;
	return(1);
}

float findmax(xv)
	float xv;
{
	float yv,dyv,d2yv;

	if (upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv) == 0)
		return(0.0);
	return(-yv);
}

float findmin(xv)
	float xv;
{
	float yv,dyv,d2yv;

	if (upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv) == 0)
		return(0.0);
	return(yv);
}

float findmaxslope(xv)
	float xv;
{
	float yv,dyv,d2yv;

	if (upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv) == 0)
		return(0.0);
	return(-dyv);
}

float findminslope(xv)
	float xv;
{
	float yv,dyv,d2yv;

	if (upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv) == 0)
		return(0.0);
	return(dyv);
}

float finddiffmin(xv)
	float xv;
{
	float yv,dyv,d2yv;

	if (upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv) == 0)
		return(0.0);
	yv -= globv + v0*(1.0-exp((globt-xv)/tau0)) +
		v1*(1.0-exp((globt-xv)/tau1));
			
	return(yv);
}

float crossval(xv)
	float xv;
{
	float yv,dyv,d2yv;

	if (upisplint(x-1,y-1,d2y-1,npts,xv,&yv,&dyv,&d2yv) == 0)
		return(0.0);
	yv -= globv;
			
	return(yv);
}

int splinfit2(x,y,lo,hi,ret,func)
	float	x[],y[],(*func)();
	Parm	*ret;
	float	lo,hi;
{
	float	t,v,dv,d2v;
	float	yv;
	int		erflag = 0;

	/* find the peak */
	yv = brent(lo,(hi + lo)/2.0,hi,func,TOL,&t,&erflag);
	if (erflag)
		return(0);
	if (upisplint(x-1,y-1,d2y-1,npts,t,&v,&dv,&d2v) == 0)
		return(0);
	ret->t = t;
	ret->v = v;
	ret->dv = dv;
	ret->d2v = d2v;
	ret->index = (int)(0.5+(t-startt)/dt);
	return(1);
}

int splinroot(x,y,lo,hi,ret,func)
	float	x[],y[],(*func)();
	Parm	*ret;
	float	lo,hi;
{
	float	t,v,dv,d2v;
	int		erflag = 0;

	t = zbrent(func,lo,hi,TOL,&erflag);
	if (erflag)
		return(0);
	if (upisplint(x-1,y-1,d2y-1,npts,t,&v,&dv,&d2v) == 0)
		return(0.0);
	ret->t = t;
	ret->v = v;
	ret->dv = dv;
	ret->d2v = d2v;
	ret->index = (int)(0.5+(t-startt)/dt);
	return(1);
}
