/*
This is part of the audio CD player library
Copyright (C)1998-99 Tony Arcieri

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.
*/

#include <config.h>

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <cdaudio.h>
#include <errno.h>
#include <unistd.h>
#include <netinet/in.h>

/* We can check to see if the CD-ROM is mounted if this is available */
#ifdef HAVE_MNTENT_H
#include <mntent.h>
#endif

#ifdef HAVE_SYS_MNTENT_H
#include <sys/mntent.h>
#endif

/* For Linux */
#ifdef HAVE_LINUX_CDROM_H
#include <linux/cdrom.h>
#define NON_BLOCKING
#endif

#ifdef HAVE_LINUX_UCDROM_H
#include <linux/ucdrom.h>
#endif

/* For FreeBSD, OpenBSD, and Solaris */
#ifdef HAVE_SYS_CDIO_H
#include <sys/cdio.h>
#endif

/* For Digital UNIX */
#ifdef HAVE_IO_CAM_CDROM_H
#include <io/cam/cdrom.h>
#endif

#ifdef CDIOCREADSUBCHANNEL
#define CDIOREADSUBCHANNEL CDIOCREADSUBCHANNEL
#endif


#ifndef IRIX_CDAUDIO
/*
Because of Irix's different interface, most of this program is
completely ignored when compiling under Irix.
*/

/* Return the version of libcdaudio */
void
#if __STDC__
cd_version(char *buffer, int len)
#else
cd_version(buffer, len)
   char *buffer;
   int len;
#endif
{
   snprintf(buffer, len, "%s %s", PACKAGE, VERSION);
}

/* Initialize the CD-ROM for playing audio CDs */
int
#if __STDC__
cd_init_device(char *device_name)
#else
cd_init_device(device_name)
   char *device_name
#endif
{
   int cd_desc;

#ifdef HAVE_GETMNTENT
   FILE *mounts;
   struct mntent *mnt;
   char devname[256];
   struct stat st;

   if(lstat(device_name, &st) < 0)
     return -1;
   
   if(S_ISLNK(st.st_mode))
     readlink(device_name, devname, 256);
   else
     strncpy(devname, device_name, 256);

   if((mounts = setmntent(MOUNTED, "r")) == NULL)
      return -1;
      
   while((mnt = getmntent(mounts)) != NULL) {
      if(strcmp(mnt->mnt_fsname, devname) == 0) {
	 endmntent(mounts);
	 errno = EBUSY;
	 return -1;
      }
   }
   endmntent(mounts);
#endif

#ifdef NON_BLOCKING
   if((cd_desc = open(device_name, O_RDONLY | O_NONBLOCK)) < 0)
#else
   if((cd_desc = open(device_name, O_RDONLY)) < 0)
#endif
     return -1;
	
   return cd_desc;
}

/* Close a device handle and free its resources. */
int
#if __STDC__
cd_finish(int cd_desc)
#else
cd_finish(cd_desc)
   int cd_desc;
#endif
{
   close(cd_desc);
   return 0;
}

/* Update a CD status structure... because operating system interfaces vary
   so does this function. */
int
#if __STDC__
cd_stat(int cd_desc, struct disc_info *disc)
#else
cd_stat(cd_desc, disc)
   int cd_desc;
   struct disc_info *disc;
#endif
{
   /* Since every platform does this a little bit differently this gets pretty
      complicated... */
#ifdef CDIOREADTOCHEADER
   struct ioc_toc_header cdth;
#endif
#ifdef CDIOREADTOCENTRYS
   struct cd_toc_entry toc_buffer[MAX_TRACKS];
   struct ioc_read_toc_entry cdte;
#endif
#ifdef CDROMREADTOCHDR
   struct cdrom_tochdr cdth;
#endif
#ifdef CDROMREADTOCENTRY
   struct cdrom_tocentry cdte;
#endif
#ifdef CDROM_TOC_HEADER
   struct cd_toc_header th;
#endif
#ifdef CDROM_TOC_ENTRYS
   struct cd_toc_entry te[MAX_TRACKS];
   struct cd_toc toc;
#endif
   struct disc_status status;
   int readtracks, pos;
   
   if(cd_poll(cd_desc, &status) < 0)
     return -1;
 
   if(!status.status_present) {
      disc->disc_present = 0;
      return 0;
   }
   
   /* Read the Table Of Contents header */
#ifdef CDIOREADTOCHEADER
   if(ioctl(cd_desc, CDIOREADTOCHEADER, &cdth) < 0)
     return -1;
   
   disc->disc_first_track = cdth.starting_track;
   disc->disc_total_tracks = cdth.ending_track;
#endif
#ifdef CDROMREADTOCHDR
   if(ioctl(cd_desc, CDROMREADTOCHDR, &cdth) < 0)
     return -1;
   
   disc->disc_first_track = cdth.cdth_trk0;
   disc->disc_total_tracks = cdth.cdth_trk1;
#endif
#ifdef CDROM_TOC_HEADER
   if(ioctl(cd_desc, CDROM_TOC_HEADER, &th) < 0)
     return -1;
   
   disc->disc_first_track = th.th_starting_track;
   disc->disc_total_tracks = th.th_ending_track;
#endif
   
   /* Read the Table Of Contents */
#ifdef CDIOREADTOCENTRYS
   cdte.address_format = CD_MSF_FORMAT;
   cdte.starting_track = 0;
   cdte.data = toc_buffer;
   cdte.data_len = sizeof(toc_buffer);
   
   if(ioctl(cd_desc, CDIOREADTOCENTRYS, &cdte) < 0)
     return -1;
   
   for(readtracks = 0; readtracks <= disc->disc_total_tracks; readtracks++) {
      disc->disc_track[readtracks].track_pos.minutes = cdte.data[readtracks].addr.msf.minute;
      disc->disc_track[readtracks].track_pos.seconds = cdte.data[readtracks].addr.msf.second;
      disc->disc_track[readtracks].track_pos.frames = cdte.data[readtracks].addr.msf.frame;
      disc->disc_track[readtracks].track_type = CDAUDIO_TRACK_AUDIO;
   }
   
   cdte.address_format = CD_LBA_FORMAT;
   
   if(ioctl(cd_desc, CDIOREADTOCENTRYS, &cdte) < 0)
     return -1;
   
   for(readtracks = 0; readtracks < disc->disc_total_tracks; readtracks++)
     disc->disc_track[readtracks].track_lba = ntohl(cdte.data[readtracks].addr.lba);
   disc->disc_track[disc->disc_total_tracks].track_lba = ntohl(cdte.data[disc->disc_total_tracks].addr.lba) + 1;
#endif
#ifdef CDROMREADTOCENTRY
   for(readtracks = 0; readtracks <= disc->disc_total_tracks; readtracks++) {
      cdte.cdte_track = (readtracks == disc->disc_total_tracks) ? CDROM_LEADOUT : readtracks + 1;
      cdte.cdte_format = CDROM_MSF;
      if(ioctl(cd_desc, CDROMREADTOCENTRY, &cdte) < 0)
	return -1;
	 
      disc->disc_track[readtracks].track_pos.minutes = cdte.cdte_addr.msf.minute;
      disc->disc_track[readtracks].track_pos.seconds = cdte.cdte_addr.msf.second;
      disc->disc_track[readtracks].track_pos.frames = cdte.cdte_addr.msf.frame;
      disc->disc_track[readtracks].track_type = (cdte.cdte_ctrl & CDROM_DATA_TRACK) ? CDAUDIO_TRACK_DATA : CDAUDIO_TRACK_AUDIO;
      
      cdte.cdte_format = CDROM_LBA;
      if(ioctl(cd_desc, CDROMREADTOCENTRY, &cdte) < 0)
	return -1;
      
      disc->disc_track[readtracks].track_lba = cdte.cdte_addr.lba;
   }
#endif
#ifdef CDROM_TOC_ENTRYS
   toc.toc_address_format = CDROM_MSF_FORMAT;
   toc.toc_starting_track = 0;
   toc.toc_buffer = te;
   toc.toc_alloc_length = sizeof(te);
   
   if(ioctl(cd_desc, CDROM_ROC_ENTRYS, &toc) < 0)
     return -1;
   
   for(readtracks = 0; readtracks <= disc->disc_total_tracks; readtracks++) {
      disc->disc_track[readtracks].track_pos.minutes = te.te_absaddr.m_units;
      disc->disc_track[readtracks].track_pos.seconds = te.te_absaddr.s_units;
      disc->disc_track[readtracks].track_pos.frames = te.te_absaddr.f_units;
      disc->disc_track[readtracks].track_type = (te.te_control & CDROM_DATA_TRACK) ? CDAUDIO_TRACK_DATA : CDAUDIO_TRACK_AUDIO;
      disc->disc_track[readtracks].track_lba = disc->disc_track[readtracks].track_pos.minutes * 4500 + disc->disc_track[readtracks].track_pos.seconds * 75 + disc->disc_track[readtracks].track_pos.frames - 150;
   }
#endif   
   for(readtracks = 0; readtracks <= disc->disc_total_tracks; readtracks++) {
      if(readtracks > 0) {
	 pos = ((disc->disc_track[readtracks].track_pos.minutes * 60 + disc->disc_track[readtracks].track_pos.seconds) * 75 + disc->disc_track[readtracks].track_pos.frames) - ((disc->disc_track[readtracks - 1].track_pos.minutes * 60 + disc->disc_track[readtracks - 1].track_pos.seconds) * 75 + disc->disc_track[readtracks - 1].track_pos.frames);
	 disc->disc_track[readtracks - 1].track_length.minutes = pos / 4500;
	 disc->disc_track[readtracks - 1].track_length.seconds = (pos % 4500) / 75;
	 disc->disc_track[readtracks - 1].track_length.frames = pos % 75;
      }
   }
            
   disc->disc_length.minutes = disc->disc_track[disc->disc_total_tracks].track_pos.minutes;
   disc->disc_length.seconds = disc->disc_track[disc->disc_total_tracks].track_pos.seconds;
   disc->disc_length.frames = disc->disc_track[disc->disc_total_tracks].track_pos.frames;
   
   cd_update(disc, status);
 
   return 0;
}

int
#if __STDC__
cd_poll(int cd_desc, struct disc_status *status)
#else
cd_poll(cd_desc, status)
   int cd_desc;
   struct disc_status *status
#endif
{
#ifdef CDIOREADSUBCHANNEL
   struct ioc_read_subchannel cdsc;
   struct cd_sub_channel_info data;
#endif
#ifdef CDROMSUBCHNL
   struct cdrom_subchnl cdsc;
#endif
#ifdef CDROM_READ_SUBCHANNEL
   struct cd_sub_channel sch;
   struct cd_subc_channel_data scd;
#endif
   
#ifdef CDIOREADSUBCHANNEL
   memset(&cdsc, '\0', sizeof(cdsc));
   cdsc.data = &data;
   cdsc.data_len = sizeof(data);
   cdsc.data_format = CD_CURRENT_POSITION;
   cdsc.address_format = CD_MSF_FORMAT;
   
   if(ioctl(cd_desc, CDIOREADSUBCHANNEL, (char *)&cdsc) < 0)
#endif
#ifdef CDROMSUBCHNL
   cdsc.cdsc_format = CDROM_MSF;
   
   if(ioctl(cd_desc, CDROMSUBCHNL, &cdsc) < 0)
#endif
#ifdef CDROM_READ_SUBCHANNEL
   memset(scd, '\0', sizeof(struct cd_subc_channel_data));
   sch.sch_address_format = CDROM_MSF_FORMAT;
   sch.sch_data_format = CDROM_SUBQ_DATA;
   sch.sch_buffer = &scd;
   sch.sch_alloc_length = sizeof(struct cd_subc_channel_data);
   
   if(ioctl(cd_desc, CDROM_READ_SUBCHANNEL, &sch) < 0)
#endif
     {
	status->status_present = 0;
	
	return 0;
     }
   
   status->status_present = 1;
   
#ifdef CDIOREADSUBCHANNEL
   status->status_disc_time.minutes = data.what.position.absaddr.msf.minute;
   status->status_disc_time.seconds = data.what.position.absaddr.msf.second;
   status->status_disc_time.frames = data.what.position.absaddr.msf.frame;
   
   switch(data.header.audio_status) {
    case CD_AS_PLAY_IN_PROGRESS:
      status->status_mode = CDAUDIO_PLAYING;
      break;
    case CD_AS_PLAY_PAUSED:
      status->status_mode = CDAUDIO_PAUSED;
      break;
    case CD_AS_PLAY_COMPLETED:
      status->status_mode = CDAUDIO_COMPLETED;
      break;
    default:
      status->status_mode = CDAUDIO_NOSTATUS;
   }
#endif
#ifdef CDROMSUBCHNL
   status->status_disc_time.minutes = cdsc.cdsc_absaddr.msf.minute;
   status->status_disc_time.seconds = cdsc.cdsc_absaddr.msf.second;
   status->status_disc_time.frames = cdsc.cdsc_absaddr.msf.frame;
   
   switch(cdsc.cdsc_audiostatus) {
    case CDROM_AUDIO_PLAY:
      status->status_mode = CDAUDIO_PLAYING;
      break;
    case CDROM_AUDIO_PAUSED:
      status->status_mode = CDAUDIO_PAUSED;
      break;
    case CDROM_AUDIO_COMPLETED:
      status->status_mode = CDAUDIO_COMPLETED;
      break;
    default:
      status->status_mode = CDAUDIO_NOSTATUS;
   }
#endif
#ifdef CDROM_READ_SUBCHANNEL
   status->status_disc_time.minutes = scd.scd_position_data.scp_absaddr.m_units;
   status->status_disc_time.seconds = scd.scd_position_data.scp_absaddr.s_units;
   status->status_disc_time.frames = scd.scd_position_data.scp_absaddr.f_units;
   
   switch(scd.scd_header.sh_audio_status) {
    case AS_PLAY_IN_PROGRESS:
      status->status_mode = CDAUDIO_PLAYING;
      break;
    case AS_PLAY_PAUSED:
      status->status_mode = CDAUDIO_PAUSED;
      break;
    case AS_PLAY_COMPLETED:
      status->status_mode = CDAUDIO_COMPLETED;
      break;
    default:
      status->status_mode = CDAUDIO_NOSTATUS;
      break;
   }
#endif
   
   return 0;
}

/* Play frames from CD */
int
#if __STDC__
cd_play_frames(int cd_desc, int startframe, int endframe)
#else
cd_play_frames(cd_desc, startframe, endframe)
   int cd_desc;
   int startframe;
   int endframe;
#endif
{
#ifdef CDIOCPLAYMSF
   struct ioc_play_msf cdmsf;
#endif
#ifdef CDROMPLAYMSF
   struct cdrom_msf cdmsf;
#endif
#ifdef CDROM_PLAY_AUDIO_MSF
   struct cd_play_audio_msf msf;
#endif

#ifdef CDIOCPLAYMSF
   cdmsf.start_m = startframe / 4500;
   cdmsf.start_s = (startframe % 4500) / 75;
   cdmsf.start_f = startframe % 75;
   cdmsf.end_m = endframe / 4500;
   cdmsf.end_s = (endframe % 4500) / 75;
   cdmsf.end_f = endframe % 75;
#endif
#ifdef CDROMPLAYMSF
   cdmsf.cdmsf_min0 = startframe / 4500;
   cdmsf.cdmsf_sec0 = (startframe % 4500) / 75;
   cdmsf.cdmsf_frame0 = startframe % 75;
   cdmsf.cdmsf_min1 = endframe / 4500;
   cdmsf.cdmsf_sec1 = (endframe % 4500) / 75;
   cdmsf.cdmsf_frame1 = endframe % 75;
#endif
#ifdef CDROM_PLAY_AUDIO_MSF
   msf.msf_starting_M_unit = startframe / 4500;
   msf.msf_starting_S_unit = (startframe % 4500) / 75;
   msf.msf_starting_F_unit = startframe % 75;
   msf.msf_ending_M_unit = endframe / 4500;
   msf.msf_ending_S_unit = (endframe % 4500) / 75;
   msf.msf_ending_F_unit = endframe % 75;
#endif
   
#ifdef CDIOCSTART
   if(ioctl(cd_desc, CDIOCSTART) < 0)
     return -1;
#endif
#ifdef CDROMSTART
   if(ioctl(cd_desc, CDROMSTART) < 0)
     return -1;
#endif
   
#ifdef CDIOCPLAYMSF
   if(ioctl(cd_desc, CDIOCPLAYMSF, (char *)&cdmsf) < 0)
     return -1;
#endif
#ifdef CDROMPLAYMSF
   if(ioctl(cd_desc, CDROMPLAYMSF, &cdmsf) < 0)
     return -1;
#endif
#ifdef CDROM_PLAY_AUDIO_MSF
   if(ioctl(cd_desc, CDROM_PLAY_AUDIO_MSF, &msf) < 0)
     return -1;
#endif
   
   return 0;
}

/* Play starttrack at position pos to endtrack */
int
#if __STDC__
cd_play_track_pos(int cd_desc, int starttrack, int endtrack, int startpos)
#else
cd_play_track_pos(cd_desc, starttrack, endtrack, startpos)
   int cd_desc;
   int starttrack;
   int endtrack;
   int startpos;
#endif
{
   struct disc_timeval time;
   
   time.minutes = startpos / 60;
   time.seconds = startpos % 60;
   time.frames = 0;
   
   return cd_playctl(cd_desc, PLAY_END_TRACK | PLAY_START_POSITION, starttrack, endtrack, &time);
}

/* Play starttrack to endtrack */
int
#if __STDC__
cd_play_track(int cd_desc, int starttrack, int endtrack)
#else
cd_play_track(cd_desc, starttrack, endtrack)
   int cd_desc;
   int starttrack;
   int endtrack;
#endif
{
   return cd_playctl(cd_desc, PLAY_END_TRACK, starttrack, endtrack);
}

/* Play starttrack at position pos to end of CD */
int
#if __STDC__
cd_play_pos(int cd_desc, int track, int startpos)
#else
cd_play_pos(cd_desc, track, startpos)
   int cd_desc;
   int track;
   int startpos;
#endif
{
   struct disc_timeval time;
   
   time.minutes = startpos / 60;
   time.seconds = startpos * 60;
   time.frames = 0;
   
   return cd_playctl(cd_desc, PLAY_START_POSITION, track, &time);
}

/* Play starttrack to end of CD */
int
#if __STDC__
cd_play(int cd_desc, int track)
#else
cd_play(cd_desc, track)
   int cd_desc;
   int track;
#endif
{
   return cd_playctl(cd_desc, PLAY_START_TRACK, track);
}

/* Stop the CD, if it is playing */
int
#if __STDC__
cd_stop(int cd_desc)
#else
cd_stop(cd_desc)
   int cd_desc;
#endif
{
#ifdef CDIOCSTOP
   if(ioctl(cd_desc, CDIOCSTOP) < 0)
     return -1;
#endif
#ifdef CDROMSTOP
   if(ioctl(cd_desc, CDROMSTOP) < 0)
     return -1;
#endif
   
   return 0;
}

/* Pause the CD */
int
#if __STDC__
cd_pause(int cd_desc)
#else
cd_pause(cd_desc)
   int cd_desc;
#endif
{
#ifdef CDIOCPAUSE
   if(ioctl(cd_desc, CDIOCPAUSE) < 0)
     return -1;
#endif
#ifdef CDROMPAUSE
   if(ioctl(cd_desc, CDROMPAUSE) < 0)
     return -1;
#endif
#ifdef CDROM_PAUSE_PLAY
   if(ioctl(cd_desc, CDROM_PAUSE_PLAY) < 0)
     return -1;
#endif
   
   return 0;
}

/* Resume playing */
int
#if __STDC__
cd_resume(int cd_desc)
#else
cd_resume(cd_desc)
   int cd_desc;
#endif
{
#ifdef CDIOCRESUME
   if(ioctl(cd_desc, CDIOCRESUME) < 0)
     return -1;
#endif
#ifdef CDROMRESUME
   if(ioctl(cd_desc, CDROMRESUME) < 0)
     return -1;
#endif
#ifdef CDROM_RESUME_PLAY
   if(ioctl(cd_desc, CDROM_RESUME_PLAY) < 0)
     return -1;
#endif
   
   return 0;
}

/* Eject the tray */
int
#if __STDC__
cd_eject(int cd_desc)
#else
cd_eject(cd_desc)
   int cd_desc;
#endif
{  
#ifdef CDIOCEJECT
   if(ioctl(cd_desc, CDIOCEJECT) < 0)
     return -1;
#endif
#ifdef CDROMEJECT
   if(ioctl(cd_desc, CDROMEJECT) < 0)
     return -1;
#endif
   
   return 0;
}

/* Close the tray */
int
#if __STDC__
cd_close(int cd_desc)
#else
cd_close(cd_desc)
   int cd_desc;
#endif
{
#ifdef CDIOCCLOSE
   if(ioctl(cd_desc, CDIOCCLOSE) < 0)
     return -1;
#endif
#ifdef CDROMCLOSETRAY
   if(ioctl(cd_desc, CDROMCLOSETRAY) < 0)
     return -1;
#endif
   
   return 0;
}

/* Return the current volume setting */
int
#if __STDC__
cd_get_volume(int cd_desc, struct disc_volume *vol)
#else
cd_get_volume(cd_desc, vol)
   int cd_desc;
   struct disc_volume *vol;
#endif
{
#ifdef CDIOCGETVOL
   struct ioc_vol volume;
#endif
#ifdef CDROMVOLREAD
   struct cdrom_volctrl volume;
#endif
#ifdef CDROM_PLAYBACK_STATUS
   struct cd_playback pb;
   struct cd_playback_status ps;
#endif
   
#ifdef CDIOCGETVOL
   if(ioctl(cd_desc, CDIOCGETVOL, &volume) < 0)
     return -1;
   
   vol->vol_front.left = volume.vol[0];
   vol->vol_front.right = volume.vol[1];
   vol->vol_back.left = volume.vol[2];
   vol->vol_back.right = volume.vol[3];
#endif
#ifdef CDROMVOLREAD
   if(ioctl(cd_desc, CDROMVOLREAD, &volume) < 0)
     return -1;
      
   vol->vol_front.left = volume.channel0;
   vol->vol_front.right = volume.channel1;
   vol->vol_back.left = volume.channel2;
   vol->vol_back.right = volume.channel3;
#endif
#ifdef CDROM_PLAYBACK_STATUS
   pb.pb_buffer = &ps;
   pb.pb_alloc_length = sizeof(struct cd_playback_status);
   
   if(ioctl(cd_desc, CDROM_PLAYBACK_STATUS, &pb) < 0)
     return -1;
   
   vol->vol_front.left = ps.ps_chan0_volume;
   vol->vol_front.right = ps.ps_chan1_volume;
   vol->vol_back.left = ps.ps_chan2_volume;
   vol->vol_back.right = ps.ps_chan3_volume;
#endif
   
   return 0;
}

/* Set the volume */
int
#if __STDC__
cd_set_volume(int cd_desc, struct disc_volume vol)
#else
cd_set_volume(cd_desc, vol)
   int cd_desc;
   struct disc_volume vol;
#endif
{
#ifdef CDIOCSETVOL
   struct ioc_vol volume;
#endif
#ifdef CDROMVOLCTRL
   struct cdrom_volctrl volume;
#endif
#ifdef CDROM_PLAYBACK_CONTROL
   struct cd_playback pb;
   struct cd_playback_control pc;
#endif
   
   if(vol.vol_front.left > 255 || vol.vol_front.left < 0 || vol.vol_front.right > 255 || vol.vol_front.right < 0 || vol.vol_back.left > 255 || vol.vol_back.left < 0 || vol.vol_back.right > 255 || vol.vol_back.right < 0)
     return -1;

#ifdef CDIOCSETVOL
   volume.vol[0] = vol.vol_front.left;
   volume.vol[1] = vol.vol_front.right;
   volume.vol[2] = vol.vol_back.left;
   volume.vol[3] = vol.vol_back.right;
   
   if(ioctl(cd_desc, CDIOCSETVOL, &volume) < 0)
     return -1;
#endif
#ifdef CDROMVOLCTRL
   volume.channel0 = vol.vol_front.left;
   volume.channel1 = vol.vol_front.right;
   volume.channel2 = vol.vol_back.left;
   volume.channel3 = vol.vol_back.right;
   
   if(ioctl(cd_desc, CDROMVOLCTRL, &volume) < 0)
     return -1;
#endif
#ifdef CDROM_PLAYBACK_CONTROL
   pc.pc_chan0_select = CDROM_MAX_VOLUME;
   pc.pc_chan0_volume = vol.vol_front.left;
   pc.pc_chan1_select = CDROM_MAX_VOLUME;
   pc.pc_chan1_volume = vol.vol_front.right;
   pc.pc_chan2_select = CDROM_MAX_VOLUME;
   pc.pc_chan2_volume = vol.vol_back.left;
   pc.pc_chan3_select = CDROM_MAX_VOLUME;
   pc.pc_chan3_volume = vol.vol_back.right;
   
   pb.pb_buffer = &pc;
   pb.pb_alloc_length = sizeof(struct cd_playback_control);
   
   if(ioctl(cd_desc, CDROM_PLAYBACK_CONTROL, &pb) < 0)
     return -1;
#endif
   
   return 0;
}

#endif  /* IRIX_CDAUDIO */

/*
Because all these functions are solely mathematical and/or only make callbacks
to previously existing functions they can be used for any platform.
 */

/* Convert frames to a logical block address */
int
cd_frames_to_lba(int frames)
{
   if(frames >= 150)
     return frames - 150;
   
   return 0;
}

/* Convert a logical block address to frames */
int
cd_lba_to_frames(int lba)
{
   return lba + 150;
}

/* Convert disc_timeval to frames */
int
cd_msf_to_frames(struct disc_timeval time)
{
   return time.minutes * 4500 + time.seconds * 75 + time.frames;
}

/* Convert disc_timeval to a logical block address */
int
cd_msf_to_lba(struct disc_timeval time)
{
   if(cd_msf_to_frames(time) > 150)
     return cd_msf_to_frames(time) - 150;
   
   return 0;
}

/* Convert frames to disc_timeval */
void
cd_frames_to_msf(struct disc_timeval *time, int frames)
{
   time->minutes = frames / 4500;
   time->seconds = (frames % 4500) / 75;
   time->frames = frames % 75;
}

/* Convert a logical block address to disc_timeval */
void
cd_lba_to_msf(struct disc_timeval *time, int lba)
{
   cd_frames_to_msf(time, lba + 150);
}

/* Internal advance function */
int
#if __STDC__
__internal_cd_track_advance(int cd_desc, struct disc_info disc, int endtrack, struct disc_timeval time)
#else
__internal_cd_track_advance(int cd_desc, struct disc_info disc, endtrack, time)
   int cd_desc;
   struct disc_info disc;
   int endtrack;
   struct disc_timeval time;
#endif
{ 
   disc.disc_track_time.minutes += time.minutes;
   disc.disc_track_time.seconds += time.seconds;
   disc.disc_track_time.frames += time.frames;
   
   if(disc.disc_track_time.frames > 74) {
      disc.disc_track_time.frames -= 74;
      disc.disc_track_time.seconds++;
   }
   
   if(disc.disc_track_time.frames < 0) {
      disc.disc_track_time.frames += 75;
      disc.disc_track_time.seconds--;
   }
   
   if(disc.disc_track_time.seconds > 59) {
      disc.disc_track_time.seconds -= 59;
      disc.disc_track_time.minutes++;
   }
   if(disc.disc_track_time.seconds < 0) {
      disc.disc_track_time.seconds += 60;
      disc.disc_track_time.minutes--;
   }

   if(disc.disc_track_time.minutes < 0) {
      disc.disc_current_track--;
      if(disc.disc_current_track == 0)
        disc.disc_current_track = 1;

      return cd_play_track(cd_desc, disc.disc_current_track, endtrack);
   }

   if((disc.disc_track_time.minutes == disc.disc_track[disc.disc_current_track].track_pos.minutes && disc.disc_track_time.seconds > disc.disc_track[disc.disc_current_track].track_pos.seconds) || disc.disc_track_time.minutes > disc.disc_track[disc.disc_current_track].track_pos.minutes) {
      disc.disc_current_track++;
      if(disc.disc_current_track > endtrack)
        disc.disc_current_track = endtrack;

      return cd_play_track(cd_desc, disc.disc_current_track, endtrack);
   }

   return cd_play_track_pos(cd_desc, disc.disc_current_track, endtrack, disc.disc_track_time.minutes * 60 + disc.disc_track_time.seconds);
}

/* Advance the position within a track */
int
#if __STDC__
cd_track_advance(int cd_desc, int endtrack, struct disc_timeval time)
#else
cd_track_advance(cd_desc, endtrack, time)
   int cd_desc;
   int endtrack;
   struct disc_timeval time;
#endif
{
   struct disc_info disc;
   
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
   
   if(!disc.disc_present)
     return -1;
   
   if(__internal_cd_track_advance(cd_desc, disc, endtrack, time) < 0)
     return -1;
   
   return 0;
}

/* Advance the position within the track without preserving an endtrack */
int
#if __STDC__
cd_advance(int cd_desc, struct disc_timeval time)
#else
cd_advance(cd_desc, time)
    int cd_desc;
    struct disc_timeval time;
#endif
{
   struct disc_info disc;

   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
   
   if(__internal_cd_track_advance(cd_desc, disc, disc.disc_total_tracks, time) < 0)
     return -1;
   
   return 0;
}

/* Update information in a disc_info structure using a disc_status structure */
int
#if __STDC__
cd_update(struct disc_info *disc, struct disc_status status)
#else
cd_update(disc, status)
   struct disc_info *disc;
   struct disc_status status;
#endif
{
   int pos;
   
   if(!(disc->disc_present = status.status_present))
     return -1;
   
   disc->disc_mode = status.status_mode;
   memcpy(&disc->disc_time, &status.status_disc_time, sizeof(struct disc_timeval));

   disc->disc_current_track = 0;
   while(disc->disc_current_track < disc->disc_total_tracks && (disc->disc_time.minutes * 60 + disc->disc_time.seconds) * 75 + disc->disc_time.frames >= (disc->disc_track[disc->disc_current_track].track_pos.minutes * 60 + disc->disc_track[disc->disc_current_track].track_pos.seconds) * 75 + disc->disc_track[disc->disc_current_track].track_pos.frames)
     disc->disc_current_track++;

   pos = ((disc->disc_time.minutes * 60 + disc->disc_time.seconds) * 75 + disc->disc_time.frames) - ((disc->disc_track[disc->disc_current_track - 1].track_pos.minutes * 60 + disc->disc_track[disc->disc_current_track - 1].track_pos.seconds) * 75 + disc->disc_track[disc->disc_current_track - 1].track_pos.frames);

   cd_frames_to_msf(&disc->disc_track_time, pos);
   
   return 0;
}

/* Universal play control function */
int
#if __STDC__
cd_playctl(int cd_desc, int options, int start_track, ...)
#else
cd_playctl(cd_desc, options, ...)
   int cd_desc;
   int options;
#endif
{
   int end_track;
   struct disc_info disc;
   struct disc_timeval *startpos, *endpos, start_position, end_position;
   va_list arglist;

   va_start(arglist, start_track);
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;

   if(options & PLAY_END_TRACK)
     end_track = va_arg(arglist, int);

   if(options & PLAY_START_POSITION)
     startpos = va_arg(arglist, struct disc_timeval *);

   if(options & PLAY_END_POSITION)
     endpos = va_arg(arglist, struct disc_timeval *);
   
   va_end(arglist);
   
   if(options & PLAY_START_POSITION) {
      start_position.minutes = disc.disc_track[start_track - 1].track_pos.minutes + startpos->minutes;
      start_position.seconds = disc.disc_track[start_track - 1].track_pos.seconds + startpos->seconds;
      start_position.frames = disc.disc_track[start_track - 1].track_pos.frames + startpos->frames;
   } else {
      start_position.minutes = disc.disc_track[start_track - 1].track_pos.minutes;
      start_position.seconds = disc.disc_track[start_track - 1].track_pos.seconds;
      start_position.frames = disc.disc_track[start_track - 1].track_pos.frames;
   }

   if(options & PLAY_END_TRACK) {
      if(options & PLAY_END_POSITION) {
	 end_position.minutes = disc.disc_track[end_track].track_pos.minutes + endpos->minutes;
	 end_position.seconds = disc.disc_track[end_track].track_pos.seconds + endpos->seconds;
	 end_position.frames = disc.disc_track[end_track].track_pos.frames + endpos->frames;
      } else {
	 end_position.minutes = disc.disc_track[end_track].track_pos.minutes;
	 end_position.seconds = disc.disc_track[end_track].track_pos.seconds;
	 end_position.frames = disc.disc_track[end_track].track_pos.frames;
      }
   } else {
      end_position.minutes = disc.disc_track[disc.disc_total_tracks].track_pos.minutes;
      end_position.seconds = disc.disc_track[disc.disc_total_tracks].track_pos.seconds;
      end_position.frames = disc.disc_track[disc.disc_total_tracks].track_pos.frames;
   }
   
   return cd_play_frames(cd_desc, (start_position.minutes * 60 + start_position.seconds) * 75 + start_position.frames, (end_position.minutes * 60 + end_position.seconds) * 75 + end_position.frames);
}
