/********************************************************************************
*                                                                               *
*                               F o n t   O b j e c t                           *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXFont.cpp,v 1.7 1999/11/17 20:44:29 jeroen Exp $                        *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXFont.h"


/*
  Notes:
  
  - Interpretation of the hints:
  
      FONTPITCH_DEFAULT     No preference for pitch
      FONTPITCH_FIXED       If specified, match for fixed pitch fonts are strongly preferred
      FONTPITCH_VARIABLE    If specified, match for variable pitch font strongly preferred
 
      FONTHINT_DONTCARE     No hints given
      FONTHINT_DECORATIVE   Ye Olde Fonte
      FONTHINT_MODERN       Monospace fonts such as courier and so on
      FONTHINT_ROMAN        Serif font such as times
      FONTHINT_SCRIPT       Cursive font/script
      FONTHINT_SWISS        Sans serif font such as swiss, helvetica, arial
      FONTHINT_SYSTEM       Raster based fonts, typically monospaced
 
      FONTHINT_X11          Set internally to force X11-style font specification for font name
      
      FONTHINT_SCALABLE     Strong emphasis on scalable fonts; under Windows, this means
                            TrueType fonts are desired

      FONTHINT_POLYMORPHIC  Strong emphasis on polymorphic fonts; under Windows, this means
                            TrueType fonts desired also
  
  - For font size, the largest font not larger than the indicated size is taken; 
    rationale is that larger fonts may make applications too big for the screen, but 
    smaller ones are OK.
      
  - FONTENCODING_DEFAULT means we don't care about the encoding; if another value is
    specified, we'll insist/prefer the encoding must match.
      
  - FONTSLANT_ITALIC is a cursive typeface for some fonts; FONTSLANT_OBLIQUE is the same
    basic font written at an angle; for many fonts, FONTSLANT_ITALIC and FONTSLANT_OBLIQUE
    means pretty much the same thing.
      
  - FONTWEIGHT_DONTCARE means we don't care about the typeface weight; if non-null, we'll
    try to get as close as we can.

  - Alternative face names:
  
      helvetica   arial, swiss
      times       roman
      courier     modern

    [Perhaps we can come up with a few additional equivalents to try before failing].

  - We have hidden TEXTMETRICS or XFontStruct knowlegde from the public interface
  
  - XFontStruct.ascent+XFontStruct.descent is the height of the font, as far as line
    spacing goes.  XFontStruct.max_bounds.ascent+XFontStruct.max_bounds.descent is 
    larger, as some characters can apparently extend beyond ascent or descent!!
    
  - We should assume success first, i.e. only fall back on complicated matching if
    stuff fails; in many cases, fonts are picked from FXFontSelector and already known
    to exist...
    
  - Registry section FONTSUBSTITUTIONS can be used to map typeface names to platform
    specific typeface names.
    
    e.g.
    [FONTSUBSTITUTIONS]
    arial = helvetica
    swiss = helvetica
    
    To make Windows programs work elsewhere w/o mods to source code...
 
  - Text txfm matrix [a b c d] premultiplies.

  - Should we perhaps build our own tables of font metrics? This might make
    things simpler for the advanced stuff, and be conceivably a lot faster
    under MS-Windows [no need to SelectObject() all the time just to get some
    info; also, this could be useful in case the drawing surface is not a 
    window].

  - Somehow, create() of fonts assumes its for the screen; how about other
    surfaces?

  - Need some API to make OpenGL fonts out of the 2D fonts...

  - Matching algorithm should favor bitmapped fonts over scalable ones [as the
    latter may not be optimized easthetically; also, the matching algorithm should
    not weight resolution as much.

  - Use PANOSE numbers for matching fonts...

  - FOR THE MOMENT we're creating a dummy DC to keep the font locked into the GDI
    for MUCH quicker access to text metrics.  Soon however we want to just build
    our own font metrics tables and determine the metrics entirely with client-side
    code.  This will [predictably] be the fastest possible method as it will not
    involve context switches...
    
  - UNICODE means registry and encoding are set to iso10646-1
*/


// X11 
#ifndef FX_NATIVE_WIN32


// XLFD Fields
#define XLFD_FOUNDRY      0
#define XLFD_FAMILY       1
#define XLFD_WEIGHT       2
#define XLFD_SLANT        3
#define XLFD_SETWIDTH     4
#define XLFD_ADDSTYLE     5
#define XLFD_PIXELSIZE    6
#define XLFD_POINTSIZE    7
#define XLFD_RESOLUTION_X 8
#define XLFD_RESOLUTION_Y 9
#define XLFD_SPACING      10
#define XLFD_AVERAGE      11
#define XLFD_REGISTRY     12
#define XLFD_ENCODING     13


// Match factors
#define ENCODING_FACTOR   256
#define PITCH_FACTOR      128
#define RESOLUTION_FACTOR 64 
#define SCALABLE_FACTOR   32 
#define POLY_FACTOR       16 
#define SIZE_FACTOR       8  
#define WEIGHT_FACTOR     4  
#define SLANT_FACTOR      2  
#define SETWIDTH_FACTOR   1  

// Hint mask
#define FONTHINT_MASK (FONTHINT_DECORATIVE|FONTHINT_MODERN|FONTHINT_ROMAN|FONTHINT_SCRIPT|FONTHINT_SWISS|FONTHINT_SYSTEM)

// Convenience macros
#define EQUAL1(str,c)     (str[0]==c && str[1]=='\0')
#define EQUAL2(str,c1,c2) (str[0]==c1 && str[1]==c2 && str[2]=='\0')
#define ISISO8859(str)    (str[0]=='i' && str[1]=='s' && str[2]=='o' && str[3]=='8' && str[4]=='8' && str[5]=='5' && str[6]=='9' && str[7]=='\0')
#define ISMULTIBYTE(str)  (((str[0]=='k' && str[1]=='s' && str[2]=='c') || (str[0]=='j' && str[1]=='i' && str[2]=='s' && str[3]=='x') || (str[0]=='g' && str[1]=='b')))

// A glyph exists for the given character
#define HASCHAR(font,ch)  ((((XFontStruct*)font)->min_char_or_byte2 <= (FXuint)ch) && ((FXuint)ch <= ((XFontStruct*)font)->max_char_or_byte2))



// MS-Windows
#else

// A glyph exists for the given character
#define HASCHAR(font,ch)  ((((TEXTMETRIC*)font)->tmFirstChar <= (FXuint)ch) && ((FXuint)ch <= ((TEXTMETRIC*)font)->tmLastChar))


// Character set encoding
static BYTE FXFontEncoding2CharSet(FXuint encoding){
  switch(encoding){
    case FONTENCODING_DEFAULT:
      return DEFAULT_CHARSET; 
    case FONTENCODING_TURKISH:
      return TURKISH_CHARSET; 
    case FONTENCODING_BALTIC:
      return BALTIC_CHARSET; 
    case FONTENCODING_RUSSIAN:
      return RUSSIAN_CHARSET; 
    case FONTENCODING_ARABIC:
      return ARABIC_CHARSET; 
    case FONTENCODING_GREEK:
      return GREEK_CHARSET; 
    case FONTENCODING_HEBREW:
      return HEBREW_CHARSET; 
    case FONTENCODING_THAI:
      return THAI_CHARSET; 
    case FONTENCODING_EASTEUROPE:
      return EASTEUROPE_CHARSET;
    case FONTENCODING_USASCII:
      return ANSI_CHARSET;
    default:
      FXASSERT(FALSE); // Unrecognized encoding
    }
  return DEFAULT_CHARSET;
  }


// Character set encoding
static FXuint CharSet2FXFontEncoding(BYTE lfCharSet){
  switch(lfCharSet){
    case DEFAULT_CHARSET:
    case OEM_CHARSET:
    case SYMBOL_CHARSET:
    case MAC_CHARSET:
      return FONTENCODING_DEFAULT;
    case ANSI_CHARSET:
      return FONTENCODING_USASCII;
    case TURKISH_CHARSET:
      return FONTENCODING_TURKISH; 
    case BALTIC_CHARSET:
      return FONTENCODING_BALTIC; 
    case RUSSIAN_CHARSET:
      return FONTENCODING_RUSSIAN; 
    case ARABIC_CHARSET:
      return FONTENCODING_ARABIC; 
    case GREEK_CHARSET:
      return FONTENCODING_GREEK; 
    case HEBREW_CHARSET:
      return FONTENCODING_HEBREW; 
    case THAI_CHARSET:
      return FONTENCODING_THAI;
    case EASTEUROPE_CHARSET:
      return FONTENCODING_EASTEUROPE;
    default:
      FXASSERT(FALSE); // Unrecognized character set
    }
  return FONTENCODING_DEFAULT;
  }

#endif


/*******************************************************************************/

// Object implementation
FXIMPLEMENT(FXFont,FXId,NULL,0)


// Deserialization
FXFont::FXFont(){
  font=(void*)-1;
  }


// Construct font from X11 font spec
FXFont::FXFont(FXApp* a,const FXString& nm):FXId(a),name(nm){
  FXTRACE((100,"FXFont::FXFont %08x\n",this));
  size=0;
  weight=0;
  slant=0;
  encoding=FONTENCODING_DEFAULT;
  setwidth=FONTSETWIDTH_DONTCARE;
#ifndef FX_NATIVE_WIN32
  hints=FONTHINT_X11;               // X11 font string method
#else
  hints=0;
  dc=NULL;
#endif
  font=NULL;
  }


// Construct a font with given face name, size in points(pixels), weight, slant, character set encoding, setwidth, and hints
FXFont::FXFont(FXApp* a,const FXString& face,FXuint sz,FXuint wt,FXuint sl,FXuint enc,FXuint setw,FXuint h):FXId(a),name(face){
  FXTRACE((100,"FXFont::FXFont %08x\n",this));
  size=10*sz;
  weight=wt;
  slant=sl;
  encoding=enc;
  setwidth=setw;
  hints=(h&~FONTHINT_X11);          // System-independent method
  font=NULL;
#ifdef FX_NATIVE_WIN32
  dc=NULL;
#endif
  }


// Construct font from font description
FXFont::FXFont(FXApp* a,const FXFontDesc& fontdesc):FXId(a){
  FXTRACE((100,"FXFont::FXFont %08x\n",this));
  name=fontdesc.face;
  size=fontdesc.size;
  weight=fontdesc.weight;
  slant=fontdesc.slant;
  encoding=fontdesc.encoding;
  setwidth=fontdesc.setwidth;
  hints=fontdesc.flags;             // We may get system dependent fonts
  font=NULL;
#ifdef FX_NATIVE_WIN32
  dc=NULL;
#endif
  }
  

// Get font description
void FXFont::getFontDesc(FXFontDesc& fontdesc){
  strncpy(fontdesc.face,name.text(),31);
  fontdesc.face[31]=0;
  fontdesc.size=size;
  fontdesc.weight=weight;
  fontdesc.slant=slant;
  fontdesc.encoding=encoding;
  fontdesc.setwidth=setwidth;
  fontdesc.flags=hints;
  }
  

#ifndef FX_NATIVE_WIN32

// Get list of font names matching pattern
static char **listfontnames(Display *dpy,const char* pattern,int& numfnames){
  int maxfnames=256;
  char **fnames;
  for(;;){
    fnames=XListFonts(dpy,pattern,maxfnames,&numfnames);
    if((fnames==NULL) || (numfnames<maxfnames)) break;
    XFreeFontNames(fnames);
    maxfnames<<=1;
    }
  return fnames;
  }


// Return number of fonts matching name
static int matchingfonts(Display *dpy,const char *pattern){
  char **fnames;
  int numfnames;
  fnames=listfontnames(dpy,pattern,numfnames);
  XFreeFontNames(fnames);
  if(numfnames>0) FXTRACE((100,"matched: %s\n",pattern));
  return numfnames;
  }


// Parse font name into parts
static int parsefontname(char *fields[],char* fname){
  int f=0;
  if(fname && *fname++ == '-'){
    while(*fname){
      fields[f++]=fname;
      if(f>XLFD_ENCODING) return 1;
      fname=strchr(fname,'-');
      if(!fname) break;
      *fname++='\0';
      }
    }
  return 0;
  }


// Try find matching font
char* FXFont::findmatch(char *fontname,const char* family){
  FXchar pattern[300],fname[300],*field[14],*reg,*enc,*wgt,*setw,spacing,*italic,**fnames;
  FXint bestf,bestdweight,bestdsize,bestvalue,bestscalable,bestxres,bestyres;
  FXint screenres,xres,yres;
  FXint dweight,w,scalable,polymorphic,dsize;
  FXint numfnames,f,value;
  FXuint sw,sz;
  
  // Get fonts matching the pattern
  sprintf(pattern,"-*-%s-*-*-*-*-*-*-*-*-*-*-*-*",family);
  fnames=listfontnames(getApp()->display,pattern,numfnames);
  if(!fnames) return NULL;
  
  // Init match values
  bestf=-1;
  bestvalue=0;
  bestdsize=10000000;
  bestdweight=10000000;
  bestscalable=0;
  bestxres=75;
  bestyres=75;
  fname[299]=0;
  
    
  // Perhaps override screen resolution via registry
  screenres=getApp()->reg().readUnsignedEntry("SETTINGS","screenres",100);
  
  // Validate
  if(screenres<50) screenres=50;
  if(screenres>200) screenres=200;

  FXTRACE((150,"Matching Fonts for screenres=%d :\n",screenres));

  // Loop over all fonts to find the best match  
  for(f=0; f<numfnames; f++){
    strncpy(fname,fnames[f],299);
    if(parsefontname(field,fname)){
      
      // This font's match value
      value=0;
      scalable=0;
      polymorphic=0;
      dsize=1000000;
      dweight=1000;
      
      // Scalable
      if(EQUAL1(field[XLFD_PIXELSIZE],'0') && EQUAL1(field[XLFD_POINTSIZE],'0') && EQUAL1(field[XLFD_AVERAGE],'0')){
        value+=SCALABLE_FACTOR;
        scalable=1;
        }
      else{
        if(!(hints&FONTHINT_SCALABLE)) value+=SCALABLE_FACTOR;
        }
      
      // Polymorphic
      if(EQUAL1(field[XLFD_WEIGHT],'0') || EQUAL1(field[XLFD_SETWIDTH],'0') || EQUAL1(field[XLFD_SLANT],'0') || EQUAL1(field[XLFD_ADDSTYLE],'0')){
        value+=POLY_FACTOR;
        polymorphic=1;
        }
      else{
        if(!(hints&FONTHINT_POLYMORPHIC)) value+=POLY_FACTOR;
        }
        
      // Check pitch
      spacing=field[XLFD_SPACING][0];
      if(hints&FONTPITCH_FIXED){
        if(spacing=='m' || spacing=='c' || spacing=='M' || spacing=='C') value+=PITCH_FACTOR;
        }
      else if(hints&FONTPITCH_VARIABLE){
        if(spacing=='p' || spacing=='P') value+=PITCH_FACTOR;
        }
      else{
        value+=PITCH_FACTOR;
        }

      // Check weight
      if(weight==FONTWEIGHT_DONTCARE){
        dweight=0;
        //value+=WEIGHT_FACTOR;
        }
      else{
        wgt=field[XLFD_WEIGHT];
        if(fxstrcasecmp(wgt,"medium")==0) w=FONTWEIGHT_MEDIUM;
        else if(fxstrcasecmp(wgt,"bold")==0) w=FONTWEIGHT_BOLD;
        else if(fxstrcasecmp(wgt,"demibold")==0) w=FONTWEIGHT_DEMIBOLD;
        else if(fxstrcasecmp(wgt,"demi")==0) w=FONTWEIGHT_DEMIBOLD;
        else if(fxstrcasecmp(wgt,"black")==0) w=FONTWEIGHT_BLACK;
        else if(fxstrcasecmp(wgt,"light")==0) w=FONTWEIGHT_LIGHT;
        else if(fxstrcasecmp(wgt,"normal")==0) w=FONTWEIGHT_NORMAL;
        else w=FONTWEIGHT_NORMAL;
        dweight=w-weight;
        dweight=FXABS(dweight);
        //if(dweight==0) value+=WEIGHT_FACTOR;
        }
      
      // Check slant
      if(slant==FONTSLANT_DONTCARE){
        value+=SLANT_FACTOR;
        }
      else{
        italic=field[XLFD_SLANT];
        if(slant==FONTSLANT_ITALIC){
          if(italic[0]=='i' || italic[0]=='I') value+=SLANT_FACTOR;
          }
        else if(slant==FONTSLANT_OBLIQUE){
          if(italic[0]=='o' || italic[0]=='O') value+=SLANT_FACTOR;
          }
        else if(slant==FONTSLANT_REVERSE_ITALIC){
          if((italic[0]=='r' || italic[0]=='R') && (italic[1]=='i'|| italic[1]=='I')) value+=SLANT_FACTOR;
          } 
        else if(slant==FONTSLANT_REVERSE_OBLIQUE){
          if((italic[0]=='r' || italic[0]=='R') && (italic[1]=='o'|| italic[1]=='O')) value+=SLANT_FACTOR;
          }
        else if(slant==FONTSLANT_REGULAR){
          if(italic[0]=='r' || italic[0]=='R') value+=SLANT_FACTOR;
          }
        }
      
      // Check SetWidth
      if(setwidth==FONTSETWIDTH_DONTCARE){
        value+=SETWIDTH_FACTOR;
        }
      else{
        setw=field[XLFD_SETWIDTH];
        if(fxstrcasecmp(setw,"ultracondensed")==0) sw=FONTSETWIDTH_ULTRACONDENSED;
        else if(fxstrcasecmp(setw,"extracondensed")==0) sw=FONTSETWIDTH_EXTRACONDENSED;
        else if(fxstrcasecmp(setw,"condensed")==0) sw=FONTSETWIDTH_CONDENSED;
        else if(fxstrcasecmp(setw,"narrow")==0) sw=FONTSETWIDTH_CONDENSED;
        else if(fxstrcasecmp(setw,"compressed")==0) sw=FONTSETWIDTH_CONDENSED;
        else if(fxstrcasecmp(setw,"semicondensed")==0) sw=FONTSETWIDTH_SEMICONDENSED;
        else if(fxstrcasecmp(setw,"medium")==0) sw=FONTSETWIDTH_MEDIUM;
        else if(fxstrcasecmp(setw,"normal")==0) sw=FONTSETWIDTH_MEDIUM;
        else if(fxstrcasecmp(setw,"regular")==0) sw=FONTSETWIDTH_MEDIUM;
        else if(fxstrcasecmp(setw,"semiexpanded")==0) sw=FONTSETWIDTH_SEMIEXPANDED;
        else if(fxstrcasecmp(setw,"demiexpanded")==0) sw=FONTSETWIDTH_SEMIEXPANDED;
        else if(fxstrcasecmp(setw,"expanded")==0) sw=FONTSETWIDTH_EXPANDED;
        else if(fxstrcasecmp(setw,"wide")==0) sw=FONTSETWIDTH_EXTRAEXPANDED;
        else if(fxstrcasecmp(setw,"extraexpanded")==0) sw=FONTSETWIDTH_EXTRAEXPANDED;
        else if(fxstrcasecmp(setw,"ultraexpanded")==0) sw=FONTSETWIDTH_ULTRAEXPANDED;
        else sw=FONTSETWIDTH_MEDIUM;
        if(setwidth==sw) value+=SETWIDTH_FACTOR;
        }
        
      // Check encoding
      reg=field[XLFD_REGISTRY];
      enc=field[XLFD_ENCODING];
      switch(encoding){
        case FONTENCODING_DEFAULT:
          if(!ISMULTIBYTE(reg)){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_ISO_8859_1:
          if(ISISO8859(reg) && EQUAL1(enc,'1')){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_ISO_8859_2:
          if(ISISO8859(reg) && EQUAL1(enc,'2')){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_ISO_8859_3:
          if(ISISO8859(reg) && EQUAL1(enc,'3')){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_ISO_8859_4:
          if(ISISO8859(reg) && EQUAL1(enc,'4')){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_ISO_8859_5:
          if(ISISO8859(reg) && EQUAL1(enc,'5')){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_ISO_8859_6:
          if(ISISO8859(reg) && EQUAL1(enc,'6')){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_ISO_8859_7:
          if(ISISO8859(reg) && EQUAL1(enc,'7')){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_ISO_8859_8:
          if(ISISO8859(reg) && EQUAL1(enc,'8')){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_ISO_8859_11:
          if(ISISO8859(reg) && EQUAL2(enc,'1','1')){
            value+=ENCODING_FACTOR;
            }
          break;
        case FONTENCODING_KOI_8:
          if(strcmp(reg,"koi8")==0 && EQUAL1(enc,'r')){
            value+=ENCODING_FACTOR;
            }
          break;
        }

      // If the font can be rendered at any resolution, we'll render at our actual device resolution
      if(EQUAL1(field[XLFD_RESOLUTION_X],'0') && EQUAL1(field[XLFD_RESOLUTION_Y],'0')){
        xres=screenres;
        yres=screenres;
        }
      
      // Else get the resolution for which the font is designed
      else{
        xres=atoi(field[XLFD_RESOLUTION_X]);
        yres=atoi(field[XLFD_RESOLUTION_Y]);
        }

      // If scalable, we can of course get the exact size we want
      // We do not set dsize to 0, as we prefer a bitmapped font that gets within
      // 10% over a scalable one that's exact, as the bitmapped fonts look much better
      // at small sizes than scalable ones...
      if(scalable){
        value+=SIZE_FACTOR;
        dsize=size/10;
        }
      
      // Otherwise, we try to get something close
      else{
        
        // We correct for the actual screen resolution; if the font is rendered at a
        // 100 dpi, and we have a screen with 90dpi, the actual point size of the font 
        // should be multiplied by (100/90).
        sz=(yres*atoi(field[XLFD_POINTSIZE]))/screenres;    
        
        // We strongly prefer the largest pointsize not larger than the desired pointsize
        if(sz<=size){
          value+=SIZE_FACTOR;
          dsize=size-sz;
          }
        
        // But if we can't get that, we'll take anything thats close...
        else{
          dsize=sz-size;
          }
        }
        
      FXTRACE((160,"%4d: match=%-3x dw=%-3d ds=%3d sc=%d py=%d xres=%-3d yres=%-3d xlfd=%s\n",f,value,dweight,dsize,scalable,polymorphic,xres,yres,fnames[f]));

      // How close is the match?
      if((value>bestvalue) || ((value==bestvalue) && (dsize<bestdsize)) || ((value==bestvalue) && (dsize==bestdsize) && (dweight<bestdweight))){
        bestvalue=value;
        bestdsize=dsize;
        bestdweight=dweight;
        bestscalable=scalable;
        bestxres=xres;
        bestyres=yres;
        bestf=f;
        }
      }
    }
  if(0<=bestf){
    if(!bestscalable){
      strncpy(fontname,fnames[bestf],299);
      }
    else{
      strncpy(fname,fnames[bestf],299);
      parsefontname(field,fname);
      
      // Build XLFD, correcting for possible difference between font resolution and screen resolution
      sprintf(fontname,"-%s-%s-%s-%s-%s-%s-*-%d-%d-%d-%s-*-%s-%s",field[XLFD_FOUNDRY],field[XLFD_FAMILY],field[XLFD_WEIGHT],field[XLFD_SLANT],field[XLFD_SETWIDTH],field[XLFD_ADDSTYLE],(bestyres*size)/screenres,bestxres,bestyres,field[XLFD_SPACING],field[XLFD_REGISTRY],field[XLFD_ENCODING]);
      
      // This works!! draw text sideways:- but need to get some more experience with this
      //sprintf(fontname,"-%s-%s-%s-%s-%s-%s-*-[0 64 ~64 0]-%d-%d-%s-*-%s-%s",field[XLFD_FOUNDRY],field[XLFD_FAMILY],field[XLFD_WEIGHT],field[XLFD_SLANT],field[XLFD_SETWIDTH],field[XLFD_ADDSTYLE],screenxres,screenyres,field[XLFD_SPACING],field[XLFD_REGISTRY],field[XLFD_ENCODING]);
      }
    FXTRACE((150,"Best Font:\n"));
    FXTRACE((150,"%4d: match=%3x dw=%-3d ds=%-3d sc=%d py=%d xres=%-3d yres=%-3d xlfd=%s\n",bestf,bestvalue,bestdweight,bestdsize,bestscalable,polymorphic,bestxres,bestyres,fontname));
    XFreeFontNames(fnames);
    return fontname;
    }
  XFreeFontNames(fnames);
  return NULL;
  }


// Try load the best matching font
char* FXFont::findbestfont(char *fontname){
  char *match=NULL;

  // Try specified font family first
  if(!name.empty()){
    match=findmatch(fontname,getApp()->reg().readStringEntry("FONTSUBSTITUTIONS",name.text(),name.text()));
    }

  // Try swiss if we didn't have a match yet
  if(!match && ((hints&(FONTHINT_SWISS|FONTHINT_SYSTEM)) || !(hints&FONTHINT_MASK))){
    match=findmatch(fontname,getApp()->reg().readStringEntry("FONTSUBSTITUTIONS","helvetica","helvetica"));
    }

  // Try roman if we didn't have a match yet
  if(!match && ((hints&FONTHINT_ROMAN) || !(hints&FONTHINT_MASK))){
    match=findmatch(fontname,getApp()->reg().readStringEntry("FONTSUBSTITUTIONS","times","times"));
    }

  // Try modern if we didn't have a match yet
  if(!match && ((hints&FONTHINT_MODERN) || !(hints&FONTHINT_MASK))){
    match=findmatch(fontname,getApp()->reg().readStringEntry("FONTSUBSTITUTIONS","courier","courier"));
    }

  // Try modern if we didn't have a match yet
  if(!match && ((hints&FONTHINT_DECORATIVE) || !(hints&FONTHINT_MASK))){
    match=findmatch(fontname,getApp()->reg().readStringEntry("FONTSUBSTITUTIONS","gothic","gothic"));
    }

  return fontname;
  }


// Try these fallbacks for swiss hint
const char* swissfallback[]={
  "-*-helvetica-bold-r-*-*-*-120-*-*-*-*-*-*",
  "-*-lucida-bold-r-*-*-*-120-*-*-*-*-*-*",
  "-*-helvetica-medium-r-*-*-*-120-*-*-*-*-*-*",
  "-*-lucida-medium-r-*-*-*-120-*-*-*-*-*-*",
  "-*-helvetica-*-*-*-*-*-120-*-*-*-*-*-*",
  "-*-lucida-*-*-*-*-*-120-*-*-*-*-*-*",
  NULL
  };
  
  
// Try these fallbacks for times hint
const char* romanfallback[]={
  "-*-times-bold-r-*-*-*-120-*-*-*-*-*-*",
  "-*-charter-bold-r-*-*-*-120-*-*-*-*-*-*",
  "-*-times-medium-r-*-*-*-120-*-*-*-*-*-*",
  "-*-charter-medium-r-*-*-*-120-*-*-*-*-*-*",
  "-*-times-*-*-*-*-*-120-*-*-*-*-*-*",
  "-*-charter-*-*-*-*-*-120-*-*-*-*-*-*",
  NULL
  };


// Try these fallbacks for modern hint
const char* modernfallback[]={
  "-*-courier-bold-r-*-*-*-120-*-*-*-*-*-*",
  "-*-lucidatypewriter-bold-r-*-*-*-120-*-*-*-*-*-*",
  "-*-courier-medium-r-*-*-*-120-*-*-*-*-*-*",
  "-*-lucidatypewriter-medium-r-*-*-*-120-*-*-*-*-*-*",
  "-*-courier-*-*-*-*-*-120-*-*-*-*-*-*",
  "-*-lucidatypewriter-*-*-*-*-*-120-*-*-*-*-*-*",
  NULL
  };

  
// Try these final fallbacks
const char* finalfallback[]={
  "7x13",
  "8x13",
  "7x14",
  "8x16",
  "9x15",
  NULL
  };


// See which fallback font exists
const char* FXFont::fallbackfont(){
  const char *fname=NULL;
  int i=0;
  
  // Try swiss if we wanted swiss, or if we don't care
  if((hints&FONTHINT_SWISS) || !(hints&FONTHINT_MASK)){
    while((fname=swissfallback[i])!=NULL){
      if(matchingfonts(getApp()->display,fname)>0) break;
      i++;
      }
    }
  
  // Try roman if we wanted roman, or if we don't care
  if(!fname && ((hints&FONTHINT_ROMAN) || !(hints&FONTHINT_MASK))){
    while((fname=romanfallback[i])!=NULL){
      if(matchingfonts(getApp()->display,fname)>0) break;
      i++;
      }
    }
  
  // Try modern if we wanted modern, or if we don't care
  if(!fname && ((hints&FONTHINT_MODERN) || !(hints&FONTHINT_MASK))){
    while((fname=modernfallback[i])!=NULL){
      if(matchingfonts(getApp()->display,fname)>0) break;
      i++;
      }
    }
  
  // Try final fallback fonts
  if(!fname){
    while((fname=finalfallback[i])!=NULL){
      if(matchingfonts(getApp()->display,fname)>0) break;
      i++;
      }
    if(!fname) fname="fixed";
    }
  
  return fname;
  }


#endif


/*******************************************************************************/
  

// Create font
void FXFont::create(){
  if(!xid){
    FXTRACE((100,"%s::create %08x\n",getClassName(),this));
    
#ifndef FX_NATIVE_WIN32

    // Gotta have display open!
    if(!getApp()->display){ fxerror("%s::create: trying to create font before opening display.\n",getClassName()); }

    // X11 font specification
    if(hints&FONTHINT_X11){
  
      // Should have non-NULL font name
      if(!name.text()){ fxerror("%s::create: font name should not be NULL.\n",getClassName()); }
    
      FXTRACE((150,"%s::create: X11 font: %s\n",getClassName(),name.text()));

      // Try load the font
      font=XLoadQueryFont(getApp()->display,name.text());
      }
    
    // Platform independent specification
    else{
      FXTRACE((150,"%s::create: face: %s size: %d weight: %d slant: %d encoding: %d hints: %04x\n",getClassName(),name.text()?name.text():"none",size,weight,slant,encoding,hints));
      
      // Try load best font
      char fontname[300];
      font=XLoadQueryFont(getApp()->display,findbestfont(fontname));
      }
    
    // If we still don't have a font yet, try fallback fonts
    if(!font){
      font=XLoadQueryFont(getApp()->display,fallbackfont());
      }
    
    // Remember font id
    if(font){ xid=((XFontStruct*)font)->fid; }
  
    // At this point, we should examine the font properties
    
//  if(font){
//    XFontStruct *fs=(XFontStruct*) font;
//    int i;
//    char *val;
//    if(fs->properties){
//      for(i=0; i<fs->n_properties; i++){
//        fprintf(stderr,"ATOM = %d (%s)",fs->properties[i].name,XGetAtomName(getApp()->display,fs->properties[i].name));
//          fprintf(stderr," = %d\n",fs->properties[i].card32);
//        }
//      }
//    }
#else
    
    // Windows will not support the X11 font string specification method
    if(hints&FONTHINT_X11){ fxerror("%s::create: this method of font specification not supported under Windows.\n",getClassName()); }
    
    // Hang on to this for text metrics functions
    dc=CreateCompatibleDC(NULL);

    // Now fill in the fields
    LOGFONT lf;
    lf.lfHeight=fxpointsize_to_height(dc,size);
    lf.lfWidth=0;
    lf.lfEscapement=0;
    lf.lfOrientation=0;
    lf.lfWeight=weight;
    if((slant==FONTSLANT_ITALIC) || (slant==FONTSLANT_OBLIQUE))
      lf.lfItalic=TRUE;
    else
      lf.lfItalic=FALSE;
    lf.lfUnderline=FALSE;
    lf.lfStrikeOut=FALSE;
    
    // Character set encoding
    lf.lfCharSet=FXFontEncoding2CharSet(encoding);
    
    // Other hints
    lf.lfOutPrecision=OUT_DEFAULT_PRECIS;
    if(hints&FONTHINT_SYSTEM) lf.lfOutPrecision=OUT_RASTER_PRECIS;
    if(hints&FONTHINT_SCALABLE) lf.lfOutPrecision=OUT_TT_PRECIS;
    if(hints&FONTHINT_POLYMORPHIC) lf.lfOutPrecision=OUT_TT_PRECIS;
    lf.lfClipPrecision=CLIP_DEFAULT_PRECIS;
    lf.lfQuality=DEFAULT_QUALITY;
    lf.lfPitchAndFamily=DEFAULT_PITCH|FF_DONTCARE;
    if(hints&FONTPITCH_FIXED) lf.lfPitchAndFamily=FIXED_PITCH;
    if(hints&FONTPITCH_VARIABLE) lf.lfPitchAndFamily=VARIABLE_PITCH;
    if(hints&FONTHINT_DECORATIVE) lf.lfPitchAndFamily&=FF_DECORATIVE;
    if(hints&FONTHINT_MODERN) lf.lfPitchAndFamily&=FF_MODERN;
    if(hints&FONTHINT_ROMAN) lf.lfPitchAndFamily&=FF_ROMAN;
    if(hints&FONTHINT_SCRIPT) lf.lfPitchAndFamily&=FF_SCRIPT;
    if(hints&FONTHINT_SWISS) lf.lfPitchAndFamily&=FF_SWISS;
    
    // Font substitution
    if(!name.empty())
      strcpy(lf.lfFaceName,getApp()->reg().readStringEntry("FONTSUBSTITUTIONS",name.text(),name.text()));
    else
      strcpy(lf.lfFaceName,"");

    // Here we go!
    xid=CreateFontIndirect(&lf);
    if(xid){
      FXCALLOC(&font,TEXTMETRIC,1);
      SelectObject(dc,xid);
      GetTextMetrics(dc,(TEXTMETRIC*)font);
      }

#endif
    
    // Uh-oh, we failed
    if(!xid){ fxerror("%s::create: unable to create font.\n",getClassName()); }
    }
  }


// Detach font
void FXFont::detach(){
  if(xid){
    FXTRACE((100,"%s::detach %08x\n",getClassName(),this));
    FXASSERT(font);
#ifndef FX_NATIVE_WIN32
    // Free font struct w/o doing anything else...
#else
    FXFREE(&font);
#endif
    font=NULL;
    xid=0;
    }
  }


// Destroy font
void FXFont::destroy(){
  if(xid){
    FXTRACE((100,"%s::destroy %08x\n",getClassName(),this));
    FXASSERT(font);
#ifndef FX_NATIVE_WIN32
    XFreeFont(getApp()->display,(XFontStruct*)font);
#else
    DeleteObject((HFONT)xid);
    FXFREE(&font);
#endif
    font=NULL;
    xid=0;
    }
  }


// Does font have given character glyph?
FXbool FXFont::hasChar(FXchar ch) const {
  if(!font){ fxerror("%s::hasChar: font has not been created.\n",getClassName()); }
  return HASCHAR(font,ch);
  }


// Get font leading [that is lead-ing as in Pb!]
FXint FXFont::getFontLeading() const {
  if(!font){ fxerror("%s::getFontLeading: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  return ((XFontStruct*)font)->ascent+((XFontStruct*)font)->descent-((XFontStruct*)font)->max_bounds.ascent-((XFontStruct*)font)->max_bounds.descent;
#else
  return ((TEXTMETRIC*)font)->tmExternalLeading;
#endif
  }


// Get font line spacing [height+leading]
FXint FXFont::getFontSpacing() const {
  if(!font){ fxerror("%s::getFontSpacing: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  return ((XFontStruct*)font)->ascent+((XFontStruct*)font)->descent;
#else
  return ((TEXTMETRIC*)font)->tmHeight; // tmHeight includes font point size plus internal leading
#endif
  }


// Left bearing
FXint FXFont::leftBearing(FXchar ch) const {
  if(!font){ fxerror("%s::leftBearing: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  if(((XFontStruct*)font)->per_char){
    if(!HASCHAR(font,ch)){ ch=((XFontStruct*)font)->default_char; }
    return ((XFontStruct*)font)->per_char[(FXuint)ch-((XFontStruct*)font)->min_char_or_byte2].lbearing;
    }
  return ((XFontStruct*)font)->max_bounds.lbearing;
#else
  return 0; /// FIX ME
#endif
  }


// Right bearing
FXint FXFont::rightBearing(FXchar ch) const {
  if(!font){ fxerror("%s::rightBearing: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  if(((XFontStruct*)font)->per_char){
    if(!HASCHAR(font,ch)){ ch=((XFontStruct*)font)->default_char; }
    return ((XFontStruct*)font)->per_char[(FXuint)ch-((XFontStruct*)font)->min_char_or_byte2].rbearing;
    }
  return ((XFontStruct*)font)->max_bounds.rbearing;
#else
  return 0; /// FIX ME
#endif
  }


// Is it a mono space font
FXbool FXFont::isFontMono() const {
  if(!font){ fxerror("%s::isFontMono: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  return ((XFontStruct*)font)->per_char==NULL;
#else
  return !(((TEXTMETRIC*)font)->tmPitchAndFamily&TMPF_FIXED_PITCH);
#endif
  }


// Get font width
FXint FXFont::getFontWidth() const {
  if(!font){ fxerror("%s::getFontWidth: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  return ((XFontStruct*)font)->max_bounds.width;
#else
  return ((TEXTMETRIC*)font)->tmMaxCharWidth;
#endif
  }


// Get font height
FXint FXFont::getFontHeight() const {
  if(!font){ fxerror("%s::getFontHeight: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  return ((XFontStruct*)font)->ascent+((XFontStruct*)font)->descent;  // This is wrong!
  //return ((XFontStruct*)font)->max_bounds.ascent+((XFontStruct*)font)->max_bounds.descent;  // This is right!
#else
  return ((TEXTMETRIC*)font)->tmHeight;
#endif
  }


// Get font ascent
FXint FXFont::getFontAscent() const {
  if(!font){ fxerror("%s::getFontAscent: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  return ((XFontStruct*)font)->ascent;
#else
  return ((TEXTMETRIC*)font)->tmAscent;
#endif
  }


// Get font descent
FXint FXFont::getFontDescent() const {
  if(!font){ fxerror("%s::getFontDescent: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  return ((XFontStruct*)font)->descent;
#else
  return ((TEXTMETRIC*)font)->tmDescent;
#endif
  }


// Text width
FXint FXFont::getTextWidth(const FXchar *text,FXuint n) const {
  if(!text && n){ fxerror("%s::getTextWidth: NULL string argument\n",getClassName()); }
  if(!font){ fxerror("%s::getTextWidth: font has not been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  return XTextWidth((XFontStruct*)font,text,n);
#else
  SIZE size;
  FXASSERT(dc!=NULL);
  GetTextExtentPoint32(dc,text,n,&size);
  return size.cx;
#endif
  }


// Text height
FXint FXFont::getTextHeight(const FXchar *text,FXuint n) const {
  if(!font){ fxerror("%s::getTextHeight: font has not been created.\n",getClassName()); }
  if(!text && n){ fxerror("%s::getTextHeight: NULL string argument\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  XCharStruct chst; int dir,asc,desc;
  XTextExtents((XFontStruct*)font,text,n,&dir,&asc,&desc,&chst);
  return asc+desc;
#else
  SIZE size;
  FXASSERT(dc!=NULL);
  GetTextExtentPoint32(dc,text,n,&size);
  return size.cy;
#endif
  }


/*******************************************************************************/


// Function to sort by name, weight, slant, and size
static int comparefont(const void * a,const void *b){
  register FXFontDesc *fa=(FXFontDesc*)a;
  register FXFontDesc *fb=(FXFontDesc*)b;
#ifndef FX_NATIVE_WIN32
  int cmp=strcmp(fa->face,fb->face);
  return cmp>0 || (cmp==0 && (fa->weight>fb->weight || (fa->weight==fb->weight && (fa->slant > fb->slant || (fa->slant==fb->slant && fa->size>fb->size)))));
#else
  return strcmp(fa->face,fb->face);
#endif
  }


#ifndef FX_NATIVE_WIN32
  
// List all fonts matching hints
FXbool FXFont::listFonts(FXFontDesc*& fonts,FXuint& numfonts,const FXString& face,FXuint wt,FXuint sl,FXuint sw,FXuint en,FXuint h){
  const char* charsetencoding[]={"*-*","iso8859-1","iso8859-2","iso8859-3","iso8859-4","iso8859-5","iso8859-6","iso8859-7","iso8859-8","iso8859-11","koi8-r"};
  int screenres,xres,yres;
  FXuint size,weight,slant,encoding,setwidth,flags;   
  char pattern[300],fname[300],*field[14];  
  char **fnames,*scal,*pit,*wgt,*slt,*set,*reg,*enc;
  const char *facename;
  int numfnames,f,j,addit;

  fonts=NULL;
  numfonts=0;
  
  // Gotta have display open!
  if(!FXApp::app){ fxerror("FXFont::listFonts: no application object.\n"); }
  if(!FXApp::app->display){ fxerror("FXFont::listFonts: trying to list fonts before opening display.\n"); }

  // Screen resolution may be overidden by registry
  screenres=FXApp::app->reg().readUnsignedEntry("SETTINGS","screenxres",100);

  // Validate
  if(screenres<50) screenres=50;
  if(screenres>200) screenres=200;

  FXTRACE((150,"Listing Fonts for screenres=%d:\n",screenres));
  
  if(en>FONTENCODING_KOI_8) en=FONTENCODING_DEFAULT;

  // Define pattern to match against
  if(h&FONTHINT_X11){
    facename="*";
    if(!face.empty()) facename=face.text();
    strcpy(pattern,facename);
    }
  
  // Match XLFD fonts; try to limit the number by using
  // some of the info we already have acquired.
  else{
    scal="*";
    if(h&FONTHINT_SCALABLE) scal="0";
    facename="*";
    if(!face.empty()) facename=face.text();
    sprintf(pattern,"-*-%s-*-*-*-*-%s-%s-*-*-*-%s-%s",facename,scal,scal,scal,charsetencoding[en]);
    }
  
  // Get list of all font names
  fnames=listfontnames(FXApp::app->display,pattern,numfnames);
  if(!fnames) return FALSE;
  
  // Make room to receive face names
  FXMALLOC(&fonts,FXFontDesc,numfnames);
  if(!fonts){ XFreeFontNames(fnames); return FALSE; }

  // Add all matching fonts to the list
  for(f=0; f<numfnames; f++){
    strncpy(fname,fnames[f],299);
    
    // XLFD font name; parse out unique face names
    if(parsefontname(field,fname)){
      
      flags=0;
      
      // Get pitch
      pit=field[XLFD_SPACING];
      if(pit[0]=='m' || pit[0]=='M' || pit[0]=='c' || pit[1]=='C') flags|=FONTPITCH_FIXED;
      else if(pit[0]=='p' || pit[0]=='P') flags|=FONTPITCH_VARIABLE;
      
      // Skip this font if pitch does not match
      if((h&FONTPITCH_FIXED) && !(flags&FONTPITCH_FIXED)) continue;
      if((h&FONTPITCH_VARIABLE) && !(flags&FONTPITCH_VARIABLE)) continue;
      
      // Get weight
      wgt=field[XLFD_WEIGHT];
      if(fxstrcasecmp(wgt,"medium")==0) weight=FONTWEIGHT_MEDIUM;
      else if(fxstrcasecmp(wgt,"bold")==0) weight=FONTWEIGHT_BOLD;
      else if(fxstrcasecmp(wgt,"demibold")==0) weight=FONTWEIGHT_DEMIBOLD;
      else if(fxstrcasecmp(wgt,"demi")==0) weight=FONTWEIGHT_DEMIBOLD;
      else if(fxstrcasecmp(wgt,"black")==0) weight=FONTWEIGHT_BLACK;
      else if(fxstrcasecmp(wgt,"light")==0) weight=FONTWEIGHT_LIGHT;
      else if(fxstrcasecmp(wgt,"regular")==0) weight=FONTWEIGHT_REGULAR;
      else if(fxstrcasecmp(wgt,"normal")==0) weight=FONTWEIGHT_NORMAL;
      else weight=FONTWEIGHT_DONTCARE;
      
      // Skip if no match
      if((wt!=FONTWEIGHT_DONTCARE) && (wt!=weight)) continue;
      
      // Get slant
      slt=field[XLFD_SLANT];
      if(slt[0]=='i' || slt[0]=='I') slant=FONTSLANT_ITALIC;
      else if(slt[0]=='o' || slt[0]=='O') slant=FONTSLANT_OBLIQUE;
      else if((slt[0]=='r' || slt[0]=='R') && (slt[1]=='i'|| slt[1]=='I')) slant=FONTSLANT_REVERSE_ITALIC;
      else if((slt[0]=='r' || slt[0]=='R') && (slt[1]=='o'|| slt[1]=='O')) slant=FONTSLANT_REVERSE_OBLIQUE;
      else if(slt[0]=='r' || slt[0]=='R') slant=FONTSLANT_REGULAR;
      else slant=FONTSLANT_DONTCARE;
      
      // Skip if no match
      if((sl!=FONTSLANT_DONTCARE) && (sl!=slant)) continue;
      
      // Get SetWidth
      set=field[XLFD_SETWIDTH];
      if(fxstrcasecmp(set,"ultracondensed")==0) setwidth=FONTSETWIDTH_ULTRACONDENSED;
      else if(fxstrcasecmp(set,"extracondensed")==0) setwidth=FONTSETWIDTH_EXTRACONDENSED;
      else if(fxstrcasecmp(set,"condensed")==0) setwidth=FONTSETWIDTH_CONDENSED;
      else if(fxstrcasecmp(set,"narrow")==0) setwidth=FONTSETWIDTH_CONDENSED;
      else if(fxstrcasecmp(set,"compressed")==0) setwidth=FONTSETWIDTH_CONDENSED;
      else if(fxstrcasecmp(set,"semicondensed")==0) setwidth=FONTSETWIDTH_SEMICONDENSED;
      else if(fxstrcasecmp(set,"medium")==0) setwidth=FONTSETWIDTH_MEDIUM;
      else if(fxstrcasecmp(set,"normal")==0) setwidth=FONTSETWIDTH_MEDIUM;
      else if(fxstrcasecmp(set,"regular")==0) setwidth=FONTSETWIDTH_MEDIUM;
      else if(fxstrcasecmp(set,"semiexpanded")==0) setwidth=FONTSETWIDTH_SEMIEXPANDED;
      else if(fxstrcasecmp(set,"demiexpanded")==0) setwidth=FONTSETWIDTH_SEMIEXPANDED;
      else if(fxstrcasecmp(set,"expanded")==0) setwidth=FONTSETWIDTH_EXPANDED;
      else if(fxstrcasecmp(set,"wide")==0) setwidth=FONTSETWIDTH_EXTRAEXPANDED;
      else if(fxstrcasecmp(set,"extraexpanded")==0) setwidth=FONTSETWIDTH_EXTRAEXPANDED;
      else if(fxstrcasecmp(set,"ultraexpanded")==0) setwidth=FONTSETWIDTH_ULTRAEXPANDED;
      else setwidth=FONTSETWIDTH_DONTCARE;
      
      // Skip if no match
      if((sw!=FONTSETWIDTH_DONTCARE) && (sw!=setwidth)) continue;
      
      // Get encoding
      reg=field[XLFD_REGISTRY];
      enc=field[XLFD_ENCODING];
      if(ISISO8859(reg)){
        if(EQUAL1(enc,'1')) encoding=FONTENCODING_ISO_8859_1;
        else if(EQUAL1(enc,'2')) encoding=FONTENCODING_ISO_8859_2;
        else if(EQUAL1(enc,'3')) encoding=FONTENCODING_ISO_8859_2;
        else if(EQUAL1(enc,'4')) encoding=FONTENCODING_ISO_8859_4;
        else if(EQUAL1(enc,'5')) encoding=FONTENCODING_ISO_8859_5;
        else if(EQUAL1(enc,'6')) encoding=FONTENCODING_ISO_8859_6;
        else if(EQUAL1(enc,'7')) encoding=FONTENCODING_ISO_8859_7;
        else if(EQUAL1(enc,'8')) encoding=FONTENCODING_ISO_8859_8;
        else if(EQUAL2(enc,'1','1')) encoding=FONTENCODING_ISO_8859_11;
        else encoding=FONTENCODING_DEFAULT;
        }
      else if(strcmp(reg,"koi8")==0){
        if(EQUAL1(enc,'r')) encoding=FONTENCODING_KOI_8;
        else encoding=FONTENCODING_DEFAULT;
        }
      else{
        encoding=FONTENCODING_DEFAULT;
        }
      
      // Skip if no match
      if((en!=FONTENCODING_DEFAULT) && (en!=encoding)) continue;
      
      // Scalable
      if(EQUAL1(field[XLFD_PIXELSIZE],'0') && EQUAL1(field[XLFD_POINTSIZE],'0') && EQUAL1(field[XLFD_AVERAGE],'0')){
        flags|=FONTHINT_SCALABLE;
        }
      
      // Polymorphic
      if(EQUAL1(field[XLFD_WEIGHT],'0') || EQUAL1(field[XLFD_SETWIDTH],'0') || EQUAL1(field[XLFD_SLANT],'0') || EQUAL1(field[XLFD_ADDSTYLE],'0')){
        flags|=FONTHINT_POLYMORPHIC;
        }
        
      // Get Font resolution
      if(EQUAL1(field[XLFD_RESOLUTION_X],'0') && EQUAL1(field[XLFD_RESOLUTION_Y],'0')){
        xres=screenres;
        yres=screenres;
        }
      else{
        xres=atoi(field[XLFD_RESOLUTION_X]);
        yres=atoi(field[XLFD_RESOLUTION_Y]);
        }

      // Get size, corrected for screen resolution
      if(!(flags&FONTHINT_SCALABLE)){
        size=(yres*atoi(field[XLFD_POINTSIZE]))/screenres;    
        }
      else{
        size=0;
        }

      // Dump what we have found out
      FXTRACE((160,"Font=%s weight=%d slant=%d size=%3d setwidth=%d encoding=%d\n",field[XLFD_FAMILY],weight,slant,size,setwidth,encoding));
      
      addit=1;
      
      // If NULL face name, just list one of each face
      if(face.empty()){
        for(j=numfonts-1; j>=0; j--){
          if(strcmp(field[XLFD_FAMILY],fonts[j].face)==0){
            addit=0;
            break;
            }
          }
        }

      if(addit){
        strncpy(fonts[numfonts].face,field[XLFD_FAMILY],31); 
        fonts[numfonts].face[31]=0;
        fonts[numfonts].size=size;
        fonts[numfonts].weight=weight;
        fonts[numfonts].slant=slant;
        fonts[numfonts].encoding=encoding;
        fonts[numfonts].setwidth=setwidth;
        fonts[numfonts].flags=flags;
        numfonts++;
        }
      }
    
    // X11 font, add it to the list
    else{
      strncpy(fonts[numfonts].face,fnames[f],31);
      fonts[numfonts].face[31]=0;
      fonts[numfonts].size=0;
      fonts[numfonts].weight=FONTWEIGHT_DONTCARE;
      fonts[numfonts].slant=FONTSLANT_DONTCARE;
      fonts[numfonts].encoding=FONTENCODING_DEFAULT;
      fonts[numfonts].setwidth=FONTSETWIDTH_DONTCARE;
      fonts[numfonts].flags=FONTHINT_X11;
      numfonts++;
      }
    }
  
  // Any fonts found?
  if(numfonts==0){
    FXFREE(&fonts);
    XFreeFontNames(fnames);
    return FALSE;
    }
  
  // Realloc to shrink the block
  FXRESIZE(&fonts,FXFontDesc,numfonts);
  
  // Sort them by name, weight, slant, and size respectively
  qsort(fonts,numfonts,sizeof(FXFontDesc),comparefont);
  
//   FXTRACE((150,"%d fonts:\n",numfonts));
//   for(f=0; f<numfonts; f++){
//     FXTRACE((150,"Font=%s weight=%d slant=%d size=%3d setwidth=%d encoding=%d\n",fonts[f].face,fonts[f].weight,fonts[f].slant,fonts[f].size,fonts[f].setwidth,fonts[f].encoding));
//     }
//   FXTRACE((150,"\n\n"));
  
  // Free the font names
  XFreeFontNames(fnames);
  return TRUE;
  }  


#else


// Yuk. Need to get some data into the callback function.
struct FXFontStore {
  HDC hdc;
  FXFontDesc *fonts;
  FXuint numfonts;
  FXuint size;
  FXFontDesc desc;
  };

  
// Callback function for EnumFontFamiliesEx()
static int CALLBACK EnumFontFamExProc(const LOGFONT *lf,const TEXTMETRIC *lptm,DWORD FontType,LPARAM lParam){
  FXASSERT(lf!=0);
  FXASSERT(lptm!=0);
  FXFontStore *pFontStore=reinterpret_cast<FXFontStore*>(lParam);
  FXASSERT(pFontStore!=0);

  // Get pitch
  FXuint flags=FONTPITCH_DEFAULT;
  if(lf->lfPitchAndFamily&FIXED_PITCH) flags|=FONTPITCH_FIXED;
  if(lf->lfPitchAndFamily&VARIABLE_PITCH) flags|=FONTPITCH_VARIABLE;

  // Skip if no match
  FXuint h=pFontStore->desc.flags;
  if((h&FONTPITCH_FIXED) && !(flags&FONTPITCH_FIXED)) return 1;
  if((h&FONTPITCH_VARIABLE) && !(flags&FONTPITCH_VARIABLE)) return 1;
  
  // Get weight (also guess from the name)
  FXuint weight=lf->lfWeight;
  if(strstr(lf->lfFaceName," Bold")!=NULL) weight=FONTWEIGHT_BOLD;
  if(strstr(lf->lfFaceName," Demi")!=NULL) weight=FONTWEIGHT_DEMIBOLD;
  if(strstr(lf->lfFaceName," Light")!=NULL) weight=FONTWEIGHT_LIGHT;
  if(strstr(lf->lfFaceName," Medium")!=NULL) weight=FONTWEIGHT_MEDIUM;

  // Skip if weight doesn't match
  FXuint wt=pFontStore->desc.weight;
  if((wt!=FONTWEIGHT_DONTCARE) && (wt!=weight)) return 1;

  // Get slant
  FXuint slant=FONTSLANT_REGULAR;
  if(lf->lfItalic==TRUE) slant=FONTSLANT_ITALIC;
  if(strstr(lf->lfFaceName," Italic")!=NULL) slant=FONTSLANT_ITALIC;
  if(strstr(lf->lfFaceName," Roman")!=NULL) slant=FONTSLANT_REGULAR;

  // Skip if no match
  FXuint sl=pFontStore->desc.slant;
  if((sl!=FONTSLANT_DONTCARE) && (sl!=slant)) return 1;

  // Get set width (also guess from the name)
  FXuint setwidth=FONTSETWIDTH_DONTCARE;
  if(strstr(lf->lfFaceName," Cond")!=NULL) setwidth=FONTSETWIDTH_CONDENSED;
  if(strstr(lf->lfFaceName," Ext Cond")!=NULL) setwidth=FONTSETWIDTH_EXTRACONDENSED;

  // Skip if no match
  FXuint sw=pFontStore->desc.setwidth;
  if((sw!=FONTSETWIDTH_DONTCARE) && (sw!=setwidth)) return 1;

  // Get encoding
  FXuint encoding=CharSet2FXFontEncoding(lf->lfCharSet);

  // Skip if no match
  FXuint en=pFontStore->desc.encoding;
  if((en!=FONTENCODING_DEFAULT) && (en!=encoding)) return 1;

  // Is it scalable?
  if(FontType==TRUETYPE_FONTTYPE){
    flags|=FONTHINT_SCALABLE;
    }

  // Is it polymorphic?
  if(FontType==TRUETYPE_FONTTYPE){
    flags|=FONTHINT_POLYMORPHIC;
    }

  // Initial allocation of storage?
  if(pFontStore->numfonts==0){
    FXMALLOC(&pFontStore->fonts,FXFontDesc,50);
    if(pFontStore->fonts==0) return 0;
    pFontStore->size=50;
    }
  
  // Grow the array if needed
  if(pFontStore->numfonts>=pFontStore->size){
    FXRESIZE(&pFontStore->fonts,FXFontDesc,pFontStore->size+50);
    if(pFontStore->fonts==0) return 0;
    pFontStore->size+=50;
    }

  FXFontDesc *fonts=pFontStore->fonts;
  FXuint numfonts=pFontStore->numfonts;

  strncpy(fonts[numfonts].face,lf->lfFaceName,31);
  fonts[numfonts].face[31]=0;
  fonts[numfonts].size=fxheight_to_pointsize(pFontStore->hdc,lf->lfHeight);
  fonts[numfonts].weight=weight;
  fonts[numfonts].slant=slant;
  fonts[numfonts].encoding=encoding;
  fonts[numfonts].setwidth=setwidth;
  fonts[numfonts].flags=flags;

  pFontStore->fonts=fonts;
  pFontStore->numfonts++;

  // Must return 1 to continue enumerating fonts
  return 1;
  }


// List all fonts matching hints
FXbool FXFont::listFonts(FXFontDesc*& fonts,FXuint& numfonts,const FXString& face,FXuint wt,FXuint sl,FXuint sw,FXuint en,FXuint h){
  // Initialize return values
  fonts=NULL;
  numfonts=0;

  // This data gets passed into the callback function
  FXFontStore fontStore;
  HDC hdc=GetDC(GetDesktopWindow());
  SaveDC(hdc);
  fontStore.hdc=hdc;
  fontStore.fonts=fonts;
  fontStore.numfonts=numfonts;
  fontStore.desc.weight=wt;
  fontStore.desc.slant=sl;
  fontStore.desc.setwidth=sw;
  fontStore.desc.encoding=en;
  fontStore.desc.flags=h;

  // Fill in the appropriate fields of the LOGFONT structure. Note that
  // EnumFontFamiliesEx() only examines the lfCharSet, lfFaceName and
  // lpPitchAndFamily fields of this struct.
  LOGFONT lf;
  lf.lfCharSet=FXFontEncoding2CharSet(en);
  if(face.text()){
    FXASSERT(face.length()<LF_FACESIZE);
    strcpy(lf.lfFaceName,face.text());
    }
  else
    strcpy(lf.lfFaceName,"");
  lf.lfPitchAndFamily=0; // Should be MONO_FONT for Hebrew and Arabic?

  // Start enumerating!
  EnumFontFamiliesEx(hdc,&lf,EnumFontFamExProc,reinterpret_cast<LPARAM>(&fontStore),0);
  RestoreDC(hdc,-1);
  ReleaseDC(GetDesktopWindow(),hdc);

  // Copy stuff back from the store
  fonts=fontStore.fonts;
  numfonts=fontStore.numfonts;

  // Any fonts found?
  if(numfonts==0){
    FXFREE(&fonts);
    return FALSE;
    }

  // Realloc to shrink the block
  FXRESIZE(&fonts,FXFontDesc,numfonts);
  
  // Sort them by name, weight, slant, and size respectively
  qsort(fonts,numfonts,sizeof(FXFontDesc),comparefont);
  
//   FXTRACE((150,"%d fonts:\n",numfonts));
//   for(FXuint f=0; f<numfonts; f++){
//     FXTRACE((150,"Font=%s weight=%d slant=%d size=%3d setwidth=%d encoding=%d\n",fonts[f].face,fonts[f].weight,fonts[f].slant,fonts[f].size,fonts[f].setwidth,fonts[f].encoding));
//     }
//   FXTRACE((150,"\n\n"));

  return TRUE;
  }

#endif


/*******************************************************************************/


// Save font to stream
void FXFont::save(FXStream& store) const {
  FXId::save(store);
  store << name;
  store << size;
  store << weight;
  store << slant;
  store << encoding;
  store << setwidth;
  store << hints;
  }


// Load font from stream; create() should be called later
void FXFont::load(FXStream& store){
  FXId::load(store);
  store >> name;
  store >> size;
  store >> weight;
  store >> slant;
  store >> encoding;
  store >> setwidth;
  store >> hints;
  }  


// Clean up
FXFont::~FXFont(){
  FXTRACE((100,"FXFont::~FXFont %08x\n",this));
#ifndef FX_NATIVE_WIN32
  if(font){XFreeFont(getApp()->display,(XFontStruct*)font);}
#else
  if(font){FXFREE(&font);DeleteObject((HFONT)xid);}
  DeleteDC(dc);
#endif
  font=(void*)-1;
  }
  
