/********************************************************************************
*                                                                               *
*                            V i s u a l   C l a s s                            *
*                                                                               *
*********************************************************************************
* Copyright (C) 1999 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXVisual.cpp,v 1.3 1999/11/02 22:15:15 jeroen Exp $                      *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXObjectList.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"


/*
  Notes:
  
  - Purpose is to provide the illusion of having True Color on all systems.
  - The visual is what determines how an RGB tuple maps to a color on the device.
  - Thus, the visual also knows the colormap, and perhaps other lookup mechanisms.
  - When a drawable is created, it is created with a certain visual.
  - When painting, graphic contexts are specific for a certain visual; hence,
    the visual should probably keep track of the gc's.
  - FIRST set up pseudo color ramp, then read it back and set up the tables.
    This way, table setup can be reused for read-only colormaps [StaticColor]
    also...
  - We try to match the deepest visual not deeper than the specified depth.
  - For some info on visual setup, see:
  
	http://www.wolfram.com/~cwikla/articles/txa/visual.html
	http://www.wolfram.com/~cwikla/articles/txa/xcc.1.html
	http://www.wolfram.com/~cwikla/articles/txa/xcc.2.html
        
  - Freshly constructed FXVisual sets maxcolors to 1000000 in anticipation
    of private colormap. [FXApp however sets the default FXVisual's maximum
    to a lower value as the colormap is shared between lots of apps].
  - Always check for Standard Colormap first [even for default colormap], as
    that (a) makes initialization simpler, and (b) may give us the preferred
    colors to grab on that system [*** Too bad this does not work! ***].
  - Find closest depth to the given depth hint
  - RGB Ordering:
  
       RGB 111      > | R  G  B
       BGR 000      --+--------
       RBG 110      R | x  4  2
       GBR 001      G |    x  1
       BRG 100      B |       x
       GRB 011

  - SGI Infinite Reality may have up to 12 bits for red, green, blue each!
 */



/*******************************************************************************/

#ifndef FX_NATIVE_WIN32

// Standard dither kernel
const FXuint FXVisual::dither[16]={
   0*16,  8*16,  2*16, 10*16,
  12*16,  4*16, 14*16,  6*16,
   3*16, 11*16,  1*16,  9*16,
  15*16,  7*16, 13*16,  5*16,
  };

#endif   

/*******************************************************************************/
     
     
// Object implementation
FXIMPLEMENT(FXVisual,FXId,NULL,0)



// Deserialization
FXVisual::FXVisual(){
  flags=0;
  hint=1;
  depth=0;
  info=NULL;
  numcolors=0;
  numred=0;
  numgreen=0;
  numblue=0;
  maxcolors=1000000;
  rgborder=0;
  type=VISUALTYPE_UNKNOWN;
#ifndef FX_NATIVE_WIN32
  visual=NULL;
  colormap=0;
  alloced=NULL;
  nalloced=0;
  red_to_pix=NULL;
  green_to_pix=NULL;
  blue_to_pix=NULL;
  pix_to_red=NULL;
  pix_to_green=NULL;
  pix_to_blue=NULL;
  redmax=0;
  greenmax=0;
  bluemax=0;
  redmask=0;
  greenmask=0;
  bluemask=0;
  redshift=0;
  greenshift=0;
  blueshift=0;
  freemap=FALSE;
  gc=0;
#else
  hPalette=NULL;
#endif
  }


// Construct 
FXVisual::FXVisual(FXApp* a,FXuint flgs,FXuint d):FXId(a,0){
  FXTRACE((100,"FXVisual::FXVisual %08x\n",this));
  flags=flgs;
  hint=FXMAX(1,d);
  depth=0;
  info=NULL;
  numcolors=0;
  numred=0;
  numgreen=0;
  numblue=0;
  maxcolors=1000000;
  rgborder=0;
  type=VISUALTYPE_UNKNOWN;
#ifndef FX_NATIVE_WIN32
  visual=NULL;
  colormap=0;
  alloced=NULL;
  nalloced=0;
  red_to_pix=NULL;
  green_to_pix=NULL;
  blue_to_pix=NULL;
  pix_to_red=NULL;
  pix_to_green=NULL;
  pix_to_blue=NULL;
  redmax=0;
  greenmax=0;
  bluemax=0;
  redmask=0;
  greenmask=0;
  bluemask=0;
  redshift=0;
  greenshift=0;
  blueshift=0;
  freemap=FALSE;
  gc=0;
#else
  hPalette=NULL;
  info=NULL;
  pixelformat=0;
#endif
  }


#ifndef FX_NATIVE_WIN32

/*******************************************************************************/

// X11 Internal helper functions


// Make GC for this visual
GC FXVisual::makegc(){
  XGCValues gval;
  FXID drawable;
  GC gg;
  
  gval.fill_style=FillSolid;
  gval.graphics_exposures=True;
  
  // Monochrome gc; create a temporary pixmap of depth 1
  if(visual==NULL){
    FXTRACE((100,"%s::init: gc for monochrome pixmap\n",getClassName()));
    drawable=XCreatePixmap(getApp()->display,XDefaultRootWindow(getApp()->display),1,1,1);
    gg=XCreateGC(getApp()->display,drawable,GCFillStyle|GCGraphicsExposures,&gval);
    XFreePixmap(getApp()->display,drawable);
    }

  // For default visual; this is easy as we already have a matching window
  else if(visual==DefaultVisual(getApp()->display,DefaultScreen(getApp()->display))){
    FXTRACE((100,"%s::init: gc for default visual\n",getClassName()));
    gg=XCreateGC(getApp()->display,XDefaultRootWindow(getApp()->display),GCFillStyle|GCGraphicsExposures,&gval);
    }

  // For arbitrary visual; create a temporary pixmap of the same depth as the visual
  else{
    FXTRACE((100,"%s::init: gc for non-default visual\n",getClassName()));
    drawable=XCreatePixmap(getApp()->display,XDefaultRootWindow(getApp()->display),1,1,depth);
    gg=XCreateGC(getApp()->display,drawable,GCFillStyle|GCGraphicsExposures,&gval);
    XFreePixmap(getApp()->display,drawable);
    }

  return gg;
  }


// Find shift amount
static inline FXuint findshift(unsigned long mask){
  register FXuint sh=0;
  while(!(mask&(1<<sh))) sh++;
  return sh;
  }


// Get number of bits in n
static inline FXuint findnbits(unsigned long n){
  register FXuint nb=0;
  while(n){nb+=(n&1);n>>=1;}
  return nb;
  }



// Setup for true color
void FXVisual::setuptruecolor(){
  FXuint i,mapsize;
  
  // Get map size
  mapsize=visual->map_entries;

  // Arrangement of pixels
  redmask=visual->red_mask;
  greenmask=visual->green_mask;
  bluemask=visual->blue_mask;
  redshift=findshift(redmask);
  greenshift=findshift(greenmask);
  blueshift=findshift(bluemask);
  if(redmask>greenmask) rgborder|=4;
  if(redmask>bluemask) rgborder|=2;
  if(greenmask>bluemask) rgborder|=1;
  redmax=redmask>>redshift;
  greenmax=greenmask>>greenshift;
  bluemax=bluemask>>blueshift;

  // Allocate tables 
  FXMALLOC(&pix_to_red,FXuchar,redmax+1);
  FXMALLOC(&pix_to_green,FXuchar,greenmax+1);
  FXMALLOC(&pix_to_blue,FXuchar,bluemax+1);
  
  numred=redmax+1;
  numgreen=greenmax+1;
  numblue=bluemax+1;
  numcolors=numred*numgreen*numblue;
  
  // Allocate tables for forward map
  FXMALLOC(&red_to_pix,FXPixel,numred);
  FXMALLOC(&green_to_pix,FXPixel,numgreen);
  FXMALLOC(&blue_to_pix,FXPixel,numblue);
  
  // RGB -> Pixel maps
  for(i=0; i<=redmax; i++) red_to_pix[i] = i<<redshift;
  for(i=0; i<=greenmax; i++) green_to_pix[i] = i<<greenshift;
  for(i=0; i<=bluemax; i++) blue_to_pix[i] = i<<blueshift;
   
  // Fill in reverse pixel to color mapping tables
  for(i=0; i<=redmax; i++){ pix_to_red[i]=(255*i)/redmax; }
  for(i=0; i<=greenmax; i++){ pix_to_green[i]=(255*i)/greenmax; }
  for(i=0; i<=bluemax; i++){ pix_to_blue[i]=(255*i)/bluemax; }
    
  // What did we get
  FXTRACE((100,"True color:\n"));
  FXTRACE((100,"  visual id    = 0x%02x\n",visual->visualid));
  FXTRACE((100,"  depth        = %d\n",depth));
  FXTRACE((100,"  rgborder     = %d\n",rgborder));
  FXTRACE((100,"  map_entries  = %d\n",mapsize));
  FXTRACE((100,"  numcolors    = %d\n",numcolors));
  FXTRACE((100,"  redmax       = %3d; redmask   =%08x; redshift   = %-2d\n",redmax,redmask,redshift));
  FXTRACE((100,"  greenmax     = %3d; greenmask =%08x; greenshift = %-2d\n",greenmax,greenmask,greenshift));
  FXTRACE((100,"  bluemax      = %3d; bluemask  =%08x; blueshift  = %-2d\n",bluemax,bluemask,blueshift));

  // Set type  
  type=VISUALTYPE_TRUE;
  }



// Setup direct color
void FXVisual::setupdirectcolor(){
  XColor *table,color;
  FXuint allocedcolor,mapsize,maxcols,i,j,r,g,b,emax,rr,gg,bb;
  FXuint bestmatchr,bestmatchg,bestmatchb;
  FXdouble mindist,dist;
  FXuint gottable=0;
  FXuint rm,gm,bm,em;
  
  mapsize=visual->map_entries;

  // Arrangement of pixels
  redmask=visual->red_mask;
  greenmask=visual->green_mask;
  bluemask=visual->blue_mask;
  redshift=findshift(redmask);
  greenshift=findshift(greenmask);
  blueshift=findshift(bluemask);
  if(redmask>greenmask) rgborder|=4;
  if(redmask>bluemask) rgborder|=2;
  if(greenmask>bluemask) rgborder|=1;
  redmax=redmask>>redshift;
  greenmax=greenmask>>greenshift;
  bluemax=bluemask>>blueshift;

  // Allocate tables 
  FXMALLOC(&pix_to_red,FXuchar,redmax+1);
  FXMALLOC(&pix_to_green,FXuchar,greenmax+1);
  FXMALLOC(&pix_to_blue,FXuchar,bluemax+1);
  
  rm=redmax;
  gm=greenmax;
  bm=bluemax;
  em=FXMAX3(rm,gm,bm);
 
  // Maximum number of colors to allocate
  maxcols=FXMIN(maxcolors,mapsize);
  
  // No more allocations than allowed
  if(redmax>=maxcols) redmax=maxcols-1;
  if(greenmax>=maxcols) greenmax=maxcols-1;
  if(bluemax>=maxcols) bluemax=maxcols-1;
  
  numred=redmax+1;
  numgreen=greenmax+1;
  numblue=bluemax+1;
  numcolors=numred*numgreen*numblue;
  emax=FXMAX3(redmax,greenmax,bluemax);

  // Room for allocated pixels:- must later be freed
  FXMALLOC(&alloced,FXPixel,maxcols);
  nalloced=0;
  
  // Allocate tables for forward map
  FXMALLOC(&red_to_pix,FXPixel,numred);
  FXMALLOC(&green_to_pix,FXPixel,numgreen);
  FXMALLOC(&blue_to_pix,FXPixel,numblue);

  // Allocate color table
  FXMALLOC(&table,XColor,mapsize);

  // Allocate ramp 
  for(i=0; i<=emax; i++){   // FIXME: We should try to allocate MONOTONICALLY increasing ramp [for OpenGL]
    r=(redmax*i+emax/2)/emax;
    g=(greenmax*i+emax/2)/emax;
    b=(bluemax*i+emax/2)/emax;
    color.red=(r*65535)/redmax;
    color.green=(g*65535)/greenmax;
    color.blue=(b*65535)/bluemax;

    // First try just using XAllocColor
    allocedcolor=XAllocColor(getApp()->display,colormap,&color);
    if(allocedcolor==0){

      // Get colors in the map
      if(!gottable){
        for(j=0; j<mapsize; j++){
          rr=(rm*j+em/2)/em;
          gg=(gm*j+em/2)/em;
          bb=(bm*j+em/2)/em;
          table[j].pixel=(rr<<redshift) | (gg<<greenshift) | (bb<<blueshift);
          }
        XQueryColors(getApp()->display,colormap,table,mapsize);
        gottable=1;
        }

      // Find best match for red
      for(mindist=1.0E10,bestmatchr=0,j=0; j<mapsize; j++){
        dist=fabs(color.red-table[j].red);
        if(dist<mindist){ bestmatchr=j; mindist=dist; if(mindist==0.0) break; }
        }

      // Find best match for green
      for(mindist=1.0E10,bestmatchg=0,j=0; j<mapsize; j++){
        dist=fabs(color.green-table[j].green);
        if(dist<mindist){ bestmatchg=j; mindist=dist; if(mindist==0.0) break; }
        }
      
      // Find best match for blue
      for(mindist=1.0E10,bestmatchb=0,j=0; j<mapsize; j++){
        dist=fabs(color.blue-table[j].blue);
        if(dist<mindist){ bestmatchb=j; mindist=dist; if(mindist==0.0) break; }
        }
      
      // Now try to allocate this color
      color.red=table[bestmatchr].red;
      color.green=table[bestmatchg].green;
      color.blue=table[bestmatchb].blue;

      // Try to allocate the closest match color.  This should only
      // fail if the cell is read/write.  Otherwise, we're incrementing
      // the cell's reference count.
      allocedcolor=XAllocColor(getApp()->display,colormap,&color);
      if(!allocedcolor){
        color.red=table[bestmatchr].red;
        color.green=table[bestmatchg].green;
        color.blue=table[bestmatchb].blue;
        color.pixel=(table[bestmatchr].pixel&redmask) | (table[bestmatchg].pixel&greenmask) | (table[bestmatchb].pixel&bluemask);
        }
      }


    // Keep track of allocated colors
    if(allocedcolor) alloced[nalloced++]=color.pixel;

    // Split pixel into its red, green, blue parts    
    rr=color.pixel&redmask;
    gg=color.pixel&greenmask;
    bb=color.pixel&bluemask;
    
    FXTRACE((100,"Alloc %3d %3d %3d (%6d %6d %6d) pixel=%08x rpx=%08x gpx=%08x bpx=%08x\n",r,g,b,color.red,color.green,color.blue,color.pixel,rr,gg,bb));

    // RGB to pixel mapping
    red_to_pix[r]=rr;
    green_to_pix[g]=gg;
    blue_to_pix[b]=bb;
    }

  // Has to be true
  FXASSERT(nalloced<=maxcols);

  // Read back entire colormap, also colors not allocated by us
  for(j=0; j<mapsize; j++){
    rr=(rm*j+em/2)/em;
    gg=(gm*j+em/2)/em;
    bb=(bm*j+em/2)/em;
    table[j].pixel=(rr<<redshift) | (gg<<greenshift) | (bb<<blueshift);
    }
  XQueryColors(getApp()->display,colormap,table,mapsize);
  
  // Build backward ramp
  for(j=0; j<mapsize; j++){
    rr=(table[j].pixel&redmask)>>redshift;
    gg=(table[j].pixel&greenmask)>>greenshift;
    bb=(table[j].pixel&bluemask)>>blueshift;
    pix_to_red[rr]=table[j].red/257;
    pix_to_green[gg]=table[j].green/257;
    pix_to_blue[bb]=table[j].blue/257;
    }
  
  // Free table
  FXFREE(&table);

  // What did we get
  FXTRACE((100,"Direct color:\n"));
  FXTRACE((100,"  visual id    = 0x%02x\n",visual->visualid));
  FXTRACE((100,"  depth        = %d\n",depth));
  FXTRACE((100,"  rgborder     = %d\n",rgborder));
  FXTRACE((100,"  map_entries  = %d\n",mapsize));
  FXTRACE((100,"  numcolors    = %d\n",numcolors));
  FXTRACE((100,"  redmax       = %3d; redmask   =%08x; redshift   = %-2d\n",redmax,redmask,redshift));
  FXTRACE((100,"  greenmax     = %3d; greenmask =%08x; greenshift = %-2d\n",greenmax,greenmask,greenshift));
  FXTRACE((100,"  bluemax      = %3d; bluemask  =%08x; blueshift  = %-2d\n",bluemax,bluemask,blueshift));

  // Set type  
  type=VISUALTYPE_TRUE;
  }



// Setup for pseudo color
void FXVisual::setuppseudocolor(){
  XColor *table,color;
  FXdouble mindist,dist,dr,dg,db;
  FXuint r,g,b,mapsize,bestmatch,allocedcolor,maxcols,j;
  FXuint gottable=0;
  mapsize=visual->map_entries;

  // How many colors to allocate
  maxcols=FXMIN(maxcolors,mapsize);
  
  // Find a product of r*g*b which will fit the available map.
  // We prefer b+1>=g and g>=r>=b; start with 6x7x6 or 252 colors.
  numred=6;
  numgreen=7;
  numblue=6;
  while(numred*numgreen*numblue>maxcols){
    if(numblue==numred && numblue==numgreen) numblue--; 
    else if(numred==numgreen) numred--; 
    else numgreen--;
    }
  
  // Set stuff up
  numcolors=numred*numgreen*numblue;
  redmax=numred-1;
  greenmax=numgreen-1;
  bluemax=numblue-1;
    
  // Room for allocated pixels:- must later be freed
  FXMALLOC(&alloced,FXPixel,maxcols);
  nalloced=0;
  
  // Allocate tables
  FXMALLOC(&red_to_pix,FXPixel,numcolors);
  FXMALLOC(&pix_to_red,FXuchar,mapsize);
  FXMALLOC(&pix_to_green,FXuchar,mapsize);
  FXMALLOC(&pix_to_blue,FXuchar,mapsize);

  // Allocate color table
  FXMALLOC(&table,XColor,mapsize);
  
  // Allocate color ramp
  for(r=0; r<=redmax; r++){
    for(g=0; g<=greenmax; g++){
      for(b=0; b<=bluemax; b++){
    
        // Color to get
        color.red=(r*65535)/redmax;
        color.green=(g*65535)/greenmax;
        color.blue=(b*65535)/bluemax;
        
        // First try just using XAllocColor
        allocedcolor=XAllocColor(getApp()->display,colormap,&color);
        if(allocedcolor==0){

          // Get colors in the map
          if(!gottable){
            for(j=0; j<mapsize; j++) table[j].pixel=j;
            XQueryColors(getApp()->display,colormap,table,mapsize);
            gottable=1;
            }

          // Find best match
          for(mindist=1.0E10,bestmatch=0,j=0; j<mapsize; j++){
            dr=color.red-table[j].red;
            dg=color.green-table[j].green;
            db=color.blue-table[j].blue;
            dist=dr*dr+dg*dg+db*db;
            if(dist<mindist){
              bestmatch=j;
              mindist=dist;
              if(mindist==0.0) break;
              }
            }

          // Return result
          color.red=table[bestmatch].red;
          color.green=table[bestmatch].green;
          color.blue=table[bestmatch].blue;

          // Try to allocate the closest match color.  This should only
          // fail if the cell is read/write.  Otherwise, we're incrementing
          // the cell's reference count.
          allocedcolor=XAllocColor(getApp()->display,colormap,&color);
          if(!allocedcolor){
            color.pixel=bestmatch;
            color.red=table[bestmatch].red;
            color.green=table[bestmatch].green;
            color.blue=table[bestmatch].blue;
            }
          }

        // Keep track of allocated colors
        if(allocedcolor) alloced[nalloced++]=color.pixel;

        // Add color into table
        red_to_pix[mix(r,g,b)]=color.pixel;
        }
      }
    }

  // Has to be true
  FXASSERT(nalloced<=maxcols);
  
  // Read colormap just allocated, as well as colors not allocated by us
  for(j=0; j<mapsize; j++){ table[j].pixel=j; }
  XQueryColors(getApp()->display,colormap,table,mapsize);
  
  // Fill in reverse pixel to color mapping tables
  for(j=0; j<mapsize; j++){
    pix_to_red[j]=table[j].red/257;
    pix_to_green[j]=table[j].green/257;
    pix_to_blue[j]=table[j].blue/257;
    }

  FXFREE(&table);
  
  // What did we get
  FXTRACE((100,"Pseudo color display:\n"));
  FXTRACE((100,"  visual id    = 0x%02x\n",visual->visualid));
  FXTRACE((100,"  depth        = %d\n",depth));
  FXTRACE((100,"  map_entries  = %d\n",mapsize));
  FXTRACE((100,"  numcolors    = %d\n",numcolors));
  FXTRACE((100,"  redmax       = %d\n",redmax));
  FXTRACE((100,"  greenmax     = %d\n",greenmax));
  FXTRACE((100,"  bluemax      = %d\n",bluemax));

  // Set type  
  type=VISUALTYPE_INDEX;
  }



// Setup for static color
void FXVisual::setupstaticcolor(){
  FXushort rc[256],gc[256],bc[256];
  FXuint mapsize,i,nr,ng,nb,r,g,b,j;
  XColor *table;
  
  mapsize=visual->map_entries;

  // Allocate tables 
  FXMALLOC(&red_to_pix,FXPixel,mapsize);
  FXMALLOC(&pix_to_red,FXuchar,mapsize);
  FXMALLOC(&pix_to_green,FXuchar,mapsize);
  FXMALLOC(&pix_to_blue,FXuchar,mapsize);

  // Allocate color table
  FXMALLOC(&table,XColor,mapsize);
  
  // Read back table
  for(i=0; i<mapsize; i++) table[i].pixel=i;
  XQueryColors(getApp()->display,colormap,table,mapsize);
  
  // Counts
  for(i=0; i<256; i++){ rc[i]=gc[i]=bc[i]=0; red_to_pix[i]=0; }

  // See if the colors are clustering
  for(i=0; i<mapsize; i++){
    rc[table[i].red/257]++;
    gc[table[i].green/257]++;
    bc[table[i].blue/257]++;
    }
  nr=ng=nb=0;
  for(i=0; i<256; i++){
    if(rc[i]) nr++;
    if(gc[i]) ng++;
    if(bc[i]) nb++;
    }
  
  // Color space is properly partitioned
  if(nr*ng*nb<=mapsize){
    redmax=nr-1;
    greenmax=ng-1;
    bluemax=nb-1;
    for(i=0; i<mapsize; i++){
      r=(redmax*table[i].red+32767)/65535;
      g=(greenmax*table[i].green+32767)/65535;
      b=(bluemax*table[i].blue+32767)/65535;
      red_to_pix[mix(r,g,b)]=i;
      }
    }
  
  // Fill in reverse pixel to color mapping tables
  for(j=0; j<mapsize; j++){
    pix_to_red[j]=table[j].red/257;
    pix_to_green[j]=table[j].green/257;
    pix_to_blue[j]=table[j].blue/257;
    }

  FXFREE(&table);
  
  numred=redmax+1;
  numgreen=greenmax+1;
  numblue=bluemax+1;
  numcolors=numred*numgreen*numblue;
  
  // What did we get
  FXTRACE((100,"Static color:\n"));
  FXTRACE((100,"  visual id    = 0x%02x\n",visual->visualid));
  FXTRACE((100,"  depth        = %d\n",depth));
  FXTRACE((100,"  map_entries  = %d\n",mapsize));
  FXTRACE((100,"  numcolors    = %d\n",numcolors));
  FXTRACE((100,"  redmax       = %d\n",redmax));
  FXTRACE((100,"  greenmax     = %d\n",greenmax));
  FXTRACE((100,"  bluemax      = %d\n",bluemax));

  // Set type  
  type=VISUALTYPE_INDEX;
  }



// Setup for gray scale
void FXVisual::setupgrayscale(){
  XColor *table,color;
  FXdouble mindist,dist,dr,dg,db;
  FXuint allocedcolor,g,bestmatch,mapsize,maxcols,j;
  FXuint gottable=0;
  
  mapsize=visual->map_entries;

  // How many to allocate
  maxcols=FXMIN(mapsize,maxcolors);
  
  // Colors
  numcolors=maxcols;
  redmax=numcolors-1;
  numred=numcolors;
  
  // Room for allocated pixels:- must later be freed
  FXMALLOC(&alloced,FXPixel,maxcols);
  nalloced=0;
  
  // Allocate tables 
  FXMALLOC(&red_to_pix,FXPixel,numcolors);
  FXMALLOC(&pix_to_red,FXuchar,mapsize);
  FXMALLOC(&pix_to_green,FXuchar,mapsize);
  FXMALLOC(&pix_to_blue,FXuchar,mapsize);
  
  // Allocate color table
  FXMALLOC(&table,XColor,mapsize);

  // Allocate gray ramp
  for(g=0; g<numcolors; g++){
    
    // Color to get
    color.red=color.green=color.blue=(g*65535)/redmax;

    // First try just using XAllocColor
    allocedcolor=XAllocColor(getApp()->display,colormap,&color);
    if(allocedcolor==0){

      // Get colors in the map
      if(!gottable){
        for(j=0; j<mapsize; j++) table[j].pixel=j;
        XQueryColors(getApp()->display,colormap,table,mapsize);
        gottable=1;
        }

      // Find best match
      for(mindist=1.0E10,bestmatch=0,j=0; j<mapsize; j++){
        dr=color.red-table[j].red;
        dg=color.green-table[j].green;
        db=color.blue-table[j].blue;
        dist=dr*dr+dg*dg+db*db;
        if(dist<mindist){
          bestmatch=j;
          mindist=dist;
          if(mindist==0.0) break;
          }
        }

      // Return result
      color.red=table[bestmatch].red;
      color.green=table[bestmatch].green;
      color.blue=table[bestmatch].blue;

      // Try to allocate the closest match color.  This should only
      // fail if the cell is read/write.  Otherwise, we're incrementing
      // the cell's reference count.
      allocedcolor=XAllocColor(getApp()->display,colormap,&color);
      if(!allocedcolor){
        color.pixel=bestmatch;
        color.red=table[bestmatch].red;
        color.green=table[bestmatch].green;
        color.blue=table[bestmatch].blue;
        }
      }

    // Keep track of allocated colors
    if(allocedcolor) alloced[nalloced++]=color.pixel;

    // All ramps equal
    red_to_pix[g]=color.pixel;
    }
  
  // Has to be true
  FXASSERT(nalloced<=maxcols);
  
  // Read colormap just allocated, as well as colors not allocated by us
  for(j=0; j<mapsize; j++){ table[j].pixel=j; }
  XQueryColors(getApp()->display,colormap,table,mapsize);
  
  // Fill in reverse pixel to color mapping tables
  for(j=0; j<mapsize; j++){
    pix_to_red[j]=table[j].red/257;
    pix_to_green[j]=table[j].green/257;
    pix_to_blue[j]=table[j].blue/257;
    }
  
  FXFREE(&table);

  // What did we get
  FXTRACE((100,"Gray Scale:\n"));
  FXTRACE((100,"  visual id    = 0x%02x\n",visual->visualid));
  FXTRACE((100,"  depth        = %d\n",depth));
  FXTRACE((100,"  map_entries  = %d\n",mapsize));
  FXTRACE((100,"  numcolors    = %d\n",numcolors));
  FXTRACE((100,"  graymax      = %d\n",redmax));

  // Set type  
  type=VISUALTYPE_GRAY;
  }



// Setup for static gray
void FXVisual::setupstaticgray(){
  XColor *table;
  FXuint i,mapsize;
  
  mapsize=visual->map_entries;

  // Colors
  numcolors=mapsize;
  redmax=numcolors-1;
  numred=numcolors;
 
  // Allocate tables 
  FXMALLOC(&red_to_pix,FXPixel,numcolors);
  FXMALLOC(&pix_to_red,FXuchar,mapsize);
  FXMALLOC(&pix_to_green,FXuchar,mapsize);
  FXMALLOC(&pix_to_blue,FXuchar,mapsize);

  // Allocate color table
  FXMALLOC(&table,XColor,mapsize);

  // Read back fixed colormap
  for(i=0; i<mapsize; i++){ table[i].pixel=i; }
  XQueryColors(getApp()->display,colormap,table,mapsize);
  
  // RGB -> Pixel maps
  for(i=0; i<numcolors; i++) red_to_pix[i]=i;
  
  // Fill in reverse pixel to color mapping tables
  for(i=0; i<mapsize; i++){
    pix_to_red[i]=table[i].red/257;
    pix_to_green[i]=table[i].green/257;
    pix_to_blue[i]=table[i].blue/257;
    }
  
  FXFREE(&table);

  // What did we get
  FXTRACE((100,"Static Gray:\n"));
  FXTRACE((100,"  visual id    = 0x%02x\n",visual->visualid));
  FXTRACE((100,"  depth        = %d\n",depth));
  FXTRACE((100,"  map_entries  = %d\n",mapsize));
  FXTRACE((100,"  numcolors    = %d\n",numcolors));
  FXTRACE((100,"  graymax      = %d\n",numcolors-1));

  // Set type  
  type=VISUALTYPE_GRAY;
  }



// Setup for static monochrome
void FXVisual::setupstaticmono(){
  XColor table[2];
  
  // Number of colors
  numcolors=2;
  redmax=1;
  numred=2;
  
  // Allocate tables 
  FXMALLOC(&red_to_pix,FXPixel,2);
  FXMALLOC(&pix_to_red,FXuchar,2);
  FXMALLOC(&pix_to_green,FXuchar,2);
  FXMALLOC(&pix_to_blue,FXuchar,2);

  // Read colors
  table[0].pixel=0;
  table[1].pixel=1;
  XQueryColors(getApp()->display,colormap,table,2);
  
  // Find out which one's the black and which one's the white color
  if((table[0].red+table[0].green+table[0].blue) < (table[1].red+table[1].green+table[1].blue)){
    red_to_pix[0]=0;
    red_to_pix[1]=1;
    }
  else{
    red_to_pix[0]=1;
    red_to_pix[1]=0;
    }
  
  // Inverse mapping
  pix_to_red[0]=table[0].red/257;
  pix_to_green[0]=table[0].green/257;
  pix_to_blue[0]=table[0].blue/257;
  pix_to_red[1]=table[1].red/257;
  pix_to_green[1]=table[1].green/257;
  pix_to_blue[1]=table[1].blue/257;
  
  // What did we get
  FXTRACE((100,"Static monochrome:\n"));
  FXTRACE((100,"  visual id    = 0x%02x\n",visual->visualid));
  FXTRACE((100,"  depth        = %d\n",depth));
  FXTRACE((100,"  map_entries  = %d\n",2));
  FXTRACE((100,"  numcolors    = %d\n",numcolors));
  FXTRACE((100,"  black        = %d\n",red_to_pix[0]));
  FXTRACE((100,"  white        = %d\n",red_to_pix[1]));

  // Set type  
  type=VISUALTYPE_MONO;
  }



// Setup for pixmap monochrome
void FXVisual::setuppixmapmono(){
  
  // Number of colors
  numcolors=2;
  redmax=1;
  numred=2;
  
  // Allocate tables for reverse map
  FXMALLOC(&red_to_pix,FXPixel,2);
  FXMALLOC(&pix_to_red,FXuchar,2);
  FXMALLOC(&pix_to_green,FXuchar,2);
  FXMALLOC(&pix_to_blue,FXuchar,2);

  red_to_pix[0]=0;
  red_to_pix[1]=1;
  
  // Inverse mapping
  pix_to_red[0]=0;
  pix_to_green[0]=0;
  pix_to_blue[0]=0;
  pix_to_red[1]=255;
  pix_to_green[1]=255;
  pix_to_blue[1]=255;
  
  // What did we get
  FXTRACE((100,"Pixmap monochrome:\n"));
  FXTRACE((100,"  depth        = %d\n",depth));
  FXTRACE((100,"  map_entries  = %d\n",2));
  FXTRACE((100,"  numcolors    = %d\n",numcolors));
  FXTRACE((100,"  black        = 0\n"));
  FXTRACE((100,"  white        = 1\n"));

  // Set type  
  type=VISUALTYPE_MONO;
  }



// Determine colormap, then initialize it
void FXVisual::setupcolormap(){
  if(visual){
    if((flags&VISUAL_OWNCOLORMAP) || (visual!=DefaultVisual(getApp()->display,DefaultScreen(getApp()->display)))){
      colormap=XCreateColormap(getApp()->display,RootWindow(getApp()->display,0),visual,AllocNone);
      FXTRACE((100,"%s::init: allocate colormap\n",getClassName()));
      freemap=TRUE;
      }
    else{
      colormap=DefaultColormap(getApp()->display,DefaultScreen(getApp()->display));
      FXTRACE((100,"%s::init: use default colormap\n",getClassName()));
      }
    if(visual->c_class==DirectColor) setupdirectcolor();
    else if(visual->c_class==TrueColor) setuptruecolor();
    else if(visual->c_class==PseudoColor) setuppseudocolor();
    else if(visual->c_class==StaticColor) setupstaticcolor();
    else if(visual->c_class==GrayScale) setupgrayscale();
    else if(visual->c_class==StaticGray && visual->map_entries==2) setupstaticmono();
    else setupstaticgray();
    }
  else{
    FXTRACE((100,"%s::init: need no colormap\n",getClassName()));
    setuppixmapmono();
    }
  }

#else

/*******************************************************************************/

// Creates an "all-purpose" palette
// From "Programming Windows", 5th ed. by Charles Petzold
HPALETTE FXVisual::createAllPurposePalette(){
  LOGPALETTE *plp;
  HPALETTE hPalette=NULL;
  if(FXMALLOC(&plp,BYTE,sizeof(LOGPALETTE)+246*sizeof(PALETTEENTRY))){
    plp->palVersion=0x0300;
    plp->palNumEntries=247;

    // Calculate 31 gray shades, 3 of which match the standard 20 colors
    int i,G,incr;
    for(i=0, G=0, incr=8; G<=0xff; i++, G+=incr){
      plp->palPalEntry[i].peRed=(BYTE)G;
      plp->palPalEntry[i].peGreen=(BYTE)G;
      plp->palPalEntry[i].peBlue=(BYTE)G;
      plp->palPalEntry[i].peFlags=0;
      incr=(incr==9 ? 8 : 9);
      }

    // Calculate remaining 216 colors, 8 of which match the standard
    // 20 colors and 4 of which match the gray shades above
    numred=6;
    numgreen=6;
    numblue=6;
    for(int R=0; R<0xff; R+=0x33){
      for(G=0; G<=0xff; G+=0x33){
	for(int B=0; B<=0xff; B+=0x33){
	  plp->palPalEntry[i].peRed=(BYTE)R;
	  plp->palPalEntry[i].peGreen=(BYTE)G;
	  plp->palPalEntry[i].peBlue=(BYTE)B;
	  plp->palPalEntry[i].peFlags=0;
	  i++;
	  }
	}
      }

    // Create palette and we're done
    hPalette=CreatePalette(plp);
    FXFREE(&plp);
    }
  return hPalette;
  }


#endif


/*******************************************************************************/


// Initialize
void FXVisual::init(){
  if(!xid){
#ifndef FX_NATIVE_WIN32
    XVisualInfo vitemplate;
    XVisualInfo *vi;
    FXint nvi,i,d,dbest;

    // Assume the default
    visual=DefaultVisual(getApp()->display,DefaultScreen(getApp()->display));
    depth=DefaultDepth(getApp()->display,DefaultScreen(getApp()->display));
    
    // True color
    if(flags&VISUAL_TRUECOLOR){
      vitemplate.screen=DefaultScreen(getApp()->display);
      vitemplate.c_class=TrueColor;
      vi=XGetVisualInfo(getApp()->display,VisualClassMask|VisualScreenMask,&vitemplate,&nvi);
      if(vi){
        for(i=0,dbest=1000000; i<nvi; i++){
          if((vi[i].c_class==DirectColor) || (vi[i].c_class==TrueColor)){
            d=vi[i].depth-hint;
            if(d<0) d*=-100;         // Strongly prefer >= hint 
            if(d<dbest){
              dbest=d;
              visual=vi[i].visual;
              depth=vi[i].depth;
              }
            }
          }
        XFree((char*)vi);
        }
      }

    // Index color
    else if(flags&VISUAL_INDEXCOLOR){
      vitemplate.screen=DefaultScreen(getApp()->display);
      vi=XGetVisualInfo(getApp()->display,VisualScreenMask,&vitemplate,&nvi);
      if(vi){
        for(i=0,dbest=1000000; i<nvi; i++){
          if((vi[i].c_class==StaticColor) || (vi[i].c_class==PseudoColor)){
            d=vi[i].depth-hint;
            if(d<0) d*=-100;     
            if(d<dbest){
              dbest=d;
              visual=vi[i].visual;
              depth=vi[i].depth;
              }
            }
          }
        XFree((char*)vi);
        }
      }

    // Gray scale color
    else if(flags&VISUAL_GRAYSCALE){
      vitemplate.screen=DefaultScreen(getApp()->display);
      vi=XGetVisualInfo(getApp()->display,VisualScreenMask,&vitemplate,&nvi);
      if(vi){
        for(i=0,dbest=1000000; i<nvi; i++){
          if((vi[i].c_class==GrayScale) || (vi[i].c_class==StaticGray)){ 
            d=vi[i].depth-hint;
            if(d<0) d*=-100;
            if(d<dbest){
              dbest=d;
              visual=vi[i].visual;
              depth=vi[i].depth;
              }
            }
          }
        XFree((char*)vi);
        }
      }

    // Get the best (deepest) visual
    else if(flags&VISUAL_BEST){
      vitemplate.screen=DefaultScreen(getApp()->display);
      vi=XGetVisualInfo(getApp()->display,VisualScreenMask,&vitemplate,&nvi);
      if(vi){
        for(i=0,dbest=1000000; i<nvi; i++){
          d=vi[i].depth-hint;
          if(d<0) d*=-100;
          if(d<dbest){
            dbest=d;
            visual=vi[i].visual;
            depth=vi[i].depth;
            }
          }
        XFree((char*)vi);
        }
      }

    // Monochrome visual (for masks and stipples, not for windows)
    else if(flags&VISUAL_MONOCHROME){
      numcolors=2;
      visual=NULL;    // THIS SIGNIFIES MONOCHROME PIXMAP
      depth=1;
      }

    // Initialize colormap
    setupcolormap();
        
    // Make GC for this visual
    gc=makegc();

    xid=1;

#else

    // Check for palette support
    HDC hdc=GetDC(GetDesktopWindow());
    if(GetDeviceCaps(hdc,RASTERCAPS)&RC_PALETTE!=0){
      depth=GetDeviceCaps(hdc,COLORRES);
      if(depth<=8){
	hPalette=createAllPurposePalette();   // Palette
	type=VISUALTYPE_INDEX;
	}
      else if(depth==16){
        numred=32;
        numgreen=64;
        numblue=32;
	type=VISUALTYPE_TRUE;
        }
      else if(depth==15){
        numred=32;
        numgreen=32;
        numblue=32;
	type=VISUALTYPE_TRUE;
        }
      else if(depth>=24){
	numred=256;
	numgreen=256;
	numblue=256;
	type=VISUALTYPE_TRUE;
	}
      }
    else{
      int nPlanes=GetDeviceCaps(hdc,PLANES);
      int nBitsPixel=GetDeviceCaps(hdc,BITSPIXEL);
      depth=1<<(nPlanes*nBitsPixel);
      type=VISUALTYPE_UNKNOWN;
      }
    numcolors=numred*numgreen*numblue;
    ReleaseDC(GetDesktopWindow(),hdc);

    // This is just a placeholder
    xid=(void*)1;

#endif
    }
  }


// Get pixel value for color
FXPixel FXVisual::getPixel(FXColor clr){
#ifndef FX_NATIVE_WIN32
  register FXPixel pixel;
  switch(type){
    case VISUALTYPE_MONO:
      pixel=pixel_mono(FXREDVAL(clr),FXGREENVAL(clr),FXBLUEVAL(clr));
      break;
    case VISUALTYPE_TRUE:
      pixel=pixel_true(FXREDVAL(clr),FXGREENVAL(clr),FXBLUEVAL(clr));
      break;
    case VISUALTYPE_GRAY:
      pixel=pixel_gray(FXREDVAL(clr),FXGREENVAL(clr),FXBLUEVAL(clr));
      break;
    case VISUALTYPE_INDEX:
      pixel=pixel_index(FXREDVAL(clr),FXGREENVAL(clr),FXBLUEVAL(clr));
      break;
    case VISUALTYPE_UNKNOWN:
      pixel=0;
      break;
    }
  return pixel;
#else
  return PALETTERGB(FXREDVAL(clr),FXGREENVAL(clr),FXBLUEVAL(clr));
#endif
  }


// Get color value for pixel
FXColor FXVisual::getColor(FXPixel pix){
#ifndef FX_NATIVE_WIN32
  FXuint r,g,b;
  switch(type){
    case VISUALTYPE_MONO:
      r=pix_to_red[pix];
      g=pix_to_green[pix];
      b=pix_to_blue[pix];
      break;
    case VISUALTYPE_TRUE:
      r=pix_to_red[(pix&redmask)>>redshift];
      g=pix_to_green[(pix&greenmask)>>greenshift];
      b=pix_to_blue[(pix&bluemask)>>blueshift];
      break;
    case VISUALTYPE_GRAY:
      r=pix_to_red[pix];
      g=pix_to_green[pix];
      b=pix_to_blue[pix];
      break;
    case VISUALTYPE_INDEX:
      r=pix_to_red[pix];
      g=pix_to_green[pix];
      b=pix_to_blue[pix];
      break;
    case VISUALTYPE_UNKNOWN:
      r=0;
      g=0;
      b=0;
      break;
    }
  return FXRGB(r,g,b);
#else
  return PALETTEINDEX(pix);
#endif
  }


// Set maximum number of colors to allocate
void FXVisual::setMaxColors(FXuint maxcols){
  if(xid){ fxerror("%s::setMaxColors: visual already initialized.\n",getClassName()); }
  if(maxcols<2) maxcols=2;
  maxcolors=maxcols;
  }


// Save to stream
void FXVisual::save(FXStream& store) const {
  FXId::save(store);
  store << flags;
  store << depth;
  }


// Load from stream
void FXVisual::load(FXStream& store){ 
  FXId::load(store);
  store >> flags;
  store >> depth;
  }


// Destroy
FXVisual::~FXVisual(){
  FXTRACE((100,"FXVisual::~FXVisual %08x\n",this));
  if(xid){
#ifndef FX_NATIVE_WIN32
    if(nalloced){XFreeColors(getApp()->display,colormap,alloced,nalloced,0);}
    if(freemap){XFreeColormap(getApp()->display,colormap);}
    if(gc){XFreeGC(getApp()->display,gc);}
    FXFREE(&red_to_pix);
    FXFREE(&green_to_pix);
    FXFREE(&blue_to_pix);
    FXFREE(&pix_to_red);
    FXFREE(&pix_to_green);
    FXFREE(&pix_to_blue);
    FXFREE(&alloced);
#else
    if(hPalette){DeleteObject(hPalette);}
#endif
    }
  }


