/********************************************************************************
*                                                                               *
*                             S p i n   B u t t o n                             *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* Contributed by: Lyle Johnson                                                  *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXSpinner.cpp,v 1.2 1999/11/05 18:06:36 jeroen Exp $                     *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXTextField.h"
#include "FXButton.h"
#include "FXArrowButton.h"
#include "FXComposite.h"
#include "FXPacker.h"
#include "FXSpinner.h"


/*
  To do:
  - Should respond to up/down arrows.
  - should send SEL_COMMAND.
  - Should understand get/set messages.
  - Should this also be derived from FXTextField instead?
  - Sends SEL_COMMAND; should it send SEL_CHANGED for each repeat, then SEL_COMMAND
    at end?
*/

#define BUTTONWIDTH 14


/*******************************************************************************/

  
//  Message map
FXDEFMAP(FXSpinner) FXSpinnerMap[]={
  FXMAPFUNC(SEL_KEYPRESS,0,FXSpinner::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXSpinner::onKeyRelease),
  FXMAPFUNC(SEL_UPDATE,FXSpinner::ID_INCREMENT,FXSpinner::onUpdIncrement),
  FXMAPFUNC(SEL_COMMAND,FXSpinner::ID_INCREMENT,FXSpinner::onCmdIncrement),
  FXMAPFUNC(SEL_UPDATE,FXSpinner::ID_DECREMENT,FXSpinner::onUpdDecrement),
  FXMAPFUNC(SEL_COMMAND,FXSpinner::ID_DECREMENT,FXSpinner::onCmdDecrement),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,FXSpinner::onCmdSetValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTVALUE,FXSpinner::onCmdSetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTVALUE,FXSpinner::onCmdGetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTRANGE,FXSpinner::onCmdSetIntRange),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTRANGE,FXSpinner::onCmdGetIntRange),
  };

  
// Object implementation
FXIMPLEMENT(FXSpinner,FXPacker,FXSpinnerMap,ARRAYNUMBER(FXSpinnerMap))


// Construct spinner out of two buttons and a text field
FXSpinner::FXSpinner(){
  flags|=FLAG_ENABLED|FLAG_SHOWN;
  textField=(FXTextField*)-1;
  upButton=(FXArrowButton*)-1;
  downButton=(FXArrowButton*)-1;
  range[0]=-2147483648;
  range[1]=2147483647;
  incr=1;
  pos=1;
  }


// Construct spinner out of two buttons and a text field
FXSpinner::FXSpinner(FXComposite *p,FXint cols,FXObject *tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXPacker(p, opts, x,y,w,h, 0,0,0,0, 0,0){
  flags|=FLAG_ENABLED|FLAG_SHOWN;
  target=tgt;
  message=sel;
  textField=new FXTextField(this,cols,NULL,0, JUSTIFY_RIGHT|TEXTFIELD_READONLY, 0,0,0,0,pl,pr,pt,pb);
  upButton=new FXArrowButton(this,this,FXSpinner::ID_INCREMENT,FRAME_RAISED|FRAME_THICK|ARROW_UP|ARROW_REPEAT, 0,0,0,0, 0,0,0,0);
  downButton=new FXArrowButton(this,this,FXSpinner::ID_DECREMENT,FRAME_RAISED|FRAME_THICK|ARROW_DOWN|ARROW_REPEAT, 0,0,0,0, 0,0,0,0);
  range[0]=(options&SPIN_NOMIN) ? -2147483648 : 1;
  range[1]=(options&SPIN_NOMAX) ? 2147483647 : 10;
  incr=1;
  pos=1;
  }


// Get default width
FXint FXSpinner::getDefaultWidth(){
  FXint tw=0;
  if(!(options&SPIN_NOTEXT)) tw=textField->getDefaultWidth();
//  return tw+upButton->getDefaultWidth()+(border<<1);
  return tw+BUTTONWIDTH+(border<<1);
  }


// Get default height
FXint FXSpinner::getDefaultHeight(){
//   FXint th=0;
//   if(!(options&SPIN_NOTEXT)) th=textField->getDefaultHeight();
//   return MAX(th,2*upButton->getDefaultHeight())+(border<<1);
  return textField->getDefaultHeight()+(border<<1);
  }


// Create window
void FXSpinner::create(){
  FXPacker::create();
  updateText();
  }


// Recompute layout
void FXSpinner::layout(){
  FXint buttonWidth,buttonHeight,textWidth,textHeight;

  textHeight=height-2*border;
  buttonHeight=textHeight>>1;
  
  // Buttons plus the text; buttons are default width, text stretches to fill the rest
  if(!(options&SPIN_NOTEXT)){
    buttonWidth=BUTTONWIDTH;
    textWidth=width-buttonWidth-2*border;
    textField->position(border,border,textWidth,textHeight);
    upButton->position(border+textWidth,border,buttonWidth,buttonHeight);
    downButton->position(border+textWidth,height-buttonHeight-border,buttonWidth,buttonHeight);
    }
  
  // Only the buttons:- place buttons to take up the whole space!
  else{
    buttonWidth=width-2*border;
    upButton->position(border,border,buttonWidth,buttonHeight);
    downButton->position(border,height-buttonHeight-border,buttonWidth,buttonHeight);
    }
  flags&=~FLAG_DIRTY;
  }


// Respond to increment message
long FXSpinner::onUpdIncrement(FXObject* sender,FXSelector,void*){
  if(isEnabled() && pos<range[1])
    sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  else
    sender->handle(this,MKUINT(ID_DISABLE,SEL_COMMAND),NULL);
  return 1;
  }


// Respond to increment message
long FXSpinner::onCmdIncrement(FXObject*,FXSelector,void *ptr){
  if(!isEnabled()) return 0;
  increment();
  if(target) target->handle(this,MKUINT(message,SEL_COMMAND),ptr);
  return 1;
  }


// Disable decrement if at low end already
long FXSpinner::onUpdDecrement(FXObject* sender,FXSelector,void*){
  if(isEnabled() && range[0]<pos)
    sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  else
    sender->handle(this,MKUINT(ID_DISABLE,SEL_COMMAND),NULL);
  return 1;
  }


// Respond to decrement message
long FXSpinner::onCmdDecrement(FXObject*,FXSelector,void *ptr){
  if(!isEnabled()) return 0;
  decrement();
  if(target) target->handle(this,MKUINT(message,SEL_COMMAND),ptr);
  return 1;
  }


// Keyboard press
long FXSpinner::onKeyPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(!isEnabled()) return 0;
  switch(event->code){
    case KEY_Up:    
    case KEY_KP_Up:    
    case KEY_KP_Add:  
    case KEY_plus:
      increment();  
      if(target) target->handle(this,MKUINT(message,SEL_COMMAND),ptr);
      return 1;
    case KEY_Down:  
    case KEY_KP_Down:  
    case KEY_KP_Subtract: 
    case KEY_minus: 
      decrement();
      if(target) target->handle(this,MKUINT(message,SEL_COMMAND),ptr);
      return 1;
    }
  return 0;
  }


// Keyboard release
long FXSpinner::onKeyRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(!isEnabled()) return 0;
  switch(event->code){
    case KEY_Up:    
    case KEY_KP_Up:    
    case KEY_KP_Add:  
    case KEY_plus:
      return 1;
    case KEY_Down:  
    case KEY_KP_Down:  
    case KEY_KP_Subtract: 
    case KEY_minus: 
      return 1;
    }
  return 0;
  }


// Update value from a message
long FXSpinner::onCmdSetValue(FXObject*,FXSelector,void* ptr){
  setValue((FXint)(long)ptr); 
  return 1;
  }


// Update value from a message
long FXSpinner::onCmdSetIntValue(FXObject*,FXSelector,void* ptr){
  setValue(*((FXint*)ptr));
  return 1;
  }


// Obtain value from spinner
long FXSpinner::onCmdGetIntValue(FXObject*,FXSelector,void* ptr){
  *((FXint*)ptr)=getValue();
  return 1;
  }


// Update range from a message
long FXSpinner::onCmdSetIntRange(FXObject*,FXSelector,void* ptr){
  setMinMax(((FXint*)ptr)[0],((FXint*)ptr)[1]);
  return 1;
  }


// Get range with a message
long FXSpinner::onCmdGetIntRange(FXObject*,FXSelector,void* ptr){
  ((FXint*)ptr)[0]=range[0];
  ((FXint*)ptr)[1]=range[1];
  return 1;
  }


// Increment spinner
void FXSpinner::increment(){
  if(range[0]<range[1]){
    FXint oldval=pos;
    pos+=incr;
    if(options&SPIN_CYCLIC){
      pos=range[0]+(pos-range[0])%(range[1]-range[0]+1);  // Count around modulo (max-min+1)
      }
    else{
      if(pos>range[1]) pos=range[1];
      }
    if(oldval!=pos) updateText();
    }
  }


// Decrement spinner
void FXSpinner::decrement(){
  if(range[0]<range[1]){
    FXint oldval=pos;
    pos-=incr;
    if(options&SPIN_CYCLIC){
      pos+=(range[1]-range[0]+1);
      pos=range[0]+(pos-range[0])%(range[1]-range[0]+1);  // Count around modulo (max-min+1)
      }
    else{
      if(pos<range[0]) pos=range[0];
      }
    if(oldval!=pos) updateText();
    }
  }


// Update text in the field
void FXSpinner::updateText(){
  FXchar s[20];
  sprintf(s,"%d",pos);
  setText(s);
  }


// Place text straight into the text field part
void FXSpinner::setText(const FXString& text){
  textField->setText(text);
  }


// Return text in the text field part
FXString FXSpinner::getText() const {
  return textField->getText();
  }


// True if spinner is cyclic
FXbool FXSpinner::isCyclic() const {
  return (options&SPIN_CYCLIC)!=0;
  }


// Set spinner cyclic mode
void FXSpinner::setCyclic(FXbool s){
  if(s) options|=SPIN_CYCLIC; else options&=~SPIN_CYCLIC;
  }


// Enable the widget
void FXSpinner::enable(){
  if(!(flags&FLAG_ENABLED)){
    FXPacker::enable();
    textField->enable();
    upButton->enable();
    downButton->enable();
    }
  }


// Disable the widget
void FXSpinner::disable(){
  if(flags&FLAG_ENABLED){
    FXPacker::disable();
    textField->disable();
    upButton->disable();
    downButton->disable();
    }
  }


// Set minimum and maximum; fix other stuff too
void FXSpinner::setMinMax(FXint mn,FXint mx){
  if(mn>mx){ fxerror("%s::setMinMax: minimum should be lower than maximum.\n",getClassName()); }
  if(pos<mn) pos=mn;
  if(pos>mx) pos=mx;
  range[0]=mn;
  range[1]=mx;
  updateText();
  }


// Return the minimum and maximum
void FXSpinner::getMinMax(FXint& mn,FXint& mx) const {
  mn=range[0];
  mx=range[1];
  }


// Set new value
void FXSpinner::setValue(FXint value){
  if(value<range[0]) value=range[0];
  if(value>range[1]) value=range[1];
  pos=value;
  updateText();
  }


// Change value increment
void FXSpinner::setIncrement(FXint inc){
  if(inc<1){ fxerror("%s::setIncrement: negative or zero increment specified.\n",getClassName()); }
  incr=inc;
  }


// True if text supposed to be visible
FXbool FXSpinner::isTextVisible() const {
  return textField->shown();
  }


// Change text visibility
void FXSpinner::setTextVisible(FXbool s){
  FXuint opts=s?(options|SPIN_NOTEXT):(options&~SPIN_NOTEXT);
  if(options!=opts){
    s ? textField->show() : textField->hide();
    options=opts;
    recalc();
    }
  }


// Set help text
void FXSpinner::setHelpText(const FXString&  text){
  textField->setHelpText(text);
  }


// Get help text
FXString FXSpinner::getHelpText() const {
  return textField->getHelpText();
  }


// Set tip text
void FXSpinner::setTipText(const FXString&  text){
  textField->setTipText(text);
  }



// Get tip text
FXString FXSpinner::getTipText() const {
  return textField->getTipText();
  }


// Save object to stream
void FXSpinner::save(FXStream& store) const {
  FXPacker::save(store);
  store << textField;
  store << upButton;
  store << downButton;
  store << range[0] << range[1];
  store << incr;
  store << pos;
  }


// Load object from stream
void FXSpinner::load(FXStream& store){
  FXPacker::load(store);
  store >> textField;
  store >> upButton;
  store >> downButton;
  store >> range[0] >> range[1];
  store >> incr;
  store >> pos;
  }


// Destruct spinner:- trash it!
FXSpinner::~FXSpinner(){
  textField=(FXTextField*)-1;
  upButton=(FXArrowButton*)-1;
  downButton=(FXArrowButton*)-1;
  }
