/*
	PSPI.C

	Originally written by Sergey Larin.
	Corrected by Denis Chertykov.
	Module for Downloading Through the Parallel Port
*/

#include <stdio.h>

/* sys/io.h is the correct file.  If your system doesn't have that
 * file, something is broken.
 */
#ifdef __BROKEN_LINUX
#include <asm/io.h>
#else
#include <sys/io.h>
#endif

#include <unistd.h>
#include <linux/lp.h>
#include <signal.h>
#include <sys/time.h>

#include "PSPI.h"

#define BASE  parport_base	/* 0x378 by default */
#define SCK   LP_PSTROBE	/* base + 2 */
#define RESET LP_PINITP		/* base + 2 */
#define DIN   LP_PBUSY		/* base + 1 */
#define DOUT  0x1		/* base */

int parport_base =  0x378;
static long n_per_ms;

static volatile void delay_func ()
{
  volatile int i;
  for (i=0; i < 100; ++i);
}

static void
big_delay (void)
{
  static volatile int n;
  n = n_per_ms/2500;
  n = n ? n : 10;
  while (n)
    {
      delay_func ();
      --n;
    }
}

void PSPI::clk (void)
{
  outb (SCK, BASE+2); big_delay ();         /* sck = 0, reset = 0 */
  outb (SCK | RESET, BASE+2); big_delay (); /* sck = 0, reset = 1 */
  outb (SCK, BASE+2); big_delay ();         /* sck = 0, reset = 0 */
  
  outb (0, BASE+2); big_delay ();
  outb (SCK, BASE+2); big_delay ();
}

static unsigned char
send_recv (unsigned char b)
{
  int i, mask, received=0;
  
  for (i=0, mask = 0x80; i<8; i++, mask >>= 1)
    {
      outb (b & mask ? DOUT : 0, BASE); big_delay ();
      outb (0, BASE+2); big_delay ();
      if (!(inb (BASE+1) & DIN))
	received |= mask;
      outb (SCK, BASE+2); big_delay ();
    }
  return received;
}

static volatile int yes_alarm;

static void
alarm( int k )
{
  yes_alarm = 0;
}

long
calc_delay() // pause x ms
{
  long n=0;
  struct itimerval it;
  it.it_interval.tv_sec = 0;
  it.it_value.tv_sec = 0;
  it.it_interval.tv_usec = 0;
  it.it_value.tv_usec = 100000;
  signal( SIGALRM, alarm );
  setitimer( ITIMER_REAL, &it, NULL );
  yes_alarm = 1;
  while (yes_alarm)
    {
      delay_func ();
      ++n;
    }
  return n;
}

PSPI::PSPI ()
{
  port_enabled = 0;
  n_per_ms = calc_delay ();
  if (::ioperm (BASE, 8, 1) == 0)
    {
      outb (SCK, BASE+2); big_delay ();         /* sck = 0, reset = 0 */
      outb (SCK | RESET, BASE+2); big_delay (); /* sck = 0, reset = 1 */
      outb (SCK, BASE+2); big_delay ();         /* sck = 0, reset = 0 */
      port_enabled = 1;
    }
  else{perror("ioperm()");throw Error_Device("PSPI","Cannot open port.");}
}

PSPI::~PSPI ()
{
  outb (SCK | RESET, BASE+2); big_delay (); /* sck = 0, reset = 1 */
}

int PSPI::Send (unsigned char* queue, int queueSize, int rec_queueSize=-1)
{
  unsigned char *p = queue, ch;
  int i = queueSize;
  
  if (rec_queueSize==-1) { rec_queueSize = queueSize; }

  /* printf ("send(recv): ");*/
  while (i--)
    {
      /*           printf ("%02X(", (unsigned int)*p);*/
      ch = send_recv (*p);
      /*           printf ("%02X) ", (unsigned int)ch);*/
      *p++ = ch;
    }
  /*  printf ("\n");*/
  return queueSize;
}

