/*	$Id: main.c,v 1.23 1997/10/31 22:34:11 sandro Exp $	*/

/*
 * Copyright (c) 1997
 *	Sandro Sigala, Brescia, Italy.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <err.h>

#include "zile.h"
#include "extern.h"
#include "version.h"
#ifdef USE_NCURSES
#include "term_ncurses/term_ncurses.h"
#endif

#define ZILE_VERSION_STRING \
	"Zile " ZILE_VERSION " of " CONFIGURE_DATE " on " CONFIGURE_HOST

/* The current window; the first window in list. */
windowp cur_wp, head_wp;
/* The current buffer; the previous buffer; the first buffer in list. */
bufferp cur_bp, prev_bp, head_bp;
terminalp cur_tp;

/* The global editor flags. */
int thisflag, lastflag;
/* The universal argument repeat count. */
int last_uniarg = 1;

#ifdef DEBUG
/*
 * This function checks the line list of the specified buffer and
 * appends list informations to the `zile.abort' file if
 * some unexpected corruption is found.
 */
static void
check_list(windowp wp)
{
	linep lp, prevlp;

	prevlp = wp->bp->limitp;
	for (lp = wp->bp->limitp->next;; lp = lp->next) {
		if (lp->prev != prevlp || prevlp->next != lp) {
			FILE *f = fopen("zile.abort", "a");
			fprintf(f, "---------- buffer `%s' corruption\n", wp->bp->name);
			fprintf(f, "limitp = %p, limitp->prev = %p, limitp->next = %p\n", wp->bp->limitp, wp->bp->limitp->prev, wp->bp->limitp->next);
			fprintf(f, "prevlp = %p, prevlp->prev = %p, prevlp->next = %p\n", prevlp, prevlp->prev, prevlp->next);
			fprintf(f, "pointp = %p, pointp->prev = %p, pointp->next = %p\n", wp->bp->pointp, wp->bp->pointp->prev, wp->bp->pointp->next);
			fprintf(f, "lp = %p, lp->prev = %p, lp->next = %p\n", lp, lp->prev, lp->next);
			fclose(f);
			cur_tp->close();
			fprintf(stderr, "\aAborting due to internal buffer structure corruption\n");
			fprintf(stderr, "\aFor more informations read the `zile.abort' file\n");
			abort();
		}
		if (lp == wp->bp->limitp)
			break;
		prevlp = lp;
	}
}
#endif /* DEBUG */

static void
loop(void)
{
	int c;

	for (;;) {
		if (lastflag & FLAG_NEED_RESYNC)
			resync_redisplay();
#ifdef DEBUG
		check_list(cur_wp);
#endif
		cur_tp->redisplay();

		c = cur_tp->getkey();
		minibuf_clear();

		thisflag = 0;
		if (lastflag & FLAG_DEFINING_MACRO)
			thisflag |= FLAG_DEFINING_MACRO;

		process_key(c);

		if (thisflag & FLAG_QUIT_ZILE)
			break;
		if (!(thisflag & FLAG_SET_UNIARG))
			last_uniarg = 1;

		lastflag = thisflag;
	}
}

static void
select_terminal(void)
{
#ifdef USE_NCURSES
	/* Only the ncurses terminal is available for now. */
	cur_tp = ncurses_tp;
#else
#error At least one terminal should be defined.
#endif
}

static char *about_text_str[] = {
	ZILE_VERSION_STRING,
	"",
	"%Copyright (C) 1997 Sandro Sigala <ssigala@globalnet.it>%",
	"",
	"Type %C-x C-c% to exit Zile.",
	"Type %M-h h% for help; %C-x u% to undo changes.",
	"Type %M-h M-h% to show a mini help window.",
	"Type %M-h d% for information on getting the latest version.",
	"Type %M-h t% for a tutorial on using Zile.",
	"Type %C-g% anytime to cancel the current operation.",
	"",
	"%C-x% means hold the CTRL key while typing the character %x%.",
	"%M-x% means hold the META or EDIT or ALT key down while typing %x%.",
	"If there is no META, EDIT or ALT key, instead press and release",
	"the ESC key and then type %x%.",
	"Combinations like %M-h t% mean first do %M-h%, then press %t%.",
	NULL
};
static char *about_minibuf_str = "For help type %FWM-h h%E; to show a mini help window type %FWM-h M-h%E";
static char *about_wait_str = "[Press any key to leave this screen]";

static void
about_screen(void)
{
	if (lookup_bool_variable("novice-level")) {
		cur_tp->show_about(about_text_str, about_minibuf_str, about_wait_str);
		waitkey_discard(200 * 1000);
	} else {
		cur_tp->show_about(about_text_str, about_minibuf_str, NULL);
		waitkey(20 * 1000);
	}
	minibuf_clear();
}

/*
 * Output the program syntax then exit.
 */
static void
usage(void)
{
	fprintf(stderr, "usage: zile [-hqV] [file ...]\n");
	exit(1);
}

int
main(int argc, char **argv)
{
	int c;
	int hflag = 0, qflag = 0;

	select_terminal();
	cur_tp->init();

	while ((c = getopt(argc, argv, "hqV")) != -1)
		switch (c) {
		case 'h':
			hflag = 1;
			break;
		case 'q':
			qflag = 1;
			break;
		case 'V':
			fprintf(stderr, ZILE_VERSION_STRING "\n");
			exit(0);
		case '?':
		default:
			usage();
			/* NOTREACHED */
		}
	argc -= optind;
	argv += optind;

	init_variables();
	if (!qflag)
		read_rc_file();
	cur_tp->open();

	create_first_window();
	cur_tp->redisplay();
	bind_functions();

	if (argc < 1) {
		if (!hflag)
			about_screen();
	} else {
		while (*argv) {
			int line = 0;
			if (**argv == '+')
				line = atoi(*argv++ + 1);
			if (*argv)
				open_file(*argv++, line - 1);
		}
	}

	/*
	 * Show the Mini Help window if the `-h' flag was specified
	 * on command line or the novice level is enabled.
	 */
	if (hflag || lookup_bool_variable("novice-level"))
		F_toggle_minihelp_window(1);

	if (argc < 1 && lookup_bool_variable("novice-level")) {
		/* Cutted & pasted from Emacs 20.2. */
		insert_string("\
This buffer is for notes you don't want to save.\n\
If you want to create a file, visit that file with C-x C-f,\n\
then enter the text in that file's own buffer.\n\
\n");
		cur_bp->flags &= ~BFLAG_MODIFIED;
		resync_redisplay();
	}

	/*
	 * Run the main Zile loop.
	 */
	loop();

	cur_tp->close();

	return 0;
}
