;; Quail package for Ethiopic characters.
;; Copyright (C) 1994 Free Software Foundation, Inc.
;; This file is part of Mule (MULtilingual Enhancement of GNU Emacs).
;; This file contains Ethipic characters.

;; Mule is free software distributed in the form of patches to GNU Emacs.
;; You can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; Mule is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; 94.10.13 created for Mule Ver.2.1 by TAKAHASHI Naoto <ntakahas@etl.go.jp>
;;; 95.7.19 modified for Mule Ver.2.2.2 by TAKAHASHI Naoto <ntakahas@etl.go.jp>

(require 'quail)
(require 'ethio)

;;
;; toggle ethio & latin input
;;

(defvar ethio-prefer-arabic-digit t)
(make-variable-buffer-local 'ethio-prefer-arabic-digit)
(defvar ethio-prefer-latin-space t)
(make-variable-buffer-local 'ethio-prefer-latin-space)
(defvar ethio-prefer-ethiopic-space nil)
(make-variable-buffer-local 'ethio-prefer-ethiopic-space)
(defvar ethio-prefer-latin-punctuation nil)
(make-variable-buffer-local 'ethio-prefer-latin-punctuation)

(defconst arabic-digit-strings
  '("1" "2" "3" "4" "5" "6" "7" "8" "9"
    "10" "20" "30" "40" "50" "60" "70" "80" "90"
    "100" "200" "300" "400" "500" "600" "700" "800" "900"
    "1000" "2000" "3000" "4000" "5000" "6000" "7000" "8000" "9000"
    "10000" "20000" "30000" "40000" "50000" "60000" "70000" "80000" "90000"
    "100000" "200000" "300000" "400000" "500000" "600000" "700000" "80000"
    "900000" "1000000"))

(defconst ethiopic-digit-strings
  '("$(2$W(B" "$(2$X(B" "$(2$Y(B" "$(2$Z(B" "$(2$[(B" "$(2$\(B" "$(2$](B" "$(2$^(B" "$(2$_(B"
    "$(2$`(B" "$(2$a(B" "$(2$b(B" "$(2$c(B" "$(2$d(B" "$(2$e(B" "$(2$f(B" "$(2$g(B" "$(2$h(B"
    "$(2$i(B" "$(2$X$i(B" "$(2$Y$i(B" "$(2$Z$i(B" "$(2$[$i(B" "$(2$\$i(B" "$(2$]$i(B" "$(2$^$i(B" "$(2$_$i(B"
    "$(2$`$i(B" "$(2$a$i(B" "$(2$b$i(B" "$(2$c$i(B" "$(2$d$i(B" "$(2$e$i(B" "$(2$f$i(B" "$(2$g$i(B" "$(2$h$i(B"
    "$(2$j(B" "$(2$X$j(B" "$(2$Y$j(B" "$(2$Z$j(B" "$(2$[$j(B" "$(2$\$j(B" "$(2$]$j(B" "$(2$^$j(B" "$(2$_$j(B"
    "$(2$`$j(B" "$(2$a$j(B" "$(2$b$j(B" "$(2$c$j(B" "$(2$d$j(B" "$(2$e$j(B" "$(2$f$j(B" "$(2$g$j(B" "$(2$h$j(B"
    "$(2$i$j(B"))

(defconst latin-punctuation-strings
  '("." ".." "," ",," ";" ";;" ":" ":::" "*" "**"))

(defconst ethiopic-punctuation-strings
  '("$(2$P(B" "." "$(2$Q(B" "," "$(2$R(B" ";" "$(2$S(B" ":" "$(2$T(B" "*"))

(defun ethio-toggle-digit nil
  "Toggle Arabic numbers and Ethiopic numbers for keyboard input."
  (interactive)
  (if (and quail-mode
	   (string= (quail-package-name) "ethio"))
      (let ((nums (if ethio-prefer-arabic-digit
		      ethiopic-digit-strings
		    arabic-digit-strings))
	    (keys arabic-digit-strings))
	(while keys
	  (quail-defrule (car keys) (car nums))
	  (setq keys (cdr keys)
		nums (cdr nums)))
	(setq ethio-prefer-arabic-digit (not ethio-prefer-arabic-digit))
	(update-mode-lines))))

(defun ethio-toggle-space nil
  "Toggle Latin space and Ethiopic separator for keyboard input."
  (interactive)
  (if (and quail-mode
	   (string= (quail-package-name) "ethio"))
      (progn
	(cond (ethio-prefer-latin-space
	       (quail-defrule " " "$(2$N(B")
	       (setq ethio-prefer-latin-space nil
		     ethio-prefer-ethiopic-space t))
	      (ethio-prefer-ethiopic-space
	       (quail-defrule " " "$(2$O(B")
	       (setq ethio-prefer-ethiopic-space nil))
	      (t
	       (quail-defrule " " " " "ethio")
	       (setq ethio-prefer-latin-space t)))
	(update-mode-lines))))

(defun ethio-toggle-punctuation nil
  "Toggle Ethiopic punctuation and Latin punctuation for keyboard input."
  (interactive)
  (if (and quail-mode
	   (string= (quail-package-name) "ethio"))
      (let ((puncs (if ethio-prefer-latin-punctuation
		       ethiopic-punctuation-strings
		     latin-punctuation-strings))
	    (keys latin-punctuation-strings))
	(while keys
	  (quail-defrule (car keys) (car puncs))
	  (setq keys (cdr keys)
		puncs (cdr puncs)))
	(setq ethio-prefer-latin-punctuation
	      (not ethio-prefer-latin-punctuation))
	(update-mode-lines))))

(defvar quail-ethio-mode nil)
(make-variable-buffer-local 'quail-ethio-mode)

(defvar quail-ethio-mode-map nil)
(or quail-ethio-mode-map
    (let ((map (make-sparse-keymap)))
      (define-key map [f1] 'ethio-toggle-digit)
      (define-key map [f2] 'ethio-toggle-space)
      (define-key map [f3] 'ethio-toggle-punctuation)
      (setq quail-ethio-mode-map map)))

(or (assq 'quail-ethio-mode minor-mode-map-alist)
    (setq minor-mode-map-alist
	  (cons (cons 'quail-ethio-mode quail-ethio-mode-map)
		minor-mode-map-alist)))

(defun quail-ethio-mode (&optional arg)
  (interactive "P")
  (setq quail-ethio-mode
	(and quail-mode
	     (if (null arg) (not quail-ethio-mode)
	       (> (prefix-numeric-value arg) 0)))))

(quail-define-package
 "ethio"
 '("$(2"="#!&(B "
   (ethio-prefer-arabic-digit "1" "$(2$W(B")
   (ethio-prefer-latin-space "_"
			     (ethio-prefer-ethiopic-space "$(2$N(B" "$(2$O(B"))
   (ethio-prefer-latin-punctuation "." "$(2$P(B"))
 t "  KEYS AND FUNCTIONS

[F1] or `M-x ethio-toggle-digit'
  toggles Arabic digit input and Ethiopic digit input.  The
  current input mode is indicated by the character `1' or `$(2$W(B' in
  the mode line, next to the `Q-ETHIO' string.  The default is
  Arabic.
  In Ethiopic mode, you can input up to 1000000 ($(2$i$j(B) by
  continuous zeros.  To input 2000000 ($(2$X$i$j(B), for example, you
  have to type `21000000'.

[F2] or `M-x ethio-toggle-space'
  toggles space characters (one of the three below) for keyboard
  input.
    a) ` '   - Single width ASCII space.  Mode line Indicator is `_'.
    b) `$(2$N(B'  - Double width Ethiopic blank.  Mode line Indicator is `$(2$N(B'.
    c) `$(2$O(B'  - Ethiopic word separator.  Mode line Indicator is `$(2$O(B'.
  Each time you hit [F2], the space character changes a -> b -> c -> a 
  and so on.  The default is ASCII space.

[F3] or `M-x ethio-toggle-punctuation'
  toggles Latin punctuation input and Ethiopic punctuation input.
  The mode line indicator is `.' (Latin) and `$(2$P(B' (Ethiopic).
  The default is Ethiopic.
" nil t t nil nil nil nil '(quail-ethio-mode))

(qd "le"	"$(2!!(B")
(qd "lu"	"$(2!"(B")
(qd "li"	"$(2!#(B")
(qd "la"	"$(2!$(B")
(qd "lE"	"$(2!%(B")
(qd "l"		"$(2!&(B")
(qd "lo"	"$(2!'(B")
(qd "lW"	"$(2!((B")
(qd "lWa"	"$(2!((B")
(qd "lWe"	"$(2!&#L(B")
(qd "lWu"	"$(2!&#M(B")
(qd "lWi"	"$(2!&#N(B")
(qd "lWE"	"$(2!&#P(B")
(qd "lW'"	"$(2!&#M(B")

(qd "Le"	"$(2!!(B")
(qd "Lu"	"$(2!"(B")
(qd "Li"	"$(2!#(B")
(qd "La"	"$(2!$(B")
(qd "LE"	"$(2!%(B")
(qd "L"		"$(2!&(B")
(qd "Lo"	"$(2!'(B")
(qd "LW"	"$(2!((B")
(qd "LWa"	"$(2!((B")
(qd "LWe"	"$(2!&#L(B")
(qd "LWu"	"$(2!&#M(B")
(qd "LWi"	"$(2!&#N(B")
(qd "LWE"	"$(2!&#P(B")
(qd "LW'"	"$(2!&#M(B")

(qd "me"	"$(2!)(B")
(qd "mu"	"$(2!*(B")
(qd "mi"	"$(2!+(B")
(qd "ma"	"$(2!,(B")
(qd "mE"	"$(2!-(B")
(qd "m"		"$(2!.(B")
(qd "mo"	"$(2!/(B")
(qd "mW"	"$(2!0(B")
(qd "mWa"	"$(2!0(B")
(qd "mWe"	"$(2!.#L(B")
(qd "mWu"	"$(2!.#M(B")
(qd "mWi"	"$(2!.#N(B")
(qd "mWE"	"$(2!.#P(B")
(qd "mW'"	"$(2!.#M(B")

(qd "Me"	"$(2!)(B")
(qd "Mu"	"$(2!*(B")
(qd "Mi"	"$(2!+(B")
(qd "Ma"	"$(2!,(B")
(qd "ME"	"$(2!-(B")
(qd "M"		"$(2!.(B")
(qd "Mo"	"$(2!/(B")
(qd "MW"	"$(2!0(B")
(qd "MWa"	"$(2!0(B")
(qd "MWe"	"$(2!.#L(B")
(qd "MWu"	"$(2!.#M(B")
(qd "MWi"	"$(2!.#N(B")
(qd "MWE"	"$(2!.#P(B")
(qd "MW'"	"$(2!.#M(B")

(qd "re"	"$(2!1(B")
(qd "ru"	"$(2!2(B")
(qd "ri"	"$(2!3(B")
(qd "ra"	"$(2!4(B")
(qd "rE"	"$(2!5(B")
(qd "r"		"$(2!6(B")
(qd "ro"	"$(2!7(B")
(qd "rW"	"$(2!8(B")
(qd "rWa"	"$(2!8(B")
(qd "rWe"	"$(2!6#L(B")
(qd "rWu"	"$(2!6#M(B")
(qd "rWi"	"$(2!6#N(B")
(qd "rWE"	"$(2!6#P(B")
(qd "rW'"	"$(2!6#M(B")

(qd "Re"	"$(2!1(B")
(qd "Ru"	"$(2!2(B")
(qd "Ri"	"$(2!3(B")
(qd "Ra"	"$(2!4(B")
(qd "RE"	"$(2!5(B")
(qd "R"		"$(2!6(B")
(qd "Ro"	"$(2!7(B")
(qd "RW"	"$(2!8(B")
(qd "RWa"	"$(2!8(B")
(qd "RWe"	"$(2!6#L(B")
(qd "RWu"	"$(2!6#M(B")
(qd "RWi"	"$(2!6#N(B")
(qd "RWE"	"$(2!6#P(B")
(qd "RW'"	"$(2!6#M(B")

(qd "xe"	"$(2!9(B")
(qd "xu"	"$(2!:(B")
(qd "xi"	"$(2!;(B")
(qd "xa"	"$(2!<(B")
(qd "xE"	"$(2!=(B")
(qd "x"		"$(2!>(B")
(qd "xo"	"$(2!?(B")
(qd "xW"	"$(2!@(B")
(qd "xWa"	"$(2!@(B")
(qd "xWe"	"$(2!>#L(B")
(qd "xWu"	"$(2!>#M(B")
(qd "xWi"	"$(2!>#N(B")
(qd "xWE"	"$(2!>#P(B")
(qd "xW'"	"$(2!>#M(B")

(qd "be"	"$(2!A(B")
(qd "bu"	"$(2!B(B")
(qd "bi"	"$(2!C(B")
(qd "ba"	"$(2!D(B")
(qd "bE"	"$(2!E(B")
(qd "b"		"$(2!F(B")
(qd "bo"	"$(2!G(B")
(qd "bW"	"$(2!H(B")
(qd "bWa"	"$(2!H(B")
(qd "bWe"	"$(2!F#L(B")
(qd "bWu"	"$(2!F#M(B")
(qd "bWi"	"$(2!F#N(B")
(qd "bWE"	"$(2!F#P(B")
(qd "bW'"	"$(2!F#M(B")

(qd "Be"	"$(2!A(B")
(qd "Bu"	"$(2!B(B")
(qd "Bi"	"$(2!C(B")
(qd "Ba"	"$(2!D(B")
(qd "BE"	"$(2!E(B")
(qd "B"		"$(2!F(B")
(qd "Bo"	"$(2!G(B")
(qd "BW"	"$(2!H(B")
(qd "BWa"	"$(2!H(B")
(qd "BWe"	"$(2!F#L(B")
(qd "BWu"	"$(2!F#M(B")
(qd "BWi"	"$(2!F#N(B")
(qd "BWE"	"$(2!F#P(B")
(qd "BW'"	"$(2!F#M(B")

(qd "ve"	"$(2!I(B")
(qd "vu"	"$(2!J(B")
(qd "vi"	"$(2!K(B")
(qd "va"	"$(2!L(B")
(qd "vE"	"$(2!M(B")
(qd "v"		"$(2!N(B")
(qd "vo"	"$(2!O(B")
(qd "vW"	"$(2!P(B")
(qd "vWa"	"$(2!P(B")
(qd "vWe"	"$(2!N#L(B")
(qd "vWu"	"$(2!N#M(B")
(qd "vWi"	"$(2!N#N(B")
(qd "vWE"	"$(2!N#P(B")
(qd "vW'"	"$(2!N#M(B")

(qd "Ve"	"$(2!I(B")
(qd "Vu"	"$(2!J(B")
(qd "Vi"	"$(2!K(B")
(qd "Va"	"$(2!L(B")
(qd "VE"	"$(2!M(B")
(qd "V"		"$(2!N(B")
(qd "Vo"	"$(2!O(B")
(qd "VW"	"$(2!P(B")
(qd "VWa"	"$(2!P(B")
(qd "VWe"	"$(2!N#L(B")
(qd "VWu"	"$(2!N#M(B")
(qd "VWi"	"$(2!N#N(B")
(qd "VWE"	"$(2!N#P(B")
(qd "VW'"	"$(2!N#M(B")

(qd "te"	"$(2!Q(B")
(qd "tu"	"$(2!R(B")
(qd "ti"	"$(2!S(B")
(qd "ta"	"$(2!T(B")
(qd "tE"	"$(2!U(B")
(qd "t"		"$(2!V(B")
(qd "to"	"$(2!W(B")
(qd "tW"	"$(2!X(B")
(qd "tWa"	"$(2!X(B")
(qd "tWe"	"$(2!V#L(B")
(qd "tWu"	"$(2!V#M(B")
(qd "tWi"	"$(2!V#N(B")
(qd "tWE"	"$(2!V#P(B")
(qd "tW'"	"$(2!V#M(B")

(qd "ce"	"$(2!Y(B")
(qd "cu"	"$(2!Z(B")
(qd "ci"	"$(2![(B")
(qd "ca"	"$(2!\(B")
(qd "cE"	"$(2!](B")
(qd "c"		"$(2!^(B")
(qd "co"	"$(2!_(B")
(qd "cW"	"$(2!`(B")
(qd "cWa"	"$(2!`(B")
(qd "cWe"	"$(2!^#L(B")
(qd "cWu"	"$(2!^#M(B")
(qd "cWi"	"$(2!^#N(B")
(qd "cWE"	"$(2!^#P(B")
(qd "cW'"	"$(2!^#M(B")

(qd "ne"	"$(2!a(B")
(qd "nu"	"$(2!b(B")
(qd "ni"	"$(2!c(B")
(qd "na"	"$(2!d(B")
(qd "nE"	"$(2!e(B")
(qd "n"		"$(2!f(B")
(qd "no"	"$(2!g(B")
(qd "nW"	"$(2!h(B")
(qd "nWa"	"$(2!h(B")
(qd "nWe"	"$(2!f#L(B")
(qd "nWu"	"$(2!f#M(B")
(qd "nWi"	"$(2!f#N(B")
(qd "nWE"	"$(2!f#P(B")
(qd "nW'"	"$(2!f#M(B")

(qd "Ne"	"$(2!i(B")
(qd "Nu"	"$(2!j(B")
(qd "Ni"	"$(2!k(B")
(qd "Na"	"$(2!l(B")
(qd "NE"	"$(2!m(B")
(qd "N"		"$(2!n(B")
(qd "No"	"$(2!o(B")
(qd "NW"	"$(2!p(B")
(qd "NWa"	"$(2!p(B")
(qd "NWe"	"$(2!n#L(B")
(qd "NWu"	"$(2!n#M(B")
(qd "NWi"	"$(2!n#N(B")
(qd "NWE"	"$(2!n#P(B")
(qd "NW'"	"$(2!n#M(B")

(qd "ze"	"$(2!q(B")
(qd "zu"	"$(2!r(B")
(qd "zi"	"$(2!s(B")
(qd "za"	"$(2!t(B")
(qd "zE"	"$(2!u(B")
(qd "z"		"$(2!v(B")
(qd "zo"	"$(2!w(B")
(qd "zW"	"$(2!x(B")
(qd "zWa"	"$(2!x(B")
(qd "zWe"	"$(2!v#L(B")
(qd "zWu"	"$(2!v#M(B")
(qd "zWi"	"$(2!v#N(B")
(qd "zWE"	"$(2!v#P(B")
(qd "zW'"	"$(2!v#M(B")

(qd "Ze"	"$(2!y(B")
(qd "Zu"	"$(2!z(B")
(qd "Zi"	"$(2!{(B")
(qd "Za"	"$(2!|(B")
(qd "ZE"	"$(2!}(B")
(qd "Z"		"$(2!~(B")
(qd "Zo"	"$(2"!(B")
(qd "ZW"	"$(2""(B")
(qd "ZWa"	"$(2""(B")
(qd "ZWe"	"$(2!~#L(B")
(qd "ZWu"	"$(2!~#M(B")
(qd "ZWi"	"$(2!~#N(B")
(qd "ZWE"	"$(2!~#P(B")
(qd "ZW'"	"$(2!~#M(B")

(qd "de"	"$(2"#(B")
(qd "du"	"$(2"$(B")
(qd "di"	"$(2"%(B")
(qd "da"	"$(2"&(B")
(qd "dE"	"$(2"'(B")
(qd "d"		"$(2"((B")
(qd "do"	"$(2")(B")
(qd "dW"	"$(2"*(B")
(qd "dWa"	"$(2"*(B")
(qd "dWe"	"$(2"(#L(B")
(qd "dWu"	"$(2"(#M(B")
(qd "dWi"	"$(2"(#N(B")
(qd "dWE"	"$(2"(#P(B")
(qd "dW'"	"$(2"(#M(B")

(qd "Te"	"$(2"+(B")
(qd "Tu"	"$(2",(B")
(qd "Ti"	"$(2"-(B")
(qd "Ta"	"$(2".(B")
(qd "TE"	"$(2"/(B")
(qd "T"		"$(2"0(B")
(qd "To"	"$(2"1(B")
(qd "TW"	"$(2"2(B")
(qd "TWa"	"$(2"2(B")
(qd "TWe"	"$(2"0#L(B")
(qd "TWu"	"$(2"0#M(B")
(qd "TWi"	"$(2"0#N(B")
(qd "TWE"	"$(2"0#P(B")
(qd "TW'"	"$(2"0#M(B")

(qd "Ce"	"$(2"3(B")
(qd "Cu"	"$(2"4(B")
(qd "Ci"	"$(2"5(B")
(qd "Ca"	"$(2"6(B")
(qd "CE"	 "$(2"7(B")
(qd "C"		 "$(2"8(B")
(qd "Co"	"$(2"9(B")
(qd "CW"	"$(2":(B")
(qd "CWa"	"$(2":(B")
(qd "CWe"	"$(2"8#L(B")
(qd "CWu"	"$(2"8#M(B")
(qd "CWi"	"$(2"8#N(B")
(qd "CWE"	"$(2"8#P(B")
(qd "CW'"	"$(2"8#M(B")

(qd "fe"	"$(2";(B")
(qd "fu"	"$(2"<(B")
(qd "fi"	"$(2"=(B")
(qd "fa"	"$(2">(B")
(qd "fE"	"$(2"?(B")
(qd "f"		"$(2"@(B")
(qd "fo"	"$(2"A(B")
(qd "fW"	"$(2"B(B")
(qd "fWa"	"$(2"B(B")
(qd "fWe"	"$(2"@#L(B")
(qd "fWu"	"$(2"@#M(B")
(qd "fWi"	"$(2"@#N(B")
(qd "fWE"	"$(2"@#P(B")
(qd "fW'"	"$(2"@#M(B")

(qd "Fe"	"$(2";(B")
(qd "Fu"	"$(2"<(B")
(qd "Fi"	"$(2"=(B")
(qd "Fa"	"$(2">(B")
(qd "FE"	"$(2"?(B")
(qd "F"		"$(2"@(B")
(qd "Fo"	"$(2"A(B")
(qd "FW"	"$(2"B(B")
(qd "FWa"	"$(2"B(B")
(qd "FWe"	"$(2"@#L(B")
(qd "FWu"	"$(2"@#M(B")
(qd "FWi"	"$(2"@#N(B")
(qd "FWE"	"$(2"@#P(B")
(qd "FW'"	"$(2"@#M(B")

(qd "se"	"$(2"C(B")
(qd "su"	"$(2"D(B")
(qd "si"	"$(2"E(B")
(qd "sa"	"$(2"F(B")
(qd "sE"	"$(2"G(B")
(qd "s"		"$(2"H(B")
(qd "so"	"$(2"I(B")
(qd "sW"	"$(2"J(B")
(qd "sWa"	"$(2"J(B")
(qd "sWe"	"$(2"H#L(B")
(qd "sWu"	"$(2"H#M(B")
(qd "sWi"	"$(2"H#N(B")
(qd "sWE"	"$(2"H#P(B")
(qd "sW'"	"$(2"H#M(B")

(qd "Se"	"$(2"K(B")
(qd "Su"	"$(2"L(B")
(qd "Si"	"$(2"M(B")
(qd "Sa"	"$(2"N(B")
(qd "SE"	"$(2"O(B")
(qd "S"		"$(2"P(B")
(qd "So"	"$(2"Q(B")
(qd "SW"	"$(2"R(B")
(qd "SWa"	"$(2"R(B")
(qd "SWe"	"$(2"P#L(B")
(qd "SWu"	"$(2"P#M(B")
(qd "SWi"	"$(2"P#N(B")
(qd "SWE"	"$(2"P#P(B")
(qd "SW'"	"$(2"P#M(B")

(qd "a"		"$(2"S(B")
(qd "e"		"$(2"S(B")
(qd "u"		"$(2"T(B")
(qd "U"		"$(2"T(B")
(qd "i"		"$(2"U(B")
(qd "A"		"$(2"V(B")
(qd "E"		"$(2"W(B")
(qd "I"		"$(2"X(B")
(qd "o"		"$(2"Y(B")
(qd "O"		"$(2"Y(B")
(qd "e3"	"$(2"Z(B")

(qd "he"	"$(2"[(B")
(qd "hu"	"$(2"\(B")
(qd "hi"	"$(2"](B")
(qd "ha"	"$(2"^(B")
(qd "hE"	"$(2"_(B")
(qd "h"		"$(2"`(B")
(qd "ho"	"$(2"a(B")
(qd "hW"	"$(2"c(B")
(qd "hWe"	"$(2"b(B")
(qd "hWu"	"$(2"c(B")
(qd "hWi"	"$(2"d(B")
(qd "hWa"	"$(2"e(B")
(qd "hWE"	"$(2"f(B")

(qd "ke"	"$(2"g(B")
(qd "ku"	"$(2"h(B")
(qd "ki"	"$(2"i(B")
(qd "ka"	"$(2"j(B")
(qd "kE"	"$(2"k(B")
(qd "k"		"$(2"l(B")
(qd "ko"	"$(2"m(B")
(qd "kW"	"$(2"o(B")
(qd "kWe"	"$(2"n(B")
(qd "kWu"	"$(2"o(B")
(qd "kWi"	"$(2"p(B")
(qd "kWa"	"$(2"q(B")
(qd "kWE"	"$(2"r(B")

(qd "Ke"	"$(2"s(B")
(qd "Ku"	"$(2"t(B")
(qd "Ki"	"$(2"u(B")
(qd "Ka"	"$(2"v(B")
(qd "KE"	"$(2"w(B")
(qd "K"		"$(2"x(B")
(qd "Ko"	"$(2"y(B")
(qd "KW"	"$(2"{(B")
(qd "KWe"	"$(2"z(B")
(qd "KWu"	"$(2"{(B")
(qd "KWi"	"$(2"|(B")
(qd "KWa"	"$(2"}(B")
(qd "KWE"	"$(2"~(B")

(qd "qe"	"$(2#!(B")
(qd "qu"	"$(2#"(B")
(qd "qi"	"$(2##(B")
(qd "qa"	"$(2#$(B")
(qd "qE"	"$(2#%(B")
(qd "q"		"$(2#&(B")
(qd "qo"	"$(2#'(B")
(qd "qW"	"$(2#)(B")
(qd "qWe"	"$(2#((B")
(qd "qWu"	"$(2#)(B")
(qd "qWi"	"$(2#*(B")
(qd "qWa"	"$(2#+(B")
(qd "qWE"	"$(2#,(B")

(qd "Qe"	"$(2#-(B")
(qd "Qu"	"$(2#.(B")
(qd "Qi"	"$(2#/(B")
(qd "Qa"	"$(2#0(B")
(qd "QE"	"$(2#1(B")
(qd "Q"		"$(2#2(B")
(qd "Qo"	"$(2#3(B")
(qd "QW"	"$(2#5(B")
(qd "QWe"	"$(2#4(B")
(qd "QWu"	"$(2#5(B")
(qd "QWi"	"$(2#6(B")
(qd "QWa"	"$(2#7(B")
(qd "QWE"	"$(2#8(B")

(qd "ge"	"$(2#9(B")
(qd "gu"	"$(2#:(B")
(qd "gi"	"$(2#;(B")
(qd "ga"	"$(2#<(B")
(qd "gE"	"$(2#=(B")
(qd "g"		"$(2#>(B")
(qd "go"	"$(2#?(B")
(qd "gW"	"$(2#A(B")
(qd "gWe"	"$(2#@(B")
(qd "gWu"	"$(2#A(B")
(qd "gWi"	"$(2#B(B")
(qd "gWa"	"$(2#C(B")
(qd "gWE"	"$(2#D(B")

(qd "we"	"$(2#E(B")
(qd "wu"	"$(2#F(B")
(qd "wi"	"$(2#G(B")
(qd "wa"	"$(2#H(B")
(qd "wE"	"$(2#I(B")
(qd "w"		"$(2#J(B")
(qd "wo"	"$(2#K(B")

(qd "W"		"$(2#M(B")
(qd "We"	"$(2#L(B")
(qd "Wu"	"$(2#M(B")
(qd "Wi"	"$(2#N(B")
(qd "Wa"	"$(2#O(B")
(qd "WE"	"$(2#P(B")

(qd "wW"	"$(2#M(B")
(qd "wWe"	"$(2#L(B")
(qd "wWu"	"$(2#M(B")
(qd "wWi"	"$(2#N(B")
(qd "wWa"	"$(2#O(B")
(qd "wWE"	"$(2#P(B")

(qd "`se"	"$(2#Q(B")
(qd "`su"	"$(2#R(B")
(qd "`si"	"$(2#S(B")
(qd "`sa"	"$(2#T(B")
(qd "`sE"	"$(2#U(B")
(qd "`s"	"$(2#V(B")
(qd "`so"	"$(2#W(B")
(qd "`sW"	"$(2"J(B")
(qd "`sWa"	"$(2"J(B")
(qd "`sWe"	"$(2#V#L(B")
(qd "`sWu"	"$(2#V#M(B")
(qd "`sWi"	"$(2#V#N(B")
(qd "`sWE"	"$(2#V#P(B")
(qd "`sW'"	"$(2#V#M(B")

(qd "s2e"	"$(2#Q(B")
(qd "s2u"	"$(2#R(B")
(qd "s2i"	"$(2#S(B")
(qd "s2a"	"$(2#T(B")
(qd "s2E"	"$(2#U(B")
(qd "s2"	"$(2#V(B")
(qd "s2o"	"$(2#W(B")
(qd "s2W"	"$(2"J(B")
(qd "s2Wa"	"$(2"J(B")
(qd "s2We"	"$(2#V#L(B")
(qd "s2Wu"	"$(2#V#M(B")
(qd "s2Wi"	"$(2#V#N(B")
(qd "s2WE"	"$(2#V#P(B")
(qd "s2W'"	"$(2#V#M(B")

(qd "sse"	"$(2#Q(B")
(qd "ssu"	"$(2#R(B")
(qd "ssi"	"$(2#S(B")
(qd "ssa"	"$(2#T(B")
(qd "ssE"	"$(2#U(B")
(qd "ss"	"$(2#V(B")
(qd "sso"	"$(2#W(B")
(qd "ssW"	"$(2"J(B")
(qd "ssWa"	"$(2"J(B")
(qd "ssWe"	"$(2#V#L(B")
(qd "ssWu"	"$(2#V#M(B")
(qd "ssWi"	"$(2#V#N(B")
(qd "ssWE"	"$(2#V#P(B")
(qd "ssW'"	"$(2#V#M(B")

(qd "`Se"	"$(2#X(B")
(qd "`Su"	"$(2#Y(B")
(qd "`Si"	"$(2#Z(B")
(qd "`Sa"	"$(2#[(B")
(qd "`SE"	"$(2#\(B")
(qd "`S"	"$(2#](B")
(qd "`So"	"$(2#^(B")
(qd "`SW"	"$(2"R(B")
(qd "`SWa"	"$(2"R(B")
(qd "`SWe"	"$(2#]#L(B")
(qd "`SWu"	"$(2#]#M(B")
(qd "`SWi"	"$(2#]#N(B")
(qd "`SWE"	"$(2#]#P(B")
(qd "`SW'"	"$(2#]#M(B")

(qd "S2e"	"$(2#X(B")
(qd "S2u"	"$(2#Y(B")
(qd "S2i"	"$(2#Z(B")
(qd "S2a"	"$(2#[(B")
(qd "S2E"	"$(2#\(B")
(qd "S2"	"$(2#](B")
(qd "S2o"	"$(2#^(B")
(qd "S2W"	"$(2"R(B")
(qd "S2Wa"	"$(2"R(B")
(qd "S2We"	"$(2#]#L(B")
(qd "S2Wu"	"$(2#]#M(B")
(qd "S2Wi"	"$(2#]#N(B")
(qd "S2WE"	"$(2#]#P(B")
(qd "S2W'"	"$(2#]#M(B")

(qd "SSe"	"$(2#X(B")
(qd "SSu"	"$(2#Y(B")
(qd "SSi"	"$(2#Z(B")
(qd "SSa"	"$(2#[(B")
(qd "SSE"	"$(2#\(B")
(qd "SS"	"$(2#](B")
(qd "SSo"	"$(2#^(B")
(qd "SSW"	"$(2"R(B")
(qd "SSWa"	"$(2"R(B")
(qd "SSWe"	"$(2#]#L(B")
(qd "SSWu"	"$(2#]#M(B")
(qd "SSWi"	"$(2#]#N(B")
(qd "SSWE"	"$(2#]#P(B")
(qd "SSW'"	"$(2#]#M(B")

(qd "`e"	"$(2#_(B")
(qd "`u"	"$(2#`(B")
(qd "`U"	"$(2#`(B")
(qd "`i"	"$(2#a(B")
(qd "`a"	"$(2#b(B")
(qd "`A"	"$(2#b(B")
(qd "`E"	"$(2#c(B")
(qd "`I"	"$(2#d(B")
(qd "`o"	"$(2#e(B")
(qd "`O"	"$(2#e(B")

(qd "e2"	"$(2#_(B")
(qd "u2"	"$(2#`(B")
(qd "U2"	"$(2#`(B")
(qd "i2"	"$(2#a(B")
(qd "a2"	"$(2#b(B")
(qd "A2"	"$(2#b(B")
(qd "E2"	"$(2#c(B")
(qd "I2"	"$(2#d(B")
(qd "o2"	"$(2#e(B")
(qd "O2"	"$(2#e(B")

(qd "`he"	"$(2#f(B")
(qd "`hu"	"$(2#g(B")
(qd "`hi"	"$(2#h(B")
(qd "`ha"	"$(2#i(B")
(qd "`hE"	"$(2#j(B")
(qd "`h"	"$(2#k(B")
(qd "`ho"	"$(2#l(B")
(qd "`hW"	"$(2"c(B")
(qd "`hWe"	"$(2"b(B")
(qd "`hWu"	"$(2"c(B")
(qd "`hWi"	"$(2"d(B")
(qd "`hWa"	"$(2"e(B")
(qd "`hWE"	"$(2"f(B")

(qd "h2e"	"$(2#f(B")
(qd "h2u"	"$(2#g(B")
(qd "h2i"	"$(2#h(B")
(qd "h2a"	"$(2#i(B")
(qd "h2E"	"$(2#j(B")
(qd "h2"	"$(2#k(B")
(qd "h2o"	"$(2#l(B")
(qd "h2W"	"$(2"c(B")
(qd "h2We"	"$(2"b(B")
(qd "h2Wu"	"$(2"c(B")
(qd "h2Wi"	"$(2"d(B")
(qd "h2Wa"	"$(2"e(B")
(qd "h2WE"	"$(2"f(B")

(qd "hhe"	"$(2#f(B")
(qd "hhu"	"$(2#g(B")
(qd "hhi"	"$(2#h(B")
(qd "hha"	"$(2#i(B")
(qd "hhE"	"$(2#j(B")
(qd "hh"	"$(2#k(B")
(qd "hho"	"$(2#l(B")
(qd "hhW"	"$(2"c(B")
(qd "hhWe"	"$(2"b(B")
(qd "hhWu"	"$(2"c(B")
(qd "hhWi"	"$(2"d(B")
(qd "hhWa"	"$(2"e(B")
(qd "hhWE"	"$(2"f(B")

(qd "`ke"	"$(2#m(B")
(qd "`ku"	"$(2#n(B")
(qd "`ki"	"$(2#o(B")
(qd "`ka"	"$(2#p(B")
(qd "`kE"	"$(2#q(B")
(qd "`k"	"$(2#r(B")
(qd "`ko"	"$(2#s(B")

(qd "k2e"	"$(2#m(B")
(qd "k2u"	"$(2#n(B")
(qd "k2i"	"$(2#o(B")
(qd "k2a"	"$(2#p(B")
(qd "k2E"	"$(2#q(B")
(qd "k2"	"$(2#r(B")
(qd "k2o"	"$(2#s(B")

(qd "kke"	"$(2#m(B")
(qd "kku"	"$(2#n(B")
(qd "kki"	"$(2#o(B")
(qd "kka"	"$(2#p(B")
(qd "kkE"	"$(2#q(B")
(qd "kk"	"$(2#r(B")
(qd "kko"	"$(2#s(B")

(qd "Xe"	"$(2#t(B")
(qd "Xu"	"$(2#u(B")
(qd "Xi"	"$(2#v(B")
(qd "Xa"	"$(2#w(B")
(qd "XE"	"$(2#x(B")
(qd "X"		"$(2#y(B")
(qd "Xo"	"$(2#z(B")

(qd "He"	"$(2#{(B")
(qd "Hu"	"$(2#|(B")
(qd "Hi"	"$(2#}(B")
(qd "Ha"	"$(2#~(B")
(qd "HE"	"$(2$!(B")
(qd "H"		"$(2$"(B")
(qd "Ho"	"$(2$#(B")

(qd "ye"	"$(2$$(B")
(qd "yu"	"$(2$%(B")
(qd "yi"	"$(2$&(B")
(qd "ya"	"$(2$'(B")
(qd "yE"	"$(2$((B")
(qd "y"		"$(2$)(B")
(qd "yo"	"$(2$*(B")

(qd "Ye"	"$(2$$(B")
(qd "Yu"	"$(2$%(B")
(qd "Yi"	"$(2$&(B")
(qd "Ya"	"$(2$'(B")
(qd "YE"	"$(2$((B")
(qd "Y"		"$(2$)(B")
(qd "Yo"	"$(2$*(B")

(qd "De"	"$(2$+(B")
(qd "Du"	"$(2$,(B")
(qd "Di"	"$(2$-(B")
(qd "Da"	"$(2$.(B")
(qd "DE"	"$(2$/(B")
(qd "D"		"$(2$0(B")
(qd "Do"	"$(2$1(B")

(qd "je"	"$(2$2(B")
(qd "ju"	"$(2$3(B")
(qd "ji"	"$(2$4(B")
(qd "ja"	"$(2$5(B")
(qd "jE"	"$(2$6(B")
(qd "j"		"$(2$7(B")
(qd "jo"	"$(2$8(B")

(qd "Je"	"$(2$2(B")
(qd "Ju"	"$(2$3(B")
(qd "Ji"	"$(2$4(B")
(qd "Ja"	"$(2$5(B")
(qd "JE"	"$(2$6(B")
(qd "J"		"$(2$7(B")
(qd "Jo"	"$(2$8(B")

(qd "Ge"	"$(2$9(B")
(qd "Gu"	"$(2$:(B")
(qd "Gi"	"$(2$;(B")
(qd "Ga"	"$(2$<(B")
(qd "GE"	"$(2$=(B")
(qd "G"		"$(2$>(B")
(qd "Go"	"$(2$?(B")

(qd "Pe"	"$(2$@(B")
(qd "Pu"	"$(2$A(B")
(qd "Pi"	"$(2$B(B")
(qd "Pa"	"$(2$C(B")
(qd "PE"	"$(2$D(B")
(qd "P"		"$(2$E(B")
(qd "Po"	"$(2$F(B")

(qd "pe"	"$(2$G(B")
(qd "pu"	"$(2$H(B")
(qd "pi"	"$(2$I(B")
(qd "pa"	"$(2$J(B")
(qd "pE"	"$(2$K(B")
(qd "p"		"$(2$L(B")
(qd "po"	"$(2$M(B")

(qd "'"		"")
(qd "''"	"'")
(qd "::"	"$(2$O(B")
(qd "."		"$(2$P(B")
(qd ".."	".")
(qd ","		"$(2$Q(B")
(qd ",,"	",")
(qd ";"		"$(2$R(B")
(qd ";;"	";")
(qd ":"		"$(2$S(B")
(qd ":::"	":")
(qd "*"		"$(2$T(B")
(qd "**"	"*")
(qd "<<"	"$(2$U(B")
(qd ">>"	"$(2$V(B")

;; These two are necessary for toggling commands.
(qd "1"		"1")
(qd " "		" ")

(quail-setup-current-package)
