/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/* FLASK */

#include "sidtab.h"
#include "extension.h"
#include "services.h"

#define SIDTAB_HASH(key) \
(key % SIDTAB_SIZE)

#define SIDTAB_EQ(key1,key2) \
(key1 == key2)

static void 
sid_datum_destroy(sid_datum_t *datum)
{
	hashtab_map(datum->children.table, invalidate_child, 0);
	hashtab_destroy(datum->children.table);
	datum->children.table = 0;
	if (datum->child_val_to_name) {
		FREE(datum->child_val_to_name,
		     (datum->children.nprim / CVTN_NPRIM_INCR + 1) * CVTN_NPRIM_INCR * sizeof(char *));
	}
	context_destroy(&datum->context);
}


int 
sidtab_insert(sidtab_t * h, security_id_t key, sid_datum_t * datum)
{
	int             hvalue;
	sidtab_ptr_t     cur, newnode;


	if (!h)
		return -ENOMEM;

	hvalue = SIDTAB_HASH(key);
	cur = h->htable[hvalue];
	while (cur != NULL && !SIDTAB_EQ(cur->key, key))
		cur = cur->next;

	if (cur != NULL)
		return -EEXIST;

	newnode = (sidtab_ptr_t) malloc(sizeof(struct sidtab_node_t));
	if (newnode == NULL)
		return -ENOMEM;
	newnode->key = key;
	newnode->datum = *datum;
	newnode->next = h->htable[hvalue];
	h->htable[hvalue] = newnode;

	h->nel++;
	return 0;
}


int 
sidtab_remove(sidtab_t *h, security_id_t key)
{
	int             hvalue;
	sidtab_ptr_t   cur, last;


	if (!h)
		return -ENOENT;

	hvalue = SIDTAB_HASH(key);
	last = NULL;
	cur = h->htable[hvalue];
	while (cur != NULL && !SIDTAB_EQ(cur->key, key)) {
		last = cur;
		cur = cur->next;
	}

	if (cur == NULL)
		return -ENOENT;

	if (last == NULL)
		h->htable[hvalue] = cur->next;
	else
		last->next = cur->next;

	sid_datum_destroy(&cur->datum);

	free(cur);
	return 0;
}


sid_datum_t  *
sidtab_search(sidtab_t * h, security_id_t key)
{
	int             hvalue;
	sidtab_ptr_t     cur;


	if (!h)
		return NULL;

	hvalue = SIDTAB_HASH(key);
	cur = h->htable[hvalue];
	while (cur != NULL && !SIDTAB_EQ(cur->key, key))
		cur = cur->next;

	if (cur == NULL)
		return NULL;

	return &cur->datum;
}


int 
sidtab_map(sidtab_t * h,
	     int (*apply) (security_id_t key,
			sid_datum_t * d,
			void *args),
	  void *args)
{
	int             i, ret;
	sidtab_ptr_t     cur;


	if (!h)
		return 0;

	for (i = 0; i < SIDTAB_SIZE; i++) {
		cur = h->htable[i];
		while (cur != NULL) {
			ret = apply(cur->key, &cur->datum, args);
			if (ret)
				return ret;
			cur = cur->next;
		}
	}
	return 0;
}


void 
sidtab_map_remove_on_error(sidtab_t *h,
			   int (*apply) (security_id_t sid,
					 sid_datum_t *d,
					 void *args),
			   void *args)
{
	int             i, ret;
	sidtab_ptr_t   last, cur, temp;


	if (!h)
		return;

	for (i = 0; i < SIDTAB_SIZE; i++) {
		last = NULL;
		cur = h->htable[i];
		while (cur != NULL) {
			ret = apply(cur->key, &cur->datum, args);
			if (ret) {
				if (last) {
					last->next = cur->next;
				} else {
					h->htable[i] = cur->next;
				}

				temp = cur;
				cur = cur->next;
				sid_datum_destroy(&temp->datum);
				free(temp);
			} else {
				last = cur;
				cur = cur->next;
			}
		}
	}

	return;
}


int 
sidtab_init(sidtab_t * h)
{
	int             i;


	for (i = 0; i < SIDTAB_SIZE; i++)
		h->htable[i] = (sidtab_ptr_t) NULL;
	h->nel = 0;
	return 0;
}


int 
sidtab_context_to_sid(sidtab_t * s,
		      ss_context_t * context,
		      security_id_t * out_sid)
{
	security_id_t	key;
	sid_datum_t	*datum;
	sidtab_ptr_t     cur;
	int             i, ret;


	*out_sid = SECSID_NULL;

	/* check to see if there is already a sid for this context */
	for (i = 0; i < SIDTAB_SIZE; i++) {
		cur = s->htable[i];
		while (cur != NULL) {
			if (context_cmp(&cur->datum.context, context)) {
				*out_sid = cur->key;
				return 0;
			}
			cur = cur->next;
		}
	}

	/* no sid exists; need to create a new entry in the sid table */
	key = s->nel + 1;
	if (key == 0)
		return -1;
	datum = (sid_datum_t *) MALLOC(sizeof(sid_datum_t));
	if (!datum) {
		return -1;
	}
	if (!context_cpy(&datum->context, context)) {
		FREE(datum, sizeof(sid_datum_t));
		return -1;
	}
	datum->children.table = 0;
	datum->children.nprim = 0;
	datum->child_val_to_name = 0;

	ret = sidtab_insert(s, key, datum);
	if (ret) {
		context_destroy(&datum->context);
		FREE(datum, sizeof(sid_datum_t));
		return -1;
	}
 
	*out_sid = key;
	return 0;
}

/* FLASK */
