/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
#include "policydb.h"

int 
policydb_init(policydb_t * p)
{
	memset(p, 0, sizeof(policydb_t));

	p->sids = (symtab_t *) MALLOC(sizeof(symtab_t));
	if (!p->sids)
		return -1;
	if (symtab_init(p->sids))
		return -1;

	p->commons = (symtab_t *) MALLOC(sizeof(symtab_t));
	if (!p->commons)
		return -1;
	if (symtab_init(p->commons))
		return -1;

	p->classes = (symtab_t *) MALLOC(sizeof(symtab_t));
	if (!p->classes)
		return -1;
	if (symtab_init(p->classes))
		return -1;


	p->avdefs = (avdeftab_t *) MALLOC(sizeof(avdeftab_t));
	if (!p->avdefs)
		return -1;
	if (avdeftab_init(p->avdefs))
		return -1;

	p->roles = (symtab_t *) MALLOC(sizeof(symtab_t));
	if (!p->roles)
		return -1;
	if (symtab_init(p->roles))
		return -1;

	p->types = (symtab_t *) MALLOC(sizeof(symtab_t));
	if (!p->types)
		return -1;
	if (symtab_init(p->types))
		return -1;

	p->transitions = (trtab_t *) MALLOC(sizeof(trtab_t));
	if (!p->transitions)
		return -1;
	if (trtab_init(p->transitions))
		return -1;

	p->levels = (symtab_t *) MALLOC(sizeof(symtab_t));
	if (!p->levels)
		return -1;
	if (symtab_init(p->levels))
		return -1;

	p->cats = (symtab_t *) MALLOC(sizeof(symtab_t));
	if (!p->cats)
		return -1;
	if (symtab_init(p->cats))
		return -1;

	p->users = (symtab_t *) MALLOC(sizeof(symtab_t));
	if (!p->users)
		return -1;
	if (symtab_init(p->users))
		return -1;

	p->rules = (avtab_t *) MALLOC(sizeof(avtab_t));
	if (!p->rules)
		return -1;
	if (avtab_init(p->rules))
		return -1;

	p->provided = (prvavtab_t *) MALLOC(sizeof(prvavtab_t));
	if (!p->provided)
		return -1;
	if (prvavtab_init(p->provided))
		return -1;

	return 0;
}


static int 
make_rvtns(hashtab_key_t key, hashtab_datum_t datum, void *datap)
{
	policydb_t     *p;
	role_datum_t   *role;


	role = (role_datum_t *) datum;
	p = (policydb_t *) datap;

	p->role_val_to_name[role->value - 1] = (char *) key;
	p->role_val_to_struct[role->value - 1] = role;

	return 0;
}


static int 
make_tvtn(hashtab_key_t key, hashtab_datum_t datum, void *datap)
{
	char          **tvtnp;
	type_datum_t   *typdatum;


	typdatum = (type_datum_t *) datum;
	tvtnp = (char **) datap;

	if (!typdatum->isalias)
		tvtnp[typdatum->info->value - 1] = (char *) key;

	return 0;
}


static int 
make_svtn(hashtab_key_t key, hashtab_datum_t datum, void *datap)
{
	char          **svtnp;
	level_datum_t  *levdatum;


	levdatum = (level_datum_t *) datum;
	svtnp = (char **) datap;

	if (!levdatum->isalias)
		svtnp[levdatum->level->sens - 1] = (char *) key;

	return 0;
}


static int 
make_cvtn(hashtab_key_t key, hashtab_datum_t datum, void *datap)
{
	char          **cvtnp;
	cat_datum_t    *catdatum;


	catdatum = (cat_datum_t *) datum;
	cvtnp = (char **) datap;

	if (!catdatum->isalias)
		cvtnp[catdatum->value - 1] = (char *) key;

	return 0;
}


static int 
make_uvtns(hashtab_key_t key, hashtab_datum_t datum, void *datap)
{
	policydb_t     *p;
	user_datum_t   *usrdatum;


	usrdatum = (user_datum_t *) datum;
	p = (policydb_t *) datap;

	p->user_val_to_name[usrdatum->value - 1] = (char *) key;
	p->user_val_to_struct[usrdatum->value - 1] = usrdatum;

	return 0;
}


static int 
set_ncons(avtab_key_t * avkey, avtab_datum_t * avdatum, void *datap)
{
	avtab_t        *rules = (avtab_t *) datap;

	if (avdatum->avdefdatum->constraints)
		rules->ncons++;

	return 0;
}


int 
policydb_transform(policydb_t * p)
{
	p->role_val_to_name = (char **) MALLOC(p->roles->nprim * sizeof(char *));
	p->role_val_to_struct = (role_datum_t **) MALLOC(p->roles->nprim * sizeof(role_datum_t *));
	if (!p->role_val_to_name || !p->role_val_to_struct)
		return -1;
	if (hashtab_map(p->roles->table, make_rvtns, p))
		return -1;

	p->type_val_to_name = (char **) MALLOC(p->types->nprim * sizeof(char *));
	if (!p->type_val_to_name)
		return -1;
	if (hashtab_map(p->types->table, make_tvtn, p->type_val_to_name))
		return -1;

	p->sens_val_to_name = (char **) MALLOC(p->levels->nprim * sizeof(char *));
	if (!p->sens_val_to_name)
		return -1;
	if (hashtab_map(p->levels->table, make_svtn, p->sens_val_to_name))
		return -1;

	p->cat_val_to_name = (char **) MALLOC(p->cats->nprim * sizeof(char *));
	if (!p->cat_val_to_name)
		return -1;
	if (hashtab_map(p->cats->table, make_cvtn, p->cat_val_to_name))
		return -1;

	p->user_val_to_name = (char **) MALLOC(p->users->nprim * sizeof(char *));
	if (!p->user_val_to_name)
		return -1;
	p->user_val_to_struct = (user_datum_t **)
		MALLOC(p->users->nprim * sizeof(user_datum_t *));
	if (!p->user_val_to_struct)
		return -1;
	if (hashtab_map(p->users->table, make_uvtns, p))
		return -1;

	avtab_map(p->rules, set_ncons, p->rules);

	return 0;
}


static int 
sid_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	initial_sid_datum_t *siddatum;


	siddatum = (initial_sid_datum_t *) datum;
	root_context_destroy(&siddatum->context);
	FREE(key, strlen(key) + 1);
	FREE(datum, sizeof(initial_sid_datum_t));
	return 0;
}

static int 
perm_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	FREE(key, strlen(key) + 1);
	FREE(datum, sizeof(perm_datum_t));
	return 0;
}

static int 
common_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	symtab_t       *comdatum;

	FREE(key, strlen(key) + 1);
	comdatum = (symtab_t *) datum;
	hashtab_map(comdatum->table, perm_destroy, 0);
	hashtab_destroy(comdatum->table);
	FREE(datum, sizeof(symtab_t));
	return 0;
}


static int 
class_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	FREE(key, strlen(key) + 1);
	FREE(datum, sizeof(class_datum_t));
	return 0;
}

static int 
role_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	role_datum_t   *role;

	FREE(key, strlen(key) + 1);
	role = (role_datum_t *) datum;
	ebitmap_destroy(&role->dominates);
	ebitmap_destroy(&role->types);
	FREE(datum, sizeof(role_datum_t));
	return 0;
}

static int 
type_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	type_datum_t   *typdatum;

	FREE(key, strlen(key) + 1);
	typdatum = (type_datum_t *) datum;
	if (!typdatum->isalias)
		FREE(typdatum->info, sizeof(type_info_t));
	FREE(datum, sizeof(type_datum_t));
	return 0;
}

static int 
level_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	level_datum_t  *levdatum;

	FREE(key, strlen(key) + 1);
	levdatum = (level_datum_t *) datum;
	if (!levdatum->isalias) {
		ebitmap_destroy(&levdatum->level->cat);
		FREE(levdatum->level, sizeof(mls_level_t));
	}
	FREE(datum, sizeof(level_datum_t));
	return 0;
}

static int 
cat_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	FREE(key, strlen(key) + 1);
	FREE(datum, sizeof(cat_datum_t));
	return 0;
}

static int 
range_destroy(queue_element_t e, void *p)
{
	mls_range_t    *r;

	r = (mls_range_t *) e;
	ebitmap_destroy(&r->low.cat);
	ebitmap_destroy(&r->high.cat);
	return 0;
}

static int 
user_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	user_datum_t   *usrdatum;


	FREE(key, strlen(key) + 1);
	usrdatum = (user_datum_t *) datum;
	ebitmap_destroy(&usrdatum->roles);
	queue_map(usrdatum->ranges, range_destroy, 0);
	queue_destroy(usrdatum->ranges);
	FREE(datum, sizeof(user_datum_t));
	return 0;
}

static int 
tr_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	FREE(key, sizeof(tr_key_t));
	FREE(datum, sizeof(tr_datum_t));
	return 0;
}

static int 
prvav_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	FREE(key, sizeof(prvav_key_t));
	FREE(datum, sizeof(prvav_datum_t));
	return 0;
}

void 
policydb_destroy(policydb_t * p)
{
	if (p->role_val_to_name)
		FREE(p->role_val_to_name, p->roles->nprim * sizeof(char *));
	if (p->role_val_to_struct)
		FREE(p->role_val_to_struct, p->roles->nprim * sizeof(char *));
	if (p->type_val_to_name)
		FREE(p->type_val_to_name, p->types->nprim * sizeof(char *));
	if (p->sens_val_to_name)
		FREE(p->sens_val_to_name, p->levels->nprim * sizeof(char *));
	if (p->cat_val_to_name)
		FREE(p->cat_val_to_name, p->cats->nprim * sizeof(char *));
	if (p->user_val_to_name)
		FREE(p->user_val_to_name, p->users->nprim * sizeof(char *));
	if (p->user_val_to_struct)
		FREE(p->user_val_to_struct, p->users->nprim * sizeof(user_datum_t *));
	if (p->sids) {
		hashtab_map(p->sids->table, sid_destroy, 0);
		hashtab_destroy(p->sids->table);
		FREE(p->sids, sizeof(symtab_t));
	}
	if (p->commons) {
		hashtab_map(p->commons->table, common_destroy, 0);
		hashtab_destroy(p->commons->table);
		FREE(p->commons, sizeof(symtab_t));
	}
	if (p->classes) {
		hashtab_map(p->classes->table, class_destroy, 0);
		hashtab_destroy(p->classes->table);
		FREE(p->classes, sizeof(symtab_t));
	}
	if (p->avdefs) {
		avdeftab_destroy(p->avdefs);
		FREE(p->avdefs, sizeof(avdeftab_t));
	}
	if (p->roles) {
		hashtab_map(p->roles->table, role_destroy, 0);
		hashtab_destroy(p->roles->table);
		FREE(p->roles, sizeof(symtab_t));
	}
	if (p->types) {
		hashtab_map(p->types->table, type_destroy, 0);
		hashtab_destroy(p->types->table);
		FREE(p->types, sizeof(symtab_t));
	}
	if (p->transitions) {
		hashtab_map(p->transitions->table, tr_destroy, 0);
		hashtab_destroy(p->transitions->table);
		FREE(p->transitions, sizeof(trtab_t));
	}
	if (p->levels) {
		hashtab_map(p->levels->table, level_destroy, 0);
		hashtab_destroy(p->levels->table);
		FREE(p->levels, sizeof(symtab_t));;
	}
	if (p->cats) {
		hashtab_map(p->cats->table, cat_destroy, 0);
		hashtab_destroy(p->cats->table);
		FREE(p->cats, sizeof(symtab_t));
	}
	if (p->users) {
		hashtab_map(p->users->table, user_destroy, 0);
		hashtab_destroy(p->users->table);
		FREE(p->users, sizeof(symtab_t));
	}
	if (p->rules) {
		avtab_destroy(p->rules);
		FREE(p->rules, sizeof(avtab_t));
	}
	if (p->provided) {
		hashtab_map(p->provided->table, prvav_destroy, 0);
		hashtab_destroy(p->provided->table);
		FREE(p->provided, sizeof(prvavtab_t));
	}
	return;
}
