/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/* FLASK */

%{
#include "policydb.h"
#include "services.h"
#include "sidtab.h"
#include "extavtab.h"
#include "extension.h"

parent_info_t parent_info;

extern char exttext[];
extern int exterror(char *msg);

static char errormsg[255];

static int insert_separator(int);
static int insert_id(char *,int);

static int add_child(void);
static int remove_child(void);

#define CHILD_CHILD	1
#define CHILD_PARENT	2
#define PARENT_CHILD	3
static int define_permissions(int);
static int define_transition(int);
%}

%token CLASS
%token PARENT
%token CHILD
%token ADD
%token REMOVE
%token ALLOW
%token SID_TRANSITION
%token IDENTIFIER
%%
extension		: adds
	                | removes
	                | permissions
	                | transitions
	                | adds permissions
	                | adds permissions transitions
	                | permissions transitions
	 		;
removes			: remove_def 
			| removes remove_def
			;
remove_def		: REMOVE CHILD identifier ';'
			{if (remove_child()) return -1;}
			;
adds			: add_def 
			| adds add_def
			;
add_def			: ADD CHILD identifier ';'
			{if (add_child()) return -1;}
			;
permissions		: permission_def
			| permissions permission_def
			;
permission_def		: ALLOW identifier identifier ':' identifier perms_def ';'
			{if (define_permissions(CHILD_CHILD)) return -1;}
                        | ALLOW identifier PARENT ':' identifier perms_def ';'
			{if (define_permissions(CHILD_PARENT)) return -1;}
                        | ALLOW PARENT identifier ':' identifier perms_def ';'
			{if (define_permissions(PARENT_CHILD)) return -1;}
	                ;
perms_def		: '{' identifier_list '}'
	                | identifier
			;
transitions		: transition_def 
			| transitions transition_def
			;
transition_def		: SID_TRANSITION identifier identifier ':' identifier identifier ';'
                        {if (define_transition(CHILD_CHILD)) return -1;}
                        | SID_TRANSITION identifier PARENT ':' identifier identifier ';'
                        {if (define_transition(CHILD_PARENT)) return -1;}
                        | SID_TRANSITION PARENT identifier ':' identifier identifier ';'
                        {if (define_transition(PARENT_CHILD)) return -1;}
    			;
identifier_list		: identifier
			| identifier_list identifier
			;
identifier		: IDENTIFIER
			{ if (insert_id(exttext,0)) return -1; }
%%
static int 
insert_separator(int push)
{
	int             error;

	if (push)
		error = queue_push(id_queue, 0);
	else
		error = queue_insert(id_queue, 0);

	if (error) {
		yyerror("queue overflow");
		return -1;
	}
	return 0;
}

static int 
insert_id(char *id, int push)
{
	char           *newid;
	int             error;

	newid = (char *) MALLOC(strlen(id) + 1);
	if (!newid) {
		yyerror("out of memory");
		return -1;
	}
	strcpy(newid, id);
	if (push)
		error = queue_push(id_queue, (queue_element_t) newid);
	else
		error = queue_insert(id_queue, (queue_element_t) newid);

	if (error) {
		yyerror("queue overflow");
		free(newid);
		return -1;
	}
	return 0;
}


static void 
ss_dump_av(security_class_t tclass, access_vector_t av)
{
	return;
}


static void 
do_nothing(hashtab_key_t key, hashtab_datum_t datum, void *args)
{
	return;
}

static int 
add_child(void)
{
	char           *id;
	ss_context_t    context;
	ss_child_context_t *child;
	child_datum_t  *childdatum;
	int             ret;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		exterror("no child name for child definition?");
		return -1;
	}
	childdatum = (child_datum_t *)
		hashtab_search(parent_info.datum->children.table,
			       (hashtab_key_t) id);
	if (childdatum) {
		sprintf(errormsg, "child %s is already defined for parent", id);
		exterror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	if (!parent_info.datum->children.table) {
		/* initialize the parent SID's child table */
		if (symtab_init(&parent_info.datum->children)) {
			exterror("out of memory");
			FREE(id, strlen(id) + 1);
			return -1;
		}
		parent_info.datum->child_val_to_name = (char **)
			MALLOC(CVTN_NPRIM_INCR * sizeof(char *));
		if (!parent_info.datum->child_val_to_name) {
			exterror("out of memory");
			hashtab_destroy(parent_info.datum->children.table);
			parent_info.datum->children.table = 0;
			FREE(id, strlen(id) + 1);
			return -1;
		}
	}
	/* define the child context */
	context_init(&context, FALSE);
	child = &CCONTEXT(&context);
	if (parent_info.datum->context.isroot)
		child->depth = 1;
	else
		child->depth = CCONTEXT(&parent_info.datum->context).depth + 1;
	child->parent = parent_info.sid;
	child->value = ++parent_info.datum->children.nprim;

	/* define the child datum */

	childdatum = (child_datum_t *) MALLOC(sizeof(child_datum_t));
	if (!childdatum) {
		exterror("out of memory");
		FREE(id, strlen(id) + 1);
		--parent_info.datum->children.nprim;
		return -1;
	}
	childdatum->value = child->value;
	if (sidtab_context_to_sid(&sidtab, &context, &childdatum->sid)) {
		exterror("out of memory");
		FREE(childdatum, sizeof(child_datum_t));
		FREE(id, strlen(id) + 1);
		--parent_info.datum->children.nprim;
		return -1;
	}
	ret = hashtab_insert(parent_info.datum->children.table,
			  (hashtab_key_t) id, (hashtab_datum_t) childdatum);

	if (ret == HASHTAB_PRESENT) {
		--parent_info.datum->children.nprim;
		FREE(childdatum, sizeof(child_datum_t));
		FREE(id, strlen(id) + 1);
		exterror("duplicate child definition");
		return -1;
	}
	if (ret == HASHTAB_OVERFLOW) {
		--parent_info.datum->children.nprim;
		FREE(childdatum, sizeof(child_datum_t));
		FREE(id, strlen(id) + 1);
		exterror("hash table overflow");
		return -1;
	}
	if ((parent_info.datum->children.nprim % CVTN_NPRIM_INCR) == 0) {
		unsigned int    old_size, new_size;
		char          **cvtn;


		old_size = parent_info.datum->children.nprim * sizeof(char *);
		new_size = old_size + CVTN_NPRIM_INCR * sizeof(char *);
		cvtn = (char **) malloc(new_size);
		if (!cvtn) {
			exterror("out of memory");
			hashtab_remove(parent_info.datum->children.table, id, do_nothing, 0);
			--parent_info.datum->children.nprim;
			FREE(childdatum, sizeof(child_datum_t));
			FREE(id, strlen(id) + 1);
			return -1;
		}
		memcpy(cvtn, parent_info.datum->child_val_to_name, old_size);

		parent_info.datum->child_val_to_name = cvtn;
	}
	parent_info.datum->child_val_to_name[child->value - 1] = id;

#if VERBOSE
	printf("ss:  Added SID %ld for child %s of parent SID %ld\n",
	       childdatum->sid, id, parent_info.sid);
#endif	/* VERBOSE */

	return 0;
}


int 
invalidate_child(hashtab_key_t key, hashtab_datum_t datum, void *ptr)
{
	child_datum_t  *childdatum;


	childdatum = (child_datum_t *) datum;
	sidtab_remove(&sidtab, childdatum->sid);
	FREE(key, strlen(key) + 1);
	FREE(childdatum, sizeof(child_datum_t));

	return 0;
}


static void 
destroy_name(hashtab_key_t key, hashtab_datum_t datum, void *args)
{
	FREE(key, strlen(key) + 1);
}

static int 
remove_child(void)
{
	char           *id;
	child_datum_t  *childdatum;


	id = (char *) queue_remove(id_queue);
	if (!id) {
		exterror("no child name for child definition?");
		return -1;
	}
	childdatum = (child_datum_t *)
		hashtab_search(parent_info.datum->children.table,
			       (hashtab_key_t) id);
	if (!childdatum) {
		sprintf(errormsg, "child %s is not defined for parent", id);
		exterror(errormsg);
		FREE(id, strlen(id) + 1);
		return -1;
	}
	hashtab_remove(parent_info.datum->children.table, id, destroy_name, 0);

	return invalidate_child(id, (hashtab_datum_t) childdatum, &parent_info);
}


static int 
define_permissions(int kind)
{
	char           *id, *tclassid = 0;
	extav_key_t    *extavkey;
	extav_datum_t  *extavdatum;
	avdef_datum_t  *avddatum;
	class_datum_t  *cladatum;
	perm_datum_t   *perdatum;
	child_datum_t  *childdatum;
	access_vector_t requested, allowed, decided, auditallow, auditdeny,
	                notify;
	int             ret, rc;
	unsigned int    seqno;


	extavkey = (extav_key_t *) MALLOC(sizeof(extav_key_t));
	if (!extavkey) {
		exterror("out of memory");
		return -1;
	}
	if (kind == PARENT_CHILD) {
		extavkey->source_sid = parent_info.sid;
	} else {
		id = (char *) queue_remove(id_queue);
		if (!id) {
			exterror("no source child name for permission definition?");
			FREE(extavkey, sizeof(extav_key_t));
			return -1;
		}
		childdatum = (child_datum_t *)
			hashtab_search(parent_info.datum->children.table,
				       (hashtab_key_t) id);
		if (!childdatum) {
			sprintf(errormsg, "unknown child %s used in permission definition", id);
			exterror(errormsg);
			FREE(extavkey, sizeof(extav_key_t));
			FREE(id, strlen(id) + 1);
			return -1;
		}
		FREE(id, strlen(id) + 1);
		extavkey->source_sid = childdatum->sid;
	}

	if (kind == CHILD_PARENT) {
		extavkey->target_sid = parent_info.sid;
	} else {
		id = (char *) queue_remove(id_queue);
		if (!id) {
			exterror("no target child name for permission definition?");
			FREE(extavkey, sizeof(extav_key_t));
			return -1;
		}
		childdatum = (child_datum_t *)
			hashtab_search(parent_info.datum->children.table,
				       (hashtab_key_t) id);
		if (!childdatum) {
			sprintf(errormsg, "unknown child %s used in permission definition", id);
			exterror(errormsg);
			FREE(extavkey, sizeof(extav_key_t));
			return -1;
		}
		FREE(id, strlen(id) + 1);
		extavkey->target_sid = childdatum->sid;
	}

	tclassid = (char *) queue_remove(id_queue);
	if (!tclassid) {
		exterror("no target class name for permission definition?");
		FREE(extavkey, sizeof(extav_key_t));
		return -1;
	}
	cladatum = (class_datum_t *) hashtab_search(policydbp->classes->table,
						  (hashtab_key_t) tclassid);
	if (!cladatum) {
		sprintf(errormsg, "unknown class %s used in permission definition", tclassid);
		exterror(errormsg);
		FREE(extavkey, sizeof(extav_key_t));
		FREE(tclassid, strlen(tclassid) + 1);
		return -1;
	}
	extavkey->target_class = cladatum->value;

	avddatum = avdeftab_search(policydbp->avdefs, extavkey->target_class);
	if (!avddatum) {
		sprintf(errormsg, "no access vector definition for %s\n",
			tclassid);
		exterror(errormsg);
		FREE(tclassid, strlen(tclassid) + 1);
		FREE(extavkey, sizeof(extav_key_t));
		return -1;
	}
	requested = 0;
	while ((id = queue_remove(id_queue))) {
		perdatum = (perm_datum_t *) hashtab_search(avddatum->private.table,
							(hashtab_key_t) id);
		if (!perdatum) {
			if (avddatum->common) {
				perdatum = (perm_datum_t *) hashtab_search(avddatum->common->table,
							(hashtab_key_t) id);
			}
			if (!perdatum) {
				sprintf(errormsg, "permission %s is not defined for %s", id, tclassid);
				exterror(errormsg);
				FREE(id, strlen(id) + 1);
				continue;
			}
		}
		requested |= (1 << (perdatum->value - 1));

		FREE(id, strlen(id) + 1);
	}

	FREE(tclassid, strlen(tclassid) + 1);

	/* verify that the requested perms are allowed */
	rc = security_compute_av(parent_info.sid,
				 parent_info.sid,
				 extavkey->target_class,
				 requested,
				 &allowed,
				 &decided,
#ifdef CONFIG_FLASK_AUDIT
				 &auditallow,
				 &auditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
				 &notify,
#endif

				 &seqno);
	if (rc) {
		sprintf(errormsg, "compute_access_vector failed, rc = 0x%x", rc);
		exterror(errormsg);
		FREE(extavkey, sizeof(extav_key_t));
		return -1;
	}
	if ((requested & allowed) != requested) {
		exterror("requested permissions exceed allowed permissions");
		printf("requested");
		ss_dump_av(extavkey->target_class, requested);
		printf("  allowed");
		ss_dump_av(extavkey->target_class, allowed);
		printf("\n");
		FREE(extavkey, sizeof(extav_key_t));
		return -1;
	}
	extavdatum = (extav_datum_t *)
		hashtab_search(extavtab.table, (hashtab_key_t) extavkey);
	if (extavdatum) {
		extavdatum->allowed = requested;
		FREE(extavkey, sizeof(extav_key_t));
	} else {
		extavdatum = (extav_datum_t *) MALLOC(sizeof(extav_datum_t));
		if (!extavdatum) {
			exterror("out of memory");
			FREE(extavkey, sizeof(extav_key_t));
			return -1;
		}
		extavdatum->allowed = requested;

		ret = hashtab_insert(extavtab.table,
				     (hashtab_key_t) extavkey,
				     (hashtab_datum_t) extavdatum);

		if (ret == HASHTAB_PRESENT) {
			exterror("duplicate definition for permission definition");
			FREE(extavkey, sizeof(extav_key_t));
			FREE(extavdatum, sizeof(extav_datum_t));
			return -1;
		}
		if (ret == HASHTAB_OVERFLOW) {
			exterror("hash table overflow");
			FREE(extavkey, sizeof(extav_key_t));
			FREE(extavdatum, sizeof(extav_datum_t));
			return -1;
		}
	}

	return 0;
}


static int 
define_transition(int kind)
{
	char           *id;
	exttr_key_t    *exttrkey;
	exttr_datum_t  *exttrdatum;
	child_datum_t  *childdatum;
	class_datum_t  *cladatum;
	int             ret;


	exttrkey = (exttr_key_t *) MALLOC(sizeof(exttr_key_t));
	if (!exttrkey) {
		exterror("out of memory");
		return -1;
	}
	if (kind == PARENT_CHILD) {
		exttrkey->source_sid = parent_info.sid;
	} else {
		id = (char *) queue_remove(id_queue);
		if (!id) {
			exterror("no source child name for transition definition?");
			FREE(exttrkey, sizeof(exttr_key_t));
			return -1;
		}
		childdatum = (child_datum_t *)
			hashtab_search(parent_info.datum->children.table,
				       (hashtab_key_t) id);
		if (!childdatum) {
			sprintf(errormsg, "unknown child %s used in transition definition", id);
			exterror(errormsg);
			FREE(exttrkey, sizeof(exttr_key_t));
			FREE(id, strlen(id) + 1);
			return -1;
		}
		FREE(id, strlen(id) + 1);
		exttrkey->source_sid = childdatum->sid;
	}

	if (kind == CHILD_PARENT) {
		exttrkey->target_sid = parent_info.sid;
	} else {
		id = (char *) queue_remove(id_queue);
		if (!id) {
			exterror("no target child name for transition definition?");
			FREE(exttrkey, sizeof(exttr_key_t));
			return -1;
		}
		childdatum = (child_datum_t *)
			hashtab_search(parent_info.datum->children.table,
				       (hashtab_key_t) id);
		if (!childdatum) {
			sprintf(errormsg, "unknown child %s used in transition definition", id);
			exterror(errormsg);
			FREE(exttrkey, sizeof(exttr_key_t));
			return -1;
		}
		FREE(id, strlen(id) + 1);
		exttrkey->target_sid = childdatum->sid;
	}

	id = queue_remove(id_queue);
	if (!id) {
		exterror("no class name?");
		FREE(exttrkey, sizeof(exttr_key_t));
		return -1;
	}
	cladatum = (class_datum_t *) hashtab_search(policydbp->classes->table,
						    (hashtab_key_t) id);
	if (!cladatum) {
		sprintf(errormsg, "unknown class %s used in transition definition", id);
		exterror(errormsg);
		FREE(id, strlen(id) + 1);
		FREE(exttrkey, sizeof(exttr_key_t));
		return -1;
	}
	exttrkey->object_class = cladatum->value;

	id = queue_remove(id_queue);
	if (!id) {
		exterror("no to-child name?");
		FREE(exttrkey, sizeof(exttr_key_t));
		return -1;
	}
	childdatum = (child_datum_t *)
		hashtab_search(parent_info.datum->children.table,
			       (hashtab_key_t) id);
	if (!childdatum) {
		sprintf(errormsg, "unknown child %s used in transition definition", id);
		exterror(errormsg);
		FREE(exttrkey, sizeof(exttr_key_t));
		FREE(id, strlen(id) + 1);
		return -1;
	}
	FREE(id, strlen(id) + 1);

	exttrdatum = (exttr_datum_t *)
		hashtab_search(exttrtab.table, (hashtab_key_t) exttrkey);
	if (exttrdatum) {
		*exttrdatum = childdatum->sid;
		FREE(exttrkey, sizeof(exttr_key_t));
	} else {
		exttrdatum = (exttr_datum_t *) MALLOC(sizeof(exttr_datum_t));
		if (!exttrdatum) {
			exterror("out of memory");
			FREE(exttrkey, sizeof(exttr_key_t));
			return -1;
		}
		*exttrdatum = childdatum->sid;

		ret = hashtab_insert(exttrtab.table,
				     (hashtab_key_t) exttrkey,
				     (hashtab_datum_t) exttrdatum);

		if (ret == HASHTAB_PRESENT) {
			exterror("duplicate definition for transition definition");
			FREE(exttrkey, sizeof(exttr_key_t));
			FREE(exttrdatum, sizeof(exttr_datum_t));
			return -1;
		}
		if (ret == HASHTAB_OVERFLOW) {
			exterror("hash table overflow");
			FREE(exttrkey, sizeof(exttr_key_t));
			FREE(exttrdatum, sizeof(exttr_datum_t));
			return -1;
		}
	}

	return 0;
}


/* FLASK */
