/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/* FLASK */

#include "ebitmap.h"

int 
ebitmap_or(ebitmap_t * dst, ebitmap_t * e1, ebitmap_t * e2)
{
	ebitmap_node_t *n1, *n2, *new, *prev;


	ebitmap_init(dst);

	n1 = e1->node;
	n2 = e2->node;
	prev = 0;
	while (n1 || n2) {
		new = (ebitmap_node_t *) MALLOC(sizeof(ebitmap_node_t));
		if (!new) {
			ebitmap_destroy(dst);
			return FALSE;
		}
		if (n1 && n2 && n1->startbit == n2->startbit) {
			new->startbit = n1->startbit;
			new->map = n1->map | n2->map;
			n1 = n1->next;
			n2 = n2->next;
		} else if (!n2 || (n1 && n1->startbit < n2->startbit)) {
			new->startbit = n1->startbit;
			new->map = n1->map;
			n1 = n1->next;
		} else {
			new->startbit = n2->startbit;
			new->map = n2->map;
			n2 = n2->next;
		}

		new->next = 0;
		if (prev)
			prev->next = new;
		else
			dst->node = new;
		prev = new;
	}

	dst->highbit = (e1->highbit > e2->highbit) ? e1->highbit : e2->highbit;
	return TRUE;
}


int 
ebitmap_cmp(ebitmap_t * e1, ebitmap_t * e2)
{
	ebitmap_node_t *n1, *n2;


	if (e1->highbit != e2->highbit)
		return FALSE;

	n1 = e1->node;
	n2 = e2->node;
	while (n1 && n2 &&
	       (n1->startbit == n2->startbit) &&
	       (n1->map == n2->map)) {
		n1 = n1->next;
		n2 = n2->next;
	}

	if (n1 || n2)
		return FALSE;

	return TRUE;
}


int 
ebitmap_cpy(ebitmap_t * dst, ebitmap_t * src)
{
	ebitmap_node_t *n, *new, *prev;


	ebitmap_init(dst);
	n = src->node;
	prev = 0;
	while (n) {
		new = (ebitmap_node_t *) MALLOC(sizeof(ebitmap_node_t));
		if (!new) {
			ebitmap_destroy(dst);
			return FALSE;
		}
		new->startbit = n->startbit;
		new->map = n->map;
		new->next = 0;
		if (prev)
			prev->next = new;
		else
			dst->node = new;
		prev = new;
		n = n->next;
	}

	dst->highbit = src->highbit;
	return TRUE;
}


int 
ebitmap_contains(ebitmap_t * e1, ebitmap_t * e2)
{
	ebitmap_node_t *n1, *n2;


	if (e1->highbit < e2->highbit)
		return FALSE;

	n1 = e1->node;
	n2 = e2->node;
	while (n1 && n2 && (n1->startbit <= n2->startbit)) {
		if (n1->startbit < n2->startbit) {
			n1 = n1->next;
			continue;
		}
		if ((n1->map & n2->map) != n2->map)
			return FALSE;

		n1 = n1->next;
		n2 = n2->next;
	}

	if (n2)
		return FALSE;

	return TRUE;
}


int 
ebitmap_get_bit(ebitmap_t * e, unsigned long bit)
{
	ebitmap_node_t *n;


	if (e->highbit < bit)
		return FALSE;

	n = e->node;
	while (n && (n->startbit <= bit)) {
		if ((n->startbit + MAPSIZE) > bit) {
			if (n->map & (MAPBIT << (bit - n->startbit)))
				return TRUE;
			else
				return FALSE;
		}
		n = n->next;
	}

	return FALSE;
}


int 
ebitmap_set_bit(ebitmap_t * e, unsigned long bit, int value)
{
	ebitmap_node_t *n, *prev, *new;


	prev = 0;
	n = e->node;
	while (n && n->startbit <= bit) {
		if ((n->startbit + MAPSIZE) > bit) {
			if (value) {
				n->map |= (MAPBIT << (bit - n->startbit));
			} else {
				n->map &= ~(MAPBIT << (bit - n->startbit));
				if (!n->map) {
					/* drop this node from the bitmap */

					if (!n->next) {
						/*
						 * this was the highest map
						 * within the bitmap
						 */
						if (prev)
							e->highbit = prev->startbit + MAPSIZE;
						else
							e->highbit = 0;
					}
					if (prev)
						prev->next = n->next;
					else
						e->node = n->next;

					FREE(n, sizeof(ebitmap_node_t));
				}
			}
			return TRUE;
		}
		prev = n;
		n = n->next;
	}

	if (!value)
		return TRUE;

	new = (ebitmap_node_t *) MALLOC(sizeof(ebitmap_node_t));
	if (!new)
		return FALSE;

	new->startbit = bit & ~(MAPSIZE - 1);
	new->map = (MAPBIT << (bit - new->startbit));

	if (!n)
		/* this node will be the highest map within the bitmap */
		e->highbit = new->startbit + MAPSIZE;

	if (prev) {
		new->next = prev->next;
		prev->next = new;
	} else {
		new->next = e->node;
		e->node = new;
	}

	return TRUE;
}


void 
ebitmap_destroy(ebitmap_t * e)
{
	ebitmap_node_t *n, *temp;


	if (!e)
		return;

	n = e->node;
	while (n) {
		temp = n;
		n = n->next;
		FREE(temp, sizeof(ebitmap_node_t));
	}

	e->highbit = 0;
	e->node = 0;
	return;
}

/* FLASK */
