/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/* FLASK */

#include "avtab.h"

#define AVTAB_HASH(keyp) \
((keyp->target_type + (keyp->source_type << 4) + \
(keyp->target_class << 8)) % AVTAB_SIZE)

#define AVTAB_EQ(key,keyp) \
((key.source_type == keyp->source_type) && \
 (key.target_class == keyp->target_class) && \
 (key.target_type == keyp->target_type))


int 
avtab_insert(avtab_t * h, avtab_key_t * key, avtab_datum_t * datum)
{
	int             hvalue;
	avtab_ptr_t     cur, newnode;


	if (!h)
		return -ENOMEM;

	hvalue = AVTAB_HASH(key);
	cur = h->htable[hvalue];
	while (cur != NULL && !AVTAB_EQ(cur->key, key))
		cur = cur->next;

	if (cur != NULL)
		return -EEXIST;

	newnode = (avtab_ptr_t) malloc(sizeof(struct avtab_node_t));
	if (newnode == NULL)
		return -ENOMEM;
	newnode->key = *key;
	newnode->datum = *datum;
	newnode->next = h->htable[hvalue];
	h->htable[hvalue] = newnode;

	h->nel++;
	return 0;
}


avtab_datum_t  *
avtab_search(avtab_t * h, avtab_key_t * key)
{
	int             hvalue;
	avtab_ptr_t     cur;


	if (!h)
		return NULL;

	hvalue = AVTAB_HASH(key);
	cur = h->htable[hvalue];
	while (cur != NULL && !AVTAB_EQ(cur->key, key))
		cur = cur->next;

	if (cur == NULL)
		return NULL;

	return &cur->datum;
}


void 
avtab_destroy(avtab_t * h)
{
	int             i;
	avtab_ptr_t     cur, temp;


	if (!h)
		return;

	for (i = 0; i < AVTAB_SIZE; i++) {
		cur = h->htable[i];
		while (cur != NULL) {
			temp = cur;
			cur = cur->next;
			free(temp);
		}
	}
}


int 
avtab_map(avtab_t * h,
	  int (*apply) (avtab_key_t * k,
			avtab_datum_t * d,
			void *args),
	  void *args)
{
	int             i, ret;
	avtab_ptr_t     cur;


	if (!h)
		return 0;

	for (i = 0; i < AVTAB_SIZE; i++) {
		cur = h->htable[i];
		while (cur != NULL) {
			ret = apply(&cur->key, &cur->datum, args);
			if (ret)
				return ret;
			cur = cur->next;
		}
	}
	return 0;
}


int 
avtab_init(avtab_t * h)
{
	int             i;


	for (i = 0; i < AVTAB_SIZE; i++)
		h->htable[i] = (avtab_ptr_t) NULL;
	h->nel = 0;
	h->ncons = 0;
	return 0;
}


void 
avtab_hash_eval(avtab_t * h, char *progname, char *table)
{
	int             i, chain_len, slots_used, max_chain_len;
	avtab_ptr_t     cur;


	slots_used = 0;
	max_chain_len = 0;
	for (i = 0; i < AVTAB_SIZE; i++) {
		cur = h->htable[i];
		if (cur) {
			slots_used++;
			chain_len = 0;
			while (cur) {
				chain_len++;
				cur = cur->next;
			}

			if (chain_len > max_chain_len)
				max_chain_len = chain_len;
		}
	}

	printf("%s:  %s:  %d nodes and %d slots used, longest chain length %d\n",
	       progname, table, h->nel, slots_used, max_chain_len);
}
