/*
 * Copyright (c) 1997-1999 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
#include <oskit/c/assert.h>
#include <oskit/c/stdio.h>
#include <oskit/c/stdlib.h>
#include <oskit/c/stddef.h>
#include <oskit/c/string.h>
#include <oskit/c/arpa/inet.h>
#include <oskit/com/queue.h>

#include <oskit/dev/dev.h>
#include <oskit/dev/linux.h>
#include <oskit/dev/net.h>
#include <oskit/dev/ethernet.h>
#include <oskit/dev/error.h>
#include <oskit/dev/osenv_device.h>

#include <oskit/net/bootp.h>

#include <oskit/x86/proc_reg.h>

#include "getline.h"
#include "timer.h"
#include "ether.h"
#include "netboot.h"

struct etherdev {
	oskit_etherdev_t	*dev;
	oskit_netio_t		*send_nio;
	oskit_netio_t		*recv_nio;
	oskit_devinfo_t		info;
	unsigned char		haddr[OSKIT_ETHERDEV_ADDR_SIZE];
};

unsigned long netmask;			/* in host order */
char *hostname;
int hostnamelen;			/* word aligned length of hostname */
static struct etherdev ed;
static volatile oskit_queue_t *packq;
#define PACKQ_MAX 32

char packet[ETH_MAX_PACKET];
oskit_size_t packetlen;

/*
 * This is our network receive callback.
 * We just add the buf to a queue that is emptied in eth_poll.
 *
 * Returns zero on success, an error code from <oskit/error.h> otherwise.
 */
static oskit_error_t
net_receive(void *data, oskit_bufio_t *b, oskit_size_t pkt_size)
{
	unsigned char bcastaddr[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
	unsigned char dhost[6];
	int err;
	oskit_size_t got;
	struct etherdev *dev = (struct etherdev *)data;
	unsigned flags;

	if (pkt_size < ETH_MIN_PACKET || pkt_size > ETH_MAX_PACKET) {
		printf("%s: bad packet size %d\n", dev->info.name, pkt_size);
		return OSKIT_E_DEV_BADPARAM;
	}

	/*
	 * Check the Ethernet header to see if it is addressed to us.
	 */
	err = oskit_bufio_read(b, dhost,
			       offsetof(struct ether_header, ether_dhost),
			       sizeof dhost,
			       &got);
	assert(err == 0);
	assert(got == sizeof dhost);
	if (memcmp(dhost, &dev->haddr, sizeof dev->haddr) != 0
	    && memcmp(dhost, bcastaddr, sizeof bcastaddr) != 0)
		return 0;		/* not for us */

	/*
	 * Try to add it to the queue.
	 */
	flags = get_eflags();
	cli();
	oskit_queue_enqueue(packq, b, 0);
	set_eflags(flags);

	return 0;
}

/*
 * Send an ethernet frame of given type and size.
 *
 * Returns zero on success, an error code from <oskit/error.h> otherwise.
 */
oskit_error_t
eth_transmit(char *dest,		/* destination ethernet address */
	     unsigned short type,	/* ether type */
	     unsigned short size,
	     void *buf)
{
	oskit_bufio_t *b;
	oskit_error_t err;
	oskit_size_t actual;
	struct ether_header *eth;

	/*
	 * Create a bufio and fill it in.
	 */
	b = oskit_bufio_create(sizeof *eth + size);
	if (!b)
		return OSKIT_E_OUTOFMEMORY;

	/* Write the header. */
	err = oskit_bufio_map(b, (void **)&eth, 0, sizeof *eth);
	if (err)
		return err;
	memcpy(&eth->ether_dhost, dest, ETHER_ADDR_SIZE);
	memcpy(&eth->ether_shost, arptable[ARP_CLIENT].node, ETHER_ADDR_SIZE);
	eth->ether_type = htons(type);
	err = oskit_bufio_unmap(b, (void **)&eth, 0, sizeof *eth);
	if (err)
		return err;

	/* Write the data. */
	err = oskit_bufio_write(b, buf, sizeof *eth, size, &actual);
	if (err)
		return err;

	/* Send it out. */
	err = oskit_netio_push(ed.send_nio, b, sizeof *eth + size);
	oskit_bufio_release(b);

	return err;
}

/*
 * Wait until there is a frame available and fill in
 *	packet
 *	packetlen
 *
 * Return nonzero if we got something.
 */
int
eth_poll()
{
	unsigned long time = currticks() + 50; /* XXX */
	oskit_error_t err;
	oskit_bufio_t *b;
	unsigned flags;
	oskit_size_t got;
	oskit_off_t offset;

	/* Wait until the queue is nonempty and then take the first one. */
	while (time > currticks()) {
		flags = get_eflags();
		cli();
		if (oskit_queue_size(packq) == 0) {
			set_eflags(flags);
			continue;
		}
		/* Queue has something, take it. */
		oskit_queue_dequeue(packq, &b);
		set_eflags(flags);

		err = oskit_bufio_getsize(b, &offset);
		assert(err == 0);
		packetlen = (oskit_size_t)offset;
		err = oskit_bufio_read(b, packet, 0, packetlen, &got);
		assert(err == 0);
		assert(got == packetlen);
		oskit_bufio_release(b);
		return 1;
	}

	return 0;
}

/*
 * Initialize networkiness.
 *
 * Empty comments in column 1 indicate things that need to be undone upon
 * failure or in net_shutdown.
 *
 * Returns zero on success, an error code from <oskit/error.h> otherwise.
 */
oskit_error_t
net_init(oskit_osenv_t *osenv)
{
#define BUFSIZE 128
	static char buf[BUFSIZE];
	struct bootp_net_info bpi;
	oskit_error_t err;
	oskit_etherdev_t **alldevs;
	int ndev;
	int i;
	unsigned required_flags;
	oskit_osenv_device_t *osenv_device = 0;

	/* The cast is needed to make it non-volatile. */
/**/	err = oskit_bounded_com_queue_create(PACKQ_MAX,
					     (oskit_queue_t **)&packq);
	if (err)
		return err;

	oskit_linux_init_osenv(osenv);
	oskit_linux_init_net();
	oskit_dev_probe();

	/*
	 * Need the osenv_device interface for this.
	 */
	oskit_osenv_lookup(osenv,
			   &oskit_osenv_device_iid, (void *) &osenv_device);
	assert(osenv_device);
	
 retry:
	ndev = oskit_osenv_device_lookup(osenv_device,
				&oskit_etherdev_iid, (void ***)&alldevs);
	if (ndev <= 0)
		panic("no Ethernet adaptors found!");

	/*
	 * Try bootp on each dev and use the first one that succeeds.
	 */
	ed.dev = NULL;
	memset(&bpi, 0, sizeof bpi);
	for (i = 0; i < ndev; i++) {
		if (ed.dev) {
			/* Have choice, but must release the others anyway. */
			oskit_etherdev_release(alldevs[i]);
			continue;
		}

		if (bootp(alldevs[i], &bpi) != 0) {
			oskit_etherdev_release(alldevs[i]);
			continue;
		}

		required_flags = (BOOTP_NET_IP |
				  BOOTP_NET_NETMASK |
				  BOOTP_NET_GATEWAY |
				  BOOTP_NET_HOSTNAME);
		if ((bpi.flags & required_flags) != required_flags) {
#define MISSING(flag, name) if ((bpi.flags & flag) == 0) \
			printf("bootp did not supply %s\n", name)
			MISSING(BOOTP_NET_IP, "my IP address");
			MISSING(BOOTP_NET_NETMASK, "my netmask");
			MISSING(BOOTP_NET_GATEWAY, "gateway address");
			MISSING(BOOTP_NET_HOSTNAME, "my hostname");
#undef	MISSING
			oskit_etherdev_release(alldevs[i]);
			continue;
		}

/**/		ed.dev = alldevs[i];
	}
	if (ed.dev == NULL) {
		printf("No bootp server found for any interfaces!  Try again? [n] ");
		getline(buf, BUFSIZE);
		if (buf[0] == 'y' || buf[0] == 'Y')
			goto retry;
		oskit_queue_release(packq);
		return OSKIT_E_FAIL;
	}

	/*
	 * Now we know our bootp struct has all we need,
	 * so we copy the info out.
	 */
	bpi.ip.s_addr = ntohl(bpi.ip.s_addr);
	memcpy(&arptable[ARP_CLIENT].ipaddr, &bpi.ip.s_addr,
	       sizeof bpi.ip.s_addr);

	bpi.netmask.s_addr = ntohl(bpi.netmask.s_addr);
	memcpy(&netmask, &bpi.netmask.s_addr,
	       sizeof bpi.netmask.s_addr);

	bpi.gateway.addr[0].s_addr = ntohl(bpi.gateway.addr[0].s_addr);
	memcpy(&arptable[ARP_GATEWAY].ipaddr, &bpi.gateway.addr[0].s_addr,
	       sizeof bpi.gateway.addr[0].s_addr);

	/* Hostnamelen needs to be word aligned. */
	hostname = (char *)mustcalloc(strlen(bpi.hostname) + 3, 1);
	strcpy(hostname, bpi.hostname);
	hostnamelen = (strlen(hostname) + 3) & ~3;

	/*
	 * Fill in `ed' with info about this dev.
	 */
/**/	ed.recv_nio = oskit_netio_create(net_receive, &ed);
	if (ed.recv_nio == NULL)
		panic("unable to create recv_nio\n");
	oskit_etherdev_getaddr(ed.dev, ed.haddr);
	memcpy(arptable[ARP_CLIENT].node, ed.haddr, sizeof(ed.haddr));
	err = oskit_etherdev_getinfo(ed.dev, &ed.info);
	if (err)
		panic("Error(0x%08x) getting info from ethercard %d", err, i);

	/*
	 * Open it.
	 */
/**/	err = oskit_etherdev_open(ed.dev, 0, ed.recv_nio, &ed.send_nio);
	if (err)
		panic("Error(0x%08x) opening ethercard %d", err, i);

	return 0;
}

void
net_shutdown()
{
	/* Undo net_init in reverse order. */
	oskit_netio_release(ed.send_nio);
	oskit_netio_release(ed.recv_nio);
	oskit_etherdev_release(ed.dev);
	oskit_queue_release(packq);
}
