/*
 * Copyright (c) 1998, 1999 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#include <mom/c/libcast.h>
#include <mom/c/libpres_c.h>

#include <mom/c/pfe.hh>

/*
 * The method fills the `pg_state::prim_collections' array with descriptions
 * of this presentation generator's primitive and built-in presented types.
 *
 * Different presentation generators may use these primitive types in slightly
 * different ways; it is up to the specific PG to decide how to associate AOI
 * types with various presented types.  But since most presentations have
 * very similar notions about their sets of primitive types (e.g., similar
 * notions for `short' and `long' and `unsigned' integers), it is useful for
 * the PG library to provide a facility for defining these common, ``basic''
 * presented types.
 */
void pg_state::make_prim_collections()
{
	p_type_collection *ptc;
	p_scope_node *psn;
	p_type_node *ptn;
	
	pres_c_mapping direct_map = pres_c_new_mapping(PRES_C_MAPPING_DIRECT);
	
	int i;
	
	/*
	 * It is problematic to make an integer `signed' just because it is not
	 * unsigned.  (`signed x' is not always equivalent to `x'.)  The
	 * reverse is also true, but is not so problematic in practice.  What
	 * we really need are flag bits in the AOI integer like those we have
	 * for AOI characters.
	 *
	 * Similarly, `long int' is not the same as `int'.
	 */
	const cast_primitive_modifier int_signed = 0; /* CAST_MOD_SIGNED */
	const cast_primitive_modifier int_long   = 0; /* CAST_MOD_LONG */
	
	/*****/
	
	/* Initialize `prim_collections'. */
	for (i = 0; i < PRIM_COLLECTION_MAX; ++i)
		prim_collections[i] = 0;
	
	/*********************************************************************/
	
	/*
	 * Create the ``definition'' presentations of the types, i.e., the
	 * basic presentations of the types.
	 */
	
	/* Define the presented type corresponding to `void'. */
	ptc = new p_type_collection;
	ptc->set_name("void");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_type(CAST_TYPE_VOID));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_VOID] = ptc;
	
	/* Define the presented type corresponding to `boolean'. */
	ptc = new p_type_collection;
	ptc->set_name("boolean");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_CHAR, CAST_MOD_UNSIGNED));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_BOOLEAN] = ptc;
	
	/* Define the presented type corresponding to `char'. */
	ptc = new p_type_collection;
	ptc->set_name("char");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_CHAR, 0));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_CHAR] = ptc;
	
	/* Define the presented type corresponding to `signed char'. */
	ptc = new p_type_collection;
	ptc->set_name("schar");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_CHAR, CAST_MOD_SIGNED));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_SCHAR] = ptc;
	
	/* Define the presented type corresponding to `unsigned char'. */
	ptc = new p_type_collection;
	ptc->set_name("uchar");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_CHAR, CAST_MOD_UNSIGNED));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_UCHAR] = ptc;
	
	/* Define the presented type corresponding to `octet'. */
	ptc = new p_type_collection;
	ptc->set_name("octet");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_CHAR, CAST_MOD_UNSIGNED));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_OCTET] = ptc;
	
	/* Define the presented type corresponding to `short'. */
	ptc = new p_type_collection;
	ptc->set_name("short");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_INT,
					 (int_signed | CAST_MOD_SHORT)));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_SHORT] = ptc;
	
	/* Define the presented type corresponding to `unsigned short'. */
	ptc = new p_type_collection;
	ptc->set_name("ushort");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_INT,
					 (CAST_MOD_UNSIGNED | CAST_MOD_SHORT))
		);
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_USHORT] = ptc;
	
	/* Define the presented type corresponding to `long'. */
	ptc = new p_type_collection;
	ptc->set_name("long");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_INT,
					 (int_signed | int_long)));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_LONG] = ptc;
	
	/* Define the presented type corresponding to `unsigned long'. */
	ptc = new p_type_collection;
	ptc->set_name("ulong");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_INT,
					 (CAST_MOD_UNSIGNED | int_long)));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_ULONG] = ptc;
	
	/* Define the presented type corresponding to `long long'. */
	ptc = new p_type_collection;
	ptc->set_name("longlong");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_INT,
					 (int_signed | CAST_MOD_LONG_LONG))
		);
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_LONGLONG] = ptc;
	
	/* Define the presented type corresponding to `unsigned long long'. */
	ptc = new p_type_collection;
	ptc->set_name("ulonglong");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_INT,
					 (CAST_MOD_UNSIGNED
					  | CAST_MOD_LONG_LONG))
		);
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_ULONGLONG] = ptc;
	
	/* Define the presented type corresponding to `float'. */
	ptc = new p_type_collection;
	ptc->set_name("float");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_FLOAT, 0));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_FLOAT] = ptc;
	
	/* Define the presented type corresponding to `double'. */
	ptc = new p_type_collection;
	ptc->set_name("double");
	
	psn = new p_scope_node;
	psn->set_name("default");
	ptc->add_scope(psn);
	
	ptn = new p_type_node;
	ptn->set_name("definition");
	ptn->set_format("%s");
	ptn->set_type(cast_new_prim_type(CAST_PRIM_DOUBLE, 0));
	ptn->set_mapping(direct_map);
	ptc->add_type("default", ptn);
	
	prim_collections[PRIM_COLLECTION_DOUBLE] = ptc;
	
	/*
	 * XXX --- No standard presentations for these ``primitive'' types:
	 *
	 * PRIM_COLLECTION_ENUM
	 * PRIM_COLLECTION_STRING
	 * PRIM_COLLECTION_TYPE_TAG
	 * PRIM_COLLECTION_TYPED_ANY
	 */
	
	/*********************************************************************/
	
	/*
	 * At this point, some presentation generators create additional
	 * presentations of the typed listed above (e.g., ``out_pointer''
	 * versions of the types).  But we don't need to do so.
	 */
}

/* End of file. */

