/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                    Copyright (c) 1994,1995,1996                       */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                       Author :  Paul Taylor                           */
/*                       Date   :  April 1994                            */
/*-----------------------------------------------------------------------*/
/*                      EST_Track Class source file                      */
/*                                                                       */
/*=======================================================================*/
#include <fstream.h>
#include <iostream.h>
#include <math.h>
#include "EST_unix.h"
#include "EST_Track.h"
#include "EST_string_aux.h"
#include "track_io.h"

static float dummy=0.0;

EST_Track::EST_Track()
{
    default_vals();
}

EST_Track::EST_Track(const EST_Track &a)
{ 
    default_vals();
    copy(a);
}

EST_Track::EST_Track(int n_frames, int n_channels)
{
    default_vals();
    pa.resize(n_frames, n_channels);
    pt.resize(n_frames);
    pv.resize(n_frames);
    p_field_names.resize(n_channels);
    p_num_frames = n_frames;
    p_num_channels = n_channels;
    int d = 1;
    pv.fill(d);
}

EST_Track::~EST_Track(void)
{
  if (p_map)
    {
      p_map->dec_refcount();
      if(p_map->is_unreferenced())
	delete p_map;
    }
}

void EST_Track::default_field_names()
{
    for (int i = 0; i < p_num_channels; ++i)
	set_field_name("track" + itoString(i), i);
}

void EST_Track::default_vals(void)
{
    p_space_type = "FIXED";
    p_break_type = "MANY";
    p_file_type = "unset";
    p_contour_type = EST_ContourType::OTHER;
    amax = amin = 0.0;		// Not essential and are set as need be
    color = "";
    pa.resize(0, 0);
    pt.resize(0);
    pv.resize(0);
    p_field_names.resize(0);
    p_num_frames = 0;
    p_num_channels = 0;
    p_name = "";
    p_map = NULL;
}

int EST_Track::track_break(int i) const // does location i hold a break?
{
    return (!pv(i));
}    

void EST_Track::set_break(int i) // make location i hold a break
{
    if (i > p_num_frames)
	cerr << "Requested setting of break value of the end of the array\n";
    
    pv(i) = 0;
}    

void EST_Track::set_num_frames(int i)
{
  p_num_frames = (i < 0) ? 0 : i; // don't allow negative numbers
}    

void EST_Track::assign_map(EST_TrackMap *map)
{
  if (p_map)
    {
      p_map->dec_refcount();
      if(p_map->is_unreferenced())
	delete p_map;
    }
  p_map = map;
}

void EST_Track::create_map(EST_ChannelNameMap &names)
{
  EST_TrackMap *map = new EST_TrackMap;

  map->make_refcounted();
  map->inc_refcount();

  for (int i = 0; i < num_channels(); i++)
    {
      EST_ChannelType type = names.token(p_field_names(i));

      if (type != channel_unknown)
	map->set(type, i);
    }

  assign_map(map);
}

void EST_Track::set_value(int i) // make location i hold a value
{
    pv(i) = 1;
}    

const EST_String EST_Track::field_name(int i, const EST_ChannelNameMap &map, int strings_override) const
{
  EST_ChannelType type = channel_unknown;

  if (strings_override && p_field_names(i) != "")
      return p_field_names(i);
  else if (p_map && ((type = p_map->channel_type(i)) != channel_unknown))
    {
      const char *name = map.name(type);
      if (!name)
	name  = EST_default_channel_names.name(type);
      return EST_String(name);
    }
  else if (!strings_override && p_field_names(i) != "")
      return p_field_names(i);
  else 
    return "track" + itoString(i);
}    

void EST_Track::set_field_name(const EST_String &fn, int i)
{
    p_field_names(i) = fn;
}    

ostream& operator << (ostream& s, const EST_Track &tr)
{
    int i, j;
    for (i = 0; i < tr.p_num_frames; ++i)
    {
	s << tr.t(i);
	for (j = 0; j < tr.p_num_channels; ++j)
	    s << "\t" << tr(i, j);
	s << "\t" << !tr.track_break(i) << endl;
    }
    return s;
}

void EST_Track::copy(const EST_Track& a)
{
    copy_setup(a);
    pa = a.pa;
    pt = a.pt;
    pv = a.pv;
    p_num_frames = a.p_num_frames;
    p_num_channels = a.p_num_channels;
    if (p_map)
      {
       p_map->dec_refcount();
       if(p_map->is_unreferenced())
	 delete p_map;
      }
       
    if ((p_map = a.p_map))
      p_map->inc_refcount();
}

void EST_Track::copy_setup(const EST_Track& a)
{
    p_space_type = a.p_space_type;
    p_break_type = a.p_break_type;
    p_file_type = a.p_file_type;
    p_contour_type = a.p_contour_type;
    p_name = a.p_name;
    p_field_names = a.p_field_names;
    amin = a.amin;
    amax = a.amax;
    if (p_map)
      {
       p_map->dec_refcount();
       if(p_map->is_unreferenced())
	 delete p_map;
      }
    if ((p_map = a.p_map))
      p_map->inc_refcount();
}    

void EST_Track::resize(int new_num_frames, int new_num_channels)
{
    pa.resize(new_num_frames, new_num_channels);
    pt.resize(new_num_frames);
    pv.resize(new_num_frames);
    p_field_names.resize(new_num_channels);
    for (int i = p_num_frames; i < new_num_frames; ++i)
	pv(i) = 1;

    p_num_frames = new_num_frames;
    p_num_channels = new_num_channels;
}

EST_Track& EST_Track::operator+=(const EST_Track &a)  // add to existing track
{
    int i, j, k;

    if (p_num_frames == 0)	// i.e. no existing EST_Track to add to
    {
	*this = a;
	return *this;
    }
    
    if (a.num_channels() != p_num_channels)
    {
	cerr << "Error: Tried to add " << a.num_channels() << 
	    " channel EST_Track to "<<p_num_channels << " channel EST_Track\n";
	return *this;
    }
    
    int old_num = num_frames();
    this->resize(a.num_frames()+ this->num_frames(), this->num_channels());
    for (i = 0, j = old_num; i < a.num_frames(); ++i, ++j)
    {
	for (k = 0; k < p_num_channels; ++k)
	    pa(j, k) = a(i, k);
	pt(j) = a.t(i);
	pv(j) = a.val(i);
    }
    
    return *this;
}

EST_Track& EST_Track::operator|=(const EST_Track &a)
{				// add to existing track in parallel
    int i, j, k;
    
    if (p_num_channels == 0)	// i.e. no existing EST_Track to add to
    {
	*this = a;
	return *this;
    }
    
    if (a.num_frames() != p_num_frames)
    {
	cerr << "Error: Tried to add " << a.num_frames() << 
	    " channel EST_Track to "<<p_num_frames<< " channel EST_Track\n";
	return *this;
    }
    
    int old_num = num_channels();
    this->resize(a.num_frames(), this->num_channels() + a.num_channels());
    for (i = 0, j = old_num; i < a.num_channels(); ++i, ++j)
	for (k = 0; k < p_num_frames; ++k)
	    pa(k, j) = a(k, i);
    
    return *this;
}

EST_Track &EST_Track::operator=(const EST_Track& a)
{
    copy(a);
    return *this;
}    


bool EST_Track::has_channel(const char *name) const
{
  int c;

  for (c=0; c<p_num_channels; c++)
    if (field_name(c) == name)
      return TRUE;

  return FALSE;
}


float &EST_Track::a(int i, int c)
{   // frames are rows, tracks are columns
    return pa(i, c);
}

float &EST_Track::a(int i, EST_ChannelType type, int offset)
{ 
    short c = NO_SUCH_CHANNEL;

    if (p_map && ((c = p_map->get(type)) != NO_SUCH_CHANNEL))
	return pa(i, c+offset);
    else
    {
	cerr << "no channel '" << EST_default_channel_names.name(type) << "' = " << type << "\n";
	abort();
    }

    return dummy;
}

float &EST_Track::a(int i, const char *name, int offset)
{ 
    int c;

    for (c=0; c<p_num_channels; c++)
	if (field_name(c) == name)
	    return pa(i, c+offset);

    cerr << "no channel '" << name << "'\n";
    abort();
    return dummy;
}

#define EPSILON (0.0001)

float &EST_Track::a(float t, int c, EST_InterpType interp)
{
    static float ia = 0.0;

    if (interp == it_nearest)
	return pa(index(t), c);
    else if (interp == it_linear)
    {
	int i = index_below(t);
	if (i < 0)
	    return a(0,c);

	float n = a(i,c), n1 = a(i+1,c);
	float tn = pt(i), tn1 = pt(i+1);
	ia = n + (n1-n)*(t-tn)/(tn1-tn);
	return ia;
    }
    else if (interp == it_linear_nz)
    {
	int i = index_below(t);
	if (i < 0)
	    return a(0,c);

	float n = a(i,c), n1 = a(i+1,c);
	if (fabs(n) < EPSILON || fabs(n1) < EPSILON)
	    return pa(index(t), c);
	float tn = pt(i), tn1 = pt(i+1);
	ia = n + (n1-n)*(t-tn)/(tn1-tn);
	return ia;
    }
    return ia;
}

float &EST_Track::a(float t, EST_ChannelType type, EST_InterpType interp)
{ 
    short c = NO_SUCH_CHANNEL;

    if (p_map && (c = p_map->get(type)) != NO_SUCH_CHANNEL)
	return a(t, c, interp);
    else
    {
	cerr << "no channel '" << EST_default_channel_names.name(type) << "' = " << type << "\n";
	abort();
    }
    return dummy;
}

float &EST_Track::t(int i)
{
    return pt(i);
}

float EST_Track::ms_t(int i) const
{
    return pt(i) * 1000.0;
}

int EST_Track::index(float x) const
{
  // cout << "  look for " << x << "\n";
    for (int i = 1; i < p_num_frames; ++i)
	if (x < pt(i))
	{
	  // cout << "     at " << i << " time " << pt(i) << "\n";
	    if (fabs(x - pt(i)) < fabs(x - pt(i - 1)))
		return i;
	    else
		return i - 1;
	}
    
    return p_num_frames -1;
}

int EST_Track::index_below(float x) const
{
    for (int i = 1; i < p_num_frames; ++i)
	if (x < pt(i))
	  return i - 1;

    return p_num_frames-1;
}

int EST_Track::val(int i) const
{
    return pv(i);
}
/*
   "p_space_type" indicates whether the x-axis values are evenly spaced
   (FIXED) or spaced arbitrarily (VARI).
   
   "p_break_type" describes the break format. F0 contours are seldom
   continuous - often breaks occur due to unvoicing etc. These are a
   marked  in the data arrays by break values, "i_break" for ints
   and "f_break" for floats. The "p_break_type" field specifies whether
   a break is represented by a single break value, or as a break
   value for every frame. eg
   (SINGLE)
   800  100
   810  105
   BREAK BREAK
   850  130
   860  135
   or
   (MANY)
   800  100
   810  105
   820 BREAK
   830 BREAK
   840 BREAK
   850  130
   860  135
   
   In the MANY case, only the y value is specified as a break, in the
   SINGLE case the x value may or may not be specified as a break. For
   this reason, when checking for breaks, it is useful to only rely
   on the y value being set to the i_break value. Not that if the break_type
   is MANY and the space_type is FIXED, you dont really need x-axis
   values.
   
   Different functions naturally work better on different representations
   and that is why all these different types are supported. A
   general function mod_cont() is upplied to change from one
   type to another. Not all conversions are currently
   supported however.
   
   */

float EST_Track::end() const
{
    return (pt(prev_non_break(p_num_frames)));
}
float EST_Track::start() const
{
    return (track_break(0) ? pt(next_non_break(0)) : pt(0));
}

float EST_Track::shift() const
{
    int j1 = 0;
    int j2 = 0;
    
    if (p_space_type != "FIXED")
    {
	cerr << "Tried to take shift from non-fixed contour " << p_space_type 
	    << endl;
	return -1.0;
    }
    //    cout << "shift\n";
    do
    {
	j1 = next_non_break(++j1);
	j2 = next_non_break(j1);
	//	cout << "j1:" << j1 << " j2:" << j2 << endl;
    }
    while ((j2 != 0) && (j2 != (j1 +1)));
    
    if (j2 == 0)
    {
	if (p_num_frames > 1)
	    return pt(1) - pt(0);
	else
	    cerr << "Couldn't determine  shift size\n";
    }
    return (pt(j2) - pt(j1));
}

/* tries to find the next value that isnt a break. Dont really
   know what to do on a fail, so just return 0 */

int EST_Track::next_non_break(int j) const
{
    int i = j;
    for (++i; i < p_num_frames; ++i)
    {
	//	cout << "i: " << i << " " << value[i] << endl;
	if (!track_break(i))
	    return i;
    }
    
    return 0;
}

/* give the current point, returns the previous non-break */

int EST_Track::prev_non_break(int j) const
{
    int i = j;
    for (--i; i >= 0 ; --i)
	if (!track_break(i))
	    return i;
    return 0;
}

void EST_Track::change_type(float nshift, const EST_String &nbreak)
{
    if (nshift != 0.0)
    {
      if (p_space_type != "FIXED" || nshift != shift())
	sample(nshift);
      p_space_type = "FIXED";
    }
    
    if ((nbreak != "") && (nbreak != p_break_type))
    {
	if (nbreak == "MANY")
	    pad_breaks();
	else if (nbreak == "SINGLE")
	    rm_excess_breaks();
	else
	    cerr << "change_type: Break type " << nbreak << "unknown \n";
    }
}

void EST_Track::channel_to_time(int channel, float scale)
{  

  for(int i=0; i < num_frames(); i++)
    {
      t(i) = a(i,channel) * scale;
    }
  set_space_type("VARI");
}

void EST_Track::channel_to_time(EST_ChannelType c, float scale)
{
    int channel = NO_SUCH_CHANNEL;

    if (p_map && (channel = p_map->get(c)) != NO_SUCH_CHANNEL)
    {
	channel_to_time(channel, scale);
	return;
    }
    else
    {
	cerr << "no channel '" << EST_default_channel_names.name(c) << "' = " << c << "\n";
	abort();
    }
}

void EST_Track::channel_to_time(const EST_String c_name, float scale)
{
    for (int c=0; c<p_num_channels; c++)
	if (field_name(c) == c_name)
	{
	    channel_to_time(c, scale);
	    return;
	}

    cerr << "no channel named '" << c_name << "'\n";
    abort();
}

void EST_Track::channel_to_time_lengths(int channel, float scale)
{  
    float tt=0;
    for(int i=0; i < num_frames(); i++)
    {
	// cout << "c_t_t " << i << " " << tt << "\n";
	t(i) = tt;
	tt += a(i,channel) * scale;
    }
    set_space_type("VARI");
}

void EST_Track::channel_to_time_lengths(EST_ChannelType c, float scale)
{
    int channel = NO_SUCH_CHANNEL;
    
    if (p_map && (channel = p_map->get(c)) != NO_SUCH_CHANNEL)
    {
	channel_to_time_lengths(channel, scale);
	return;
    }
    else
    {
	cerr << "no channel '" << EST_default_channel_names.name(c) << "' = " << c << "\n";
	abort();
    }
}

void EST_Track::channel_to_time_lengths(const EST_String c_name, float scale)
{
    for (int c=0; c<p_num_channels; c++)
	if (field_name(c) == c_name)
	{
	    channel_to_time_lengths(c, scale);
	    return;
	}
    
    cerr << "no channel named '" << c_name << "'\n";
    abort();
}

void EST_Track::sample(float f_interval)
{
    EST_FVector nt;
    EST_FMatrix na;
    EST_IVector nv;
    int i, j, n;
    
    n = (int) rint(((end())/ f_interval));
    
    nt.resize(n);
    na.resize(n, p_num_channels);
    nv.resize(n);
    
    // REORG - can this be replaced with fill_time()?
    for (i = 0; i < n; ++i)
	nt(i) = (float) ((i + 1) * f_interval);
    
    for (i = 0; i < n; ++i)
    {
	nv(i) = interp_value(nt(i), f_interval);
	for (j = 0; j < p_num_channels; ++j)
	    na(i, j) = nv(i) ? interp_amp(nt(i), j, f_interval): 0.0;
    }
    
    pt = nt;
    pa = na;
    pv = nv;
    p_num_frames = n;
    p_break_type = "MANY";
    p_space_type = "FIXED";
}

float EST_Track::interp_amp(float x, int c, float f)
{
    int i;
    float x1, x2, y1, y2, m;
    
    for (i = 0; i < p_num_frames; ++i)
	if ((pt(i) + (f / 2.0))> x)
	    break;
    
    if (i == p_num_frames)
	return pa(i - 1, c);
    if (i == 0)
	return pa(0, c);
    
    if (track_break(i) && track_break(i - 1))
	return 0.0;
    
    if (track_break(i))
	return pa(i - 1, c);
    
    else if (track_break(i - 1))
	return pa(i, c);
    
    x1 = pt(i - 1);
    y1 = pa(i - 1, c);
    x2 = pt(i);
    y2 = pa(i, c);
    
    m =  (y2 - y1) / (x2 -x1);
    return ((x - x1) * m) + y1;
}		

int EST_Track::interp_value(float x, float f)
{
    int i;
    int p, n;
    float cf;
    
    if (p_space_type == "FIXED")
	cf = shift();
    else
	cf = estimate_shift(x);
    
    for (i = 0; i < p_num_frames; ++i)
	if ((pt(i) + (f / 2.0))> x)
	    break;
    // This was:    
    //    for (i = 0; i < p_num_frames; ++i)
    //	if (pt[i] > x)
    //	    break;
    
    if (i == 0)		// must be a break for the first value. (can't have i -1).
	return FALSE;
    
    if ((!track_break(i)) && (!track_break(i -1)))
	return TRUE;
    
    p = prev_non_break(i);
    n = next_non_break(i);
    
    if ((x < pt(p) + (cf / 2.0)) || (x > pt(n) - (cf / 2.0)))
	return TRUE;		// rounding at edges
    
    return FALSE;
    
}

float EST_Track::estimate_shift(float x)
{
    int i, j;
    for (j = 0; j < p_num_frames; ++j)
	if (pt(j) > x)
	    break;
    
    for (i = j; i > 0; --i)
	if ((!track_break(i)) && (!track_break(i - 1)))
	    return pt(i) - pt(i - 1);
    
    for (i = j; i < p_num_frames - 1; ++i)
	if ((!track_break(i)) && (!track_break(i + 1)))
	    return pt(i + 1) - pt(i);
    
    return 5.0;		// default value
}

void EST_Track::fill_time(float t, int start)
{
    for (int i = 0; i < p_num_frames; ++i)
	pt(i) = t * (float) (i + start);
}

void EST_Track::fill_amp(float f)
{
    for (int i = 0; i < p_num_frames; ++i)
	for (int j = 0; j < p_num_channels; ++j)
	    pa(i, j) = f;
}

void EST_Track::rm_excess_breaks()
{
    int i, j, k;
    EST_FVector nt;
    EST_IVector nv;
    EST_FMatrix na;
    
    na.resize(p_num_frames, p_num_channels);
    nt.resize(p_num_frames);
    nv.resize(p_num_frames);
    
    for (i = 0; track_break(i); ++i); //rm leading breaks
    
    for (j = 0; i < p_num_frames; ++i, ++j)
    {
	for (k = 0; k < p_num_channels; ++k)
	    na(j, k) = pa(i, k);
	nt(j) = pt(i);
	nv(j) = pv(i);
	while ((!nv(j)) && (!pv(i + 1)))
	    ++i;
    }
    pt = nt;
    pa = na;
    pv = nv;
    for (--j; track_break(j); --j) // "rm" trailing breaks
	;
    p_num_frames = j + 1;
    pt.resize(p_num_frames);
    pa.resize(p_num_frames, p_num_channels);
    pv.resize(p_num_frames);
    
    p_break_type = "SINGLE";
}    

void EST_Track::rm_trailing_breaks()
{
    if (p_num_frames <=0 )
	return;
    int i, j, k;
    EST_FMatrix ta;
    
    ta.resize(p_num_frames, p_num_channels);
    
    for (i = 0; i < p_num_frames; ++i)
	if (!track_break(i))
	    break;
    
    for (j = 0; i < p_num_frames; ++i, ++j)
    {
	pt(j) = pt(i);
	for (k = 0; k < p_num_channels; k++)
	    ta(j, k) = pa(i, k);
	pv(j) = pv(i);
    }
    for (--j; track_break(j); --j);
    
    p_num_frames = j + 1;
    
    ta.resize(p_num_frames, p_num_channels);
    
    pa = ta;
    pt.resize(p_num_frames);
    //    pa.resize(p_num_frames, p_num_channels);
    pv.resize(p_num_frames);
}    

void EST_Track::add_trailing_breaks()
{
    int i, j, k;
    EST_FVector nt;
    EST_FMatrix na;
    int new_num = p_num_frames;
    
    if (!track_break(0))
	new_num++;
    if (!track_break(p_num_frames - 1))
	new_num++;
    
    if (new_num == p_num_frames) /*ie trailing breaks already there */
	return;
    
    nt.resize(new_num);
    na.resize(new_num, p_num_channels);
    
    j = 0;
    if (!track_break(j))
	set_break(j);
    
    for (i = 0; i < p_num_frames; ++i, ++j)
    {
	nt(j) = pt(i);
	for (k = 0; k < p_num_channels; ++k)
	    na(j, k) = pa(i, k);
    }
    
    if (!track_break(p_num_frames - 1))
	set_break(j);
    
    pt = nt;
    pa = na;
    p_num_frames = new_num;
    pt.resize(p_num_frames);
    pa.resize(p_num_frames, p_num_channels);
}    

void EST_Track::pad_breaks()
{
    if (p_break_type == "MANY")
	return;
    
    if (p_space_type != "FIXED")
    {
	cerr << "Can only operate on fixed data\n";
	return;
    }
    
    EST_FVector nt;
    EST_FMatrix na;
    EST_IVector nv;
    int i, j, k, n;
    
    n = (int)(((end())/ shift()) + 1.0);
    int s = int(start()/ shift());
    
    for (i = 0; i < n; ++i)
    {
	nt(i) = (float) (i * shift());
	for (k = 0; k < p_num_channels; ++k)
	    na(i, k) = 0.0;
	nv(i) = 0;
    }
    
    for (i = 0, j = s; j < n; ++i, ++j)
    {
	if (track_break(i))
	{
	    for (; nt(j) < pt(i + 1); ++j);
	    --j;
	}
	else
	{
	    nv(j) = 1;
	    for (k = 0; k < p_num_channels; ++k)
		na(j, k) = pa(i, k);
	}
    }
    nv(j) = 1;
    for (k = 0; k < p_num_channels; ++k)
	na(j, k) = pa(i, k);
    
    pt = nt;
    pa = na;
    pv = nv;
    p_num_frames = j;
    
    pt.resize(p_num_frames);
    pv.resize(p_num_frames);
    pa.resize(p_num_frames, p_num_channels);
    
    p_break_type = "MANY";
}    

int EST_Track::empty() const
{
    int i, num;
    
    for (i = num = 0; i < p_num_frames; ++i)
	if (val(i))
	    return 0;		// i.e. false
    
    return 1;		// i.e. true
}

EST_write_status EST_Track::save(const EST_String filename, const EST_String type)
{
    
    if (type == "")
	return save_ascii(filename, *this);
    if  (type == "esps")
	return save_esps(filename, *this);
    if  (type == "xmg")
    {
	rm_trailing_breaks();
	return save_xmg(filename, *this);
    }
    if  (type == "ascii")
	return save_ascii(filename, *this);
    if  (type == "xgraph")
	return save_xgraph(filename, *this);
    if  (type == "htk")
	return save_htk(filename, *this);
    
    cerr << "Unknown EST_Track output file type " << type << endl;
    return write_fail;
}

EST_read_status EST_Track::load(const EST_String filename, float ishift)
{
    EST_read_status r_val;
    EST_Track tmp;
    
    if ((r_val = load_esps(filename, tmp)) != wrong_format)
    {	*this = tmp;
	return r_val;
    }
    
    if ((r_val = load_xmg(filename, tmp)) != wrong_format)
    {	*this = tmp;
	return r_val;
    }
    
    if ((r_val = load_htk(filename, tmp)) != wrong_format)
    {	*this = tmp;
	return r_val;
    }
    
    //    if ((r_val = load_snns(filename, tmp, ishift)) != wrong_format)
    //    {	*this = tmp;
    //	return r_val;
    //    }
    
    //    if ((r_val = load_xgraph(filename, tmp)) != wrong_format)
    //    {	*this = tmp;
    //	return r_val;
    //    }
    
    if ((r_val = load_ascii(filename, tmp, ishift)) != wrong_format)
    {	*this = tmp;
	return r_val;
    }
    
    cerr << "Can't read track input file type for file " << filename << endl;
    return wrong_format;
}

EST_write_status EST_Track::save_channel_names(const EST_String filename)
{
    FILE *file;
    
    if ((file=fopen(filename, "wb"))==NULL)
	return write_fail;
    
    for(int c=0; c<num_channels(); c++)
	fprintf(file, "%s\n", (const char *)field_name(c));
    
    fclose(file);
    
    return write_ok;
}

EST_read_status EST_Track::load_channel_names(const EST_String filename)
{
    FILE *file;
    static const int buffer_length = 100;
    char buffer[buffer_length];
    
    if ((file=fopen(filename, "rb"))==NULL)
	return misc_read_error;
    
    for(int c=0; c<num_channels(); c++)
    {
	if (!fgets(buffer, buffer_length, file))
	    break;
	
	buffer[strlen(buffer)-1] = '\0';
	set_field_name(buffer, c);
    }
    
    
    fclose(file);
    
    return format_ok;
}
