/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Paul Taylor                               */
/*                   Date   :  March 1995                                */
/*************************************************************************/
/*                                                                       */
/*                     Energy EST_Track Algorithm                            */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include "EST.h"

void default_options(EST_Option &al);

int StrListtoFList(EST_StrList &s, EST_FList &f);

void meansd(EST_TrackList &tl, EST_FVector &m, EST_FVector &sd);

EST_read_status read_TrackList(EST_TrackList &tlist, EST_StrList &files, 
			       EST_Option &al);
EST_write_status save_ind_TrackList(EST_TrackList &tlist, EST_String &otype);

int main (int argc, char *argv[])
{
    EST_Track tr;
    EST_String out_file, fname;
    EST_StrList files, ts;
    EST_Option al, op;
    EST_FVector mean, sd;    
    EST_TrackList trlist;
    EST_TBI *p;
    int i;
    
    print_help(argc, argv);

    parse_command_line(argc, argv, "-stats \
		      -mean:s -o:s -otype:s -ops -s:f -sd:s", files, al);

    init_lib_ops(al, op);
    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    read_TrackList(trlist, files, al);

    if (!al.present("-mean") && !al.present("-sd"))
    {
	meansd(trlist, mean, sd);
	cout << "mean ";
	for (i = 0; i < mean.n(); ++i)
	    cout << mean(i) << " ";
	cout << endl;

	cout << "sd ";
	for (i = 0; i < sd.n(); ++i)
	    cout << sd(i) << " ";
	cout << endl;
	exit (0);
    }

    else if (al.present("-mean") && al.present("-sd"))
    {
	EST_FList tl;
	StringtoStrList(al.val("-mean"), ts);

	cout << "s = " << al.val("-mean") << endl;
	cout << "ts = " << ts << endl;

	StrListtoFList(ts, tl);
	mean.resize(tl.length());
	for (i = 0, p = tl.head(); p; p = next(p), ++i)
	    mean(i) = tl(p);

	cout << "tl = " << tl << endl;

	ts.clear();
	tl.clear();

	StringtoStrList(al.val("-sd"), ts);
	StrListtoFList(ts, tl);

	sd.resize(tl.length());
	for (i = 0, p = tl.head(); p; p = next(p), ++i)
	    sd(i) = tl(p);

    }
    else
    {
	cerr << "Both mean and standard deviation must be specified\n";
	exit (-1);
    }

    if (mean.n() != trlist.first().num_channels())
    {
	cerr << "Expect " << trlist.first().num_channels() << " values for mean. Got "
	    << mean.n() << endl;
	cout << mean;
	exit (-1);
    }
    if (sd.n() != trlist.first().num_channels())
    {
	cerr << "Expect " << trlist.first().num_channels() << " values for sd. Got "
	    << sd.n() << endl;
	exit (-1);
    }

    normalise(trlist, mean, sd);

    for (p = trlist.head(); p ; p = next(p))
    {
	fname = basename(trlist(p).name());
	cout << "p = " << trlist(p).name() << " base:" << fname << endl;
	trlist(p).save(fname, al.val("-otype"));
    }

    return 0;
}

void override_lib_ops(EST_Option &a_list, EST_Option &al)
{
    // general options
    a_list.override_val("sample_rate", al.val("-f", 0));
    a_list.override_val("ms_frame_length",al.val("-l", 0));
    a_list.override_val("ms_frame_shift",al.val("-s", 0));
    
    a_list.override_val("energy_smooth",al.val("-smooth", 0));
    
    // file options
    a_list.override_val("out_file_type", al.val("-otype", 0));
    a_list.override_val("in_file_type", al.val("-itype", 0));
}

void print_help(int argc, char *argv[])
{
    for (int i = 0; i < argc; ++i)
	if (strncmp(argv[i],"-help", 2) == 0)
	{
	    cout << "Usage:\n";
	    cout << "normalise  <input file> <output file> <options>\n";
	    cout << "use \"-\" to make input and output files stdin/out\n";
	    cout << "-f      <int> sample rate in Hertz\n";
	    cout << "-l      <float> frame length in ms.\n";
	    cout << "-s      <float> frame shift in ms.\n";
	    cout << "-smooth <float> amount of smoothing\n";
	    cout << "-itype  <input type: 'raw' 'vox' 'esps' 'sd' 'nist'>\n";
	    cout << "-otype  <output file type 'xmg' 'ascii' 'esps'>\n";
	    cout << "-c      <operations file> for system params \n";
	    cout << "-ops    print relevant system operations\n";
	    cout << "-N      do NOT read environment operations file\n";
	    cout << "-color  <string> Color of output line\n";
	    cout << "-h      options help\n";
	    exit (0);
	}
}
