/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  April 1995                             */
/*-----------------------------------------------------------------------*/
/*                     Change EST_Wave utility main                      */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include <iostream.h>
#include <math.h>
#include "EST_Wave.h"
#include "EST_cmd_line.h"
#include "EST_cmd_line_options.h"
#include "EST_sigpr.h"
#include "EST_wave_aux.h"
#include "EST_Stream.h"

int main (int argc, char *argv[])
{
    EST_Wave sig,sigload;
    EST_String in_file("-"), out_file("-"), op_file(""), test;
    EST_Option al;
    EST_StrList files;
    EST_TBI *p;

    parse_command_line2(argc, argv, 
       EST_String("Usage:   ")+
       "ch_wave  <input file0> <input file1> ... -o <output file> <options>\n"+
       "Summary: change/copy/combine waveform files\n"+
       "use \"-\" to make input and output files stdin/out\n"+
       "-h               Options help\n"+
       options_wave_input()+
       "\n"+
       options_wave_output()+
       "\n"+
       "-s <float>       Scaling factor (1.0 is no change)\n"+
       "-scaleN <float>  Scaling factor with normalization\n"+
       "                 waveform is maximized before scaling\n"+
       "-lpfilter <int>  Low pass filter cutoff frequency\n"+
       "-hpfilter <int>  High pass filter cutoff frequency\n"+
       "-forder <int>    Order of FIR filter used for -lpfilter and\n"+
       "                 -hpfilter (must be ODD; default is 199 to give\n"+
       "                 steep rolloff)\n"+
       "-fafter          Do filtering after other operations such as\n"+
       "                 resampling (default : filter before other \n"+
       "                 operations)\n"+
       "-info            Print information about file and header\n"+
       "-pc <string>     Combine given waveforms in parallel.  If options\n"+
       "                 is longest, pad shorter waves to longest, else if\n"+
       "                 first pad/cut to match first input waveform\n"+
       "-divide          Divide waveform with respect to -key label file\n"+
       "-ext <string>    File extension for divided waveforms\n"+
       "-extract <string>\n"+
       "                 Extract named label in -key specified label file\n"+
       "                 from waveform\n"+
       "-key <ifile>     Label file with filenames in optional field\n",
			files, al);

    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    // There will always be at least one (or stdin)
    // The first is dealt specially in case its *way* big
    if (read_wave(sig, files.first(), al) != format_ok)
	exit(-1);
    if (al.present("-info")) 
	wave_info(sig);
    // concat or parallelize remaining input files

    if (files.length() > 1)
    {
	for (p=next(files.head()); p != 0; p=next(p))
	{
	    if (read_wave(sigload, files(p), al) != format_ok)
		exit(-1);
	    if (al.present("-info")) 
		wave_info(sigload);
	    else if (al.present("-pc"))
	    {
		if ((al.val("-pc") == "longest") &&
		    (sig.num_samples() < sigload.num_samples()))
		    sig.resize(sigload.num_samples());
		sig |= sigload;
	    }
	    else
		sig += sigload;
	}
    }

    if (al.present("-info")) 
	exit(0);    // done what I've been asked to so stop

    // All input files are now in a single wave called sig

    // default is to filter before any resampling etc.
    // (this may cause problems for multiplexed data !)
    if(!al.present("-fafter")){
	if(al.present("-lpfilter"))
	    FIRlowpass_filter(sig,al.ival("-lpfilter"),al.ival("-forder"));
	if(al.present("-hpfilter"))
	    FIRhighpass_filter(sig,al.ival("-hpfilter"),al.ival("-forder"));
    }

    if (al.present("-c"))	// extract a channel from a multi-channel wave
	wave_extract_channel(sig,sig,al.ival("-c"));
    
    if (al.present("-F"))	// resample
	sig.resample(al.ival("-F"));
    
    if (al.present("-s"))	// rescale
    {
	float scale = al.fval("-s", 0);
	sig.rescale(scale);
    }
    else if (al.present("-scaleN"))	// rescale
    {
	float scale = al.fval("-scaleN", 0);
	if ((scale < 0) || (scale > 1.0))
	{
	    cerr << "ch_wave: -scaleN must be in range 0 to 1" << endl;
	    exit(-1);
	}
	sig.rescale(scale,1);
    }

    EST_Stream key;

    if (al.present("-divide"))
    {
	EST_WaveList wl;
	if (!al.present("-key"))
	{
	    cerr << "Must have key file specified when dividing waveform\n";
	    exit (-1);
	}
	if (key.load(al.val("-key")) != format_ok)
	    exit(-1);

        if (wave_divide(wl, sig, key, al.val("-ext", 0)) == -1)
	    exit(0);
	for (p = wl.head(); p; p = next(p))
	    wl(p).save(wl(p).name(), al.val("-otype", 0));
	exit(0);
    }
    else if (al.present("-extract"))
    {
	EST_Wave e;
	if (!al.present("-key"))
	{
	    cerr << "Must have key file specified when dividing waveform\n";
	    exit (-1);
	}
	if (key.load(al.val("-key")) != format_ok)
	    exit(-1);

        if (wave_extract(e, sig, key, al.val("-extract")) == -1)
	    exit (-1);
	sig = e;
    }

    // if we are filtering after other operations
    if(al.present("-fafter")){
	if(al.present("-lpfilter"))
	    FIRlowpass_filter(sig,al.ival("-lpfilter"),al.ival("-forder"));
	if(al.present("-hpfilter"))
	    FIRhighpass_filter(sig,al.ival("-hpfilter"),al.ival("-forder"));
    }

    write_wave(sig, out_file, al);
    return 0;
}
