
/*
 * LIB/BUFFER.C
 *
 * (c)Copyright 1997, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 *
 * Buffered I/O, may be used for reading or writing, but not both.
 */

#include "defs.h"

Prototype Buffer *bopen(int fd);
Prototype void bclose(Buffer *b, int closeFd);
Prototype void efree(char **pcopy, int *pcopymax);
Prototype char *bgets(Buffer *b, int *pbytes);
Prototype char *egets(Buffer *b, int *pbytes);
Prototype void bwrite(Buffer *b, const void *data, int bytes);
Prototype void bflush(Buffer *b);
Prototype int  berror(Buffer *b);
Prototype off_t btell(Buffer *b);
Prototype int bunget(Buffer *b, int n);
Prototype int bextfree(Buffer *b);

Buffer *
bopen(int fd)
{
    Buffer *b;
    int pgSize;

    b = pagealloc(&pgSize, 1);
    bzero(b, sizeof(Buffer));
    b->bu_BufMax = pgSize - offsetof(Buffer, bu_Buf[0]) - sizeof(int);
    b->bu_DataMax = b->bu_BufMax;
    b->bu_Data = b->bu_Buf;
    b->bu_Fd = fd;

    return(b);
}

void
bclose(Buffer *b, int closeFd)
{
    if (b != NULL) {
	if (closeFd && b->bu_Fd >= 0) {
	    close(b->bu_Fd);
	    b->bu_Fd = -1;
	}
	b->bu_Beg = 0;
	b->bu_NLScan = 0;
	b->bu_End = 0;
	(void)bextfree(b);
	pagefree(b, 1);
    }
}

/*
 * bgets() - get a line from the input.  Return NULL if no data
 *	     ready to read (non-blocking only), -1 on EOF, or a
 *	     pointer to the data.  If ptr[bytes-1] != '\n', a full
 *	     line could not be returned due to the limited buffer size.
 *
 *	     if a real pointer is returned, *pbytes will be non-zero. 
 */

char *
bgets(Buffer *b, int *pbytes)
{
    int i;

    if (b->bu_Beg == b->bu_End) {
	b->bu_Beg = 0;
	b->bu_NLScan = 0;
	b->bu_End = 0;
    }

    for (;;) {
	/*
	 * look for a newline, return the line as appropriate.  NLScan
	 * is an optimization so we do not rescan portions of the input
	 * buffer that we have already checked.
	 */

	for (i = b->bu_NLScan; i < b->bu_End; ++i) {
	    if (b->bu_Data[i] == '\n') {
		char *p = b->bu_Data + b->bu_Beg;

		/*
		 * terminate buffer, removing newline, or return
		 * exact size of buffer with nothing removed or terminated.
		 */

		if (pbytes) {
		    *pbytes = i + 1 - b->bu_Beg;
		} else {
		    b->bu_Data[i] = 0;
		}
		/*
		 * note: cannot reset to 0/0 or bunget will not work.
		 */
		b->bu_Beg = i + 1;
		b->bu_NLScan = b->bu_Beg;
		return(p);
	    }
	}
	b->bu_NLScan = i;

	/*
	 * If there is no room to append new data, attempt to
	 * make some.
	 */
	if (b->bu_End == b->bu_DataMax && b->bu_Beg != 0) {
	    memmove(b->bu_Data, b->bu_Data + b->bu_Beg, b->bu_End - b->bu_Beg);
	    b->bu_End -= b->bu_Beg;
	    b->bu_NLScan -= b->bu_Beg;
	    b->bu_Beg = 0;
	}

	/*
	 * If the buffer is full, we have an overflow problem.
	 */
	if (b->bu_End == b->bu_DataMax) {
	    char *p = b->bu_Data + b->bu_Beg;

	    if (pbytes == NULL) {
		syslog(LOG_ERR, "Line buffer overflow fd %d", b->bu_Fd);
		p = (void *)-1;
	    } else {
		*pbytes = b->bu_End - b->bu_Beg;
		/*
		 * note: cannot reset to 0 or bunget will not work
		 */
		b->bu_Beg = b->bu_NLScan = b->bu_End;
	    }
	    return(p);
	}
	/*
	 * otherwise read some data and repeat
	 */
	{
	    int n;

	    n = read(b->bu_Fd, b->bu_Data + b->bu_End, b->bu_DataMax - b->bu_End);

	    /*
	     * EOF
	     */
	    if (n == 0) {
		return((void *)-1);
	    }
	    if (n < 0) {
		if (errno == EAGAIN || errno == EINTR)
		    return(NULL);
		return((void *)-1);
	    }
	    b->bu_End += n;
	}
    }
    /* not reached */
    return((void *)-1);
}

/*
 * egets() - same as bgets, but will allocate an extended buffer to
 *	     fit the line if necessary.
 */

char *
egets(Buffer *b, int *pbytes)
{
    char *ptr = NULL;

    while ((ptr = bgets(b, pbytes)) != NULL && ptr != (char *)-1) {
	/*
	 * looking for newline
	 */
	if (ptr[*pbytes-1] == '\n') {
	    break;
	}
	/*
	 * if an overflow occurs, b->bu_Beg is guarenteed to be 0
	 */
	{
	    int cpyLen;
	    char *cpy = pagealloc(&cpyLen, *pbytes * 3 / 2);

	    bcopy(b->bu_Data, cpy, b->bu_End);
	    if (b->bu_Data != b->bu_Buf) 
		pagefree(b->bu_Data, b->bu_DataMax);
	    b->bu_Data = cpy;
	    b->bu_DataMax = cpyLen;
	}
    }
    return(ptr);
}

/*
 * bunget() partially undoes the most recent read.  You cannot bunget more
 * then the number of bytes in the last read.
 */

int
bunget(Buffer *b, int n)
{
    int r = -1;

    if (n <= b->bu_Beg) {
	b->bu_Beg -= n;
	r = 0;
    }
    return(r);
}

void
bwrite(Buffer *b, const void *data, int bytes)
{
    while (bytes > 0) {
	int n = b->bu_DataMax - b->bu_End;

	if (n > bytes)
	    n = bytes;
	memcpy(b->bu_Data + b->bu_End, data, n);
	b->bu_End += n;
	data = (const char *)data + n;
	bytes -= n;

	if (b->bu_End == b->bu_DataMax)
	    bflush(b);
    }
}

/*
 * bfreebuf() - free extended buffer if possible.  Return
 *		0 on success, -1 on failure.
 */

int
bextfree(Buffer *b)
{
    int r = 0;

    if (b->bu_Data != b->bu_Buf) {
	r = -1;
	if (b->bu_End - b->bu_Beg <= b->bu_BufMax) {
	    r = 0;
	    if (b->bu_Beg != b->bu_End)
		bcopy(b->bu_Data, b->bu_Buf, b->bu_End - b->bu_Beg);
	    b->bu_NLScan -= b->bu_Beg;
	    b->bu_End -= b->bu_Beg;
	    b->bu_Beg = 0;
	    pagefree(b->bu_Data, b->bu_DataMax);
	    b->bu_Data = b->bu_Buf;	  /* not really necessary */
	    b->bu_DataMax = b->bu_BufMax; /* not really necessary */
	}
    }
    return(r);
}

void
bflush(Buffer *b)
{
    int n;

    if (b->bu_Error == 0) {
	while (b->bu_Beg != b->bu_End) {
	    n = write(b->bu_Fd, b->bu_Data + b->bu_Beg, b->bu_End - b->bu_Beg);
	    if (n < 0) {
		b->bu_Error = 1;
		break;
	    }
	    b->bu_Beg += n;
	}
    }
    b->bu_Beg = b->bu_End = b->bu_NLScan = 0;
    (void)bextfree(b);
}

off_t
btell(Buffer *b)
{
    return(lseek(b->bu_Fd, 0L, 1) + (b->bu_End - b->bu_Beg));
}

int
berror(Buffer *b)
{
    return(b->bu_Error);
}

