/************************************************************
 *                                                          *
 *  Permission is hereby granted  to  any  individual   or  *
 *  institution   for  use,  copying, or redistribution of  *
 *  the xgobi code and associated documentation,  provided  *
 *  that   such  code  and documentation are not sold  for  *
 *  profit and the  following copyright notice is retained  *
 *  in the code and documentation:                          *
 *        Copyright (c) 1990, ..., 1996 Bellcore            *
 *                                                          *
 *  We welcome your questions and comments, and request     *
 *  that you share any modifications with us.               *
 *                                                          *
 *    Deborah F. Swayne            Dianne Cook              *
 *   dfs@research.att.com       dicook@iastate.edu          *
 *      (973) 360-8423    www.public.iastate.edu/~dicook/   *
 *                                                          *
 *                    Andreas Buja                          *
 *                andreas@research.att.com                  *
 *              www.research.att.com/~andreas/              *
 *                                                          *
 ************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <signal.h>
#include <math.h>
#include "xincludes.h"
#include "xgobitypes.h"
#include "xgobivars.h"
#include "xgobiexterns.h"

Boolean gotone = False;

/*
 * Some people's distributions of the X libraries don't contain these
 * two routines, so here they are:
*/

#if !defined(XlibSpecificationRelease) || XlibSpecificationRelease < 5
XrmDatabase
XrmGetDatabase(display)
    Display *display;
{
  return display->db;
}
void
XrmSetDatabase(display, database)
    Display *display;
    XrmDatabase database;
{
  display->db = database;
}
#endif

/*ARGSUSED*/
static void
stdin_empty(arg)
  int arg;
{
  if (!gotone)
  {
    fprintf(stderr, "xgobi requires a filename or some data from stdin\n");
    exit(0);
  }
}

void
strip_suffixes(xg)
  xgobidata *xg;
{
/*
 * Find the name of the data file excluding certain suffixes:
 * .bin, .missing, .dat
*/
  int i, nchars;
  static char *suffix[] = {
    ".bin",
    ".missing",
    ".dat"
  };

  sprintf(xg->datafname, "%s", xg->datafilename);
  for (i=0; i<3; i++)
  {
    nchars = strlen(xg->datafilename) - strlen(suffix[i]) ;
    if (strcmp(suffix[i],
               (char *) &xg->datafilename[nchars]) == 0)
    {
      strncpy((char *) xg->datafname, (char *) xg->datafilename, nchars);
      xg->datafname[nchars] = '\0';
      break;
    }
  }
}


Boolean
find_data_start(fp)
  FILE *fp;
{
  int ch;
  Boolean morelines = True;
  Boolean comment_line = True;

  while (comment_line)
  {
    /* skip white space */
    while (1)
    {
      ch = getc(fp);
      if (ch == '\t' || ch == ' ' || ch == '\n')
        ;
      else
        break;
    }

    /*
     * If we've crept up on an EOF, set morelines to False.
    */
    if (ch == EOF)
    {
      morelines = False;
      break;
    }

    /* Comment lines must begin with a punctuation character */
    else if (ispunct(ch) && ch != '-' && ch != '+' && ch != '.')
    {
      fprintf(stderr, "Skipping a comment line beginning with '%c'\n", ch);
      while ((ch = getc(fp)) != '\n')
        ;
    }
    else if (isalpha(ch) && ch != 'n' && ch != 'N')
    {
      fprintf(stderr, "Comment lines must begin with # or %%;\n");
      fprintf(stderr, "I found a line beginning with '%c'\n", ch);
      exit(1);
    }
    else
    {
      comment_line = False;
      ungetc(ch, fp);
    }
  }

  return(morelines);
}

void
read_binary(fp, xg)
  FILE *fp;
  xgobidata *xg;
{
  int i, j, nr, nc;
  int onesize;

  fread((char *) &nr, sizeof(nr), 1, fp);
  fread((char *) &nc, sizeof(nc), 1, fp);

  xg->nrows = xg->nrows_in_plot = nr;
  xg->ncols_used = nc;
  xg->ncols = nc + 1;

  xg->raw_data = (float **) XtMalloc((Cardinal) xg->nrows * sizeof(float *));
  for (i=0; i<xg->nrows; i++)
    xg->raw_data[i] = (float *) XtMalloc((Cardinal) xg->ncols * sizeof(float));

  onesize = sizeof(float);
  for (i=0; i<xg->nrows; i++)
    for (j=0; j<xg->ncols_used; j++)
    {
      if (fread((char *) &xg->raw_data[i][j], onesize, 1, fp) != 1)
      {
        fprintf(stderr, "problem in reading the binary data file\n");
        fclose(fp);
        exit(0);
      }
    }

  if (fclose(fp) == EOF)
    fprintf(stderr, "binary_read: error in fclose");
}

void
read_ascii(fp, xg)
  FILE *fp;
  xgobidata *xg;
{
  register int ch;
  int i, j, jrows, jcols, fs;
  int nitems;
  float row1[NCOLS];
  short row1_missing[NCOLS];
  int nblocks;
  char word[64];

/*
 * Read in the first row of the data file and calculate ncols.
*/
  xg->ncols_used = 0;
  while ( (ch = getc(fp)) != '\n')
  {
    if (!gotone)
      gotone = True;

    /*
     * If xgobi encounters a line whose first non-blank
     * character is a punctuation character (except for +,-,.),
     * call it a comment line and skip to the end of the line.
    */
    if (ch == '\t' || ch == ' ')
      ;

    else if (!isdigit(ch) && ch != 'n' && ch != 'N' &&
              ch != '-' && ch != '+' && ch != '.')
    {
      if (xg->ncols_used > 0)
      {
        fprintf(stderr, "Sorry, xgobi accepts only numeric data\n");
        exit(0);
      }
      else if (ispunct(ch))
      {
        fprintf(stderr, "Skipping a comment line beginning with '%c'\n", ch);
        while ((ch = getc(fp)) != '\n')
        {
          /* In case the file consists of text only: */
          if (ch == EOF)
          {
            fprintf(stderr,
              "read_array: This file seems to contain no numeric data.\n");
            fclose(fp);
            exit(0);
          }
        }
      }
      else {
        fprintf(stderr, "xgobi can't handle a line beginning with '%c'\n", ch);
        fclose(fp);
        exit(0);
      }
    }
    else
    {
      if ( ungetc(ch, fp) == EOF || fscanf(fp, "%s", word) < 0 )
      {
        fprintf(stderr,
          "read_array: error in reading first row of data\n");
        fclose(fp);
        exit(0);
      }
      else
      {
        if ( strcasecmp(word, "na") == 0 || strcmp(word, ".") == 0)
        {
          xg->nmissing++;
          if (!xg->missing_values_present)
            xg->missing_values_present = True;
          row1_missing[xg->ncols_used] = 1;
          row1[xg->ncols_used] = 0.0;
        }
        else
        {
          row1_missing[xg->ncols_used] = 0;
          row1[xg->ncols_used] = (float) atof(word);
        }
        xg->ncols_used++ ;

        if (xg->ncols_used >= NCOLS)
        {
          fprintf(stderr,
           "This file has more than %d columns.  In order to read it in,\n",
            NCOLS);
          fprintf(stderr,
           "reset NCOLS in xgobitypes.h and recompile.\n");
        }
      }
    }
  }

  xg->ncols = xg->ncols_used + 1;
/*
 * Allocate space for 500 rows
*/
  xg->raw_data = (float **) XtMalloc((Cardinal) 500 * sizeof(float *));
  for (i=0; i<500; i++)
    xg->raw_data[i] = (float *) XtMalloc(
      (Cardinal) xg->ncols * sizeof(float));

/*
 * Ditto for the missing values
*/
  xg->is_missing = (short **) XtMalloc((Cardinal) 500 * sizeof(short *));
  for (i=0; i<500; i++)
    xg->is_missing[i] = (short *) XtMalloc(
      (Cardinal) xg->ncols * sizeof(short));

/*
 * Fill in the first row
*/
  for (j=0; j<xg->ncols_used; j++)
  {
    xg->raw_data[0][j] = row1[j];
    xg->is_missing[0][j] = row1_missing[j];
  }

/*
 * Read data, reallocating whenever 500 more rows are read in.
 * Determine nrows.
*/
  nblocks = 1;
  nitems = xg->ncols_used;
  xg->nrows = jrows = 1;
  jcols = 0;
  while (1)
  {

    /*
     * At the beginning of each line, skip over white space.
     * Test the first character to make sure it's a number or
     * a + or -.  (It won't ever be e, will it?)
     * ... or an "n" or an "N"
    */
    if (jcols == 0)
      if (find_data_start(fp) == False)
        break;

    fs = fscanf(fp, "%s", word);
    /*fs = fscanf(fp, "%g", &xg->raw_data[xg->nrows][jcols]);*/
    /*jcols++;*/

    if (fs == EOF)
      break;
    else if (fs < 0)
    {
      fprintf(stderr, "Problem with input data\n");
      fclose(fp);
      exit(0);
    }
    else
    {
      nitems++;

      if ( strcasecmp(word, "na") == 0 || strcmp(word, ".") == 0 )
      {
        xg->nmissing++;
        if (!xg->missing_values_present)
          xg->missing_values_present = True;
        xg->is_missing[xg->nrows][jcols] = 1;
        xg->raw_data[xg->nrows][jcols] = 0.0;
      }
      else
      {
        xg->is_missing[xg->nrows][jcols] = 0;
        xg->raw_data[xg->nrows][jcols] = (float) atof(word);
      }

      jcols++;
      if (jcols == xg->ncols_used)
      {
        jcols = 0;
        xg->nrows++;
        jrows++;
      }
      if (jrows == 500)
      {
        jrows = 0;
        nblocks++;

        xg->raw_data = (float **) XtRealloc((char *)
          xg->raw_data,
          (Cardinal) (nblocks * 500) * sizeof(float *));
        for (i=500*(nblocks-1); i<500*nblocks; i++)
          xg->raw_data[i] = (float *) XtMalloc(
            (Cardinal) xg->ncols * sizeof(float));

        /* And the missing value matrix, too ... */
        xg->is_missing = (short **) XtRealloc((char *)
          xg->is_missing,
          (Cardinal) (nblocks * 500) * sizeof(short *));
        for (i=500*(nblocks-1); i<500*nblocks; i++)
          xg->is_missing[i] = (short *) XtMalloc(
            (Cardinal) xg->ncols * sizeof(short));

      }
    }
  }

/*
 * Close the data file
*/
  if (fclose(fp) == EOF)
    fprintf(stderr, "read_array: error in fclose");

  if ( nitems != xg->nrows * xg->ncols_used )
  {
    (void) fprintf(stderr, "read_array: nrows*ncols != nitems read\n");
    (void) fprintf(stderr, "(nrows %d, ncols %d, nitems read %d)\n",
      xg->nrows, xg->ncols_used, nitems);
    exit(0);
  }
  else if (nitems == 0)
  {
    (void) fprintf(stderr, "No data read\n");
    exit(0);
  }
  else
  {
    /*
     * One last XtFree and XtRealloc to make raw_data take up exactly
     * the amount of space it needs.
    */
    for (i=xg->nrows; i<500*nblocks; i++)
      XtFree((XtPointer) xg->raw_data[i]);
    xg->raw_data = (float **) XtRealloc((XtPointer) xg->raw_data,
      (Cardinal) xg->nrows * sizeof(float *));

    /*
     * If we haven't yet encountered a missing value, free up
     * the whole matrix.  Otherwise, do as above.
    */
    if (!xg->missing_values_present)
    {
      for (i=0; i<500*nblocks; i++)
        XtFree((XtPointer) xg->is_missing[i]);
      XtFree((char *) xg->is_missing);
      xg->is_missing = (short **) NULL;
    }
    else
    {
      for (i=xg->nrows; i<500*nblocks; i++)
        XtFree((XtPointer) xg->is_missing[i]);
      xg->is_missing = (short **) XtRealloc((XtPointer) xg->is_missing,
        (Cardinal) xg->nrows * sizeof(short *));
    }

    /*
     * If the data contains only one column, add a second,
     * the numbers 1:nrows.
    */
    if (xg->ncols_used == 1)
    {
      xg->ncols_used = 2;
      xg->ncols = 3;
      for (i=0; i<xg->nrows; i++)
      {
        xg->raw_data[i] = (float *) XtRealloc(
          (XtPointer) xg->raw_data[i],
          (Cardinal) 3 * sizeof(float));
        xg->raw_data[i][1] = (float) (i+1) ;

        /* And populate a column of missing values with 0s, if needed */
        if (xg->missing_values_present)
        {
          xg->is_missing[i] = (short *) XtRealloc(
            (XtPointer) xg->is_missing[i],
            (Cardinal) 3 * sizeof(short));
          xg->is_missing[i][1] = 0 ;
        }
      }
    }
  }
}

void
read_array(xg)
  xgobidata *xg;
{
  char fname[128];
  FILE *fp;

/*
 * Check file exists and open it - for stdin no open needs to be done
 * only assigning fp to be stdin.
*/
  if (strcmp((char *) xg->datafname, "stdin") == 0)
  {
    fp = stdin;

    /*
     * If reading from stdin, set an alarm.  If after 5 seconds,
     * no data has been read, print an error message and exit.
    */
    if (fp == stdin)
    {
      alarm((unsigned int) 5);
      signal(SIGALRM, stdin_empty);
    }
    read_ascii(fp, xg);
  }
  else
  {
    /* 
     * Are we reading the missing data into xg->raw_data ?
    */
    if (strcmp(
         ".missing",
         &xg->datafilename[strlen(xg->datafilename) - strlen(".missing")]
       ) == 0)
    {
      if ((fp = fopen(xg->datafilename, "r")) != NULL)
      {
        char *title, fulltitle[256];
        xg->is_missing_values_xgobi = True;
        read_ascii(fp, xg);

        /*
         * extend the title
        */
        title = (char *) XtMalloc(256 * sizeof(char));
        XtVaGetValues(xg->shell,
          XtNtitle, (String) &title,
          NULL);
        sprintf(fulltitle, "%s (Missing)", title);
        XtVaSetValues(xg->shell,
          XtNtitle, (String) fulltitle,
          XtNiconName, (String) fulltitle,
          NULL);
      }
      else
      {
        (void) fprintf(stderr,
          "The file %s can't be opened for reading.\n", xg->datafilename);
        exit(0);
      }
    }
    else
    {
      /*
       * Try fname.bin before fname, to see whether there is a binary
       * data file available.  If there is, call read_binary().
      */
      strcpy(fname, (char *) xg->datafname);
      strcat(fname, ".bin");

      if ((fp = fopen(fname, "r")) != NULL)
        read_binary(fp, xg);

      /*
       * If not, look for an ASCII file
      */
      else
      {
        /* Try fname.dat first, then fname without a suffix */
        strcpy(fname, (char *) xg->datafname);
        strcat(fname, ".dat");
        if ((fp = fopen(fname, "r")) == NULL)
        {
          if ((fp = fopen((char *) xg->datafname, "r")) == NULL)
          {
            (void) fprintf(stderr,
              "Neither the file %s nor %s exists\n", xg->datafname, fname);
            exit(0);
          }
          else
            read_ascii(fp, xg);
        }
        else
          read_ascii(fp, xg);
      }
    }
  }
}

int
Sread_array(xg)
  xgobidata *xg;
{
  int i, j, nitems = 0;
  int ok = 0;
  FILE *fp;
  char word[64];

  if ((fp = fopen((char *) xg->datafname, "r")) == NULL) {
    (void) fprintf(stderr,
      "Sread_array: data file %s does not exist\n", xg->datafname);
    exit(0);

  }

  else
  {
    xg->raw_data = (float **) XtMalloc(
			(Cardinal) xg->nrows * sizeof(float *));
    for (i=0; i<xg->nrows; i++)
      xg->raw_data[i] = (float *) XtMalloc(
				(Cardinal) xg->ncols * sizeof(float));

    /*
     * Allocate space for missing data, and free it if
     * it isn't used.
    */
    xg->is_missing = (short **) XtMalloc(
			(Cardinal) xg->nrows * sizeof(short *));
    for (i=0; i<xg->nrows; i++)
      xg->is_missing[i] = (short *) XtMalloc(
				(Cardinal) xg->ncols * sizeof(short));

    for (i=0; i<xg->nrows; i++) {
      for (j=0; j<xg->ncols_used; j++) {
        (void) fscanf(fp, "%s", word);
        if (strcasecmp(word, "na") == 0 || strcmp(word, ".") == 0) {
          if (!xg->missing_values_present)
            xg->missing_values_present = True;
          xg->is_missing[i][j] = 1;
          xg->raw_data[i][j] = 0.0;
        } else {
          xg->is_missing[i][j] = 0;
          xg->raw_data[i][j] = (float) atof(word);
        }
        nitems++;
      }
    }

    /*
     * Test the number of items read against the dimension
     * of the array.
    */
    if (nitems == xg->nrows * xg->ncols_used)
    {
      ok = 1;
      if (unlink((char *) xg->datafname) != 0)
        fprintf(stderr, "Sread_array: error in unlink\n");
    }
    else
    {
      ok = 0;
      (void) fprintf(stderr, "Sread_array: nrows*ncols != nitems\n");
      exit(0);
    }

    /*
     * Now fill some data into the extra column, which will be
     * used if groups are defined by brushing.
    */
    for (i=0; i<xg->nrows; i++) {
      xg->raw_data[i][xg->ncols_used] = 1.0;
      xg->is_missing[i][xg->ncols_used] = 0;
    }

    /*
     * If we haven't yet encountered a missing value, free up
     * the whole matrix.
    */
    if (!xg->missing_values_present)
    {
      for (i=0; i<xg->nrows; i++)
        XtFree((XtPointer) xg->is_missing[i]);
      XtFree((char *) xg->is_missing);
      xg->is_missing = (short **) NULL;
    }
  }

  return(ok);
}


void
fill_extra_column(xg)
  xgobidata *xg;
{
  int i;

  for (i=0; i<xg->nrows; i++)
    xg->raw_data[i][xg->ncols_used] = 1.0 ;
}

void
find_root_name_of_data(fname, title)
  char *fname, *title;
/*
 * Strip off preceding directory names and find the final filename
 * to use in resetting the title and iconName.
*/
{
  char *pf;
  int j = 0;

  pf = fname;
  while (fname[j] != '\0')
  {
    if (fname[j] == '/')
      pf = &fname[j+1];
    j++;
  }

  (void) strcpy(title, pf);
}

void
set_title_and_icon(fname, xg)
  char *fname;
  xgobidata *xg;
/*
 * If the user hasn't specified a title in a general resource
 * file, a data-specific resource file, or on the command
 * line, and if the data is being read in from a file, then
 * use the name of the file to compose the title and iconName
 * resources.
*/
{
  char fulltitle[150], usertitle[120], rootfname[100];
  char *str_type[50];
  XrmValue value;

  if (XrmGetResource((XrmDatabase) XrmGetDatabase(display),
    "xgobi.title", "XGobi.Title", str_type, &value))
  {
    (void) strncpy(usertitle, value.addr, (int) value.size);
  }

  if (strcmp(fname, "stdin") == 0)  /* If input is from stdin */
  {
    strcpy(fulltitle, "XGobi");

    /* Hmm.  This will have to be tested in Clone_XGobi() */
    xg->datarootname = (char *) NULL;
  }
  else  /* If reading from a file */
  {
    /* Used in Clone_XGobi() as well as in one option here */
    find_root_name_of_data(fname, rootfname);
    xg->datarootname = XtMalloc((unsigned)
      (strlen(rootfname) + 1) * sizeof(char));
    strcpy(xg->datarootname, rootfname);

    strcpy(fulltitle, "XGobi: ");

    /* usertitle is by default XGobi, the class name */
    if (usertitle && ((int)strlen(usertitle) > 0))
    {
      /*
       * If the title is exactly XGobi, add the data name
       * to fulltitle
      */
      if (strcmp(usertitle, "XGobi") == 0)
        strcat(fulltitle, rootfname);

      /* If the title contains XGobi, just use it */
      else if (strncmp("XGobi", usertitle, 5) == 0)
        strcpy(fulltitle, usertitle);

      /* else add it to fulltitle */
      else
        strcat(fulltitle, usertitle);
    }
    else  /* If usertitle is NULL */
    {
      find_root_name_of_data(fname, rootfname);
      strcat(fulltitle, rootfname);
    }
  }

  sprintf(xg->title, fulltitle);
  XtVaSetValues(xg->shell,
    XtNtitle, (String) fulltitle,
    XtNiconName, (String) fulltitle,
    NULL);
}

int
read_extra_resources(data_in)
/*
 * Read in the data-specific resource file.
*/
  char *data_in;
{
  char fname[128];
  int found = 0;
  int ok = 1;
  FILE *fp;
  static char *suffix = ".resources";

  if (data_in != "" && strcmp(data_in, "stdin") != 0) {
    (void) strcpy(fname, data_in);
    (void) strcat(fname, suffix);
    if ( (fp = fopen(fname,"r")) != NULL)
      found = 1;
  }

  if (found) {
    XrmDatabase newdb;

    if ( (newdb = (XrmDatabase) XrmGetFileDatabase(fname)) != NULL ) {
      XrmDatabase dispdb;

      dispdb = XrmGetDatabase(display);
      XrmMergeDatabases(newdb, &dispdb);
      XrmSetDatabase(display, dispdb);
    }
    else {
      ok = 0;
      fprintf(stderr,
        "read_extra_resources: problem reading data-specific resource file\n");
      exit(0);
    }

  /*
   * Close the data file
  */
    if (fclose(fp) == EOF)
      fprintf(stderr, "read_extra_resources: error in fclose");
  }
  return(ok);
}
