/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "../configure.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <X11/Intrinsic.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "sysdeps.h"

#ifndef INADDR_NONE
#  define INADDR_NONE (0xfffffffful)    /* a hack, hope it works */
#endif

#ifndef USE_POLL
#  if defined(SYSV) && !defined(__hpux)
#    define USE_POLL 1
#  else
#    define USE_POLL 0
#  endif
#endif

#if USE_POLL
#  include <sys/poll.h>
#endif

#ifndef USE_FIONBIO
#  if defined(sco) && sco
#    define USE_FIONBIO  1
#  else
#    define USE_FIONBIO  0
#  endif
#endif

#if USE_FIONBIO
#  include <sys/ioctl.h>
#endif

static long set_nonblock(int fd)
{
#if !USE_FIONBIO
    long	flags;

    flags = fcntl(fd, F_GETFL);
    if (flags >= 0)
	flags = fcntl(fd, F_SETFL, flags | O_NONBLOCK);
    if (flags < 0)
	perror("knews: fcntl");

    return flags;
#else
    int	yes = 1;

    yes = ioctl(fd, FIONBIO, &yes);
    if (yes < 0)
	perror("knews: ioctl");

    return yes;
#endif
}

/*
 *  FreeBSD doesn't define EIPROGRESS #if _POSIX_SOURCE...
 */
int would_block(int fd, int err_no)
{
    return (
#ifdef EWOULDBLOCK
	    err_no == EWOULDBLOCK ||
#endif
#ifdef EINPROGRESS
	    err_no == EINPROGRESS ||
#endif
	    err_no == EAGAIN);
}

char *error_string(int err_no)
{
    switch (err_no) {
#ifdef ECONNREFUSED
    case ECONNREFUSED:
	return "Connection refused";
#endif
#ifdef ENETUNREACH
    case ENETUNREACH:
	return "Network unreachable";
#endif
#ifdef ETIMEDOUT
    case ETIMEDOUT:
	return "Connection timed out";
#endif
#ifdef EADDRNOTAVAIL
    case EADDRNOTAVAIL:
	return "Address not available";
#endif
    default:
	break;
    }

    return NULL;
}

extern int 		disp_conn_num;
extern XtAppContext	app_cont;

#define MASK (XtIMXEvent | XtIMTimer | XtIMAlternateInput)

int do_wait(int *fd, int rd)
{
    int			tmp;
#if USE_POLL
    struct pollfd	fds[2];
#else
    fd_set		read_fds;
    fd_set		write_fds;
    int			maxfdp1;

    maxfdp1 = (*fd > disp_conn_num ? *fd : disp_conn_num) + 1;
#endif

    for (;;) {
	while (XtAppPending(app_cont) & MASK) {
	    XtAppProcessEvent(app_cont, MASK);
	    if (*fd < 0)
		return -1;
	}

#if USE_POLL
	fds[0].fd = disp_conn_num;
	fds[0].events = POLLIN;
	fds[0].revents = 0;
	fds[1].fd = *fd;
	fds[1].events = rd ? POLLIN : POLLOUT;
	fds[1].revents = 0;

	do {
	    tmp = poll(fds, 2, -1);
	} while (tmp < 0 && errno == EINTR);

	if (tmp < 0) {
	    perror("knews: poll");
	    return -1;
	}

	if (fds[1].revents)
	    break;
	if (fds[0].revents) {
	    XtAppProcessEvent(app_cont, MASK);
	    if (*fd < 0)
		return -1;
	}
#else
	FD_ZERO(&read_fds);
	FD_ZERO(&write_fds);
	FD_SET(disp_conn_num, &read_fds);
	if (rd)
	    FD_SET(*fd, &read_fds);
	else
	    FD_SET(*fd, &write_fds);

	do {
	    tmp = select(maxfdp1,
#ifdef __hpux  /* don't ask */
			 (int *)&read_fds, rd ? NULL : (int *)&write_fds,
#else
			 &read_fds, rd ? NULL : &write_fds,
#endif
			 NULL, NULL);
	} while (tmp < 0 && errno == EINTR);

	if (tmp < 0) {
	    perror("knews: select");
	    return -1;
	}

	if (FD_ISSET(*fd, &read_fds) || FD_ISSET(*fd, &write_fds))
	    break;
	if (FD_ISSET(disp_conn_num, &read_fds)) {
	    XtAppProcessEvent(app_cont, MASK);
	    if (*fd < 0)
		return -1;
	}
#endif
    }

    return 0;
}

/***********************************************************************/

static struct sockaddr_in	serv_addr;

int get_host(char *host)
{
    int			port;
    char		*c;
    unsigned long	l;

    port = 0;
    c = strchr(host, ':');
    if (c) {
	*c = '\0';
	if (c[1] >= '0' && c[1] <= '9')
	    port = atoi(c + 1);
    }

    if (port != 0)
	port = htons(port);
    else {
	struct servent	*service;

	service = getservbyname("nntp", "tcp");
	if (!service) {
	    if (c)
		*c = ':';
	    return GET_HOST_NO_SERVICE;
	}
	port = service->s_port;
    }

    memset(&serv_addr, 0, sizeof serv_addr);
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_port = port;

    if ((l = inet_addr(host)) != INADDR_NONE) {
	memcpy(&serv_addr.sin_addr, &l, sizeof l);
	if (c)
	    *c = ':';
    } else {
	struct hostent	*hp;

	hp = gethostbyname(host);
	if (c)
	    *c = ':';
	if (!hp)
	    return GET_HOST_NO_HOST;
	serv_addr.sin_family = hp->h_addrtype;
	memcpy(&serv_addr.sin_addr, hp->h_addr, hp->h_length);
    }

    return 0;
}

int open_socket(void)
{
    int	fd;
    int temp;
    int alive = 1;

    do {
	fd = socket(AF_INET, SOCK_STREAM, 0);
    } while(fd < 0 && errno == EINTR);

    if (fd < 0) {
	perror("knews: socket");
	return -1;
    }

    if (set_nonblock(fd) < 0) {
	close(fd);
	return -1;
    }

    do {
	temp = setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE,
			  (char *)&alive, sizeof alive);
    } while (temp < 0 && errno == EINTR);

    if (temp < 0) {
	perror("knews: setsockopt");
	close(fd);
	return -1;
    }

    return fd;
}

int connect_socket(int fd)
{
    int	temp;

    do {
	temp = connect(fd, (struct sockaddr *)&serv_addr, sizeof serv_addr);
    } while (temp < 0 && errno == EINTR);

    return temp;
}


int open_duplex(int *fd)
{
#if !defined(HAVE_SOCKETPAIR) || HAVE_SOCKETPAIR
    int	temp;

    do {
	temp = socketpair(AF_UNIX, SOCK_STREAM, 0, fd);
    } while (temp < 0 && errno == EINTR);

    if (temp < 0) {
	perror("knews: socketpair");
	return -1;
    }

    if (set_nonblock(fd[0]) < 0) {
	close(fd[0]);
	close(fd[1]);
	return -1;
    }

    return 0;
#else
    fprintf(stderr, "knews: compiled without HAVE_SOCKETPAIR, "
	    "can't open connection!\n");
    return -1;
#endif
}

/*************************************************************************/

char *get_mailhostname(void)
{
    char		buffer[1024];
    struct hostent      *hent;
    char		**loop;

    if (gethostname(buffer, sizeof buffer - 1) != 0)
	return NULL;

    buffer[sizeof buffer - 1] = '\0';
    if (strchr(buffer, '.'))
	return XtNewString(buffer);

    if (buffer[0] == '\0')
	return NULL;

    hent = gethostbyname(buffer);
    if (!hent)
	return NULL;

    if (hent->h_name && strchr(hent->h_name, '.'))
	return XtNewString(hent->h_name);

    if (hent->h_aliases)
	for (loop = hent->h_aliases ; *loop ; loop++)
	    if (strchr(*loop, '.'))
		return XtNewString(*loop);

    return NULL;
}

/*************************************************************************/

#if defined(HAVE_MEMMOVE) && !HAVE_MEMMOVE
void *memmove(void *dest, const void *src, size_t n)
{
    bcopy(src, dest, n);
    return dest;
}
#endif
