/*
    YPS-0.2, NIS-Server for Linux
    Copyright (C) 1994  Tobias Reber

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Modified for use with ypserv by Thorsten Kukuk <kukuk@uni-paderborn.de>
*/

static char rcsid[] = "$Id: ypxfr_clnt.c,v 1.2 1996/03/31 12:07:00 kukuk Exp $";

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include "system.h"

#if HAVE_BROKEN_LINUX_LIBC
#define xdr_domainname _xdr_domainname
#define xdr_ypreq_key _xdr_ypreq_key
#define xdr_ypreq_nokey _xdr_ypreq_nokey
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include "yp.h"
#include <rpcsvc/ypclnt.h>
#include <memory.h>
#ifdef HAVE_RPC_CLNT_SOC_H
#include <rpc/clnt_soc.h>
#endif

#include "ypxfr.h"

static struct sockaddr_in ServerAddress;
static CLIENT *UdpClient=NULL, *TcpClient=NULL;
static struct timeval TIMEOUT = { 25, 0 };

ypresp_key_val *ypproc_first_2( ypreq_nokey *argp, CLIENT *clnt)
{
  static ypresp_key_val res;
 
  memset((char*)&res, 0, sizeof(res));
  if (clnt_call(clnt, YPPROC_FIRST, (xdrproc_t) xdr_ypreq_key,
		(void *) argp, (xdrproc_t) xdr_ypresp_key_val,
		(void *) &res, TIMEOUT) != RPC_SUCCESS)
    return (NULL);
  
  return (&res);
}

void *ypproc_clear_2( void *argp, CLIENT *clnt)
{
  static char res;
  
  memset((char*)&res, 0, sizeof(res));
  if (clnt_call(clnt, YPPROC_CLEAR, (xdrproc_t) xdr_void, argp, 
		(xdrproc_t) xdr_void, &res, TIMEOUT) != RPC_SUCCESS)
    return (NULL);
  
  return ((void *)&res);
}

ypresp_all *ypproc_all_2( ypreq_nokey *argp, CLIENT *clnt)
{
  static ypresp_all res;
  
  memset(&res, 0, sizeof(res));
  if (clnt_call(clnt, YPPROC_ALL, (xdrproc_t) xdr_ypreq_nokey, 
		(void *) argp, (xdrproc_t) ypxfr_xdr_ypresp_all,
		(void *) &res, TIMEOUT) != RPC_SUCCESS)
    return (NULL);
    
  return (&res);
}
 
ypresp_master *ypproc_master_2( ypreq_nokey *argp, CLIENT *clnt)
{
  static ypresp_master res;
  
  memset(&res, 0, sizeof(res));
  if (clnt_call(clnt, YPPROC_MASTER, (xdrproc_t) xdr_ypreq_nokey,
		(void *) argp, (xdrproc_t) xdr_ypresp_master,
		(void *) &res, TIMEOUT) != RPC_SUCCESS)
    return (NULL);
  
  return (&res);
}

ypresp_maplist *ypproc_maplist_2( domainname *argp, CLIENT *clnt)
{
  static ypresp_maplist res;
  
  memset((char*)&res, 0, sizeof(res));
  if (clnt_call(clnt, YPPROC_MAPLIST, (xdrproc_t) xdr_domainname,
		(void *) argp, (xdrproc_t) xdr_ypresp_maplist,
		(void *) &res, TIMEOUT) != RPC_SUCCESS)
    return (NULL);
  
  return (&res);
}

static struct sockaddr_in *__do_ypbind(domainname d)
{
   static struct sockaddr_in resp;
   int rc;
   ypbind_resp r;
   CLIENT *localBindClient;
   struct sockaddr_in localAddr;
   int s;
   struct timeval t={5,0}, tott={25,0};

   s=RPC_ANYSOCK;
   memset(&localAddr, '\0', sizeof localAddr);
   localAddr.sin_family = AF_INET;
   localAddr.sin_addr.s_addr=htonl(INADDR_LOOPBACK);
   localBindClient=clntudp_create(&localAddr, YPBINDPROG, YPBINDVERS, tott, &s);
   if (!localBindClient) {
      clnt_pcreateerror("");
      return NULL;
   }
   
   rc=clnt_call(localBindClient, YPBINDPROC_DOMAIN, (xdrproc_t) xdr_domainname, 
		(char *)&d, (xdrproc_t) xdr_ypbind_resp, (char *)&r, t);
   if (rc) {
      clnt_perrno(rc);
      return NULL;
   }

   switch (r.ypbind_status) {
   case YPBIND_FAIL_VAL:
      switch(r.ypbind_resp_u.ypbind_error) {
      case YPBIND_ERR_ERR:
         fprintf(stderr, "YPBINDPROC_DOMAIN: Internal error\n");
         break;
      case YPBIND_ERR_NOSERV:
         fprintf(stderr, "YPBINDPROC_DOMAIN: No bound server for passed domain\n");
         break;
      case YPBIND_ERR_RESC:
         fprintf(stderr, "YPBINDPROC_DOMAIN: System resource allocation failure\n");
         break;
      default:
         fprintf(stderr, "YPBINDPROC_DOMAIN: Unknown error\n");
         break;
      }
      return NULL;
   case YPBIND_SUCC_VAL:
      {
         struct ypbind_binding *y=&r.ypbind_resp_u.ypbind_bindinfo;
         memset(&resp, '\0', sizeof resp);
         resp.sin_family=AF_INET;
         resp.sin_addr=*(struct in_addr *)(y->ypbind_binding_addr);
         return &resp;
      }
   }
   return NULL;
}

void
__yp_unbind(char *DomainName)
{
   if (UdpClient) clnt_destroy(UdpClient);
   UdpClient=NULL;
   if (TcpClient) clnt_destroy(TcpClient);
   TcpClient=NULL;
}
 
int ypbind_host(struct sockaddr_in *ServerAddress, char *DomainName)
{
   struct sockaddr_in UdpServerAddress, TcpServerAddress;
   int UdpSockp, TcpSockp;
   static struct timeval Wait = { 5, 0 };
 
   if (UdpClient || TcpClient) __yp_unbind(DomainName);
 
   memcpy(&UdpServerAddress, ServerAddress, sizeof(*ServerAddress));
   UdpServerAddress.sin_port=0;
   UdpSockp=(RPC_ANYSOCK);
   memcpy(&TcpServerAddress, ServerAddress, sizeof(*ServerAddress));
   TcpServerAddress.sin_port=0;
   TcpSockp=(RPC_ANYSOCK);
   if ((UdpClient=clntudp_create(&UdpServerAddress, YPPROG, YPVERS,
      Wait, &UdpSockp))==NULL) {
      clnt_pcreateerror("UdpClient");
      return(YPERR_RPC);
   }
   if ((TcpClient=clnttcp_create(&TcpServerAddress, YPPROG, YPVERS,
      &TcpSockp, 0, 0))==NULL) {
      clnt_pcreateerror("TcpClient");
      return(YPERR_RPC);
   }
   return(0);
 
}
 
int
__yp_bind(char *DomainName)
{
  static domainname domain;
  
  domain=DomainName;
  {
    struct sockaddr_in *s=__do_ypbind(DomainName);
    if (!s) return(YPERR_DOMAIN);
    ServerAddress=*s;
  }
  return (ypbind_host(&ServerAddress, DomainName));
}

int __yp_all( char *DomainName, char *MapName, struct ypall_callback *CallBack)
{
  static ypreq_nokey req;
  ypresp_all *resp;
  extern struct ypall_callback *xdr_ypall_callback;
  int Status;
  
  do {
    if (TcpClient==NULL)
      if ((Status=__yp_bind(DomainName))) return(Status);
    
    req.domain=DomainName;
    req.map=MapName;
    xdr_ypall_callback=CallBack;
    if ((resp=ypproc_all_2(&req, TcpClient))==NULL) {
      clnt_perror(TcpClient, "ypall");
      __yp_unbind(DomainName);
    }
  } while(resp==NULL);
  switch (resp->ypresp_all_u.val.stat) {
  case YP_TRUE:
  case YP_NOMORE:
    Status=0;
    break;
  default:
    Status=ypprot_err(resp->ypresp_all_u.val.stat);
  }
  clnt_freeres(TcpClient, (xdrproc_t) ypxfr_xdr_ypresp_all, (void *) resp);
  return(Status);
}

int __yp_master( char *DomainName, char *MapName, char **OutMaster)
{
   static ypreq_nokey req;
   ypresp_master *resp;
   int Status;
 
   do {
     if (UdpClient==NULL)
       if ((Status=__yp_bind(DomainName))) return(Status);
 
     req.domain=DomainName;
     req.map=MapName;
     if ((resp=ypproc_master_2(&req, UdpClient))==NULL) 
       {
         clnt_perror(UdpClient, "ypmaster");
         __yp_unbind(DomainName);
       }
   } while(resp==NULL);
   if (resp->stat!=YP_TRUE) 
     {
       Status=ypprot_err(resp->stat);
     } 
   else 
     {
       if (((*OutMaster)=malloc(strlen(resp->peer)+1))!=NULL) 
	 {
	   strcpy(*OutMaster, resp->peer);
	   Status=0;
	 } 
       else 
	 {
	   Status=YPERR_RESRC;
	 }
     }
   clnt_freeres(UdpClient, (xdrproc_t) xdr_ypresp_master, (void *) resp);
   return(Status);
}

int _yp_clear(char *DomainName)
{
  void *resp;
  int Status;
  
  do {
    if (UdpClient==NULL)
      if ((Status=__yp_bind(DomainName))) return(Status);
    if ((resp=ypproc_clear_2(NULL, UdpClient))==NULL) 
      {
	clnt_perror(UdpClient, "_yp_clear");
	__yp_unbind(DomainName);
      }
  } while(resp==NULL);
  return 0;
}

int _yp_maplist(char *DomainName, ypmaplist **OutMaplist)
{
  ypresp_maplist *resp;
  ypmaplist *m;
  ypmaplist *r=NULL;
  int Status;
  
  do {
    if (UdpClient==NULL)
      if ((Status=__yp_bind(DomainName))) return(Status);
    if ((resp=ypproc_maplist_2(&DomainName, TcpClient))==NULL) {
         clnt_perror(UdpClient, "_yp_maplist");
         __yp_unbind(DomainName);
      }
   } while(resp==NULL);
   if (resp->stat!=YP_TRUE) {
      Status=ypprot_err(resp->stat);
   } else {
      Status=0;
      for (m=resp->maps; m; m=m->next) {
         struct ypmaplist *rr=r;
         r=malloc(sizeof (struct ypmaplist));
         if (!r) {
            Status=YPERR_RESRC;
            break;
         }
         r->map=malloc(strlen(m->map)+1);
         if (!r->map) {
            Status=YPERR_RESRC;
            break;
         }
         strcpy(r->map, m->map);
         r->next=rr;
      }
      (*OutMaplist)=NULL;
      if (!Status) while (r) {
         struct ypmaplist *rr=r->next;
         r->next=*OutMaplist;
         (*OutMaplist)=r;
         r=rr;
      }
   }
   clnt_freeres(UdpClient, (xdrproc_t) xdr_ypresp_maplist, (void *) resp);

   return(Status);
}

char *ypxfr_err_string(enum ypxfrstat y) 
{
  switch(y) 
    {
    case YPXFR_SUCC:	return "Success";
    case YPXFR_AGE:	return "Master's version not newer";
    case YPXFR_NOMAP:	return "Can't find server for map";
    case YPXFR_NODOM:	return "Domain not supported";
    case YPXFR_RSRC:	return "Local resource alloc failure";
    case YPXFR_RPC:	return "RPC failure talking to server";
    case YPXFR_MADDR:	return "Can't get master address";
    case YPXFR_YPERR:	return "YP server/map db error";
    case YPXFR_BADARGS: return "Request arguments bad";
    case YPXFR_DBM:	return "Local dbm operation failed";
    case YPXFR_FILE:	return "Local file I/O operation failed";
    case YPXFR_SKEW:	return "Map version skew during transfer";
    case YPXFR_CLEAR:	return "Can't send \"Clear\" req to local ypserv";
    case YPXFR_FORCE:	return "No local order number in map  use -f flag.";
    case YPXFR_XFRERR:  return "ypxfr error";
    case YPXFR_REFUSED: return "Transfer request refused by ypserv";
    }
  return "Unknown Error, should not happen";
}

