/*
 * open.c - Open a VT to run a new command (or shell).
 * Copyright (c) 1994 by Jon Tombs <jon@gtex02.us.es>
 * Copyright (c) 1996 by Dominik Kubla, <kubla@Uni-Mainz.DE>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#ifndef lint
const char rcsId[] = "$Id$";
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/vt.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <linux/tty.h>

static void usage(int);

/*  VT naming convention.  */
#define VTNAME "/dev/tty%d"

/*  Program name.  */
const char progname[] = "open";

/*  Program version.  */
const char version[] = "1.3";

/*  Name of the virtual console device.  */
const char console[] = "/dev/tty0";

/*  Flag set by `--login'.  */
static int login_flag = 0;

/*  Flag set by `--switchto'.  */
static int switch_flag = 0;

/*  Flag set by `--verbose'.  */
static int verbose_flag = 0;

/*  Flag set by `--waitfor'.  */
static int wait_flag = 0;

/*  VT number selected by `--vtnum'.  Default is next available.  */
static int vt_number = -1;

/*  GNU long options.  */
static struct option lopts[] =
{
	{"vtnum", required_argument, 0, 0},
	{"login", no_argument, &login_flag, 1},
	{"switchto", no_argument, &switch_flag, 1},
	{"verbose", no_argument, &verbose_flag, 1},
	{"waitfor", no_argument, &wait_flag, 1},
	{"help", no_argument, NULL, 'H'},
	{"version", no_argument, NULL, 'V'},
	{NULL, no_argument, NULL, 0}
};

int main(int argc, char **argv)
{
	int fd = 0;
	int opt, pid;
	struct vt_stat vt;
	char vtname[sizeof VTNAME + 2];		/* allow 999 possible VTs */
	char *cmd, *def_cmd = NULL;

	while (1) {
		int idx = 0;

		opt = getopt_long(argc, argv, "c:lsvw", lopts, &idx);

		/*  Detect end of options.  */
		if (opt == -1)
			break;

		switch (opt) {
		case 0:
               		if (idx != 0)
                		break;
		case 'c':
			vt_number = (int) atol(optarg);
			if (vt_number < 0 || (getuid() ? vt_number > MAX_NR_USER_CONSOLES : vt_number > MAX_NR_CONSOLES)) {
				fprintf(stderr, "open: %s invalid vt number\n", optarg);
				exit(1);
			}
			/* close security holes - until we can do this safely */
			(void) setuid(getuid());
			break;
		case 'l':
			login_flag = 1;
			break;
		case 's':
			switch_flag = 1;
			break;
		case 'v':
			verbose_flag = 1;
			break;
		case 'w':
			wait_flag = 1;
			break;
		case 'H':
			usage(0);
		case 'V':
			printf("%s version %s\n", progname, version);
			exit(0);
		default:
			usage(1);

		}
	}


	if (!(argc > optind)) {
		def_cmd = getenv("SHELL");
		if (def_cmd == NULL)
			usage(0);
	}
	if (vt_number == -1) {
		if ((fd = open("/dev/tty0", O_WRONLY, 0)) < 0) {
			perror("open: Failed to open /dev/tty0\n");
			return (2);
		}
		if ((ioctl(fd, VT_OPENQRY, &vt_number) < 0) || (vt_number == -1)) {
			perror("open: Cannot find a free VT\n");
			close(fd);
			return (3);
		}
		if (ioctl(fd, VT_GETSTATE, &vt) < 0) {
			perror("open: can't get VTstate\n");
			close(fd);
			return (4);
		}
	}
	sprintf(vtname, VTNAME, vt_number);

	/* We assume getty has made any in use VT non accessable */
	if (access(vtname, R_OK | W_OK) < 0) {
		fprintf(stderr, "open: Failed to open %s read/write (%s)\n", vtname,
			strerror(errno));
		return (5);
	}
	if (!geteuid()) {
		uid_t uid = getuid();
		chown(vtname, uid, getgid());
		setuid(uid);
	}
	if (verbose_flag)
		fprintf(stderr, "open: using VT %s\n", vtname);

	if (def_cmd)
		cmd = malloc(strlen(def_cmd) + 2);
	else
		cmd = malloc(strlen(argv[optind]) + 2);

	if (login_flag)
		strcpy(cmd, "-");
	else
		cmd[0] = '\0';

	if (def_cmd)
		strcat(cmd, def_cmd);
	else
		strcat(cmd, argv[optind]);

	if (login_flag)
		argv[optind] = cmd++;


	if ((pid = fork()) == 0) {
		/* leave current vt */
		if (setsid() < 0) {
			fprintf(stderr, "open: Unable to set new session (%s)\n",
				strerror(errno));
		}
		close(0);
		close(1);
		close(2);
		close(fd);

		/* and grab new one */
		if ((fd = open(vtname, O_RDWR)) == -1) {	/* Shouldn't happen */
			_exit(4);	/* silently die */
		}
		dup(fd);
		dup(fd);

		if (switch_flag) {
			/* 
			 * Can't tell anyone if any of these fail, so throw away
			 * the return values 
			 */
			(void) ioctl(fd, VT_ACTIVATE, vt_number);
			/* wait to be really sure we have switched */
			(void) ioctl(fd, VT_WAITACTIVE, vt_number);
		}
		if (def_cmd)
			execlp(cmd, def_cmd, NULL);
		else
			execvp(cmd, &argv[optind]);
	}
	if (pid < 0) {
		perror("open: fork() error");
		return (6);
	}
	if (wait_flag) {
		wait(NULL);
		if (switch_flag) {	/* Switch back... */
			(void) ioctl(fd, VT_ACTIVATE, vt.v_active);
			/* wait to be really sure we have switched */
			(void) ioctl(fd, VT_WAITACTIVE, vt.v_active);
		}
	}
	close(fd);
	return 0;
}


static void usage(int s)
{
	if (s != 0) {
		fprintf(stderr, "Try `%s --help' for further information.\n", progname);
		exit(1);
	} else {
		printf("Usage: %s [OPTION] [PROGRAM]\n\n", progname);
		printf("  -c VT, --vtnum=VT  use the given VT number\n");
		printf("  -l   , --login     make command a login shell\n");
		printf("  -s   , --switch    switch VT when starting the command\n");
		printf("  -v   , --verbose   be a bit more verbose\n");
		printf("  -w   , --wait      wait for completion of command\n");
		printf("\n\nIf PROGRAM is omitted then the shell of the user is run.\n");
		exit(0);
	}
}
