/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** exprNode.h
*/

# ifndef EXPRNODE_H
# define EXPRNODE_H

/*
** expression Nodes:
**
**     ctype typ --- type of expression
**
**     union _val
**      { long    ival;
**        char    cval;
**        double  fval;
**        cstring sval;
**      }   *val --- value, if known.  if unknown, val = (_val *)0
**
**     storeRef sref --- storage referred to by expression, of storeRef_null
**
**     cstring  etext --- to get nice error messages, punt for now!
*/

/* in exprNode_type: typedef struct _exprNode *exprNode; */

typedef enum 
{ 
  XPR_PARENS,     XPR_ASSIGN,  XPR_CALL,     XPR_EMPTY,    XPR_VAR, 
  XPR_OP,         XPR_POSTOP,  XPR_PREOP,    XPR_SIZEOFT,  XPR_SIZEOF, 
  XPR_OFFSETOF,   XPR_CAST,    XPR_FETCH,    XPR_VAARG,    XPR_ITER,  
  XPR_FOR,        XPR_FORPRED, XPR_GOTO,     XPR_CONTINUE, XPR_BREAK, 
  XPR_RETURN,     XPR_NULLRETURN, XPR_COMMA, XPR_COND,     XPR_IF,  XPR_IFELSE,
  XPR_DOWHILE,    XPR_WHILE,   XPR_STMT,     XPR_STMTLIST, XPR_SWITCH,
  XPR_INIT,       XPR_FACCESS, XPR_ARROW,    XPR_CONST,    XPR_STRINGLITERAL, 
  XPR_NUMLIT,     XPR_BODY,    XPR_NODE,     XPR_ITERCALL, XPR_TOK,
  XPR_WHILEPRED,  XPR_CASE,	 XPR_FTCASE,   XPR_DEFAULT,  XPR_FTDEFAULT,
  XPR_BLOCK,      XPR_INITBLOCK
} exprKind; 

typedef struct _exprOffsetof
{
  /*@only@*/ qtype q;
  /*@only@*/ cstring field;
} *exprOffsetof;

typedef struct _exprPair
{
  /*@only@*/ exprNode a;
  /*@only@*/ exprNode b;
} *exprPair;

typedef struct _exprTriple
{
  /*@only@*/ exprNode pred;
  /*@only@*/ exprNode tbranch;
  /*@only@*/ exprNode fbranch;
} *exprTriple;

typedef struct _exprIter
{
  /*@dependent@*/ /*@observer@*/ uentry sname;
  /*@only@*/    exprNodeList args;
  /*@only@*/    exprNode body;
  /*@dependent@*/ /*@observer@*/ uentry ename;
} *exprIter;

typedef struct _exprCall
{
  /*@only@*/ exprNode fcn;
  /*@only@*/ exprNodeList args;
} *exprCall;

typedef struct _exprIterCall
{
  /*@dependent@*/ /*@exposed@*/ uentry iter;
  /*@only@*/   exprNodeList args;
} *exprIterCall;

typedef struct _exprOp
{
  /*@only@*/ exprNode a;
  /*@only@*/ exprNode b;
  lltok      op;
} *exprOp;

typedef struct _exprField
{
  /*@only@*/ exprNode rec;
  /*@only@*/ cstring  field;
} *exprField;

typedef struct _exprUop
{
  /*@only@*/ exprNode a;
  lltok      op;
} *exprUop;

typedef struct _exprCast
{
  /*@only@*/ exprNode exp;
	     lltok    tok;
	     qtype    q;
} *exprCast;

typedef struct _exprInit
{
  /*@only@*/ exprNode exp;
	     idDecl   id;
} *exprInit;

typedef /*@null@*/ union _exprData
{
  cstring literal;
  cstring id;
  lltok *tok;
  qtype qt;    /* sizeof(type) */
  /* use for any 2-operator (comma, arrayFetch, case, stmt) */
  exprPair pair; 
  exprOp op;    /* pair + operator */
  exprUop uop;
  exprInit init;
  exprIter iter;
  exprCall call;
  exprIterCall itercall;
  exprCast cast;
  exprNode single;
  exprField field;
  exprTriple triple; /* ifelse, ternary op, for pred */
  exprOffsetof offset;
} *exprData;

/*@constant null exprData exprData_undefined; @*/
# define exprData_undefined  ((exprData) NULL)

struct _exprNode
{
  bool isJumpPoint BOOLBITS; /* expr can be reached non-sequentially */
  bool canBreak BOOLBITS;    /* expr can break (has break, continue) */
  bool mustBreak BOOLBITS;

  ctype typ;
  exitkind exitCode;    

  multiVal val;
  /*@exposed@*/ sRef sref;
  sRefSet uses;   /* sRef's used by this expression */
  sRefSet sets;   /* sRef's set by this expression */
  sRefSet msets;  /* sRef's possibly set (implicit out params, etc.) */

  guardSet guards;
  exprKind kind;

  fileloc loc;
  /*@relnull@*/ exprData edata;
  cstring etext;
} ;

/*@constant null exprNode exprNode_undefined; @*/
# define exprNode_undefined ((exprNode)NULL)

extern /*@falsenull@*/ bool exprNode_isDefined (exprNode e) /*@*/ ;
extern /*@unused@*/ /*@truenull@*/ bool exprNode_isUndefined (exprNode e) /*@*/ ;
extern /*@truenull@*/ bool exprNode_isError (exprNode e) /*@*/ ;

# define exprNode_isDefined(e)        ((e) != exprNode_undefined)
# define exprNode_isUndefined(e)      ((e) == exprNode_undefined)
# define exprNode_isError(e)          ((e) == exprNode_undefined)

extern /*@dependent@*/ /*@exposed@*/ guardSet 
  exprNode_getGuards (/*@sef@*/ exprNode e) /*@*/ ;
# define exprNode_getGuards(e) \
  (exprNode_isDefined(e) ? (e)->guards : guardSet_undefined)

extern ctype exprNode_getType (/*@sef@*/ exprNode e) /*@*/ ;
# define exprNode_getType(e) \
  (exprNode_isDefined(e) ? (e)->typ : ctype_unknown)

extern /*@unused@*/ bool exprNode_isInParens (/*@sef@*/ exprNode e) /*@*/ ;
# define exprNode_isInParens(e) \
  (exprNode_isDefined(e) && (e)->kind == XPR_PARENS)

extern bool exprNode_isStringLiteral (/*@sef@*/ exprNode e);
# define exprNode_isStringLiteral(e) \
  (exprNode_isDefined(e) && (e)->kind == XPR_STRINGLITERAL)

extern /*@unused@*/ bool exprNode_knownIntValue (/*@sef@*/ exprNode e) /*@*/ ;
# define exprNode_knownIntValue(e) \
  (exprNode_isDefined(e) && multiVal_isInt ((e)->val))

extern /*@unused@*/ bool exprNode_knownStringValue (/*@sef@*/ exprNode e) /*@*/ ;
# define exprNode_knownStringValue(e) \
  (exprNode_isDefined(e) && multiVal_isString ((e)->val))

extern bool exprNode_hasValue (/*@sef@*/ exprNode e) /*@*/ ;
# define exprNode_hasValue(e) \
  (exprNode_isDefined(e) && multiVal_isDefined ((e)->val))

extern /*@exposed@*/ multiVal exprNode_getValue (/*@sef@*/ exprNode e) /*@*/ ;
# define exprNode_getValue(e) \
  (exprNode_isDefined(e) ? ((e)->val) : multiVal_undefined)

extern /*@observer@*/ cstring exprNode_unparseFirst (exprNode e) /*@*/ ;
extern /*@observer@*/ guardSet exprNode_getForGuards (exprNode pred) /*@*/ ;
extern bool exprNode_isNullValue (exprNode e) /*@*/ ;
extern /*@exposed@*/ sRef exprNode_getSref (exprNode e) /*@*/ ;
extern /*@observer@*/ uentry exprNode_getUentry (exprNode e) 
   /*@globals internalState@*/ ;
extern void exprNode_produceGuards (exprNode pred) /*@modifies pred@*/ ;
extern /*@observer@*/ fileloc exprNode_loc (exprNode e) /*@*/ ;
extern exprNode
  exprNode_charLiteral (char c, cstring text, /*@only@*/ fileloc loc) /*@*/ ;
extern /*@observer@*/ exprNode exprNode_makeMustExit (void) /*@*/ ;
extern exprNode 
  exprNode_cond (/*@keep@*/ exprNode pred, /*@keep@*/ exprNode ifclause, 
		 /*@keep@*/ exprNode elseclause) /*@*/ ;
extern exprNode exprNode_makeError(void) /*@*/ ;

extern exprNode exprNode_makeInitBlock (lltok lbrace, /*@only@*/ exprNodeList a) /*@*/ ;

extern exprNode exprNode_functionCall (/*@only@*/ exprNode e, 
				       /*@only@*/ exprNodeList a) /*@*/ ;
extern /*@notnull@*/ exprNode 
  exprNode_fromIdentifier (/*@observer@*/ uentry c) /*@globals internalState@*/ ;
extern exprNode exprNode_fromUIO (cstring c) /*@globals internalState@*/ ;
extern exprNode exprNode_fieldAccess (/*@only@*/ exprNode s, /*@only@*/ cstring f) /*@*/ ;
extern exprNode exprNode_arrowAccess (/*@only@*/ exprNode s, /*@only@*/ cstring f) /*@*/ ;
extern exprNode exprNode_postOp (/*@only@*/ exprNode e, /*@only@*/ lltok op) 
          /*@modifies e@*/ ;
extern exprNode exprNode_preOp (/*@only@*/ exprNode e, /*@only@*/ lltok op) /*@*/ ;
extern exprNode exprNode_addParens (/*@only@*/ lltok lpar, /*@only@*/ exprNode e) /*@*/ ;
extern exprNode exprNode_offsetof (/*@only@*/ qtype q, /*@only@*/ cstring id) /*@*/ ;
extern exprNode exprNode_sizeofType (/*@only@*/ qtype c) /*@*/ ;
extern exprNode exprNode_sizeofExpr (/*@only@*/ exprNode e) /*@*/ ;
extern exprNode 
  exprNode_op (/*@only@*/ exprNode e1, /*@keep@*/ exprNode e2, /*@only@*/ lltok op) /*@*/ ;
extern exprNode 
  exprNode_assign (/*@only@*/ exprNode e1, /*@only@*/ exprNode e2, /*@only@*/ lltok op) ;
extern exprNode
  exprNode_arrayFetch (/*@only@*/ exprNode arr, 
		       /*@only@*/ exprNode ind) 
  /*@modifies arr, ind@*/ ;

extern void exprNode_free (/*@only@*/ exprNode e) ;
extern exprNode
  exprNode_vaArg (/*@only@*/ lltok tok, /*@only@*/ exprNode arg, /*@only@*/ qtype qt) 
  /*@globals internalState@*/ ;

extern exprNode 
  exprNode_stringLiteral (/*@only@*/ cstring t, /*@only@*/ fileloc loc) /*@*/ ;
extern exprNode exprNode_comma (/*@only@*/ exprNode e1, /*@only@*/ exprNode e2)  /*@*/ ;
extern exprNode 
  exprNode_labelStatement (cstring label, /*@returned@*/ exprNode stmt);
extern exprNode 
  exprNode_notReached (/*@returned@*/ exprNode stmt);

extern exprNode 
  exprNode_caseStatement (/*@only@*/ exprNode test, /*@only@*/ exprNode stmt,
			  bool fallThrough);
extern exprNode exprNode_concat (/*@only@*/ exprNode e1, /*@only@*/ exprNode e2);
extern exprNode exprNode_createTok (/*@only@*/ lltok t) /*@*/ ;
extern exprNode exprNode_statement (/*@only@*/ exprNode e);
extern exprNode exprNode_makeBlock (/*@only@*/ exprNode e);
extern exprNode exprNode_if (/*@only@*/ exprNode pred, /*@only@*/ exprNode tclause);
extern exprNode 
  exprNode_ifelse (/*@only@*/ exprNode pred, /*@only@*/ exprNode tclause, 
		   /*@only@*/ exprNode eclause);
extern exprNode exprNode_switch (/*@only@*/ exprNode e, /*@only@*/ exprNode s);
extern exprNode exprNode_while (/*@keep@*/ exprNode t, /*@keep@*/ exprNode b);
extern exprNode exprNode_doWhile (/*@only@*/ exprNode t, /*@only@*/ exprNode b);
extern /*@notnull@*/ /*@only@*/ exprNode exprNode_goto (/*@only@*/ cstring label);
extern exprNode exprNode_continue (/*@only@*/ lltok t, int qcontinue);
extern exprNode exprNode_break (/*@only@*/ lltok t, int qbreak);
extern exprNode exprNode_nullReturn (/*@only@*/ lltok t);
extern exprNode exprNode_return (/*@only@*/ exprNode e);
extern /*@dependent@*/ /*@observer@*/ cstring 
  exprNode_unparse (exprNode e) /*@*/ ; 

extern bool exprNode_isCharLit (exprNode e) /*@*/ ;
extern bool exprNode_isNumLit (exprNode e) /*@*/ ;

extern exprNode 
  exprNode_makeInitialization (/*@only@*/ idDecl t, /*@only@*/ exprNode e);
extern bool exprNode_matchType (ctype ct, exprNode e);

extern /*@notnull@*/ /*@only@*/ exprNode 
  exprNode_defaultStatement (/*@only@*/ lltok def, /*@only@*/ exprNode stmt,
			     bool fallThrough);
extern exprNode 
  exprNode_iter (/*@observer@*/ uentry name, /*@only@*/ exprNodeList alist, 
		 /*@only@*/ exprNode body, /*@observer@*/ uentry end);
extern exprNode exprNode_iterId (/*@observer@*/ uentry c);
extern            exprNode exprNode_iterExpr (/*@returned@*/ exprNode e);
extern exprNode exprNode_iterNewId (/*@only@*/ cstring s);
extern exprNode 
  exprNode_iterStart (/*@observer@*/ uentry name, /*@only@*/ exprNodeList alist);
extern exprNode 
  exprNode_numLiteral (ctype c, /*@temp@*/ cstring t, 
		       /*@only@*/ fileloc loc, long val); 
extern void exprNode_initMod (void) /*@modifies internalState@*/ ;
extern exprNode exprNode_for (/*@keep@*/ exprNode inc, /*@keep@*/ exprNode body);
extern exprNode 
  exprNode_forPred (/*@only@*/ exprNode init, 
		    /*@only@*/ exprNode test, /*@only@*/ exprNode inc);
extern exprNode 
  exprNode_floatLiteral (double d, ctype ct, 
			 cstring text, /*@only@*/ fileloc loc);
extern /*@notnull@*/ exprNode exprNode_createId (/*@observer@*/ uentry c);
extern exprNode exprNode_cast (/*@only@*/ lltok tok, /*@only@*/ exprNode e, /*@only@*/ qtype q);
extern bool exprNode_matchLiteral (ctype expected, exprNode e);
extern void exprNode_checkUseParam (exprNode current);
extern void exprNode_checkSet (exprNode e, sRef s);
extern void exprNode_checkMSet (exprNode e, sRef s);
extern exprNode exprNode_checkExpr (/*@returned@*/ exprNode e);
extern bool exprNode_mustEscape (exprNode e);
extern bool exprNode_mayEscape (exprNode e);
extern exprNode exprNode_whilePred (/*@only@*/ exprNode test);
extern exprNode 
  exprNode_updateLocation (/*@returned@*/ exprNode e, /*@temp@*/ fileloc loc);
extern void exprNode_freeShallow (/*@only@*/ exprNode e);
extern void exprNode_destroyMod (void) /*@modifies internalState@*/ ;
extern bool exprNode_isAssign (exprNode e) /*@*/ ;

# else
# error "Multiple include"
# endif

















