/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** symtable.h
*/

# ifndef SYMTABLE_H
# define SYMTABLE_H

/* There are 3 separate namespaces being maintained in this symbol
table: (N1) for C objects, functions, typedef names and enumeration
constants.  (N2) for C tags.  (N3) for LSL operators.  I re-use the
old linked list to represent N1 since it is here and it is convenient
for representing scopes.  Two more new hashtables are added to store
N2 and N3, each keyed on the lsymbol of the identifier. */

/*@constant observer char *BEGINSYMTABLE;@*/
# define BEGINSYMTABLE "%LCLSymbolTable"

/*@constant observer char *SYMTABLEEND;@*/
# define SYMTABLEEND "%LCLSymbolTableEnd"

typedef long unsigned symbolKey;

typedef struct _fctInfo {
  ltoken id;
  bool	export;            /* all HOF maps to one sort kind: SRT_HOF */
  /*@only@*/ signNode signature;      /* in terms of sorts, no HOF */
  /*@only@*/ pairNodeList globals;    /* list of (sort globalnameSym) */
} *fctInfo;

typedef struct _typeInfo {
    ltoken	id;
    sort	basedOn; 
    bool	abstract;   /* TRUE means LCL abstract type */
    bool	modifiable; /* TRUE indicates the abstract type is mutable */
    bool	export;     /* if TRUE, export it from this module */
} *typeInfo;

typedef enum 
{
  VRK_CONST,  /* an LCL constant */
  VRK_ENUM,   /* an enumeration constant, 
		 a special kind of LCL constant */
  VRK_VAR,    /* an LCL variable */
  VRK_PRIVATE,/* a variable private to current function/module */
              /* function private overrides module private, usual scoping
		 rules */
  VRK_GLOBAL, /* in globals of a function */ 
  VRK_LET,    /* local let variable in a function decl */
  VRK_PARAM,  /* formal parameters of a function/claim */
  VRK_QUANT   /* bound by a quantifier */
  } varKind;

typedef struct _varInfo {
    ltoken	id;
    sort	sort;
    varKind	kind;
    bool	export; /* if TRUE, export it from this module */
} *varInfo;

/* The same lsymbol can originate from different files:
   an LCL file or an LSL file.  Each occurrence in an LCL file appears
   as one of {opId, anyOp, logicalOp, simpleOp, eqOp, MULOP} ltoken. 
   Each occurrence in an LSL file appears as an lslOp AST. */ 

typedef struct _opInfo {
  /*@only@*/ nameNode name;
  /*@only@*/ sigNodeSet signatures;
} *opInfo;

typedef struct _tagInfo {
    ltoken	id;
    tagKind     kind; 
    bool        imported;
    sort sort;
    /*@reldef@*/ union
      {
	/*@null@*/ /*@dependent@*/ ltokenList enums;
        /*@null@*/ stDeclNodeList decls;
      } content;
} *tagInfo;

typedef enum {
  IK_SORT, IK_OP, IK_TAG
  } infoKind;

typedef struct {
  infoKind kind;
  union {
    tagInfo tag;
    sort sort;
    opInfo op;} content;
} htData;

typedef enum { 
  SPE_GLOBAL, SPE_FCN, SPE_QUANT, SPE_CLAIM,
  SPE_ABSTRACT, SPE_INVALID
  } scopeKind;

typedef struct _scopeInfo {
    scopeKind	kind;
} *scopeInfo;

/*
** A simple hash table implementation: Given a key K and a 
** pointer P to a piece of data, associate K with P.
** 
** Assumes that K is unsigned int, and data supports
**   3 operations: 
**   data_key: htData * -> unsigned int
*/

typedef struct _htEntry {
  /*@only@*/ htData *data;
  /*@only@*/ struct _htEntry *next;
} htEntry;

/* Each bucket is a linked list of entries */

typedef htEntry bucket;
typedef /*@relnull@*/ /*@only@*/ bucket *o_bucket;

typedef struct {
  unsigned int count;
  unsigned int size;
  /*@relnull@*/ /*@only@*/ o_bucket *buckets; 
} symHashTable;

/* Local to implementation */

typedef struct _symtableStruct *symtable;

extern /*@falsenull@*/ bool typeInfo_exists(/*@null@*/ typeInfo ti);
# define typeInfo_exists(ti)  ((ti) != NULL)

extern /*@falsenull@*/ bool varInfo_exists(/*@null@*/ varInfo vi);
# define varInfo_exists(vi)   ((vi) != NULL)

extern /*@falsenull@*/ bool tagInfo_exists(/*@null@*/ tagInfo oi);
# define tagInfo_exists(ti)   ((ti) != NULL)

extern /*@falsenull@*/ bool opInfo_exists(/*@null@*/ opInfo oi);
# define opInfo_exists(oi)    ((oi) != NULL)

extern /*@only@*/ symtable symtable_new (void) /*@*/ ;
extern void symtable_enterScope (symtable st, /*@keep@*/ scopeInfo si);
extern void symtable_exitScope(symtable st);

extern bool symtable_enterFct (symtable st, /*@only@*/ fctInfo fi);
extern void symtable_enterType (symtable st, /*@only@*/ typeInfo ti);

/* not only --- it is copied! */
extern bool symtable_enterVar (symtable st, /*@temp@*/ varInfo vi);
extern void symtable_enterOp (symtable st,
			      /*@only@*/ /*@notnull@*/ nameNode n, 
			      /*@owned@*/ sigNode oi);
extern bool symtable_enterTag (symtable st, /*@only@*/ tagInfo ti);
extern bool symtable_enterTagForce(symtable st, /*@only@*/ tagInfo ti);

extern bool symtable_exists(symtable st, lsymbol i);

extern /*@observer@*/ /*@null@*/ typeInfo symtable_typeInfo(symtable st, lsymbol i);
extern /*@observer@*/ /*@null@*/ varInfo symtable_varInfo(symtable st, lsymbol i);
extern /*@observer@*/ /*@null@*/ varInfo symtable_varInfoInScope(symtable st, lsymbol i);
extern /*@observer@*/ /*@null@*/ opInfo symtable_opInfo(symtable st, 
							/*@notnull@*/ nameNode n);
extern /*@observer@*/ /*@null@*/ tagInfo symtable_tagInfo(symtable st, lsymbol  i);

extern void symtable_export(symtable st, bool yesNo);
extern void symtable_dump(symtable st, FILE *f, bool lco);

extern void symtable_import(tsource *imported, ltoken tok, mapping *map);

extern /*@unused@*/ void symtable_printStats (symtable s);

extern lsymbol lsymbol_sortFromType (symtable, lsymbol);
extern cstring tagKind_unparse (tagKind k);
extern lsymbol lsymbol_translateSort (mapping *m, lsymbol s);
extern void varInfo_free (/*@only@*/ varInfo v);

extern /*@only@*/ lslOpSet 
  symtable_opsWithLegalDomain (symtable sym, /*@temp@*/ /*@null@*/ nameNode n, 
			       sortSetList argSorts, sort qual);
extern /*@observer@*/ sigNodeSet 
  symtable_possibleOps (symtable symtab, nameNode n);
extern bool symtable_opExistsWithArity(symtable sym, nameNode n, int arity);

extern void symtable_free (/*@only@*/ symtable stable);

# else
# error "Multiple include"
# endif

