/* $Id: latex-fmt.cc,v 1.9 1997/03/31 22:55:41 dps Exp $ */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/stat.h>
#include "interface.h"
#include "lib.h"
#include "latex-table.h"
#include "fmt-latex.h"

static const cmap tex_map[]=
{
    { '\n', "\\\\\n" },		     // Newline
    { 0x1E, "-" },		     // Unbreakable join
    { 0x1F, "\\-" },		     // Soft hypen
    { '#', "{\\#}" },		     // #
    { '$', "{\\$}" },		     // $
    { '%', "{\\%}" },		     // %  (5th element)
    { '&', "{\\&}" },		     // & 
    { '@', "{\\@}" },		     // @
    { '\\', "$\\backslash$" },	     // \
    { '^', "{\\^}" },		     // ^
    { '_', "{\\_}" },		     // _  (10th element)
    { '{', "{\\{}" },		     // {
    { '}', "{\\}}" },		     // }
    { '~', "{\\~}" },		     // ~
    { 0x85, "\\unskip\\ldots" },     // Dots
    { 0x91, "{`}" },		     // 91 = left quote (15th element)
    { 0x92, "{'}" },		     // 92 = right quote
    { 0x93, "``" },		     // 93 = opening double quote
    { 0x94, "''" },		     // 94 = closing double quote
    { 0x96, "--" },		     // em-dash
    { 0x97, "---" },		     // en-dash (20th element)
    { 0x99, "${}^{\\rm TM}$" },	     // Trademark
    { 0xA0, "~" },		     // Unbreakable space
    { 0xA3, "$\\leq$" },	     // <= came out as A3, also pounds
    { 0xA9, "{\\copyright}" },	     // Copyright
    { 0xAB, "$<\\!\\!<$" },	     // Openning << quotes (25th element)
    { 0xAE, "{\\reg}" },	     // reserved sign
    { 0xB3, "$\\geq$" },	     // Greater than or = came out as B3
    { 0xBB, "$>\\!\\!>$" },	     // Closing >> quotes (28th element)
};

tblock *__latex_do_map(const char *s)
{
    tblock *out;

    out=map_string(s, tex_map);
    return out;
}

/* Preamble */
static void preamble(const tok_seq::tok *t, const docfmt *fmt, FILE *out,
		    void *d)
{
    time_t now;
    struct stat st;
    char *tnow, *tdoc;

    t=t;
    d=d;

    now=time(NULL);
    tnow=(fmt->date)(now);

    if (fstat(fileno(out), &st)==-1)
	st.st_mtime=now;
    tdoc=fmt->date(st.st_mtime);

    fprintf(out,
	    "%% Generated by word2x on %s\n"
	    "%%\n"
	    "\\date{%s}\n"
	    "\\documentclass{article}\n"
	    "\\usepackage{amstext}\n"
	    "\\def\\reg{\\setbox0\\hbox{$\\mathchar\"20D$}%%\n"
	    "\\hbox to 0pt{\\hbox to \\wd0{\\hfill\\,\\rm R\\hfill}\\hss}%%\n"
	    "$\\mathchar\"20D$}\n"
	    "\\begin{document}\n",
	    tnow, tdoc);
    free(tnow);
    free(tdoc);
}

/* Postamble */
static void postamble(const tok_seq::tok *t, const docfmt *fmt, FILE *out,
		    void *d)
{ 
    struct latex_data *dp;
    fmt=fmt;
    t=t;
    dp=(struct latex_data *) d;

    if (dp->list_flg)
	fputs("\\end{itemize}\n", out);
    fputs("\\end{document}\n", out);
}

/* Allocate local data */
static void *allocate_latex(void)
{
    struct latex_data *tdata;

    tdata=new(struct latex_data);
    tdata->tabl=NULL;
    tdata->last_tc=NULL;
    tdata->unit_type=1;
    tdata->list_flg=0;
    return tdata;
}

/* Free local data */
static void free_latex(void *d)
{
    struct latex_data *tdata;

    tdata=(struct latex_data *) d;
    if (tdata->tabl!=NULL)
	delete(tdata->tabl);
}


/* Find part number and return it or -1 if no number */
int get_part_num(const char *st, const char *fence)
{
    int n;

    while (st<fence)
    {
	if (isspace(*st))
	{
	    st++;
	    continue;
	}
	if (isdigit(*st))
	{
	    n=0;
	    while (st<fence && isdigit(*st))
	    {
		n=n*10+(*st)-'0';
		st++;
	    }
	    if (!isspace(*st))
		return -1;
	    else
		return n;

	}
	if (isupper(*st) && isspace(*(st+1)))
	    return (*st)-'A'+1;

	/* Nothing else understood at this time */
	return -1;
    }
    return -1;
}


/*
 * Paragraphs are easy, but get complicated due to need to work out
 * which things are actually chapter headings, etc
 */
static void fold_para(const tok_seq::tok *tok, const docfmt *fmt, FILE *out,
		      void *d)
{
    tblock *b, *ts, op;
    const char *s, *t, *pt;
    struct latex_data *dp;
    int do_add, has_num, i;

    static const char *const sects[]=
    {
	"chapter", "section", "subsection", "subsubsection"
    };

    dp=(struct latex_data *) d;
    do_add=0;
    dp->par_flg=1;

    /* Even this much is under 100%!! */
    pt=(tok->data.d);

    if (*pt=='\0' && dp->list_flg==1)
	return;

    if (dp->last_tc!=NULL)
    {
	if (strcmp(dp->last_tc, pt)==0)
	{
	    op.add("\\addcontentsline{toc}{");
	    op.add(sects[dp->unit_type]); op.add("}{");
	    op.add(dp->last_tc); op.add("}\n\\");
	    op.add(sects[dp->unit_type]);
	    op.add("*{");
	    do_add=1;
	}

	else
	{
	    s=dp->last_tc+strlen(dp->last_tc)-strlen(pt);
	    if (strcmp(s, pt)==0)
	    {
		/* Find type */
		for (i=0; (unsigned) i<N(sects)-1; i++)
		{
		    if (strncasecmp(dp->last_tc, sects[i],
				    strlen(sects[i]))==0)
			break;
		}
		t=dp->last_tc+strlen(sects[i]);
		has_num=get_part_num(t,s);
		if (has_num==-1)
		{
		    op.add("\\addcontentsline{toc}{");
		    op.add(sects[i]); op.add("}{");
		    op.add(dp->last_tc);
		    op.add("}\n");
		}
		op.add('\\');
		op.add(sects[i]);
		op.add((has_num!=-1) ? "{" : "*{");
		if ((unsigned) i<N(sects)-1)
		    dp->unit_type=i+1;
		else
		    dp->unit_type=i;
		do_add=1;
	    }
	}
	free((void *) dp->last_tc);
	dp->last_tc=NULL;
    }
    else
    {
	if ((tok->data.d)[0]==(char) BULLET_CODE)
	{
	    if (dp->list_flg==0)
		fputs("\\begin{itemize}\n", out);
	    op.add("\\item ");
	    dp->list_flg=1;
	    pt++;
	}
	else
	{
	    if (dp->list_flg)
	    {
		fputs("\\end{itemize}\n\n", out);
		dp->list_flg=0;
	    }
	    if (strlen(pt)>0 && strlen(pt)<PAR_TRESHOLD_LEN)
	    {
		op.add("\\noindent{\\bf ");
		do_add=1;
	    }
	}
    }
   
    ts=map_string(pt, tex_map);
    op.add(*ts);
    if (do_add)
	op.add('}');
    delete(ts);

    b=word_wrap(op, "\n", "\n", fmt->maxline, 0);
    fputs((*b), out);
    delete(b);
}

/* End of paragraph */
static void end_para(const tok_seq::tok *tok, const docfmt *fmt, FILE *out,
		     void *d)
{
    struct latex_data *dp;

    tok=tok;
    fmt=fmt;
    dp=(struct latex_data *) d;
    dp->par_flg=0;
        
    fputs("\n\n", out);
}

/* Start a table === allocate table and initialise */
static void alloc_tbl(const tok_seq::tok *t, const docfmt *fmt, FILE *out,
		      void *d)
{
    struct latex_data *tdata;

    out=out;
    fmt=fmt;
    tdata=(struct latex_data *) d;
    tdata->col=0;
    tdata->row=0;
    tdata->tabl=new(latex_table)(t->data.table.cols, t->data.table.rows);
    tdata->par_flg=0;
}


/* End of a table==print the table */
static void format_tbl(const tok_seq::tok *t, const docfmt *fmt, FILE *out,
		       void *d)
{
    struct latex_data *tdata;

    t=t;
    tdata=(struct latex_data *) d;
    tdata->tabl->print_table(fmt->maxline, out); // Print table
    delete(tdata->tabl);
    tdata->tabl=NULL;
}

/* start row==set column to 0 */
static void start_row(const tok_seq::tok *t, const docfmt *fmt, FILE *out,
		      void *d)
{ 
    struct latex_data *tdata;

    out=out;
    fmt=fmt;
    t=t;
    tdata=(struct latex_data *) d;
    tdata->col=0;
}

/* end row==add one to row */
static void inc_row(const tok_seq::tok *t, const docfmt *fmt, FILE *out,
		    void *d)
{ 
    struct latex_data *tdata;

    fmt=fmt;
    t=t;
    out=out;
    tdata=(struct latex_data *) d;
    tdata->row++;
}


/* Start field === set field */
static void set_field(const tok_seq::tok *t, const docfmt *fmt, FILE *out,
		      void *d)
{
    struct latex_data *tdata;

    tdata=(struct latex_data *) d;
    out=out;
    fmt=fmt;
    tdata->tabl->set(tdata->col, tdata->row, t->data.d);
}

/* end field==add one to col */
static void inc_col(const tok_seq::tok *t, const docfmt *fmt, FILE *out,
		    void *d)
{ 
    struct latex_data *tdata;

    out=out;
    fmt=fmt;
    t=t;
    tdata=(struct latex_data *) d;
    tdata->col++;
}

/* pointers to the functions that do the work */
docfmt latexfmt=
{
    76,				    // Width
    "\n",			    // Use \n as line ends
    allocate_latex,		    // Allocate space
    free_latex,			    // Free text
    {
	{ preamble, postamble },    // End and start of document---do nothing
	{ fold_para, end_para },    // Paragraph
	{ alloc_tbl, format_tbl },  // Start/end table
	{ set_field, inc_col },	    // Start/end field
	{ start_row, inc_row },	    // Start/end row
	{ ltx_embed, null_proc },   // Throw away embed messages
	{ null_proc, null_proc }    // Do not understanding anything else
    }
};




