/*
** Copyright (C) 1998,1999,2000,2001 Martin Roesch <roesch@clark.net>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: spp_http_decode.c,v 1.26 2001/08/13 16:21:24 roesch Exp $ */
/* spp_http_decode 
 * 
 * Purpose:
 *
 * This preprocessor normalizes HTTP requests from remote machines by
 * converting any %XX character substitutions to their ASCII equivalent.
 * This is very useful for doing things like defeating hostile attackers
 * trying to stealth themselves from IDSs by mixing these substitutions 
 * in with the request.
 *
 * Arguments:
 *   
 * This plugin takes a list of integers representing the TCP ports that the
 * user is interested in having normalized
 * a "-unicode" disables the IIS unicode check that is enabled by default
 * a "-cginull" disables the CGI NULL check that is enabled by default
 *
 * Effect:
 *
 * Changes the data in the packet payload to a plain ASCII representation 
 * and changes p->dsize to reflect the new (smaller) payload size.
 *
 * Comments:
 *
 * It could be interesting to generate an alert based on the number of
 * characters converted for a single packet, through some sort of threshold
 * setting.
 *
 *
 * Changes:
 * Wed Mar 28 19:35:48 ICT 2001
 * Added more accurate IIS attacks detection suggested by Koji Shikata.
 * (his credits go to: Arai , Micky , Tariki , Minz , All of security-talk-ML users)
 *
 */
#include "spp_http_decode.h"
#include "codes.h"
#include <ctype.h>

#define MODNAME "spp_http_decode"

#define NOUNICODE "-unicode"
#define NOCGINULL "-cginull"

int check_iis_unicode = 1;
int check_cgi_null = 1;
int http_decode_ignore = 0;

extern int do_detect;

/* struct in_addr naddr; */

typedef struct _serverNode  /* for keeping track of our network's servers */
{
    IpAddrSet *address;
   /*
    * u_long address; u_long netmask;
    */
    char ignoreFlags;
    struct _serverNode *nextNode;
}           ServerNode;

/* For portscan-ignorehosts */
IpAddrSet *HttpAllocAddrNode();
void CreateNodeList(u_char *);
IpAddrSet *HttpIgnoreAllocAddrNode(ServerNode *);
void HttpIgnoreParseIP(char *, ServerNode *);
int IsAServer(Packet *);

ServerNode *serverLst;

extern char *file_name;
extern int file_line;

/* Instantiate the list of ports we're going to watch */
PortList HttpDecodePorts;

/*
 * Function: SetupHttpDecode()
 *
 * Purpose: Registers the preprocessor keyword and initialization 
 *          function into the preprocessor list.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupHttpDecode()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterPreprocessor("http_decode", HttpDecodeInit);
    RegisterPreprocessor("http_decode_ignore", HttpDecodeInitIgnore);

    DebugMessage(DEBUG_PLUGIN, "Preprocessor: HttpDecode in setup...\n");
}


/*
 * Function: HttpDecodeInit(u_char *)
 *
 * Purpose: Processes the args sent to the preprocessor, sets up the
 *          port list, links the processing function into the preproc
 *          function list
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void HttpDecodeInit(u_char *args)
{
    DebugMessage(DEBUG_PLUGIN, "Preprocessor: HttpDecode Initialized\n");

    /* parse the argument list into a list of ports to normalize */
    SetPorts(args);
    init_codes();

    /* Set the preprocessor function into the function list */
    AddFuncToPreprocList(PreprocUrlDecode);
}

void HttpDecodeInitIgnore(u_char *args)
{
    DebugMessage(DEBUG_PLUGIN,"Preprocessor: HttpDecodeIgnore Initialized\n");
    http_decode_ignore = 1;
    DebugMessage(DEBUG_PLUGIN,"Hosts: %s\n", args);
    CreateNodeList(args);
}


/*
 * Function: SetPorts(char *)
 *
 * Purpose: Reads the list of port numbers from the argument string and 
 *          parses them into the port list data struct
 *
 * Arguments: portlist => argument list
 *
 * Returns: void function
 *
 */
void SetPorts(char *portlist)
{
    char **toks;
    int num_toks;
    int num_ports = 0;
    int num;

    if(portlist == NULL || *portlist == '\0')
    {
        FatalError("ERROR %s (%d)=> No arguments to http_decode "
                   "preprocessor!\n", file_name, file_line);
    }

    /* tokenize the argument list */
    toks = mSplit(portlist, " ", 31, &num_toks, '\\');

    /* convert the tokens and place them into the port list */
    for(num = 0; num < num_toks; num++)
    {
        if(!strncasecmp(NOUNICODE, toks[num], sizeof NOUNICODE))
        {
            check_iis_unicode = 0;
        }
        else if(!strncasecmp(NOCGINULL, toks[num], sizeof NOCGINULL))
        {
            check_cgi_null = 0;
        }
        else if(isdigit((int)toks[num][0]))
        {
            HttpDecodePorts.ports[num_ports++] = atoi(toks[num]);
        }
        else
        {
            FatalError("ERROR %s(%d) => Unknown argument to http_decode "
                       "preprocessor: \"%s\"\n", 
                       file_name, file_line, toks[num]);
        }
    }   

    HttpDecodePorts.num_entries = num_ports;

    DebugMessage(DEBUG_PLUGIN, "Decoding HTTP on %d ports: ", 
                 HttpDecodePorts.num_entries);

    for(num_ports = 0; num_ports < HttpDecodePorts.num_entries; num_ports++)
    {
        DebugMessage(DEBUG_PLUGIN, "%d ", HttpDecodePorts.ports[num_ports]);
    }

    DebugMessage(DEBUG_PLUGIN, "\n");
}


/*
 * Function: PreprocUrlDecode(Packet *)
 *
 * Purpose: Inspects the packet's payload for "Escaped" characters and 
 *          converts them back to their ASCII values.  This function
 *          is based on the NCSA code and was contributed by Michael Henry!
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 *
 */
void PreprocUrlDecode(Packet *p)
{
    char *url;       /* this is where the converted data will be written */
    char *index;     /* this is the index pointer to walk thru the data */
    char *end;       /* points to the end of the payload, for loop control */
    char *end5;      /* points to end - 5 */
    u_int16_t psize; /* payload size */
    int i;           /* loop counter */
    char logMessage[180];
    int temp;
    int temp2;
    int temp3;
    int unicode;
    int first_space_found = 0;
    int uri_set = 0;
    int uri_finished = 0;
    u_int16_t code_index = 0;
    Event event;

    DebugMessage(DEBUG_PLUGIN, "http decoder init on %d bytes\n", p->dsize);

    /* check to make sure we're talking TCP and that the TWH has already
       completed before processing anything */
    if(!PacketIsTCP(p))
    {
        DebugMessage(DEBUG_PLUGIN, "It isn't TCP session traffic\n");
        return;
    }

    if(!IsTcpSessionTraffic(p))
    {
        DebugMessage(DEBUG_PLUGIN, "It isn't TCP session traffic\n");
        return;
    }

    /* check the port against the decode port list */
    for(i = 0; i < HttpDecodePorts.num_entries; i++)
    {
        if(HttpDecodePorts.ports[i] == p->dp)
        {
            /* on match, normalize the data */
            DebugMessage(DEBUG_PLUGIN, "Got HTTP traffic (%d bytes @ %p)!\n", 
                         p->dsize, p->data);
            DebugMessage(DEBUG_PLUGIN, "%s\n", p->data);

            /* setup the pointers */
            url =   (char *) p->data;
            index = (char *) p->data;
            end =   (char *) p->data + p->dsize;
            end5 =  end - 5;
            psize = (u_int16_t) (p->dsize);
            first_space_found = 0;
            uri_set = 0;
            uri_finished = 0;

            /* If it has a source addres from the http_decode_ignore */
            if(http_decode_ignore)
            {
                if(IsAServer(p))
                {
                    return;
                }
            }

            /* walk thru each char in the payload */
            while(index < end)
            {
                /* if it's potentially "escaped" and not too close to the 
                 * end of the payload and we're processing the URI
                 */
                if((index < end - 2) && (*index == '%') && !uri_finished)
                {
                    unicode = -1;
                    temp = (((nibble((char)((int)*(index+1)))<<4) +
                                nibble((char)((int)*(index+2)))) & 0xff);

                    DebugMessage(DEBUG_PLUGIN, "First byte is %x.\n",temp);

                    /* 0xbf - 0xe0 range */
                    if ( (index < end5) && (temp > 0xbf) && (temp < 0xe0) &&
                            (*(index+3) == '%') )
                    {
                        temp2 = (((nibble((char)((int)*(index+4)))<<4) +
                                    nibble((char)((int)*(index+5)))) & 0xff);

                        DebugMessage(DEBUG_PLUGIN,"Second byte is: %x.\n", 
                                temp2);

                        unicode = (temp & 0x1f) <<6|(temp2 & 0x3f);
                        index +=6;
                        url += 2;
                        psize -= 4;
                    }
                    /* 0xdf - 0xf0 range */
                    else if ((index < end5 && (temp > 0xdf) &&
                                (temp < 0xf0) && (*(index + 3) == '%') ) &&
                            !uri_finished)
                    {
                        temp2 = (((nibble((char)((int)*(index+4)))<<4) +
                                    nibble((char)((int)*(index+5)))) & 0xff);

                        DebugMessage(DEBUG_PLUGIN,"Second byte is: %x.\n", 
                                temp2);

                        index +=6;
                        url +=  2;
                        psize -= 4;

                        if ((*(index) == '%') && (index < (end - 2)))
                        {
                            temp3 = (((nibble((char)((int)*(index+1)))<<4) +
                                        nibble((char)((int)*(index+2)))) & 0xff);
                            DebugMessage(DEBUG_PLUGIN,"Third byte is: %x.\n", 
                                    temp3);
                            unicode = (temp & 0x0f) << 4 |
                                (temp2 & 0x3f) << 6|
                                (temp3 & 0x3f);
                            index += 3;
                            url ++;
                            psize -= 2;
                        }
                    }
                    else if((index < end5) && (*(index+1) == 'u' || 
                                *(index+1) == 'U'))
                    {
                        temp = (((nibble((char)((int)*(index+2)))<<4) +
                                    nibble((char)((int)*(index+3)))) & 0xff);
                        temp2 = (((nibble((char)((int)*(index+4)))<<4) +
                                    nibble((char)((int)*(index+5)))) & 0xff);

                        if(temp == 0)
                        {
                            *url = (char)temp2;
                        }
                        else
                        {
                            code_index = (u_int16_t) ((temp & 0xff)<< 8 |
                                         (temp2 & 0xff));
                            *url = (char)codes[code_index];
                        }

                        index += 6;
                        url++;
                        psize -= 5;
                    }
                    /* anything else that is valid hex */
                    else if (isxdigit((int)*(index+1)) && 
                            isxdigit((int)*(index+2)) &&
                            !uri_finished)
                    {
                        *url = (char) 0x000000FF & temp;
                        index +=3;
                        url++;
                        psize -=2;
                    }
                    else
                    {
                        *url = *index;

                        if(first_space_found && *url != ' ' && !uri_set)
                        {
                            p->URI.uri = index;
                            uri_set = 1;
                            DebugMessage(DEBUG_PLUGIN, "set URI at %p\n", 
                                    p->URI.uri);
                        }

                        url++;
                        index++;
                    }        

                    DebugMessage(DEBUG_PLUGIN,"Unicode byte is %x.\n",unicode);

                    if (((unicode == 0x2f) || 
                                (unicode == 0x5c) ||
                                (unicode == 0x2e)) && 
                            check_iis_unicode && do_detect)
                    {
                        snprintf(logMessage, sizeof(logMessage) - 1,
                                MODNAME ": ISS Unicode attack detected");
                        SetEvent(&event, GENERATOR_SPP_HTTP_DECODE, 
                                HTTP_DECODE_UNICODE_ATTACK, 1, 0, 0, 0);
                        CallAlertFuncs(p, logMessage, NULL, &event);
                        CallLogFuncs(p, logMessage, NULL, &event);
                        DebugMessage(DEBUG_PLUGIN, 
                                "Unicode attack detected: %x\n", unicode);
                        do_detect = 0;
                    }

                    if (unicode == 0 && check_cgi_null && do_detect)
                    {
                        snprintf(logMessage, sizeof(logMessage),
                                MODNAME ": CGI Null byte attack detected");
                        SetEvent(&event, GENERATOR_SPP_HTTP_DECODE, 
                                HTTP_DECODE_CGINULL_ATTACK, 1, 0, 0, 0);
                        CallAlertFuncs(p, logMessage, NULL, &event);
                        CallLogFuncs(p, logMessage, NULL, &event);
                        DebugMessage(DEBUG_PLUGIN, "NULL byte attack "
                                "detected: %x\n", unicode);
                        do_detect = 0;
                    }

                }
                else
                {
                    *url = *index;

                    if(*url == ' ' && !first_space_found)
                    {
                        first_space_found = 1;
                        DebugMessage(DEBUG_PLUGIN, "found first space...\n");
                    }

                    if(first_space_found && *index != ' ' && !uri_set)
                    {
                        p->URI.uri = index;
                        uri_set = 1;
                        DebugMessage(DEBUG_PLUGIN, "set URI at %p\n", 
                                p->URI.uri);
                    }

                    /* ugly code?  I got your ugly code *RIGHT HERE*! */
                    if(!uri_finished && uri_set && 
                            (*url == ' ' || *url == '\n'))
                    {
                        char *foo = p->URI.uri;

                        p->URI.length = (index - foo);
                        uri_finished  = 1;
                        DebugMessage(DEBUG_PLUGIN, "URI length: %d\n", 
                                p->URI.length);
                        DebugMessage(DEBUG_PLUGIN, "URI data (%p):\n", 
                                p->URI.uri);
#ifdef DEBUG
                        ClearDumpBuf();
                        PrintNetData(stdout, p->URI.uri, p->URI.length);
                        ClearDumpBuf();
#endif
                        break;
                    }

                    url++;
                    index++;
                }
            }

            /* 
             * the length might not get set for things like gfx, so we should
             * set it on the off hand that a uri pointer was set
             */
            if(!uri_finished && uri_set)
            {
                char *foo = p->URI.uri;
                p->URI.length = (end - foo);
                uri_finished  = 1;
            }

            /* set the payload size to reflect the new size */ 
            p->dsize = psize;

#ifdef DEBUG
            DebugMessage(DEBUG_PLUGIN,"New size: %d\n", p->dsize);
            DebugMessage(DEBUG_PLUGIN,"converted data:\n");
            PrintNetData(stdout, p->data, psize);
            ClearDumpBuf();
#endif

            return;
        }
    }
}


/*
 * Function: nibble(char)
 *
 * Purpose: converts a hex or 32base character into a value 
 *          in the range 0..15 (or 0..31)
 *
 * Arguments: what => the character in question
 *
 * Returns: The converted character or -1 if the character is
 *          not 16base (hex) or 32base (?)
 *
 */
int nibble(char what)
{
    if(isdigit((int)what)) return what - '0';

    if(isalpha((int)what)) return toupper((int)what) - 'A' + 10;

    return -1;
}

void CreateNodeList(u_char * servers)
{
    char **toks;
    int num_toks;
    int num_servers = 0;
    ServerNode *currentServer;
    int i;

    currentServer = NULL;
    serverLst = NULL;

    if(servers == NULL)
    {
        FatalError(MODNAME ": ERROR: %s (%d)=> No arguments to "
                   "http_decode_ignore preprocessor!\n", file_name, file_line);
    }

    /* tokenize the argument list */
    toks = mSplit(servers, " ", 31, &num_toks, '\\');

    /* convert the tokens and place them into the server list */
    for(num_servers = 0; num_servers < num_toks; num_servers++)
    {
        if(currentServer != NULL)
        {
            currentServer->nextNode = (ServerNode *) calloc(sizeof(ServerNode),
                                                            sizeof(char));
            currentServer = currentServer->nextNode;
        }
        else
        {
            currentServer = (ServerNode *) calloc(sizeof(ServerNode), 
                                                  sizeof(char));
            serverLst = currentServer;
        }

        HttpIgnoreParseIP(toks[num_servers], currentServer);

        currentServer->nextNode = NULL;
    }

    for(i = 0; i < num_toks; i++)
    {
        free(toks[i]);
    }
}

void HttpIgnoreParseIP(char *addr, ServerNode* server)
{
    char **toks;
    int num_toks;
    int i;
    IpAddrSet *tmp_addr;
    int global_negation_flag;
    char *tmp;

    if(*addr == '!')
    {
        global_negation_flag = 1;
        addr++;
    }

    if(*addr == '$')
    {
        if((tmp = VarGet(addr + 1)) == NULL)
        {
            FatalError("ERROR %s (%d) => Undefined variable %s\n", file_name,
                    file_line, addr);
        }
    }
    else
    {
        tmp = addr;
    }

    if (*tmp == '[')
    {
        *(strrchr(tmp, (int)']')) = 0; /* null out the en-bracket */

        toks = mSplit(tmp+1, ",", 128, &num_toks, 0);

        for(i = 0; i < num_toks; i++)
        {
            tmp_addr = HttpIgnoreAllocAddrNode(server);

            ParseIP(toks[i], tmp_addr);
        }

        for(i=0;i<num_toks;i++)
            free(toks[i]);
    }
    else
    {
        tmp_addr = HttpIgnoreAllocAddrNode(server);

        ParseIP(tmp, tmp_addr);
    }
}

IpAddrSet *HttpIgnoreAllocAddrNode(ServerNode * server)
{
    IpAddrSet *idx;     /* IP struct indexing pointer */

    if(server->address == NULL)
    {
        server->address = (IpAddrSet *) calloc(sizeof(IpAddrSet), sizeof(char));

        if(server->address == NULL)
        {
            FatalError("[!] ERROR: Unable to allocate space for "
                       "http_decode IP addr\n");
        }
        return server->address;
    }

    idx = server->address;

    while(idx->next != NULL)
    {
        idx = idx->next;
    }

    idx->next = (IpAddrSet *) calloc(sizeof(IpAddrSet), sizeof(char));

    idx = idx->next;

    if(idx == NULL)
    {
        FatalError("[!] ERROR: Unable to allocate space for"
                   " http_decode IP address\n");
    }

    return idx;
}

/* Check if packet originated from a machine we have been told to ignore
   SYN and UDP "scans" from, presumably because it's a server.
*/
int IsAServer(Packet * p)
{
    ServerNode *currentServer = serverLst;

    while(currentServer)
    {
        /*
         * Return 1 if the source addr is in the serverlist, 0 if nothing is
         * found.
         */
        if(CheckAddrPort(currentServer->address, 0, 0, p, 
                    (ANY_SRC_PORT | currentServer->ignoreFlags), CHECK_SRC))
        {
            return(1);
        }

        currentServer = currentServer->nextNode;
    }

    return(0);
}



