/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* good-ole-line: serial interface stuff                                      */
/*============================================================================*/

/*
 * serialhandler
 *
 * input: anchor - pointer to vpnd global data
 *
 * return: 1 if line is open, else 0
 *
 * This procedure opens a serial line and sets it to
 * raw state at the requested speed (speed, 8N1) or
 * it closes the serial line, if it is open.
 */

int serialhandler(VPN *anchor)
{
	struct termios t;	/* tty settings			*/
	int i;			/* general purpose usage	*/
	int f;			/* lock file read access	*/
	char lockfile[32];	/* lock file pathname		*/
	char pid[16];		/* owner pid of lockfile	*/
	char *ptr;		/* lock file pathname creation	*/


	/* debug message */

	ENTER("serialhandler");

	/* prepare lock file pathname creation */

	if((ptr=strrchr(anchor->serialdev,'/'))!=NULL)ptr++;
	else ptr=anchor->serialdev;

	/* if line is open, close it */

	if(anchor->serial!=-1)goto delete;

	/* create lock file pathname, handle errors */

	if(strlen(ptr)>16||!*ptr)GOTO("device name length",err1);
	strcpy(lockfile,"/var/lock/LCK..");
	strcat(lockfile,ptr);

	/* try to create lock file, handle errors */

	while((anchor->lock=open(lockfile,O_EXCL|O_CREAT|O_RDWR,0644))==-1)
	{
		/* if the lock file already existed */

		if(errno==EEXIST)
		{
			/* try to access lock file, handle errors */

			if((f=open(lockfile,O_RDONLY))==-1)
				JUMP("lock file access",err1);

			/* read owner pid from lock file, close lock file */

			i=read(f,pid,11);
			close(f);

			/* handle read errors */

			if(i<=0)GOTO("lock file pid read",err1);

			/* get numeric pid */

			pid[i]=0;
			i=atoi(pid);

			/* if the owning process doesn't exist */

			if(kill(i,0)==-1)if(errno==ESRCH)
			{
				/* delete stale lock file, handle errors */

				if(unlink(lockfile))if(errno!=ENOENT)
					JUMP("stale lock delete",err1);

				/* retry locking */

				continue;
			}

			/* signal error as serial device is already locked */

			GOTO("device already locked",err1);
		}

		/* signal error, if lock can't be created */

		else JUMP("lock file create",err1);
	}

	/* write own pid to lock file */

	sprintf(pid,"%10d\n",getpid());
	write(anchor->lock,pid,11);

	/* open serial device, react to errors */

	if((anchor->serial=
	    open(anchor->serialdev,O_RDWR|O_NONBLOCK|O_NOCTTY))==-1)
		JUMP("open",err2);

	/* save tty device attributes, react to errors */

	if(tcgetattr(anchor->serial,&(anchor->line))<0)JUMP("tcgetattr",err3);

	/* change file protection for privacy, handle errors */

	if(fchmod(anchor->serial,0))JUMP("fchmod",err3);

	/* set dtr line, react to errors */

	if(ioctl(anchor->serial,TIOCMGET,&i)<0)JUMP("ioctl(TIOCMGET)",err4);
	i|=TIOCM_DTR;
	if(ioctl(anchor->serial,TIOCMSET,&i)<0)JUMP("ioctl(TIOCMSET)",err4);

	/* set tty device attributes, react to errors */

	t.c_cflag=CS8|CREAD|HUPCL|(anchor->rtscts?CRTSCTS:0)|
		(anchor->localline?CLOCAL:0);
	t.c_iflag=IGNBRK|IGNPAR;
	t.c_oflag=0;
	t.c_lflag=0;
	for(i=0;i<NCCS;i++)t.c_cc[i]=0;
	t.c_cc[VMIN]=1;
	t.c_cc[VTIME]=0;
	cfsetispeed(&t,anchor->speed);
	cfsetospeed(&t,anchor->speed);
	if(tcsetattr(anchor->serial,TCSAFLUSH,&t)<0)JUMP("tcsetattr",err5);

	/* signal success */

	goto err1;

	/* flush buffers, restore line settings */

delete:	if(tcsetattr(anchor->serial,TCSAFLUSH,&(anchor->line))<0)
		ERRNO("tcsetattr");

	/* drop dtr line */

err5:	if(ioctl(anchor->serial,TIOCMGET,&i)>=0)
	{
		i&=~TIOCM_DTR;
		if(ioctl(anchor->serial,TIOCMSET,&i)<0)ERRNO("ioctl(TIOCMSET)");
	}
	else ERRNO("ioctl(TIOCMGET)");

	/* restore previous file permissions */

err4:	fchmod(anchor->serial,anchor->normalmode);

	/* close device file */

err3:	if(close(anchor->serial))ERRNO("close");

	/* mark device file as closed */

	anchor->serial=-1;

	/* close lock file */

err2:	if(close(anchor->lock))ERRNO("close lockfile");

	/* mark lock file as closed */

	anchor->lock=-1;

	/* remove lock file */

	strcpy(lockfile,"/var/lock/LCK..");
	strcat(lockfile,ptr);
	if(unlink(lockfile))ERRNO("unlink lockfile");

	/* debug message */

err1:	LEAVE("serialhandler");

	/* signal success or error */

	return (anchor->serial!=-1?1:0);
}
