# Etext.pm - Project Gutenberg etext class definition for Gutenbook
# http://www.gutenbook.org/
# Copyright (C) 2000  Lee "Lefty" Burgess
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

package PGB::Etext;

use PGB;
use PGB::File;
use PGB::Network;

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;
require AutoLoader;

@ISA = qw(Exporter AutoLoader);
$VERSION = '0.01';

sub new {
  my ($class, $etext, $key, $index_hash) = @_;
  my $self = {};
  $self->{"etext_path"} = $etext;
  if (defined($key)) {
    $self->{"etext_number"} = $key;
    $self->{"etext_wildfile"} = $index_hash->{$key}{'FILE'};
    $self->{"etext_title"} = $index_hash->{$key}{'TITLE'};
  }
  else {
    $self->{"etext_number"} = undef;
    $self->{"etext_wildfile"} = undef;
    $self->{"etext_title"} = undef;
  }
  bless($self, $class);
  $self->_initialize();
  return($self);
}

sub _initialize {
  my ($self) = @_;
  my ($array, $file);
  my $i = 0;                     # for title grabbing from file, below
  
  if (-f $self->{"etext_path"}){
    $array = file_get($self->{"etext_path"});
    BUG_ME(0, "New Etext object with " . $self->{"etext_path"} . " lines: " . scalar(@{$array}) );
    ($file, undef) = fileparse($self->{"etext_path"});
    $self->{"etext_filename"} = $file;
  }
  elsif ($self->{"etext_path"} =~ /^\w{1,2}tp:/) {
    $array = http_get($self->{"etext_path"});
    BUG_ME(0, "New Etext object with " . $self->{"etext_path"} . " lines: " . scalar(@{$array}) );
    $self->{"etext_path"} = substr($self->{"etext_path"}, 6);
    ($file, undef) = fileparse($self->{"etext_path"});
    $self->{"etext_filename"} = $file;
    $self->{"etext_path"} = $plist_hash->{"library_dir"} . "/$file";
    if ($plist_hash->{"always_download_etexts"}) {
      file_save($self->{"etext_path"}, $array);
    }
  }
  else {
    # TODO: this should cope better? Send it to a support group...
    return;
  }
  
  $self->{"etext_array"} = $array;
  $self->{"line_holder"} = 0;
  $self->{"search_holder"} = undef;
  $self->{"last_match"} = undef;
  
  $self->{"page_lines"} = 22 if $plist_hash->{"main_window_small"};
  $self->{"page_lines"} = 30 if $plist_hash->{"main_window_medium"};
  $self->{"page_lines"} = 42 if $plist_hash->{"main_window_large"};
  $self->{"total_lines"} = int(scalar(@{$array}));
  $self->{"total_pages"} = int($self->{"total_lines"} / $self->{"page_lines"});
  if ($self->{"total_lines"} % $self->{"page_lines"}) {
    $self->{"total_pages"}++;
  }
  
  # This in case we are opening from file rather than from index
  if (!defined($self->{"etext_title"})) {
    $self->{"etext_title"} = $array->[$i];
    while ($self->{"etext_title"} !~ /project/i && $i < $self->{"total_lines"}) {
      $i++;
      $self->{"etext_title"} = $array->[$i];
    }
    chomp($self->{"etext_title"});  
    $self->{"etext_title"} =~ s/\r?$//;
    $self->{"etext_title"} =~ s/\s+$//;
  }
  
  # hack alert!  this is so the page numbers get set correctly
  # when directional is first invoked
  $self->{"page_left"} = -1;
  $self->{"page_right"} = 0;
  
  return;
}

sub destroy {
  my ($self) = @_;
  undef($self->{"etext_path"});
  undef($self->{"etext_array"});
  undef($self->{"etext_title"});
  undef($self->{"etext_number"});
  undef($self->{"etext_filename"});
  undef($self->{"etext_wildfile"});
  undef($self->{"line_holder"});
  undef($self->{"total_lines"});
  undef($self->{"total_pages"});
  undef($self->{"search_holder"});
  undef($self->{"last_match"});
  undef($self);
  return;
}

sub etext_array {
  my ($self) = @_;
  return($self->{"etext_array"});
}

sub etext_number {
  my ($self) = @_;
  return($self->{"etext_number"});
}

sub etext_filename {
  my ($self) = @_;
  return($self->{"etext_filename"});
}

sub etext_path {
  my ($self) = @_;
  return($self->{"etext_path"});
}

sub etext_wildfile {
  my ($self) = @_;
  return($self->{"etext_wildfile"});
}

sub directional_page {
  # the last 6 (yeesh!) parameters are GTK widgets, don't forget it
  my ($self, $direction, $label_title, $text_left, $text_right, $label_left, $label_right) = @_;
  my $array = $self->etext_array();
  
  $text_left->freeze();
  $text_right->freeze();

  if (defined($label_title)) {
    BUG_ME(0, "Setting title to " . $self->{"etext_title"});
    $label_title->set_text($self->{"etext_title"} . " (" . $self->{"total_pages"} . " pages)");
  }
  
  if ($direction) {
    # reached the end of the text
    if ($self->{"line_holder"} >= $self->{"total_lines"}) {
      $self->{"line_holder"} = ($self->{"line_holder"} - ($self->{"page_lines"} * 2));    
    }
    # normal forward paging
    else {
      $self->{"line_holder"} = $self->{"line_holder"} || 0;
      $self->{"page_left"} = $self->{"page_left"} + 2;
      $self->{"page_right"} = $self->{"page_right"} + 2;
    }
    # BUG_ME(0, "Paging forward from " .  $self->{"line_holder"} . "...");
  }
  else {
    # normal backward paging
    if ($self->{"line_holder"} >= ($self->{"page_lines"} * 4)) {
      $self->{"line_holder"} = ($self->{"line_holder"} - ($self->{"page_lines"} * 4));    
      $self->{"page_left"} = $self->{"page_left"} - 2;
      $self->{"page_right"} = $self->{"page_right"} - 2;
    }
    # reached the beginning of the text
    else {
      $self->{"line_holder"} = 0;
    }
    # BUG_ME(0, "Paging back to " . $self->{"line_holder"} . "...");
  }
  
  $text_left->backward_delete($text_left->get_length());
  $text_right->backward_delete($text_right->get_length());
  
  for (my $i = $self->{"line_holder"}; $i <= $self->{"line_holder"} + ($self->{"page_lines"} - 1); $i++) {
    my $j = $i + $self->{"page_lines"};
    if (defined($array->[$i])) {
      $text_left->insert(undef, undef, undef, $array->[$i]);      
    }
    if (defined($array->[$j])) {
      $text_right->insert(undef, undef, undef, $array->[$j]);
    }
  }
  
  $label_left->set_text($self->{"page_left"});
  $label_right->set_text($self->{"page_right"});
  $self->{"line_holder"} = ($self->{"line_holder"} + ($self->{"page_lines"} * 2));
    
  BUG_ME(0, "Line holder holding at: " . $self->{"line_holder"});
  $text_left->thaw();
  $text_right->thaw();
}

sub display_page_for_position {
  # the last 4 (yeesh!) parameters are GTK widgets, don't forget it
  my ($self, $position, $searching, $text_left, $text_right, $label_left, $label_right) = @_;
  my $array = $self->etext_array();
  my $segment = ($self->{"page_lines"} * 2);
  my ($sel_start, $sel_end);
  my $side;

  $text_left->freeze();
  $text_right->freeze();
  
  # we were given a page number rather than a line number for searching
  # do a conversion to get the line position at the top of the page
  if (!$searching) {
    $position = ($position * $self->{"page_lines"}) - $self->{"page_lines"};
  }
  
  $self->{"line_holder"} = 0;
  $self->{"page_left"} = 1;
  $self->{"page_right"} = 2;
  
  while ($position >= $segment) {
    $segment += ($self->{"page_lines"} * 2);
    $self->{"line_holder"} = $segment - ($self->{"page_lines"} * 2);
    $self->{"page_left"} = $self->{"page_left"} + 2;
    $self->{"page_right"} = $self->{"page_right"} + 2;
  }
  
  $text_left->backward_delete($text_left->get_length());
  $text_right->backward_delete($text_right->get_length());
  
  my $left_pos = 0;
  my $right_pos = 0;
  for (my $i = $self->{"line_holder"}; $i <= $self->{"line_holder"} + ($self->{"page_lines"} - 1); $i++) {
    my $j = $i + $self->{"page_lines"};
    if (defined($array->[$i])) {
      my $len = length($array->[$i]);
      $text_left->insert(undef, undef, undef, $array->[$i]);      
      if ($position == $i && $searching) {
	$side = "left";
	($sel_start, $sel_end) = ($left_pos, ($left_pos + $len));
      }
      $left_pos += $len;
    }
    if (defined($array->[$j])) {
      my $len = length($array->[$j]);
      $text_right->insert(undef, undef, undef, $array->[$j]);
      if ($position == $j && $searching) {
	$side = "right";
	($sel_start, $sel_end) = ($right_pos, ($right_pos + $len));
      }
      $right_pos += $len;
    }
  }
  
  $label_left->set_text($self->{"page_left"});
  $label_right->set_text($self->{"page_right"});
  $self->{"line_holder"} = ($self->{"line_holder"} + ($self->{"page_lines"} * 2));
  
  # BUG_ME(0, "Line holder holding at: " . $self->{"line_holder"});
  $text_left->thaw();
  $text_right->thaw();

  if ($searching) {
    if ($side eq "left") {
      $text_left->select_region($sel_start, $sel_end);
    }
    else {
      $text_right->select_region($sel_start, $sel_end);
    }
  }
}

sub search_for_string {
  # the last 5 (yeesh!) parameters are GTK widgets, don't forget it
  my ($self, $string, $label_message, $text_left, $text_right, $label_left, $label_right) = @_;
  my $array = $self->etext_array();
  my $wrapped = 0;
  
  $label_message->set_text("");
  
  if (!defined($self->{"search_holder"})) {
    $self->{"search_holder"} = $self->{"line_holder"} - ($self->{"page_lines"} * 2);
  }
  
  if (!defined($self->{"last_match"})) {
    $self->{"last_match"} = 0;
  }
  
  if ($self->{"search_holder"} == $self->{"last_match"} && $self->{"search_holder"}) {
    $wrapped = 1;
    $self->{"search_holder"} = 0;
  }
  
  BUG_ME(0, "Searching for $string from " . $self->{"search_holder"} . 
	 " out of " . $self->{"total_lines"} . " last match: " . $self->{"last_match"});
  
  for (my $i = $self->{"search_holder"}; $i < $self->{"total_lines"}; $i++) {
    if ($array->[$i] =~ /($string)/i) {
      $label_message->set_text("Found in line $i");	    
      $self->display_page_for_position($i, 1, $text_left, $text_right, $label_left, $label_right);
      $self->{"search_holder"} = $i + 1;
      if ($wrapped) {
	$label_message->set_text($label_message->get() . " (wrapped)");
      }
      last;
    }
    else {
      $label_message->set_text("Not found!");	    
      $self->{"last_match"} = $self->{"search_holder"};
    }
  }
}

sub stop_search {
  my ($self) = @_;
  $self->{"search_holder"} = undef;
  $self->{"last_match"} = undef;
}

sub save_as_file {
  my ($self, $file) = @_;
  $self->{"etext_path"} = $file;
  ($self->{"etext_filename"}, undef) = fileparse($self->{"etext_path"});
  file_save($self->{"etext_path"}, $self->{"etext_array"});
}

1;
