/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */

#include "GabberWidgets.hh"

#include "ContactInterface.hh"
#include "GabberApp.hh"
#include "GabberUtility.hh"

#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>

using namespace jabberoo;
using namespace GabberUtil;

// PrettyJID - nice display of JabberID - draggable and all
PrettyJID::PrettyJID(const string& jid, const string& nickname, DisplayType dt, int display_limit)
     : _jid(jid), _nickname(nickname), _resource(JID::getResource(_jid)), _default_jid(_jid)
{
     // Figure out if we need to lookup the nickname
     if (_nickname.empty())
     {
	  // Lookup nickname - default is the username
	  _nickname = JID::getUser(_jid);
	  try {
	       _nickname = G_App->getSession().roster()[JID::getUserHost(_jid)].getNickname();
	  } catch (Roster::XCP_InvalidJID& e) {
	       // Special handling for a groupchat ID -- use the resource as the nickname
	       if (G_App->isGroupChatID(_jid))
		    _nickname = _resource;
	  }
     }

     // If this jid has no resource, we need to get the default resource
     if (_resource.empty())
     {
	  try {
	       _default_jid = G_App->getSession().presenceDB().find(_jid)->getFrom();
	  } catch (PresenceDB::XCP_InvalidJID& e) {}
     }

     // Create the HBox
     _hboxPJ = manage(new Gtk::HBox(false, 4));
     add(*_hboxPJ);

     // Locate the pixmap
     // Build a table of possible image locations
     const char* pixpath_tbl[4] = { ConfigManager::get_PIXPATH(), "./", "./pixmaps/", "../pixmaps/" };
     // Look for and load the pixmap for the specified status...if it exists
     string filename = "offline.xpm";
     for (int i = 0; i < 4; i++)
     {
	  _pix_path = string(pixpath_tbl[i]);
	  filename = _pix_path + "offline.xpm";
	  // If we find a pixmap by this name, load it and return
	  if (g_file_exists(filename.c_str()))
	  {
	       break;
	  }
     }

     // Create the event box for XMMS music
     _evtMusic = manage(new Gtk::EventBox());
     _hboxPJ   ->pack_end(*_evtMusic, false, true, 0);
     _evtMusic ->hide();

     // Create the XMMS music info pixmap
     _pixMusic = manage(new Gnome::Pixmap(_pix_path + "xmms.xpm"));
     _evtMusic ->add(*_pixMusic);
     _pixMusic ->show();

     // Create the label
     _lblPJ  = manage(new Gtk::Label("", 0.0, 0.5));
     _hboxPJ ->pack_end(*_lblPJ, true, true, 0);
     _lblPJ  ->show();

     // Create the pixmap
     _pixPJ  = manage(new Gnome::Pixmap(filename));
     _hboxPJ ->pack_end(*_pixPJ, false, true, 0);
     _pixPJ  ->show();

     _hboxPJ ->show();

     // Set the default tooltip
     string tooltip = fromUTF8(this, _jid);
     _tips.set_tip(*this, tooltip);

     // Setup DnD targets that we can receive
     GtkTargetEntry dnd_dest_targets[] = {
//	  {"text/x-jabber-roster-item", 0, 0},
	  {"text/x-jabber-id", 0, 0},
     };
     int dest_num = sizeof(dnd_dest_targets) / sizeof(GtkTargetEntry);
     gtk_drag_dest_unset(GTK_WIDGET(this->gtkobj()));
     gtk_drag_dest_set(GTK_WIDGET(this->gtkobj()), (GtkDestDefaults) (GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP), dnd_dest_targets, dest_num, (GdkDragAction) (GDK_ACTION_COPY | GDK_ACTION_MOVE));

     // Setup DnD targets that we can send 
     GtkTargetEntry dnd_source_targets[] = {
       {"text/x-jabber-roster-item", 0, 0},
//       {"text/x-vcard", 0, 0},
//       {"text/x-vcard-xml", 0, 0},
       {"text/x-jabber-id", 0, 0},
     };
     int source_num = sizeof(dnd_source_targets) / sizeof(GtkTargetEntry);
     gtk_drag_source_set(GTK_WIDGET(this->gtkobj()), (GdkModifierType) (GDK_BUTTON1_MASK | GDK_BUTTON3_MASK), dnd_source_targets, source_num, (GdkDragAction) (GDK_ACTION_COPY | GDK_ACTION_MOVE));

     drag_data_received.connect(slot(this, &PrettyJID::on_evtShow_drag_data_received));
     drag_data_get.connect(slot(this, &PrettyJID::on_evtShow_drag_data_get));    
     drag_begin.connect(slot(this, &PrettyJID::on_evtShow_drag_begin));

     // Hook up the presence event
     G_App->getSession().evtPresence.connect(slot(this, &PrettyJID::on_presence));

     // Set the default display
     set_display_type(dt, display_limit);

     // And grab any existing presence...
     try {
     	  const Presence& p = G_App->getSession().presenceDB().findExact(_default_jid);
	  on_presence(p, p.getType());
     } catch (PresenceDB::XCP_InvalidJID& e) {
     }
}

void PrettyJID::set_display_type(DisplayType dt, int display_limit)
{
     // save the display type
     _display_type = dt;

     string display_text;
     if (!is_displaying_jid() && G_App->isGroupChatID(_jid))
     {
	  display_text = substitute(_("%s from groupchat %s"), fromUTF8(_nickname), JID::getUser(_jid));
     }
     else
     {
	  switch (dt)
	  {
	  case dtNickRes:
	       if (!_resource.empty())
	       {
		    display_text = fromUTF8(_nickname) + "/" + fromUTF8(_resource);
		    break;
	       }
	       // else it'll continue on and display just the nickname
	  case dtNick:
	       display_text = fromUTF8(_nickname);
	       break;
	  case dtJID:
	       if (!_resource.empty())
	       {
		    display_text = fromUTF8(JID::getUserHost(_jid));
		    break;
	       }
	       // else it'll just display the contents of _jid
	  case dtJIDRes:
	       display_text = fromUTF8(_jid);
	       break;
	  case dtNickJIDRes:
	       if (!_resource.empty())
	       {
		    display_text = fromUTF8(_nickname) + "/" + fromUTF8(_resource);
		    display_text += " (" + fromUTF8(JID::getUserHost(_jid)) + ")";
		    break;
	       }
	       // else it'll continue on and display just the nickname
	  case dtNickJID:
	       display_text = fromUTF8(_nickname);
	       display_text += " (" + fromUTF8(JID::getUserHost(_jid)) + ")";
	       break;
	  }
     }

     // Ensure it's not too long
     if (display_limit != 0 && display_text.length() > display_limit)
	  display_text = display_text.substr(0, display_limit);

     // Finally, set the text
     _lblPJ->set_text(display_text);
}

void PrettyJID::show_label(bool show_label)
{
     if (show_label)
	  _lblPJ->show();
     else
	  _lblPJ->hide();
}

void PrettyJID::show_pixmap(bool show_pixmap)
{
     if (show_pixmap)
	  _pixPJ->show();
     else
	  _pixPJ->hide();
}

const string PrettyJID::get_nickname() const
{
     return _nickname;
}

bool PrettyJID::is_displaying_jid()
{
     switch (_display_type)
     {
     case dtNickRes:
     case dtNick:
	  return false;
     case dtJID:
     case dtJIDRes:
     case dtNickJIDRes:
     case dtNickJID:
	  return true;
     }
     return true;
}

void PrettyJID::on_presence(const Presence& p, const Presence::Type prev)
{
     // Display a notification message if this presence packet
     // is from the JID associated with this widget
     if (p.getFrom() != _default_jid)
	  return;

     // Set the labels
     string filename = _pix_path + p.getShow_str() + ".xpm";
     _pixPJ->load(filename);

     // Set the tooltip
     string tooltip;
     // if they can't see the JID, show it in the tooltip
     if (!is_displaying_jid())
	  tooltip += fromUTF8(this, _jid) + "\n";
     tooltip += getShowName(p.getShow()) + ": " + fromUTF8(this, p.getStatus());
     _tips.set_tip(*this, tooltip);

     Tag* x = p.getX("gabber:x:music:info");
     if (!x)
     {
	  x = p.getX("jabber:x:music:info");
     }
     if (x)
     {
	  string song_title = x->getTaggedCDATA("title");
	  string state = x->getTaggedCDATA("state");
	  if (!state.empty())
	  {
	       if (state == "stopped")
		    song_title += _(" [stopped]");
	       else if (state == "paused")
		    song_title += _(" [paused]");
	  }
	  _tips.set_tip(*_evtMusic, fromUTF8(this, song_title));
	  _evtMusic->show();
     }
     else
     {
	  _evtMusic->hide();
     }
}

void PrettyJID::on_evtShow_drag_data_received(GdkDragContext* drag_ctx, gint x, gint y, GtkSelectionData* data, guint info, guint time)
{
     ContactSendDlg* csd = manage(new ContactSendDlg(_jid));
     csd->push_drag_data(drag_ctx, x, y, data, info, time);
}

void PrettyJID::on_evtShow_drag_data_get(GdkDragContext* drag_ctx, GtkSelectionData* data, guint info, guint time)
{
     if (data->target == gdk_atom_intern("text/x-jabber-id", FALSE))
     {
	  string dnddata;
	  dnddata = "jabber:" + JID::getUserHost(_jid) + "\n";
	  gtk_selection_data_set(data, data->target, 8, (guchar *) dnddata.c_str(), dnddata.size() + 1);
     }
     else if (data->target == gdk_atom_intern("text/x-jabber-roster-item", FALSE))
     {
	  string dnddata;
	  try {
	       Roster::Item item = G_App->getSession().roster()[JID::getUserHost(_jid)];

	       // Convert roster item to a tag
	       Tag itemt("item");
	       itemt.putAttrib("jid", item.getJID());
	       if (!item.getNickname().empty())
		    itemt.putAttrib("name", item.getNickname());
	       for (Roster::Item::iterator it = item.begin(); it != item.end(); it++)
		    itemt.addTaggedCDATA("group", *it);

	       // Set the DnD Data
	       dnddata = itemt.getXML();
	  } catch (Roster::XCP_InvalidJID& e) {}

	  gtk_selection_data_set(data, data->target, 8, (guchar *) dnddata.c_str(), dnddata.size() + 1);
     }
}

void PrettyJID::on_evtShow_drag_begin(GdkDragContext* drag_ctx)
{
     // I'm so evil. 
     // Gtk::Pixmap doesn't wanna work right... and Gnome::Pixmap doesn't have a nice accessor...
     gtk_drag_set_icon_pixmap(drag_ctx, _pixPJ->get_colormap(), _pixPJ->gtkobj()->pixmap, _pixPJ->gtkobj()->mask, -2, -2);
}
